﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include "TestAppSimple_HeadlineScene.h"

namespace {
    // HeadlineページにHeadline自体を表示させないために -1ページ;
    constexpr int HeadlineListSize = PageCountMax - 1;

    constexpr auto BasePosX = 60.0f;
    constexpr auto BasePosY = 140.0f;
    constexpr auto LabelSpanX = 320.0f;
    constexpr auto LabelSpanY = 90.0f;
    constexpr auto ColumnCnt = 4;
};

HeadlineScene::HeadlineScene() NN_NOEXCEPT
:   m_State(State_None),
    m_SelectedPage(NextAction::Headline),
    m_CurrentSelectPos(0)
{
}

void HeadlineScene::InternalSetup() NN_NOEXCEPT
{
    for (int i = 0; i < HeadlineListSize; i++)
    {
        m_PageRangeList[i].range.pos.x = BasePosX + LabelSpanX * (i % ColumnCnt);
        m_PageRangeList[i].range.pos.y = BasePosY + LabelSpanY * (i / ColumnCnt);
        m_PageRangeList[i].range.labelStr = ActionPairList[i].Label;
        m_PageRangeList[i].page = ActionPairList[i].Action;
    }
}

void HeadlineScene::InternalHandleNPad() NN_NOEXCEPT
{
    if (m_State != State_None)
    {
        // State_None 以外は受け付けないようにしておく
        return;
    }

    auto moveInsideTable = [&](int x, int y)
    {
        auto posX = m_CurrentSelectPos % ColumnCnt;
        auto posY = m_CurrentSelectPos / ColumnCnt;
        constexpr auto RowCnt = (HeadlineListSize + ColumnCnt - 1) / ColumnCnt;
        constexpr auto RemainderRows = ColumnCnt * RowCnt - HeadlineListSize;

        // 最下部の端数処理
        const auto CellCntX = ColumnCnt - RemainderRows * ((posY + 1) * ColumnCnt / HeadlineListSize);
        const auto CellCntY = RowCnt - posX / (ColumnCnt - RemainderRows);

        posX = (posX + CellCntX + x) % CellCntX;
        posY = (posY + CellCntY + y) % CellCntY;

        return posY * ColumnCnt + posX;
    };

    if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::A::Mask))
    {
        m_State = State_Jump;
        m_SelectedPage = m_PageRangeList[m_CurrentSelectPos].page;
    }
    else if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::Right::Mask)
        || HasHidControllerAnyButtonsDown(nn::hid::NpadButton::StickLRight::Mask))
    {
        m_CurrentSelectPos = moveInsideTable(1, 0);
    }
    else if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::Left::Mask)
        || HasHidControllerAnyButtonsDown(nn::hid::NpadButton::StickLLeft::Mask))
    {
        m_CurrentSelectPos = moveInsideTable(-1, 0);
    }
    else if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::Down::Mask)
        || HasHidControllerAnyButtonsDown(nn::hid::NpadButton::StickLDown::Mask))
    {
        m_CurrentSelectPos = moveInsideTable(0, 1);
    }
    else if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::Up::Mask)
        || HasHidControllerAnyButtonsDown(nn::hid::NpadButton::StickLUp::Mask))
    {
        m_CurrentSelectPos = moveInsideTable(0, -1);
    }
}

void HeadlineScene::InternalHandleTouchScreen() NN_NOEXCEPT
{
    if (m_State != State_None)
    {
        // State_None 以外は受け付けないようにしておく
        return;
    }

    int count = 0;
    for (auto& pageRange : m_PageRangeList)
    {
        if (pageRange.range.range.IsInRange(m_PreviousTouch) == true)
        {
            m_State = State_Jump;
            m_SelectedPage = pageRange.page;
            m_CurrentSelectPos = count;
            break;
        }
        ++count;
    }
}

void HeadlineScene::InternalProcess() NN_NOEXCEPT
{
    if (m_State == State_Jump)
    {
        SetJumpPage(m_SelectedPage);

        m_State = State_None;
    }
}

void HeadlineScene::InternalDrawDebugText(nn::gfx::util::DebugFontTextWriter* writer) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(writer);

    writer->SetTextColor(White);
    writer->SetCursor(40.0f, 75.0f);
    writer->Print("[Headline]");

    writer->SetScale(1.8f, 1.8f);
    {
        writer->SetTextColor(White);
        auto x = BasePosX + LabelSpanX * (m_CurrentSelectPos % ColumnCnt);
        auto y = BasePosY + LabelSpanY * (m_CurrentSelectPos / ColumnCnt);
        writer->SetCursor(x - 30.0f, y);
        writer->Print("*");
    }

    for (auto& pageRange : m_PageRangeList)
    {
        this->WriteTouchRange(writer, &pageRange.range);
    }

    if (m_State == State_None)
    {
        writer->SetTextColor(White);
        writer->SetCursor(100.0f, 600.0f);
        writer->Print("A: Jump Page");
    }
}
