﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/account.h>
#include <nn/err.h>

#include "TestAppSimple_EcServiceTypes.h"

class EcServerAccessor
{
private:
    EcServerAccessor() NN_NOEXCEPT;

public:
    enum Command
    {
        Command_None,
        Command_SelectUser,
        Command_ConnectNetwork,
        Command_InquireWithSelectUser,
        Command_InquireOcsiRights,
        Command_ConsumeOcsiRights,
        Command_PurchaseOcsiRights,
        Command_InquireCatalogList,
        Command_InquireCatalogItem,
        Command_ShowShopCatalogItem,

        Command_CountMax
    };

    static EcServerAccessor& GetInstance() NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC(EcServerAccessor, s_Instance);
        return s_Instance;
    }

    bool IsUserValid() const NN_NOEXCEPT
    {
        return m_Uid != nn::account::InvalidUid;
    }

    bool IsProcessing() const NN_NOEXCEPT
    {
        return m_Command != Command_None;
    }

    bool IsFailed() const NN_NOEXCEPT
    {
        return m_LastResult.IsFailure();
    }

    nn::Result GetNetworkResult() const NN_NOEXCEPT
    {
        return m_NetworkResult;
    }

    const nn::account::Uid& GetUid() const NN_NOEXCEPT
    {
        return m_Uid;
    }

    const nn::account::Nickname& GetNickname() const NN_NOEXCEPT
    {
        return m_Nickname;
    }

    const nn::account::NetworkServiceAccountId& GetNsaId() const NN_NOEXCEPT
    {
        return m_NsaId;
    }

    const EcCatalog& GetCatalog() const NN_NOEXCEPT
    {
        return m_Catalog;
    }

    const EcOcsiRightList& GetOcsiRights() const NN_NOEXCEPT
    {
        return m_RightList;
    }

    bool IsOcsiRightSelected() const NN_NOEXCEPT
    {
        return m_RightList.GetSelectedCount() > 0;
    }

    void ClearNetworkResult()
    {
        m_NetworkResult = nn::ResultSuccess();
    }

    bool SwitchOcsiRight(int index) NN_NOEXCEPT
    {
        return m_RightList.Switch(index);
    }

    bool SelectConsumable(int index) NN_NOEXCEPT
    {
        return m_Catalog.SelectConsumable(index);
    }

    bool SelectConsumableItem(int index) NN_NOEXCEPT
    {
        return m_Catalog.SelectConsumableItem(index);
    }

    void Initialize() NN_NOEXCEPT;
    void Finalize() NN_NOEXCEPT;
    bool Update() NN_NOEXCEPT;
    bool Execute(Command command) NN_NOEXCEPT;

    const char* GetCurrentStatus() const NN_NOEXCEPT;
    const char* GetErrorMessage() const NN_NOEXCEPT
    {
        return m_ErrorMessage;
    }
    bool IsErrorViewable() const NN_NOEXCEPT
    {
        return m_IsErrorViewable;
    }
    void ShowErrorViewer() const NN_NOEXCEPT;

private:
    nn::account::Uid m_Uid;
    nn::account::Nickname m_Nickname;
    nn::account::NetworkServiceAccountId m_NsaId;
    EcCatalog m_Catalog;
    EcOcsiRightList m_RightList;

    Command m_Command;
    nn::Result m_LastResult;
    nn::Result m_NetworkResult;
    nn::err::ErrorCode m_ErrorCode;
    bool m_IsErrorViewable;
    char m_ErrorMessage[128];

    void UpdateLastError(const char* message, nn::Result result, bool viewable) NN_NOEXCEPT;
    void UpdateLastError(const char* message, nn::Result result, nn::err::ErrorCode code, bool viewable) NN_NOEXCEPT;

    bool SelectUser() NN_NOEXCEPT;
    bool UpdateUser() NN_NOEXCEPT;

    bool ConnectNetwork() NN_NOEXCEPT;
    bool InquireWithSelectUser() NN_NOEXCEPT;
    bool InquireOcsiRights() NN_NOEXCEPT;
    bool ConsumeOcsiRights() NN_NOEXCEPT;
    bool PurchaseOcsiRights() NN_NOEXCEPT;
    bool InquireCatalogList() NN_NOEXCEPT;
    bool InquireCatalogItem() NN_NOEXCEPT;
    bool ShowShopCatalogItem() NN_NOEXCEPT;
    bool UpdateContents() NN_NOEXCEPT;
};
