﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Assert.h>
#include <nn/util/util_FormatString.h>

#include "TestAppSimple_EcServiceConfig.h"
#include "TestAppSimple_EcServerAccessor.h"
#include "TestAppSimple_EcServiceAccount.h"
#include "TestAppSimple_EcServiceCatalog.h"
#include "TestAppSimple_EcServiceOcsi.h"
#include "TestAppSimple_MetaInfo.h"
#include "TestAppSimple_WorkContractor.h"

namespace {

    // 作業請負人
    const size_t ThreadStackSize = 64 * 1024;
    NN_OS_ALIGNAS_THREAD_STACK char g_ThreadStack[ThreadStackSize];
    WorkContractor g_Contractor(g_ThreadStack, ThreadStackSize);

    // EC 関連作業
    class UserSelector g_UserSelector;
    class NetworkConnector g_NetworkConnector;
    class EcOcsiRightPurchaser g_EcOcsiRightPurchaser;
    class EcOcsiRightConsumer g_EcOcsiRightConsumer;
    class EcCataloger g_EcCataloger;
    class EcConsumableItemShower g_EcConsumableItemShower;
}

EcServerAccessor::EcServerAccessor() NN_NOEXCEPT
{
}

void EcServerAccessor::Initialize() NN_NOEXCEPT
{
    m_Uid = nn::account::InvalidUid;
    m_Nickname.name[0] = '\n';
    m_NsaId.id = 0;
    m_Catalog.Cleanup();
    m_RightList.Cleanup();

    m_Command = Command_None;
    m_LastResult = nn::ResultSuccess();
    m_NetworkResult = nn::ResultSuccess();
    m_ErrorCode = nn::err::ErrorCode::GetInvalidErrorCode();
    m_IsErrorViewable = false;

    g_Contractor.Initialize();
}

void EcServerAccessor::Finalize() NN_NOEXCEPT
{
    g_Contractor.Finalize();
}

bool EcServerAccessor::Update() NN_NOEXCEPT
{
    auto updated = false;
    if (g_Contractor.IsFinished())
    {
        NN_ASSERT_NOT_EQUAL(m_Command, Command_None);
        m_Command = Command_None;

        updated |= UpdateUser();
        updated |= UpdateContents();
    }
    return updated;
}

bool EcServerAccessor::Execute(Command command) NN_NOEXCEPT
{
    auto started = false;
    if (!g_Contractor.IsBusy())
    {
        switch (command)
        {
        case Command_SelectUser:
            started = SelectUser();
            break;

        case Command_ConnectNetwork:
            started = ConnectNetwork();
            break;

        case Command_InquireWithSelectUser:
            started = InquireWithSelectUser();
            break;

        case Command_InquireOcsiRights:
            started = InquireOcsiRights();
            break;

        case Command_ConsumeOcsiRights:
            started = ConsumeOcsiRights();
            break;

        case Command_PurchaseOcsiRights:
            started = PurchaseOcsiRights();
            break;

        case Command_InquireCatalogList:
            started = InquireCatalogList();
            break;

        case Command_InquireCatalogItem:
            started = InquireCatalogItem();
            break;

        case Command_ShowShopCatalogItem:
            started = ShowShopCatalogItem();
            break;

        default: NN_UNEXPECTED_DEFAULT;
        }
        if (started)
        {
            m_Command = command;
            m_LastResult = nn::ResultSuccess();
            m_ErrorCode = nn::err::ErrorCode::GetInvalidErrorCode();
            m_IsErrorViewable = false;
            m_ErrorMessage[0] = '\n';
        }
    }
    return started;
}

const char* EcServerAccessor::GetCurrentStatus() const NN_NOEXCEPT
{
    const char* Status[] =
    {
        "",                     // Command_None
        "Select User",          // Command_SelectUser
        "Connect Network",      // Command_ConnectNetwork
        "Inquire",              // Command_InquireWithSelectUser
        "Inquire OcsiRights",   // Command_InquireOcsiRights
        "Consume OcsiRights",   // Command_ConsumeOcsiRights
        "Purchase OcsiRights",  // Command_PurchaseOcsiRights
        "Inquire CatalogList",  // Command_InquireCatalogList
        "Inquire CatalogItem",  // Command_InquireCatalogItem
        "Show ShopCatalogItem", // Command_ShowShopCatalogItem
    };
    NN_STATIC_ASSERT(NN_ARRAY_SIZE(Status) == static_cast<size_t>(Command_CountMax));

    const int index = static_cast<int>(m_Command);
    NN_ASSERT_GREATER_EQUAL(index, 0);
    NN_ASSERT_LESS(index, static_cast<int>(Command_CountMax));
    return Status[index];
}

void EcServerAccessor::ShowErrorViewer() const NN_NOEXCEPT
{
    if (m_ErrorCode.IsValid())
    {
        nn::err::ShowError(m_ErrorCode);
    }
    else if (m_LastResult.IsFailure() && m_IsErrorViewable)
    {
        nn::err::ShowError(m_LastResult);
    }
}

void EcServerAccessor::UpdateLastError(const char* message, nn::Result result, bool viewable) NN_NOEXCEPT
{
    if (m_LastResult.IsSuccess() && result.IsFailure())
    {
        const char* Format = "%s Failed: 0x%08x";
        auto& buf = m_ErrorMessage;
        auto value = result.GetInnerValueForDebug();
        NN_ABORT_UNLESS(nn::util::SNPrintf(buf, sizeof(buf), Format, message, value) < sizeof(buf));

        m_LastResult = result;
        m_ErrorCode = nn::err::ErrorCode::GetInvalidErrorCode();
        m_IsErrorViewable = viewable;
    }
}

void EcServerAccessor::UpdateLastError(const char* message, nn::Result result, nn::err::ErrorCode code, bool viewable) NN_NOEXCEPT
{
    if (m_LastResult.IsSuccess() && result.IsFailure())
    {
        auto& buf = m_ErrorMessage;
        auto value = result.GetInnerValueForDebug();
        if (code.IsValid())
        {
            const char* Format = "%s Failed: 0x%08x (%u-%u)";
            NN_ABORT_UNLESS(nn::util::SNPrintf(buf, sizeof(buf), Format, message, value, code.category, code.number) < sizeof(buf));
        }
        else
        {
            const char* Format = "%s Failed: 0x%08x";
            NN_ABORT_UNLESS(nn::util::SNPrintf(buf, sizeof(buf), Format, message, value) < sizeof(buf));
        }

        m_LastResult = result;
        m_ErrorCode = code;
        m_IsErrorViewable = viewable;
    }
}

bool EcServerAccessor::SelectUser() NN_NOEXCEPT
{
    g_UserSelector.Prepare(true);
    g_Contractor.Start(&g_UserSelector);
    return true;
}

bool EcServerAccessor::UpdateUser() NN_NOEXCEPT
{
    auto updated = false;
    if (g_UserSelector.IsExecuted())
    {
        auto result = g_UserSelector.GetLastResult();
        UpdateLastError("Select User", result, true);
        if (result.IsSuccess())
        {
            m_Uid = g_UserSelector.GetUid();
            m_Nickname = g_UserSelector.GetNickname();
            m_NsaId = g_UserSelector.GetNsaId();
            m_Catalog.Cleanup();
            m_RightList.Cleanup();
            updated = true;
        }
        g_UserSelector.Cleanup();
    }
    return updated;
}

bool EcServerAccessor::ConnectNetwork() NN_NOEXCEPT
{
    g_NetworkConnector.Prepare();

    Work* works[] = { &g_NetworkConnector };
    g_Contractor.Start(works, NN_ARRAY_SIZE(works));
    return true;
}

bool EcServerAccessor::InquireWithSelectUser() NN_NOEXCEPT
{
    g_UserSelector.Prepare(true);
    g_NetworkConnector.Prepare();
    g_EcOcsiRightConsumer.PrepareToInquire(&g_UserSelector);
    g_EcCataloger.PrepareToInquireList(&g_UserSelector);

    Work* works[] = { &g_UserSelector, &g_NetworkConnector, &g_EcOcsiRightConsumer, &g_EcCataloger };
    g_Contractor.Start(works, NN_ARRAY_SIZE(works));
    return true;
}

bool EcServerAccessor::InquireOcsiRights() NN_NOEXCEPT
{
    if (IsUserValid())
    {
        g_EcOcsiRightConsumer.PrepareToInquire(m_Uid);

        Work* works[] = { &g_EcOcsiRightConsumer };
        g_Contractor.Start(works, NN_ARRAY_SIZE(works));
        return true;
    }
    return false;
}

bool EcServerAccessor::ConsumeOcsiRights() NN_NOEXCEPT
{
    if (IsUserValid() && IsOcsiRightSelected())
    {
        g_EcOcsiRightConsumer.PrepareToConsume(m_Uid, m_RightList);

        Work* works[] = { &g_EcOcsiRightConsumer };
        g_Contractor.Start(works, NN_ARRAY_SIZE(works));
        return true;
    }
    return false;
}

bool EcServerAccessor::PurchaseOcsiRights() NN_NOEXCEPT
{
#if defined( TESTAPP_ENABLE_EC_ACCESS ) && defined( TESTAPP_ENABLE_EC_OCSI_PURCHASE )
    if (IsUserValid())
    {
        g_NetworkConnector.Prepare();
        g_EcOcsiRightPurchaser.Prepare(m_NsaId);

        Work* works[] = { &g_NetworkConnector, &g_EcOcsiRightPurchaser };
        g_Contractor.Start(works, NN_ARRAY_SIZE(works));
        return true;
    }
#endif // defined( TESTAPP_ENABLE_EC_ACCESS ) && defined( TESTAPP_ENABLE_EC_OCSI_PURCHASE )
    return false;
}

bool EcServerAccessor::InquireCatalogList() NN_NOEXCEPT
{
    if (IsUserValid())
    {
        g_EcCataloger.PrepareToInquireList(m_Uid);

        Work* works[] = { &g_EcCataloger };
        g_Contractor.Start(works, NN_ARRAY_SIZE(works));
        return true;
    }
    return false;
}

bool EcServerAccessor::InquireCatalogItem() NN_NOEXCEPT
{
    if (IsUserValid() && m_Catalog.IsConsumableSelected())
    {
        g_EcCataloger.PrepareToInquireItem(m_Uid, m_Catalog);

        Work* works[] = { &g_EcCataloger };
        g_Contractor.Start(works, NN_ARRAY_SIZE(works));
        return true;
    }
    return false;
}

bool EcServerAccessor::ShowShopCatalogItem() NN_NOEXCEPT
{
    if (IsUserValid() && m_Catalog.IsConsumableItemSelected())
    {
        auto applicationId = GetApplicationId();
        g_EcConsumableItemShower.PrepareToConsumableItemDetail(applicationId, m_Uid, m_Catalog);

        Work* works[] = { &g_EcConsumableItemShower };
        g_Contractor.Start(works, NN_ARRAY_SIZE(works));
        return true;
    }
    return false;
}

bool EcServerAccessor::UpdateContents() NN_NOEXCEPT
{
    auto updated = false;
    if (g_NetworkConnector.IsExecuted())
    {
        m_NetworkResult = g_NetworkConnector.GetLastResult();
        UpdateLastError("Connect Network", m_NetworkResult, false);
        g_NetworkConnector.Cleanup();
        updated = true;
    }
    if (g_EcOcsiRightPurchaser.IsExecuted())
    {
        UpdateLastError("Purchase EcOcsi Rights", g_EcOcsiRightPurchaser.GetLastResult(), true);
        g_EcOcsiRightPurchaser.Cleanup();
        updated = true;
    }
    if (g_EcOcsiRightConsumer.IsExecuted())
    {
        UpdateLastError("Access EcOcsi", g_EcOcsiRightConsumer.GetLastResult(), g_EcOcsiRightConsumer.GetErrorCode(), true);
        m_RightList.CheckAll(false);
        g_EcOcsiRightConsumer.UpdateRights(&m_RightList);
        g_EcOcsiRightConsumer.Cleanup();
        updated = true;
    }
    if (g_EcCataloger.IsExecuted())
    {
        UpdateLastError("Access EcCatalog", g_EcCataloger.GetLastResult(), g_EcCataloger.GetErrorCode(), true);
        g_EcCataloger.UpdateCatalog(&m_Catalog);
        g_EcCataloger.Cleanup();
        updated = true;
    }
    if (g_EcConsumableItemShower.IsExecuted())
    {
        UpdateLastError("Show ShopCatalogItem",  g_EcConsumableItemShower.GetLastResult(), true);
        g_EcConsumableItemShower.Cleanup();
    }
    return updated;
}
