﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using System.Collections.Generic;
using System.Threading;
using System.Threading.Tasks;
using System.Diagnostics;

namespace MakeTestApplication
{
    internal class Service
    {
        private FsFacade m_Fs;
        private CommandLineParams m_Params;
        private string outDirPath = null;
        private bool isVerbose = false;
        private bool isMakingMultiProgram = false;
        private bool isMakingMultiProgramWithPatch = false;
        public void SetVerboseMode()
        {
            isVerbose = true;
        }

        private Executor sigloToolExec;
        private TestAppBuilder testAppBuilder;

        private const string BuildOutDirName = @"BulildOut\";
        private const string BuildIntDirName = @"BuildInt\";

        public Service(CommandLineParams inParams)
        {
            m_Params = inParams;
            m_Fs = new FsFacade();
            sigloToolExec = new Executor();
            testAppBuilder = new TestAppBuilder();

            if (inParams.VerboseFlag == true)
            {
                this.SetVerboseMode();
                m_Fs.SetVerboseMode();
                sigloToolExec.SetVerboseMode();
                testAppBuilder.SetVerboseMode();
            }
        }

        private string GetExecutingAppCodePath()
        {
            // (SIGLO-46475) --code オプションが設定されていた場合はそのパスを最優先に返す
            if (m_Params.CodeDirPath != null && m_Params.CodeDirPath != string.Empty)
            {
                return m_Params.CodeDirPath;
            }

            // 実行している自アプリ(.exe)のパスに合わせたcodeディレクトリパスを返す
            var appDirPath = Path.GetDirectoryName(System.Reflection.Assembly.GetExecutingAssembly().Location);

            // 小さいサイズのcodeディレクトリパスを返すかどうかの判定
            var smallCodeDirPath = Path.Combine(appDirPath, @"scode");
            if (m_Params.TestAppBuildFlag == false &&
                m_Params.SmallCodeFlag == true &&
                Directory.Exists(smallCodeDirPath) == true)
            {
                // 小さいサイズ版のcodeディレクトリパスを返す
                return smallCodeDirPath;
            }

            return Path.Combine(appDirPath, @"code");
        }

        // (SIGLO-79140) desc ファイルパスの確定処理を関数化
        private string GetDescFilePath(SettingParams inParam)
        {
            // descファイルのパスを設定
            // デフォルトは以下ファイルパス(Application.desc)を初期値とし、もし設定されていればその設定値を使用する
            var sdkRootPath = CommonParams.GetSigloRoot();
            var descFilePath = sdkRootPath + @"Programs\Iris\Resources\SpecFiles\Application.desc";
            if (File.Exists(descFilePath) == false)
            {
                // NintendoSDK ではこちらのパスとなるはず
                descFilePath = sdkRootPath + @"Resources\SpecFiles\Application.desc";
            }

            if (string.IsNullOrEmpty(inParam.DescFilePath) == false)
            {
                // オプションとして指定されていた場合はそちらを使用する
                descFilePath = inParam.DescFilePath;
            }

            return descFilePath;
        }

        private void CheckNumericalInputValue()
        {
            if (m_Params.MultiProgram != null)
            {
                ushort num = 0;
                var result = ushort.TryParse(m_Params.MultiProgram, out num);
                if (result == false)
                {
                    // 指定値が不正な場合は例外を投げる
                    throw new Exception("--multi-program=" + m_Params.MultiProgram + " は不正です");
                }
                if (num < 2 || 256 < num)
                {
                    // 指定値が不正な場合は例外を投げる
                    throw new Exception("--multi-program オプションは 2 ～ 256 までの値を入力する必要があります");
                }
            }

            if (m_Params.InputProgramIndex != null)
            {
                byte num = 0;
                var result = byte.TryParse(m_Params.InputProgramIndex, out num);
                if (result == false)
                {
                    // 指定値が不正な場合は例外を投げる
                    throw new Exception("--program-index=" + m_Params.InputProgramIndex + " は不正です");
                }
            }
        }

        /// <summary>
        /// 入力パラメータのチェックとファイル関連の初期化処理
        /// </summary>
        public void InitInputFileParams()
        {
            // SIGLO-27640 リビルドを行わない対応
            // ビルドオプションが無効の場合のみ確認する
            if (m_Params.TestAppBuildFlag == false)
            {
                // コードディレクトリ＆内部ファイルの存在確認
                testAppBuilder.CheckExistCodeData(this.GetExecutingAppCodePath());
            }
            // 最低限の入力ファイルチェック
            m_Fs.CheckInputFilePath(m_Params);
            // 最低限の入力数値チェック
            this.CheckNumericalInputValue();
            // 作業ディレクトリのチェック・作成
            m_Fs.InitWorkDir(m_Params);
            // 出力先ディレクトリのチェック・作成
            outDirPath = m_Fs.InitOutputDir(m_Params);
        }

        /// <summary>
        /// 作業フォルダなどの後始末処理
        /// </summary>
        public void ClearWorkSpace()
        {
            if (m_Fs.IsDeleteTempDir())
            {
                m_Fs.ClearWorkDir();
            }
        }

        /// <summary>
        /// テストアプリケーションを作成するメイン処理
        /// </summary>
        public void CreateTestApplication()
        {
            // SIGLO-27640 リビルドを行わない対応
            if (m_Params.TestAppBuildFlag == true)
            {
                // ビルドオプションが有効な場合、テストアプリケーションのビルド処理を実行
                bool ret = BuildTestApplication();
                if (ret == false)
                {
                    // テストアプリのコード構築に失敗しているので処理を返しておく
                    throw new Exception("CreateTestApplication()：テストアプリのコード生成に失敗しました");
                }
            }

            // (SIGLO-78007) マルチプログラムアプリケーションを生成する場合は特別処理を実施する必要があるためフラグ変数を用意する
            isMakingMultiProgram = (m_Params.MultiProgram != null);

            List<SettingParams> settingList = null;
            if (isMakingMultiProgram)
            {
                // (SIGLO-79140) パッチ指定かどうかはここで判定する
                isMakingMultiProgramWithPatch = (m_Params.InputContentType == CommonParams.ContentTypePatch);
                if (isMakingMultiProgramWithPatch)
                {
                    // Patch 指定で初版アプリのパスが設定されていなければエラーを返す
                    if (string.IsNullOrEmpty(m_Params.OriginalApplicationFilePath))
                    {
                        throw new Exception(
                            string.Format("CreateTestApplication()：Patch 指定時は --original-application を指定する必要があります"));
                    }
                }
                // マルチプログラムアプリケーション生成用の settingList を生成する
                settingList = CreateMultiProgramSettingList();
            }
            else
            {
                // こちらは従来の処理
                settingList = CreateSettingList();
            }

            if (isVerbose == true)
            {
                Console.WriteLine("SettingParams Count = {0}", settingList.Count);
            }

            try
            {
                ProcessSettingList(settingList);
            }
            catch (Exception err)
            {
                Console.Error.WriteLine(err.StackTrace);
                Console.Error.WriteLine(err.Message);
                throw new Exception("CreateTestApplication()：ProcessSettingList()で例外が発生しました");
            }
        }

        private List<SettingParams> CreateSettingList()
        {
            List<SettingParams> settingList = null;

            if (m_Params.ConfigXmlPath != null)
            {
                // 一括設定用のXMLが指定されていた場合
                settingList = m_Fs.ParseXmlSettingFile(m_Params.ConfigXmlPath);
                if (settingList == null || settingList.Count == 0)
                {
                    throw new Exception("CreateSettingList()：XML設定ファイルに有効な設定が存在しませんでした");
                }
                return settingList;
            }

            SettingParams setting = null;
            if (m_Params.MetaFilePath == null)
            {
                // オプションにmetaファイルが設定されていない場合
                setting = new SettingParams(m_Params);

                // コマンドラインの入力情報からmetaファイルを作成
                var metaFilePath = m_Fs.CreateMetaFile(setting);
                if (metaFilePath == null)
                {
                    throw new Exception("CreateSettingList()：metaファイルの生成に失敗しました");
                }
                setting.MetaFilePath = metaFilePath;
            }
            else
            {
                // オプションにmetaファイルが設定されている場合
                setting = m_Fs.ImportInputMetaFile(m_Params);
                if (setting == null)
                {
                    throw new Exception("CreateSettingList()：入力metaファイルが不正です");
                }
            }

            // metaファイル情報に依存しないコマンドラインオプション情報を設定する
            setting.SetCommonSettingParam(m_Params);

            // descファイルパスの設定
            if (m_Params.DescFilePath != null && m_Params.DescFilePath != string.Empty)
            {
                setting.DescFilePath = m_Params.DescFilePath;
            }

            // リストにひとつだけデータを入れる形となる
            settingList = new List<SettingParams>();
            settingList.Add(setting);

            return settingList;
        }

        private List<SettingParams> CreateMultiProgramSettingList()
        {
            var settingList = new List<SettingParams>();

            int programCount = 0;
            int.TryParse(m_Params.MultiProgram, out programCount);
            // m_Params.MultiProrgram はエラーチェック済なので例外処理は考慮しない

            for (int index = 0; index < programCount; index++)
            {
                // 強制的に ProgramIndex の値を上書きする
                m_Params.InputProgramIndex = index.ToString();
                var setting = new SettingParams(m_Params);

                // (SIGLO-79140) ContentType は Application 固定とする ( Patch が設定されている場合を考慮)
                setting.ContentType = CommonParams.ContentTypeApplication;

                // コマンドラインの入力情報からmetaファイルを作成
                var metaFilePath = m_Fs.CreateMetaFile(setting);
                if (metaFilePath == null)
                {
                    throw new Exception("CreateMultiProgramSettingList()：metaファイルの生成に失敗しました");
                }

                setting.MetaFilePath = metaFilePath;

                // metaファイル情報に依存しないコマンドラインオプション情報を設定する
                setting.SetCommonSettingParam(m_Params);

                // descファイルパスの設定
                if (string.IsNullOrEmpty(m_Params.DescFilePath) == false)
                {
                    setting.DescFilePath = m_Params.DescFilePath;
                }

                var fileName = new System.Text.StringBuilder(index.ToString());
                if (setting.IsSetOutputFileName && setting.OutputFileName == fileName.ToString())
                {
                    // 出力ファイル名に同じ名前が設定されている場合を考慮
                    fileName.Append("_");
                }

                // 出力ディレクトリの設定
                var dirPath = outDirPath;
                if (string.IsNullOrEmpty(setting.OutputFilePath) == false)
                {
                    // OutputFilePath が設定されている場合はそのディレクトリ名を使用する
                    dirPath = Path.GetDirectoryName(setting.OutputFilePath);

                    // 出力パスのファイル名に同じ名前が設定されている場合を考慮
                    var inputOutFileName = Path.GetFileNameWithoutExtension(setting.OutputFilePath);
                    if (inputOutFileName == fileName.ToString())
                    {
                        fileName.Append("_");
                    }
                }
                // 個別のプログラムとして出力するファイル名"{index}.nsp"を生成 (出力ファイルと衝突する場合は"{index}_.nsp"となる)
                fileName.Append(".nsp");

                // 出力ディレクトリに一旦作成する形とする
                // 強制的に出力ファイルパスを設定しておく
                setting.OutputFilePath = Path.Combine(dirPath, fileName.ToString());

                settingList.Add(setting);
            }

            return settingList;
        }

        private void CreateMultiProgramApplication(List<SettingParams> inSettingList)
        {
            var nspFileName = m_Params.OutputFileName;
            if (string.IsNullOrEmpty(nspFileName))
            {
                var prefix = "MultiProgram";
                var setting = inSettingList[0];
                // 指定されていない場合は以下のデフォルト名作成処理に入る
                nspFileName = string.Format("{0}_{1}", prefix, setting.MetaInputId.ToLowerInvariant());

                // (SIGLO-79140) パッチ指定の場合、デフォルト名にはそれと分かるような文字列を入れておく
                if (isMakingMultiProgramWithPatch)
                {
                    nspFileName = string.Format("{0}_Patch_{1}", prefix, setting.MetaInputId.ToLowerInvariant());
                }

                if (m_Params.SmallCodeFlag == true)
                {
                    // 念のため小さいコードサイズのフラグが指定されたらファイル名にも反映させておく
                    nspFileName += "_small";
                }
            }

            var nspOutputPath = string.Format(@"{0}\{1}.nsp", outDirPath, nspFileName);
            if (string.IsNullOrEmpty(m_Params.OutputFilePath) == false)
            {
                // m_Params.OutputFilePath が設定されている場合は強制的に上書き
                nspOutputPath = m_Params.OutputFilePath;
            }

            // 個別のプログラムのファイルパスをリストにする
            var programPathList = new List<string>();
            foreach (var setting in inSettingList)
            {
                programPathList.Add(setting.OutputFilePath);
            }

            // マルチプログラムアプリケーションを生成
            try
            {
                // AuthoringToolを叩いて nsp ファイルを作成
                var result = sigloToolExec.ExecuteAuthoringToolMergeApplication(nspOutputPath, programPathList);
                if (result.IsFailure())
                {
                    Console.Error.WriteLine("ExecuteAuthoringToolMergeApplication()でエラー発生");
                    Console.Error.WriteLine("エラーメッセージ：{0}", result.ErrorMessage);
                    throw new Exception("CreateMultiProgramApplication() : ExecuteAuthoringToolMergeApplication()の実行に失敗しました");
                }

                // (SIGLO-79140) パッチ指定の場合はさらに、makepatch を実行する
                if (isMakingMultiProgramWithPatch)
                {
                    // 作成した nsp ファイルを一旦リネーム
                    var renameNspPath = nspOutputPath + ".current.nsp";
                    File.Move(nspOutputPath, renameNspPath);
                    // 少しトリッキーだが最後の削除処理のためパスリストの中に格納
                    programPathList.Add(renameNspPath);

                    // 代表として 0 番目の setting を利用する (0番目は必ず存在するはず)
                    var setting = inSettingList[0];
                    var descFilePath = this.GetDescFilePath(setting);

                    // (念のため)スレッドセーフ化のためのDescファイルのコピー
                    var directoryPath = Path.GetDirectoryName(renameNspPath);
                    var copyDstDescFilePath = Path.Combine(directoryPath, "_tempDesc_.desc");
                    File.Copy(descFilePath, copyDstDescFilePath, true);
                    // 少しトリッキーだが最後の削除処理のためパスリストの中に格納
                    programPathList.Add(copyDstDescFilePath);

                    // AuthoringTool の makepatch を叩く
                    result = sigloToolExec.ExecuteAuthoringToolMakePatchForMultiProgram(nspOutputPath, copyDstDescFilePath, renameNspPath, setting);
                    if (result.IsFailure())
                    {
                        Console.Error.WriteLine("ExecuteAuthoringToolMakePatchForMultiProgram()でエラー発生");
                        Console.Error.WriteLine("エラーメッセージ：{0}", result.ErrorMessage);
                        throw new Exception("CreateMultiProgramApplication() : ExecuteAuthoringToolMakePatchForMultiProgram()の実行に失敗しました");
                    }
                }
            }
            catch (Exception err)
            {
                Console.Error.WriteLine(err.StackTrace);
                Console.Error.WriteLine(err.Message);
                throw err;
            }
            finally
            {
                if (m_Params.NoCleanupFlag == false)
                {
                    // 削除フラグが有効な場合、念のため個別のプログラムファイルは削除しておく
                    foreach (var path in programPathList)
                    {
                        File.Delete(path);
                    }
                }
            }
        }

        private void ProcessSettingList(List<SettingParams> inList)
        {
            // Patch は元になる nsp が必要となるため、並列処理としては Patch 以外の処理完了後に実行されることを保証させる必要がある。
            // コンテンツタイプを見て Patch と Patch 以外に分別する
            var noPatchSettingList = new List<SettingParams>();
            var patchSettingList = new List<SettingParams>();
            foreach (var param in inList)
            {
                if (param.ContentType == CommonParams.ContentTypePatch)
                {
                    // ついでのエラーチェックとなるが、Patch 指定時の特別判定を行う
                    // Patch 指定で初版アプリのパスが設定されていなければエラーを返す
                    if (string.IsNullOrEmpty(param.OriginalApplicationFilePath) && string.IsNullOrEmpty(param.OriginalApplicationForCreateNspFilePath))
                    {
                        throw new Exception(
                            string.Format("ProcessSettingList()：Id={0}, Patch 指定時は --original-application を指定する必要があります", param.MetaInputId));
                    }
                    patchSettingList.Add(param);
                }
                else
                {
                    noPatchSettingList.Add(param);
                }
            }

            var sw = new Stopwatch(); // 処理時間計測用

            // 最大並列処理の設定数を確定させる
            var pallarelNum = CommonParams.DefaultPallarelNum;
            try
            {
                pallarelNum = Convert.ToInt32(m_Params.MaxParallelNum);
                if (pallarelNum <= 0)
                {
                    // ひとまず無効な値を入力された場合は デフォルトの数を設定しておく
                    pallarelNum = CommonParams.DefaultPallarelNum;
                }
            }
            catch (Exception err)
            {
                Console.Error.WriteLine(err.Message);
                Console.WriteLine("最大並列処理数の指定で例外が発生。デフォルト値({0})を適用します。", CommonParams.DefaultPallarelNum);
                pallarelNum = CommonParams.DefaultPallarelNum;
            }

            if (isVerbose == true)
            {
                Console.WriteLine("Max Parallel Num = {0}", pallarelNum);
            }

            // 最大並列数を設定する
            var opt = new ParallelOptions();
            opt.MaxDegreeOfParallelism = pallarelNum;

            sw.Start();

            // System.Threading.Tasks の Parallel の仕組みを使って並列処理を実現させる
            // まずは Patch 以外のリストを実行
            Parallel.For(0, noPatchSettingList.Count, opt, i =>
            {
                var setting = noPatchSettingList[i];
                setting.SequenceNumber = i;
                Console.WriteLine("--[ {0} / {1} ]---------------------------------", (i + 1), noPatchSettingList.Count);
                if (isVerbose == true)
                {
                    Console.WriteLine("ContentType : {0}", setting.ContentType);
                    Console.WriteLine("Size : {0}", setting.CodeSize);
                    Console.WriteLine("metaFilePath : {0}", setting.MetaFilePath);
                }

                try
                {
                    this.ProcessSetting(setting);
                }
                catch (Exception err)
                {
                    Console.Error.WriteLine(err.StackTrace);
                    Console.Error.WriteLine(err.Message);
                    throw err;
                }
            });

            // 次に Patch のリストを実行
            Parallel.For(0, patchSettingList.Count, opt, i =>
            {
                var setting = patchSettingList[i];
                setting.SequenceNumber = i;
                Console.WriteLine("--[ {0} / {1} ]-- Patch -------------------------", (i + 1), patchSettingList.Count);
                if (isVerbose == true)
                {
                    Console.WriteLine("ContentType : {0}", setting.ContentType);
                    Console.WriteLine("Size : {0}", setting.CodeSize);
                    Console.WriteLine("metaFilePath : {0}", setting.MetaFilePath);
                }

                try
                {
                    this.ProcessSetting(setting);
                }
                catch (Exception err)
                {
                    Console.Error.WriteLine(err.StackTrace);
                    Console.Error.WriteLine(err.Message);
                    throw err;
                }
            });

            if (isMakingMultiProgram)
            {
                // (SIGLO-78007) マルチプログラムアプリケーションを生成する場合は最後のマージ処理を実行
                Console.WriteLine("-- MergeApplication Process --");
                CreateMultiProgramApplication(inList);
            }

            sw.Stop();
            // 処理の並列化に伴い、最終的に全体の処理時間のみを出力させる
            Console.WriteLine("--ElapsedTime: {0} --", sw.Elapsed);
        }

        private void ProcessSetting(SettingParams inParam)
        {
            var defaultFileSize = Convert.ToInt64(CommonParams.DefaultCodeSize);
            var binFileSize = defaultFileSize;
            try
            {
                binFileSize = Convert.ToInt64(inParam.CodeSize);
                if (binFileSize <= 0)
                {
                    // ひとまず無効な値を入力された場合は デフォルトサイズ(4バイト) を設定しておく
                    binFileSize = defaultFileSize;
                }
            }
            catch (Exception err)
            {
                Console.Error.WriteLine(err.Message);
                Console.WriteLine("コードサイズの指定でエラーが発生。デフォルト値({0}バイト)を適用します。", defaultFileSize);
                binFileSize = defaultFileSize;
            }

            // SIGLO-38113
            // コンテンツタイプが SystemData または AddOnContent の指定で、--data-directory にパスが指定されていた場合、
            // 生成するランダムデータファイルのサイズは強制的にデフォルト値(4バイト)とする。
            // (無駄な処理となるファイル生成処理を極力短くするため)
            if ((inParam.ContentType == CommonParams.ContentTypeSystemData
                || inParam.ContentType == CommonParams.ContentTypeAddOnContent)
                && inParam.DataDirPath != null)
            {
                binFileSize = defaultFileSize;
            }

            // Meta ファイルにデフォルトの情報(アプリケーションのタイトルなど)を追加する
            // (関数内部で inParam の内容を見て追加するかどうかの判定処理を実施している)
            m_Fs.AddDefaultMetaParameter(inParam);

            // 一括生成時のスレッドセーフ化目的のため Meta ファイル内のファイルやディレクトリ本体を
            // 作業ディレクトリにコピーし、コピー先のパスに置き換える
            // (関数内部で inParam の内容を見て置換するかどうかの判定処理を実施している)
            m_Fs.ReplaceFilePathMetaParameter(inParam);

            if (isVerbose == true)
            {
                Console.WriteLine("Create Dummy RomFile : size = {0}", binFileSize);
            }
            // 指定サイズのランダムバイナリファイルを生成
            var datPath = m_Fs.CreateRandomBinFile(inParam, binFileSize);

            if (inParam.ContentType == CommonParams.ContentTypeSystemData)
            {
                // SystemData専用の生成処理
                ProcessSettingData(datPath, inParam);
            }
            else if (inParam.ContentType == CommonParams.ContentTypeAddOnContent)
            {
                // AddOnContent専用の生成処理
                ProcessSettingAOC(datPath, inParam, binFileSize);
            }
            else
            {
                if (m_Params.NoBuildFlag == true)
                {
                    // ビルドを実行しない旧来のプロセスを実行
                    ProcessSettingNoBuild(datPath, inParam);
                    return;
                }

                // 設定メタファイルの情報を解析
                var testAppParam = m_Fs.ParseMetaFileSetting(inParam.MetaFilePath);
                if (testAppParam == null)
                {
                    throw new Exception("ProcessSetting() : metaファイルの解析に失敗しました");
                }

                // meta情報リストの先頭にコンテンツタイプ情報を挿入しておく
                {
                    var typeInfo = new TestAppSettingParam.SettingPair("ContentType", inParam.ContentType);
                    testAppParam.MetaSetList.Insert(0, typeInfo);
                }

                // SIGLO-37650 テストアプリの作成時刻情報をさらに先頭に挿入する
                {
                    var nowTime = DateTime.Now.ToString("yyyy/MM/dd HH:mm");
                    var createdTimeInfo = new TestAppSettingParam.SettingPair("CreatedTime", nowTime);
                    testAppParam.MetaSetList.Insert(0, createdTimeInfo);
                }

                ProcessSettingAppType(datPath, inParam, testAppParam);
            }

            if (m_Params.NoCleanupFlag == false)
            {
                // 大量にアプリを一括生成する場合は作業フォルダが無駄に圧迫される可能性があるため、
                // 不要になった作業フォルダは逐次削除しておく
                Directory.Delete(inParam.ChildWorkDirPath, true);
            }
        }

        private bool BuildTestApplication()
        {
            // SIGLO-27640 リビルドを行わない対応
            // ビルドの実行
            var buildOutPath = m_Fs.GetWorkDirFullPath() + BuildOutDirName;
            var buildIntPath = m_Fs.GetWorkDirFullPath() + BuildIntDirName;

            Console.Write("TestApp Building ...");

            var retval = testAppBuilder.ExecuteMSBuild(buildOutPath, buildIntPath);
            if (retval != 0)
            {
                if (isVerbose == false)
                {
                    Console.WriteLine(" TestApp Build  Fail");
                }
                Console.Error.WriteLine("BuildTestApplication() : アプリのビルド処理に失敗しました");
                if (isVerbose == false)
                {
                    Console.Error.WriteLine("ビルドログの確認は --verbose を付けて再実行してください");
                }
                return false;
            }

            Console.WriteLine(" TestApp Build Success!");

            // ビルドしたCodeディレクトリを規定の場所(自アプリ直下のcodeディレクトリ)にコピーする
            {
                // 既存のディレクトリが存在する場合は削除する
                var codeDirPath = this.GetExecutingAppCodePath();
                if (Directory.Exists(codeDirPath) == true)
                {
                    Directory.Delete(codeDirPath, true);
                }

                var srcCodeDir = buildOutPath + @"TestAppSimple.nspd\program0.ncd\code\";
                FsFacade.CopyDirectory(srcCodeDir, codeDirPath);
            }

            // 念のため、ビルドした場所をクリーンしておく
            testAppBuilder.ExecuteMSBuild(buildOutPath, buildIntPath, true);

            return true;
        }

        private void ProcessSettingAppType(string dummyDataPath, SettingParams inParam, TestAppSettingParam appParam)
        {
            if (isVerbose == true)
            {
                appParam.DebugPrint();
            }

            var hashStr = m_Fs.GetSha1String(dummyDataPath);
            if (hashStr == null || hashStr == string.Empty)
            {
                throw new Exception(string.Format("ProcessSettingAppType() : ハッシュ(SHA1)の取得に失敗しました FilePath = {0}", dummyDataPath));
            }
            appParam.SHA1HashString = hashStr;

            if (inParam.HtmlDocumentDirPath == null || inParam.HtmlDocumentDirPath == string.Empty)
            {
                // そもそも HtmlDocument が設定されていなければ null を設定しておく
                appParam.OfflineHtmlPathList = null;
            }
            else
            {
                var list = m_Fs.GetHtmlFilePathList(inParam.HtmlDocumentDirPath);
                for (int i = 0; i < list.Count; ++i)
                {
                    list[i] = list[i].Replace('\\', '/');
                }
                appParam.OfflineHtmlPathList = list;
            }

            var tempRomDir = Path.Combine(inParam.ChildWorkDirPath, "RomDir");
            if (Directory.Exists(tempRomDir) == false)
            {
                Directory.CreateDirectory(tempRomDir);
            }

            // (SIGLO-46475) --romfs オプションが設定されていた場合はそのパスを採用する (デフォルトのROMデータ設定処理は実施しない)
            if (inParam.RomFsDirPath != null && inParam.RomFsDirPath != string.Empty)
            {
                // スレッドセーフ化のためのROMディレクトリ設定の内容を作業ディレクトリ内へコピー
                FsFacade.CopyDirectory(inParam.RomFsDirPath, tempRomDir);
            }
            else
            {
                // ランダムバイナリファイルをdataディレクトリに移動
                var outputDataFilePath = Path.Combine(tempRomDir, @"DummyData.dat");
                m_Fs.MoveFile(dummyDataPath, outputDataFilePath);
                // ダミーデータのハッシュ情報とメタ情報一覧情報(とオフラインHTMLファイルパス情報)をファイルに保存
                testAppBuilder.CreateSettingParamFile(appParam, tempRomDir);
            }

            // descファイルのパスを設定
            var descFilePath = this.GetDescFilePath(inParam);

            // スレッドセーフ化のためのDescファイルのコピー
            var copyDstDescFilePath = Path.Combine(inParam.ChildWorkDirPath, "tempDesc.desc");
            File.Copy(descFilePath, copyDstDescFilePath, true);

            if (isVerbose == true)
            {
                Console.WriteLine("Execute MakeMeta ...");
            }
            var npdmOutPath = Path.Combine(inParam.ChildWorkDirPath, "tempNpdm.npdm");
            try
            {
                // MakeMeta を叩いて npdm ファイルを作成
                var result = sigloToolExec.ExecuteMakeMeta(npdmOutPath, copyDstDescFilePath, inParam);
                if (result.IsFailure())
                {
                    Console.Error.WriteLine("ExecuteMakeMeta()でエラー発生。id={0}, Name={1}", inParam.MetaInputId, inParam.MetaName);
                    Console.Error.WriteLine("エラーメッセージ：{0}", result.ErrorMessage);
                    throw new Exception("ProcessSettingAppType() : ExecuteMakeMeta()の実行に失敗しました");
                }
            }
            catch (Exception err)
            {
                Console.Error.WriteLine(err.StackTrace);
                Console.Error.WriteLine(err.Message);
                throw err;
            }

            var srcCodeDir = this.GetExecutingAppCodePath();
            var codeDir = Path.Combine(inParam.ChildWorkDirPath, "code");
            // SIGLO-36869 Work ディレクトリに code フォルダの内容をコピーする
            FsFacade.CopyDirectory(srcCodeDir, codeDir);

            var moveNpdmFilePath = Path.Combine(codeDir, @"main.npdm");
            // ツール側で作成した npdm ファイルをdataディレクトリに移動
            m_Fs.MoveFile(npdmOutPath, moveNpdmFilePath);

            // デフォルトアイコンが設定されていれば、アイコン画像を生成しリストにパスを設定しておく
            foreach (var iconSetting in inParam.DefaultIconList)
            {
                var fileName = string.Format("_DefaultIcon_{0}_.bmp", iconSetting.Language);
                if (inParam.MultiProgram != null)
                {
                    iconSetting.IconFilePath = Path.Combine(m_Fs.GetWorkDirFullPath(), fileName);
                    // マルチプログラムの場合、ProgramIndex 0 のみワークディレクトリ直下にアイコンを生成
                    byte num = 0;
                    byte.TryParse(inParam.MetaProgramIndex, out num);
                    if (num == 0)
                    {
                        PictureCreator.CreateBitmapIconFile(iconSetting.IconFilePath, inParam, iconSetting.Language);
                    }
                }
                else
                {
                    iconSetting.IconFilePath = Path.Combine(inParam.ChildWorkDirPath, fileName);
                    PictureCreator.CreateBitmapIconFile(iconSetting.IconFilePath, inParam, iconSetting.Language);
                }
            }

            // Meta ファイルにファイル関連の設定情報を追記する
            m_Fs.AddFilePathMetaParameter(inParam);

            // (SIGLO-42685) Aobort させる Result 値を記録したファイルを生成する
            this.ProcessAbortResultSetting(inParam, tempRomDir);

            // (SIGLO-52130) ネットワーク接続要求を起動時に実行する場合はフラグ用のファイルを作成しROMに保持しておく
            if (inParam.NetworkRequestFlag)
            {
                // ROM領域にフラグ用のファイルを作成
                FsFacade.CreateNetworkRequestFile("abc", tempRomDir);
            }

            string nspFileName;
            if (inParam.IsSetOutputFileName == true)
            {
                // ファイル名が指定されている場合はそのまま使用する
                nspFileName = inParam.OutputFileName;
            }
            else
            {
                // 指定されていない場合は以下のデフォルト名作成処理に入る
                nspFileName = string.Format("{0}_{1}", inParam.ContentType, inParam.MetaProgramId.ToLowerInvariant());

                if (inParam.IsSetName == true)
                {
                    nspFileName = string.Format("{0}_{1}", nspFileName, inParam.MetaName);
                }

                if (inParam.MetaVersion != null && inParam.MetaVersion != string.Empty)
                {
                    nspFileName = string.Format("{0}_v{1}", nspFileName, inParam.MetaVersion);
                }
                else
                {
                    // 未設定の場合は v0 が設定される
                    nspFileName = string.Format("{0}_v0", nspFileName);
                }

                if (m_Params.SmallCodeFlag == true)
                {
                    // 念のため小さいコードサイズのフラグが指定されたらファイル名にも反映させておく
                    nspFileName += "_small";
                }
            }

            var nspOutPath = string.Format(@"{0}\{1}.nsp", outDirPath, nspFileName);
            // (SIGLO-48132) "出力先ファイルパス"が指定されていた場合は強制的に上書きする
            if (inParam.OutputFilePath != null && inParam.OutputFilePath != string.Empty)
            {
                nspOutPath = inParam.OutputFilePath;
            }

            // パッチの場合は AuthoringTool を2回叩くことになる
            if (inParam.ContentType == CommonParams.ContentTypePatch)
            {
                // パッチが指定された場合の処理
                var tempCurrentFilePath = Path.Combine(inParam.ChildWorkDirPath, "_current_app_.nsp");
                try
                {
                    // 修正版アプリを作成するため強制的にタイプを Application へ変更する (その後はタイプを使用しないので問題ない)
                    inParam.ContentType = CommonParams.ContentTypeApplication;
                    // AuthoringTool を叩いて修正版アプリの nsp ファイルを(一時的に)作成
                    var result = sigloToolExec.ExecuteAuthoringToolCreateNsp(tempCurrentFilePath, copyDstDescFilePath, codeDir, tempRomDir, inParam);
                    if (result.IsFailure())
                    {
                        Console.Error.WriteLine("ExecuteAuthoringToolCreateNsp()でエラー発生。id={0}, Name={1}", inParam.MetaInputId, inParam.MetaName);
                        Console.Error.WriteLine("エラーメッセージ：{0}", result.ErrorMessage);
                        throw new Exception("ProcessSettingAppType() : ExecuteAuthoringToolCreateNsp()の実行に失敗しました");
                    }

                    // さらに AuthoringTool を叩いてパッチの nsp ファイルを作成
                    result = sigloToolExec.ExecuteAuthoringToolMakePatch(nspOutPath, copyDstDescFilePath, tempCurrentFilePath, inParam);
                    if (result.IsFailure())
                    {
                        Console.Error.WriteLine("ExecuteAuthoringToolMakePatch()でエラー発生。id={0}, Name={1}", inParam.MetaInputId, inParam.MetaName);
                        Console.Error.WriteLine("エラーメッセージ：{0}", result.ErrorMessage);
                        throw new Exception("ProcessSettingAppType() : ExecuteAuthoringToolMakePatch()の実行に失敗しました");
                    }
                }
                catch (Exception err)
                {
                    Console.Error.WriteLine(err.StackTrace);
                    Console.Error.WriteLine(err.Message);
                    throw err;
                }
            }
            else
            {
                // アプリケーション作成処理はこちら
                try
                {
                    // AuthoringToolを叩いて nsp ファイルを作成
                    var result = sigloToolExec.ExecuteAuthoringToolCreateNsp(nspOutPath, copyDstDescFilePath, codeDir, tempRomDir, inParam);
                    if (result.IsFailure())
                    {
                        Console.Error.WriteLine("ExecuteAuthoringToolCreateNsp()でエラー発生。id={0}, Name={1}", inParam.MetaInputId, inParam.MetaName);
                        Console.Error.WriteLine("エラーメッセージ：{0}", result.ErrorMessage);
                        throw new Exception("ProcessSettingAppType() : ExecuteAuthoringToolCreateNsp()の実行に失敗しました");
                    }
                }
                catch (Exception err)
                {
                    Console.Error.WriteLine(err.StackTrace);
                    Console.Error.WriteLine(err.Message);
                    throw err;
                }
            }
        }

        private void ProcessAbortResultSetting(SettingParams inParam, string inOutpuDirPath)
        {
            if (string.IsNullOrEmpty(inParam.AbortResultValue))
            {
                // そもそも abort-result を設定していない場合はすぐに返る
                return;
            }

            var value = inParam.AbortResultValue;
            // AbortのResult値は16進数のフォーマット以外は設定しない
            if (System.Text.RegularExpressions.Regex.IsMatch(value, "^(0x)?([0-9a-f]{1,8}|[0-9A-F]{1,8})$") == false)
            {
                // 弾いた場合は念のためログを出しておく
                Console.WriteLine(" *** Not Set AbortResultValue - Invalid Value : {0} ***", value);
                return;
            }

            // 整形するために一旦数値を経由して文字列化する
            var num = Convert.ToUInt32(value, 16);
            if (num == 0)
            {
                // 0 の場合は成功値が設定されているとみなして弾く
                Console.WriteLine(" *** Not Set AbortResultValue - Success Value : {0} ***", value);
                return;
            }

            // "0x"付きで0埋めの小文字の形で16進数文字列を保持する
            value = string.Format("0x{0:x8}", num);
            //Console.WriteLine("AbortResultValue : {0}", value);

            // Result値が設定されている場合のみ ROM内部 にファイルを生成する
            FsFacade.CreateAbortResultValueFile(value, inOutpuDirPath);
        }

        private void ProcessSettingNoBuild(string dummyDataPath, SettingParams inParam)
        {
            string nspFileName;
            if (inParam.IsSetOutputFileName == true)
            {
                nspFileName = inParam.OutputFileName;
            }
            else
            {
                if (inParam.IsSetName == true)
                {
                    nspFileName = string.Format("{0}_{1}", inParam.MetaProgramId.ToLowerInvariant(), inParam.MetaName);
                }
                else
                {
                    nspFileName = inParam.MetaProgramId.ToLowerInvariant();
                }
            }

            var nspOutPath = string.Format("{0}\\{1}.nsp", outDirPath, nspFileName);
            // (SIGLO-48132) "出力先ファイルパス"が指定されていた場合は強制的に上書きする
            if (inParam.OutputFilePath != null && inParam.OutputFilePath != string.Empty)
            {
                nspOutPath = inParam.OutputFilePath;
            }

            try
            {
                // AuthoringToolを叩いて nsp ファイルを作成
                var result = sigloToolExec.ExecuteAuthoringToolCreateNsp(nspOutPath, dummyDataPath, inParam);
                if (result.IsFailure())
                {
                    Console.Error.WriteLine("ExecuteAuthoringToolCreateNsp()でエラー発生。id={0}, Name={1}", inParam.MetaInputId, inParam.MetaName);
                    Console.Error.WriteLine("エラーメッセージ：{0}", result.ErrorMessage);
                    throw new Exception("ProcessSettingNoBuild() : ExecuteAuthoringToolCreateNsp()の実行に失敗しました");
                }
            }
            catch (Exception err)
            {
                Console.Error.WriteLine(err.StackTrace);
                Console.Error.WriteLine(err.Message);
                throw err;
            }
        }

        private void ProcessSettingData(string dummyDataPath, SettingParams inParam)
        {
            string dirPath = Path.GetDirectoryName(dummyDataPath);
            string tempDirPath = dirPath + @"\_temp_";
            if (Directory.Exists(tempDirPath) == false)
            {
                Directory.CreateDirectory(tempDirPath);
            }

            if (inParam.DataDirPath == null)
            {
                string tempDataPath = tempDirPath + @"\SystemData.dat";
                m_Fs.MoveFile(dummyDataPath, tempDataPath);
            }
            else
            {
                // 任意のデータ(ディレクトリ)指定があればそれを設定する
                FsFacade.CopyDirectory(inParam.DataDirPath, tempDirPath);
            }

            string nspFileName;
            if (inParam.IsSetOutputFileName == true)
            {
                nspFileName = inParam.OutputFileName;
            }
            else
            {
                nspFileName = string.Format("SystemData_{0}", inParam.MetaInputId.ToLowerInvariant());
                if (inParam.MetaVersion != null && inParam.MetaVersion != string.Empty)
                {
                    nspFileName = string.Format("{0}_v{1}", nspFileName, inParam.MetaVersion);
                }
                else
                {
                    // 未設定の場合は v0 が設定される
                    nspFileName = string.Format("{0}_v0", nspFileName);
                }
            }

            var nspOutPath = string.Format("{0}\\{1}.nsp", outDirPath, nspFileName);
            // (SIGLO-48132) "出力先ファイルパス"が指定されていた場合は強制的に上書きする
            if (inParam.OutputFilePath != null && inParam.OutputFilePath != string.Empty)
            {
                nspOutPath = inParam.OutputFilePath;
            }

            try
            {
                // AuthoringToolを叩いて nsp ファイルを作成
                var result = sigloToolExec.ExecuteAuthoringToolCreateNspData(nspOutPath, tempDirPath, inParam);
                if (result.IsFailure())
                {
                    Console.Error.WriteLine("ExecuteAuthoringToolCreateNsp()でエラー発生。id={0}, Name={1}", inParam.MetaInputId, inParam.MetaName);
                    Console.Error.WriteLine("エラーメッセージ：{0}", result.ErrorMessage);
                    throw new Exception("ProcessSettingData() : ExecuteAuthoringToolCreateNsp()の実行に失敗しました");
                }
            }
            catch (Exception err)
            {
                Console.Error.WriteLine(err.StackTrace);
                Console.Error.WriteLine(err.Message);
                throw err;
            }
        }

        private void ProcessSettingAOC(string dummyDataPath, SettingParams inParam, long inBinFileSize)
        {
            string dirPath = Path.GetDirectoryName(dummyDataPath);
            string tempDirPath = dirPath + @"\_temp_";
            if (Directory.Exists(tempDirPath) == false)
            {
                Directory.CreateDirectory(tempDirPath);
            }

            // 複数設定かどうか？
            bool isMultiSetting = (inParam.AocList.Count > 1);

            if (inParam.DataDirPath == null)
            {
                if (isMultiSetting)
                {
                    // 複数個設定だった場合の処理
                    for (int i = 0; i < inParam.AocList.Count; ++i)
                    {
                        var aoc = inParam.AocList[i];
                        if (aoc.IsAlreadySetDataPath)
                        {
                            // すでに DataPath が設定済であれば次の設定へ
                            continue;
                        }

                        // 作業ディレクトリに仮のDataPath用のディレクトリを作成
                        var dataDirPath = Path.Combine(tempDirPath, string.Format("aoc_{0}", i));
                        Directory.CreateDirectory(dataDirPath);
                        // デフォルトデータの作成
                        var aocDataPath = Path.Combine(dataDirPath, "AddOnContent.dat");
                        m_Fs.CreateRandomBinFile(aocDataPath, inBinFileSize, inParam.SeedNumber);
                        var hashStr = m_Fs.GetSha1String(aocDataPath);
                        // ハッシュ値を保持するファイルの作成
                        FsFacade.CreateHashValueFile(hashStr, dataDirPath);
                        // バージョン(32bit)値を保持するファイルの作成
                        FsFacade.CreateVersionValueFile(aoc.Version, dataDirPath);

                        // 作成したデフォルトデータのパスを設定する
                        aoc.DataPath = dataDirPath;
                    }
                }
                else
                {
                    // こちらは従来の処理
                    var tempDataPath = tempDirPath + @"\AddOnContent.dat";
                    m_Fs.MoveFile(dummyDataPath, tempDataPath);
                    // (SIGLO-46577) Aocのデフォルトデータにハッシュ値を付加しておく
                    // バージョン値も保持の対象とする
                    var hashStr = m_Fs.GetSha1String(tempDataPath);
                    FsFacade.CreateHashValueFile(hashStr, tempDirPath);
                    FsFacade.CreateVersionValueFile(inParam.Version, tempDirPath);
                }
            }
            else
            {
                // 任意のデータ(ディレクトリ)指定があればそれを設定する
                FsFacade.CopyDirectory(inParam.DataDirPath, tempDirPath);

                if (isMultiSetting)
                {
                    // 複数個設定だった場合の処理
                    foreach (var aoc in inParam.AocList)
                    {
                        if (aoc.IsAlreadySetDataPath)
                        {
                            // すでに DataPath が設定済であれば次の設定へ
                            continue;
                        }

                        // DataPath が設定されていなければ、指定の任意データのディレクトリパスを設定しておく
                        aoc.DataPath = tempDirPath;
                    }
                }
            }

            // メタファイルに DataPath のタグを追加する
            // (関数内部で設定をするしないの判定を行っている)
            m_Fs.AddAocDataFilePathMetaParameter(inParam, tempDirPath);

            string nspFileName;
            if (inParam.IsSetOutputFileName == true)
            {
                nspFileName = inParam.OutputFileName;
            }
            else
            {
                if (inParam.IsEditMetaFileForCommandArgs == true)
                {
                    if (isMultiSetting)
                    {
                        // 複数個設定だった場合は特殊なので合計コンテンツ数を付与する
                        nspFileName = string.Format("AddOnContent_{0}_v{1}_Multi_{2}", inParam.MetaInputId.ToLowerInvariant(), inParam.MetaVersion, inParam.AocList.Count);
                    }
                    else
                    {
                        // こちらが基本的なデフォルト名となる
                        nspFileName = string.Format("AddOnContent_{0}_v{1}_{2}", inParam.MetaInputId.ToLowerInvariant(), inParam.MetaVersion, inParam.MetaIndex);
                    }
                }
                else
                {
                    if (isMultiSetting)
                    {
                        // 複数個設定だった場合は特殊なので名前を変えておく。念のため連番(SequenceNumber)を入れてファイル名の重複を防ぐ
                        nspFileName = string.Format("AddOnContent_Multi_{0}_TestData_{1}", inParam.AocList.Count, inParam.SequenceNumber);
                    }
                    else
                    {
                        // --meta または --config-xml 指定、かつファイル名指定が無い場合、連番(SequenceNumber)を入れてファイル名の重複を防ぐ
                        nspFileName = string.Format("AddOnContent_TestData_{0}", inParam.SequenceNumber);
                    }
                }
            }

            var nspOutPath = string.Format("{0}\\{1}.nsp", outDirPath, nspFileName);
            // (SIGLO-48132) "出力先ファイルパス"が指定されていた場合は強制的に上書きする
            if (inParam.OutputFilePath != null && inParam.OutputFilePath != string.Empty)
            {
                nspOutPath = inParam.OutputFilePath;
            }

            try
            {
                // AuthoringToolを叩いて nsp ファイルを作成
                var result = sigloToolExec.ExecuteAuthoringToolCreateNspData(nspOutPath, null, inParam);
                if (result.IsFailure())
                {
                    Console.Error.WriteLine("ExecuteAuthoringToolCreateNsp()でエラー発生。id={0}, Name={1}", inParam.MetaInputId, inParam.MetaName);
                    Console.Error.WriteLine("エラーメッセージ：{0}", result.ErrorMessage);
                    throw new Exception("ProcessSettingAOC() : ExecuteAuthoringToolCreateNsp()の実行に失敗しました");
                }
            }
            catch (Exception err)
            {
                Console.Error.WriteLine(err.StackTrace);
                Console.Error.WriteLine(err.Message);
                throw err;
            }
        }
    }
}
