﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Microsoft.VisualStudio.TestTools.UnitTesting;
using System;
using System.Threading;

namespace NintendoWare.Spy.Windows.Tests
{
    [TestClass]
    public class DataSizeUnitConverterTests
    {
        private const long KB = 1024;
        private const long MB = KB * 1024;
        private const long GB = MB * 1024;
        private const long TB = GB * 1024;

        [TestMethod]
        public void Convert()
        {
            var converter = new DataSizeUnitConverter();

            this.Test("0.00 Byte", 0L, converter);
            this.Test("100.00 Byte", 100L, converter);
            this.Test("1.00 KB", KB, converter);
            this.Test("1.01 KB", KB + 10L, converter);
            this.Test("1.01 MB", MB + 10L * KB, converter);
            this.Test("1.01 GB", GB + 10L * MB, converter);
            this.Test("1.01 TB", TB + 10L * GB, converter);

            this.Test("-100.00 Byte", -100L, converter);
            this.Test("-1.01 KB", -(KB + 10L), converter);
            this.Test("-1.01 MB", -(MB + 10L * KB), converter);
            this.Test("-1.01 GB", -(GB + 10L * MB), converter);
            this.Test("-1.01 TB", -(TB + 10L * GB), converter);

            // float
            this.Test("1.01 KB", 1f * KB + 10f, converter);
            this.Test("-1.01 KB", -(1f * KB + 10f), converter);

            // double
            this.Test("1.01 KB", 1d * KB + 10f, converter);
            this.Test("-1.01 KB", -(1d * KB + 10f), converter);

            // double.NaN, Infinity は 0 とする
            this.Test("0.00 Byte", double.NaN, converter);
            this.Test("0.00 Byte", double.NegativeInfinity, converter);
            this.Test("0.00 Byte", double.PositiveInfinity, converter);
        }

        [TestMethod]
        public void ConvertFromKb()
        {
            var converter = new DataSizeUnitConverter()
            {
                SourceUnit = DataSizeUnitConverter.UnitType.KB,
            };

            this.Test("0.00 KB", 0d, converter);
            this.Test("0.50 KB", 0.5d, converter);
            this.Test("1.00 MB", 1d * KB, converter);
            this.Test("1.01 MB", 1d * KB + 10d, converter);
            this.Test("1.01 GB", 1d * MB + 10d * KB, converter);
        }

        [TestMethod]
        public void ConvertFromKbToMb()
        {
            var converter = new DataSizeUnitConverter()
            {
                SourceUnit = DataSizeUnitConverter.UnitType.KB,
                DestinationUnit = DataSizeUnitConverter.UnitType.MB,
            };

            this.Test("0.00 MB", 0d, converter);
            this.Test("0.10 MB", 100d, converter);
            this.Test("0.99 MB", 1010d, converter);
            this.Test("1.00 MB", 1d * KB, converter);
            this.Test("1.01 MB", 1d * KB + 10d, converter);
            this.Test("1034.00 MB", 1d * MB + 10d * KB, converter);
        }

        [TestMethod]
        [ExpectedException(typeof(InvalidOperationException))]
        public void ConvertFromMbToKb()
        {
            var converter = new DataSizeUnitConverter()
            {
                SourceUnit = DataSizeUnitConverter.UnitType.MB,
                DestinationUnit = DataSizeUnitConverter.UnitType.KB,
            };

            this.Test("0.00 KB", 0d, converter);
        }

        [TestMethod]
        public void ConvertToOverTb()
        {
            var converter = new DataSizeUnitConverter()
            {
                SourceUnit = DataSizeUnitConverter.UnitType.TB,
            };

            this.Test("1024.00 TB", 1024d, converter);
        }

        private void Test(string expectedText, object value, DataSizeUnitConverter converter)
        {
            var actualText = (string)converter.Convert(value, typeof(string), null, Thread.CurrentThread.CurrentCulture);
            Assert.AreEqual(expectedText, actualText);
        }
    }
}
