﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using Microsoft.VisualStudio.TestTools.UnitTesting;
using Nintendo.Atk.Tests;
using System;
using System.Diagnostics.CodeAnalysis;
using System.IO;
using System.Linq;

namespace Nintendo.Atk.Binary
{
    /// <summary>
    /// bfstm ファイルの入力テストです。
    /// テストの実行は、x64 / x86 テストアセンブリから行います。
    /// </summary>
    public static class BfstmReaderTest
    {
        private const string BfstmDirectoryPath = @"Resources\\bfstm\";
        private const string RawDirectoryPath = @"Resources\\raw\";
        private const string ActualRawDirectoryPath = @"Actual\\raw\";

        public static void ReadWaveBinaryInfo()
        {
            using (var reader = CreateReader($"{BfstmDirectoryPath}sin2ch48kLoop.adpcm.bfstm"))
            {
                var info = reader.ReadInfo();

                Assert.AreEqual(WaveFormat.Encodings.ADPCM, info.Format.Encoding);
                Assert.AreEqual(2, info.Format.ChannelCount);
                Assert.AreEqual(48000, info.Format.Samplerate);
                Assert.AreEqual(52672, info.SampleCount);
                Assert.AreEqual(true, info.HasLoop);
                Assert.AreEqual(28672, info.LoopStartFrame);
                Assert.AreEqual(52671, info.LoopEndFrame);
                Assert.AreEqual(24000, info.OriginalLoopStartFrame);
                Assert.AreEqual(47999, info.OriginalLoopEndFrame);
            }

            using (var reader = CreateReader($"{BfstmDirectoryPath}sin-wn2ch48kLoop.pcm16.bfstm"))
            {
                var info = reader.ReadInfo();

                Assert.AreEqual(WaveFormat.Encodings.PCM16, info.Format.Encoding);
                Assert.AreEqual(4, info.Format.ChannelCount);
                Assert.AreEqual(48000, info.Format.Samplerate);
                Assert.AreEqual(48576, info.SampleCount);
                Assert.AreEqual(true, info.HasLoop);
                Assert.AreEqual(24576, info.LoopStartFrame);
                Assert.AreEqual(48575, info.LoopEndFrame);
                Assert.AreEqual(24000, info.OriginalLoopStartFrame);
                Assert.AreEqual(47999, info.OriginalLoopEndFrame);
            }

            using (var reader = CreateReader($"{BfstmDirectoryPath}sin-wn2ch48k.adpcm.bfstm"))
            {
                var info = reader.ReadInfo();

                Assert.AreEqual(WaveFormat.Encodings.ADPCM, info.Format.Encoding);
                Assert.AreEqual(4, info.Format.ChannelCount);
                Assert.AreEqual(48000, info.Format.Samplerate);
                Assert.AreEqual(48000, info.SampleCount);
                Assert.AreEqual(false, info.HasLoop);
                Assert.AreEqual(0, info.LoopStartFrame);
                Assert.AreEqual(0, info.LoopEndFrame);
                Assert.AreEqual(0, info.OriginalLoopStartFrame);
                Assert.AreEqual(0, info.OriginalLoopEndFrame);
            }
        }

        public static void DecodeAll()
        {
            DecodeAll("sin-wn2ch48kLoop.pcm16.bfstm", "sin-wn2ch48kLoop.pcm16.bfstm.ch{0}.raw.wav");
            DecodeAll("sin-wn2ch48k.adpcm.bfstm", "sin-wn2ch48k.adpcm.bfstm.ch{0}.raw.wav");
        }

        private static void DecodeAll(string srcFileName, string destFileNameFormat)
        {
            WaveBinaryInfo info;

            using (var reader = CreateReader(Path.Combine(BfstmDirectoryPath, srcFileName)))
            {
                info = reader.ReadInfo();

                foreach (var channel in Enumerable.Range(0, info.Format.ChannelCount))
                {
                    Directory.CreateDirectory(ActualRawDirectoryPath);

                    var destFileName = string.Format(destFileNameFormat, channel);
                    using (var writer = FileIOHelper.CreateBinaryWriter(Path.Combine(ActualRawDirectoryPath, destFileName)))
                    {
                        reader.DecodeAll(channel).ForEach(sample => writer.Write(sample));
                    }
                }
            }

            foreach (var channel in Enumerable.Range(0, info.Format.ChannelCount))
            {
                var fileName = string.Format(destFileNameFormat, channel);
                FileComparer.CompareFile(Path.Combine(RawDirectoryPath, fileName), Path.Combine(ActualRawDirectoryPath, fileName));
            }
        }

        private static BfstmReader CreateReader(string filePath)
        {
            return new BfstmReader(File.OpenRead(filePath));
        }
    }
}
