﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace PropertySheetsTest
{
    public class VcxprojConfiguration
    {
        public string Spec { get; }
        public string BuildType { get; }
        public string PlatformToolset { get; }
        public string Configuration { get; }
        public string Platform { get; }
        public string VsVersion { get; }

        public VcxprojConfiguration(string spec, string buildType, string toolset, string configuration, string platform, string vsVersion)
        {
            this.Spec = spec;
            this.BuildType = buildType;
            this.PlatformToolset = toolset;
            this.Configuration = configuration;
            this.Platform = platform;
            this.VsVersion = vsVersion;
        }
    }

    public static class VcxprojConfigurationGenerator
    {
        public static IEnumerable<object[]> GenerateConfigurations()
        {
            return GenerateWinPlatformConfigurations()
                .Concat(GenerateNXPlatformConfigurations());
        }

        public static IEnumerable<object[]> GenerateWinPlatformConfigurations()
        {
            return GenerateConfigurationsImpl(
                specs: new[] { "Generic", "NX" },
                buildTypes: new[] { "Debug", "Develop", "Release" },
                toolsets: new[] { "v140", "v141" },
                platforms: new[] { "Win32", "x64" },
                isWinPlatform: true
                );
        }

        public static IEnumerable<object[]> GenerateNXPlatformConfigurations()
        {
            return GenerateConfigurationsImpl(
                specs: new[] { "NX" },
                buildTypes: new[] { "Debug", "Develop", "Release" },
                toolsets: new[] { "v140" },
                platforms: new[] { "NX32", "NX64" },
                isWinPlatform: false
                );
        }

        private static IEnumerable<object[]> GenerateConfigurationsImpl(
            string[] specs, string[] buildTypes, string[] toolsets, string[] platforms, bool isWinPlatform)
        {
            return
                from spec in specs
                from buildType in buildTypes
                from toolset in toolsets
                from platform in platforms
                select new[]
                {
                    new VcxprojConfiguration(
                        spec            : spec,
                        buildType       : buildType,
                        toolset         : toolset,
                        configuration   : GetConfigurationString(spec, toolset, buildType, isWinPlatform),
                        platform        : platform,
                        vsVersion       : GetVsVersionString(toolset))
                };
        }

        private static string GetVsVersionString(string toolset)
        {
            switch (toolset)
            {
                case "v140":
                    return "14.0";
                case "v141":
                    return "15.0";
                default:
                    throw new ArgumentException($"不明な PlatformToolset です: {toolset}", nameof(toolset));
            }
        }

        private static string GetConfigurationString(string spec, string toolset, string buildType, bool isWinPlatform)
        {
            string specConfigurationString = spec != "Generic" ? $"{spec}_" : string.Empty;
            string toolsetConfigurationString()
            {
                if (!isWinPlatform)
                {
                    return string.Empty;
                }
                switch (toolset)
                {
                    case "v140":
                        return "VS2015_";
                    case "v141":
                        return "VS2017_";
                    default:
                        throw new ArgumentException($"不明な PlatformToolset です: {toolset}", nameof(toolset));
                }
            };
            return string.Concat(
                specConfigurationString,
                toolsetConfigurationString(),
                buildType);
        }
    }
}
