﻿using Microsoft.Build.Evaluation;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml.Linq;
using Xunit;

namespace PropertySheetsTest
{
    internal static class Util
    {
        /// <summary>
        /// directory およびその先祖のディレクトリであって、filename のファイル名を持つファイルが存在するディレクトリのうち、最も directory に近いもののパスを返します。
        /// </summary>
        public static string GetDirectoryNameOfFileAbove(string directory, string filename)
        {
            if (File.Exists(Path.Combine(directory, filename)))
            {
                return directory;
            }
            else
            {
                var parentDirectoryName = Path.GetDirectoryName(directory);
                if (parentDirectoryName == null)
                {
                    return null;
                }
                return GetDirectoryNameOfFileAbove(parentDirectoryName, filename);
            }
        }

        /// <summary>
        /// 辞書 dicrionary に key をキーとする値が存在すればそれを返します。存在しなければ、defaultValue を返します。
        /// </summary>
        public static TValue GetValueOrDefault<TKey, TValue>(IReadOnlyDictionary<TKey, TValue> dictionary, TKey key, TValue defaultValue)
        {
            TValue value;
            if (!dictionary.TryGetValue(key, out value))
            {
                return defaultValue;
            }

            return value;
        }

        /// <summary>
        /// path に指定した VC++ プロジェクトファイルから ProjectConfiguration 要素を読み取って返します。
        /// </summary>
        public static IEnumerable<ConfigurationPair> ReadConfigurationPairsFromVcProject(string path)
        {
            var root = XDocument.Load(path).Root;
            var ns = root.Name.Namespace;
            return root.Descendants(ns + "ProjectConfiguration")
                .Select(e =>
                    new ConfigurationPair(
                        e.Element(ns + "Configuration").Value,
                        e.Element(ns + "Platform").Value))
                .ToArray();
        }

        /// <summary>
        /// MsBuild のプロパティ・メタデータの値 value を要素ごとに分割して返します。値が空の要素は省略します。
        /// </summary>
        private static char[] msBuildValueDelimiters = new char[] { ';', '\r', '\n' };
        public static IEnumerable<string> SplitMsBuildValue(string value)
        {
            return value
                .Split(msBuildValueDelimiters, StringSplitOptions.RemoveEmptyEntries)
                .Select(x => x.Trim()).ToArray();
        }

        /// <summary>
        /// xs に ys の要素が1つ以上現れる場合に真を返します。そうではない場合、偽を返します。
        /// </summary>
        public static bool SequenceContainsAnyElement<T>(IEnumerable<T> xs, IEnumerable<T> ys)
        {
            return SequenceContainsAnyElement(xs, ys, EqualityComparer<T>.Default);
        }

        /// <summary>
        /// xs に ys の要素が1つ以上現れる場合に真を返します。そうではない場合、偽を返します。
        /// </summary>
        public static bool SequenceContainsAnyElement<T>(IEnumerable<T> xs, IEnumerable<T> ys, IEqualityComparer<T> equalityComparer)
        {
            return ys.Intersect(xs, equalityComparer).Any();
        }

        /// <summary>
        /// xs に ys の要素がすべて現れる場合に真を返します。そうではない場合、偽を返します。
        /// </summary>
        public static bool SequenceContainsElements<T>(IEnumerable<T> xs, IEnumerable<T> ys)
        {
            return SequenceContainsElements(xs, ys, EqualityComparer<T>.Default);
        }

        /// <summary>
        /// xs に ys の要素がすべて現れる場合に真を返します。そうではない場合、偽を返します。
        /// </summary>
        public static bool SequenceContainsElements<T>(IEnumerable<T> xs, IEnumerable<T> ys, IEqualityComparer<T> equalityComparer)
        {
            return !ys.Except(xs, equalityComparer).Any();
        }

        /// <summary>
        /// xs に ys の要素が ys に現れる順番ですべて現れる場合に真を返します。そうではない場合、偽を返します。
        /// </summary>
        public static bool SequenceContainsElementsInOrder<T>(IEnumerable<T> xs, IEnumerable<T> ys)
        {
            return SequenceContainsElementsInOrder(xs, ys, EqualityComparer<T>.Default);
        }

        /// <summary>
        /// xs に ys の要素が ys に現れる順番ですべて現れる場合に真を返します。そうではない場合、偽を返します。
        /// </summary>
        public static bool SequenceContainsElementsInOrder<T>(IEnumerable<T> xs, IEnumerable<T> ys, IEqualityComparer<T> equalityComparer)
        {
            return xs.Intersect(ys, equalityComparer).SequenceEqual(ys, equalityComparer);
        }

        /// <summary>
        /// パス文字列 path をファイル名（あれば）・ディレクトリ（複数）・ルートディレクトリ（あれば）に分割したものを返します。
        /// </summary>
        public static IEnumerable<string> SplitPath(string path)
        {
            return SplitPathReversed(path).Reverse();
        }

        /// <summary>
        /// パス文字列 path をファイル名（あれば）・ディレクトリ（複数）・ルートディレクトリ（あれば）に分割したものを逆順で返します。
        /// </summary>
        public static IEnumerable<string> SplitPathReversed(string path)
        {
            var xs = new List<string>();

            // DirectorySeparator で終わっている場合、それを無視する ( @"c:\hoge\\" )
            if (string.IsNullOrEmpty(Path.GetFileName(path)))
            {
                path = Path.GetDirectoryName(path);
            }

            while (!string.IsNullOrEmpty(path))
            {
                var filename = Path.GetFileName(path);
                var directory = Path.GetDirectoryName(path);
                // まだルートディレクトリに到達していないか
                if (!string.IsNullOrEmpty(filename))
                {
                    // ファイルネーム部分を追加して親ディレクトリを再帰的に処理
                    xs.Add(filename);
                }
                else
                {
                    // rooted でなくディレクトリがなくなったら終了
                    // rooted であってルートディレクトリに到達したら、それを追加して終了
                    if (directory == null)
                    {
                        xs.Add(path);
                    }
                    break;
                }
                path = directory;
            }

            return xs;
        }

        /// <summary>
        /// コマンドラインオプション文字列を引数ごとに分割します。
        /// </summary>
        #region MakeVisualStudioProject よりコピペ、テスト済み
        public static IEnumerable<string> TokenizeCommandLineOptions(string op)
        {
            using (var reader = new StringReader(op))
            {
                var tokens = new List<string>();
                var tokenBuffer = new StringBuilder();
                var isInnerQuote = false;

                int i = reader.Read();
                while (i != -1)
                {
                    char c = (char)i;
                    i = reader.Read();

                    if (c == '"')
                    {
                        isInnerQuote = !isInnerQuote;
                    }
                    else if (c == ' ')
                    {
                        if (isInnerQuote)
                        {
                            tokenBuffer.Append(c);
                        }
                        else
                        {
                            if (tokenBuffer.Length > 0)
                            {
                                tokens.Add(tokenBuffer.ToString());
                                tokenBuffer.Clear();
                            }
                        }
                    }
                    else
                    {
                        tokenBuffer.Append(c);
                    }
                }
                if (tokenBuffer.Length > 0)
                {
                    tokens.Add(tokenBuffer.ToString());
                }

                return tokens;
            }
        }
    }
    #endregion

    public class UtilTest
    {
        [Fact]
        public void SplitPathTest()
        {
            AssertUtil.AssertSequenceEqual(
                Util.SplitPathReversed(@"c:\hoge\fuga\\"),
                new string[] { "fuga", "hoge", @"c:\" });
            AssertUtil.AssertSequenceEqual(
                Util.SplitPathReversed(@"c:\hoge\fuga"),
                new string[] { "fuga", "hoge", @"c:\" });
            AssertUtil.AssertSequenceEqual(
                Util.SplitPathReversed(@"hoge\fuga"),
                new string[] { "fuga", "hoge" });
        }
    }
}
