﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>
#include <nn/util/util_BitUtil.h>
#include <nn/nn_Common.h>

#include "../../../../../Programs/Eris/Sources/Libraries/audio/audio_ResourceExclusionChecker.h"

TEST(ResourceExclusionChecker, AcquireAndRelease)
{
    // Simple acquire and release
    for(int i = 0; i < 100; ++i)
    {
        auto pChecker = nn::audio::detail::AcquireResourceExclusionChecker();
        EXPECT_NE(pChecker, nullptr);
        nn::audio::detail::ReleaseResourceExclusionChecker(pChecker);
    }

    // Max acquire and max release
    {
        nn::audio::detail::ResourceExclusionChecker* pCheckers[nn::audio::AudioRendererCountMax];

        for(int i = 0; i < nn::audio::AudioRendererCountMax; ++i)
        {
            pCheckers[i] = nn::audio::detail::AcquireResourceExclusionChecker();
        }

        EXPECT_EQ(nn::audio::detail::AcquireResourceExclusionChecker(), nullptr);

        for(int i = 0; i < nn::audio::AudioRendererCountMax; ++i)
        {
            nn::audio::detail::ReleaseResourceExclusionChecker(pCheckers[i]);
        }
    }

    // Precondition
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::detail::ReleaseResourceExclusionChecker(nullptr), "");
}

TEST(ResourceExclusionChecker, FindResourceExclusionCheckerFromRegionInConfig)
{
    void* pConfigBuffer = reinterpret_cast<void*>(0x1000);
    size_t configBufferSize = 0x100;

    auto pChecker = nn::audio::detail::AcquireResourceExclusionChecker();
    EXPECT_NE(pChecker, nullptr);

    EXPECT_EQ(nn::audio::detail::FindResourceExclusionCheckerFromRegionInConfig(pConfigBuffer), nullptr);
    EXPECT_TRUE(pChecker->RegisterConfig(pConfigBuffer, configBufferSize));

    EXPECT_EQ(nn::audio::detail::FindResourceExclusionCheckerFromRegionInConfig(pConfigBuffer), pChecker);
    EXPECT_EQ(nn::audio::detail::FindResourceExclusionCheckerFromRegionInConfig(reinterpret_cast<void*>(reinterpret_cast<uintptr_t>(pConfigBuffer) + configBufferSize - 1)), pChecker);

    nn::audio::detail::ReleaseResourceExclusionChecker(pChecker);
    EXPECT_EQ(nn::audio::detail::FindResourceExclusionCheckerFromRegionInConfig(pConfigBuffer), nullptr);

    // Precondition
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::detail::FindResourceExclusionCheckerFromRegionInConfig(nullptr), "");
}

TEST(ResourceExclusionChecker, FindResourceExclusionCheckerFromInternalHandle)
{
    void* pInternalHandle = reinterpret_cast<void*>(0x1000);

    EXPECT_EQ(nn::audio::detail::FindResourceExclusionCheckerFromInternalHandle(pInternalHandle), nullptr);
    auto pChecker = nn::audio::detail::AcquireResourceExclusionChecker();
    EXPECT_NE(pChecker, nullptr);

    EXPECT_TRUE(pChecker->RegisterInternalHandle(pInternalHandle));
    EXPECT_EQ(pChecker->GetInternalHandle(), pInternalHandle);
    EXPECT_EQ(nn::audio::detail::FindResourceExclusionCheckerFromInternalHandle(pInternalHandle), pChecker);

    nn::audio::detail::ReleaseResourceExclusionChecker(pChecker);
    EXPECT_EQ(nn::audio::detail::FindResourceExclusionCheckerFromInternalHandle(pInternalHandle), nullptr);

    // Precondition
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::detail::FindResourceExclusionCheckerFromInternalHandle(nullptr), "");

}

TEST(ResourceExclusionChecker, AccessSuccess)
{
    void* pConfigBuffer = reinterpret_cast<void*>(0x1000);
    size_t configBufferSize = 0x100;

    auto pChecker = nn::audio::detail::AcquireResourceExclusionChecker();
    EXPECT_NE(pChecker, nullptr);

    EXPECT_TRUE(pChecker->RegisterConfig(pConfigBuffer, configBufferSize));

    // Access config (1 instance)
    EXPECT_TRUE(pChecker->NotifyAccessAudioRendererConfigBegin());
    EXPECT_TRUE(pChecker->NotifyAccessAudioRendererConfigEnd());

    // Access config (2 instance)
    EXPECT_TRUE(pChecker->NotifyAccessAudioRendererConfigBegin());
    EXPECT_TRUE(pChecker->NotifyAccessAudioRendererConfigBegin());
    EXPECT_TRUE(pChecker->NotifyAccessAudioRendererConfigEnd());
    EXPECT_TRUE(pChecker->NotifyAccessAudioRendererConfigEnd());

    // Access config complicatedly (3 instance)
    EXPECT_TRUE(pChecker->NotifyAccessAudioRendererConfigBegin());
    EXPECT_TRUE(pChecker->NotifyAccessAudioRendererConfigBegin());
    EXPECT_TRUE(pChecker->NotifyAccessAudioRendererConfigEnd());
    EXPECT_TRUE(pChecker->NotifyAccessAudioRendererConfigBegin());
    EXPECT_TRUE(pChecker->NotifyAccessAudioRendererConfigEnd());
    EXPECT_TRUE(pChecker->NotifyAccessAudioRendererConfigEnd());

    // Access handle
    EXPECT_TRUE(pChecker->NotifyRequestUpdateAudioRendererBegin());
    EXPECT_TRUE(pChecker->NotifyRequestUpdateAudioRendererEnd());

    nn::audio::detail::ReleaseResourceExclusionChecker(pChecker);
}

TEST(ResourceExclusionChecker, AccessFailure)
{
    void* pConfigBuffer = reinterpret_cast<void*>(0x1000);
    size_t configBufferSize = 0x100;

    auto pChecker = nn::audio::detail::AcquireResourceExclusionChecker();
    EXPECT_NE(pChecker, nullptr);

    EXPECT_TRUE(pChecker->RegisterConfig(pConfigBuffer, configBufferSize));

    // Access config -> Access handle
    EXPECT_TRUE(pChecker->NotifyAccessAudioRendererConfigBegin());
    EXPECT_FALSE(pChecker->NotifyRequestUpdateAudioRendererBegin());
    EXPECT_FALSE(pChecker->NotifyRequestUpdateAudioRendererEnd());
    EXPECT_TRUE(pChecker->NotifyAccessAudioRendererConfigEnd());

    // Access handle -> Access config
    EXPECT_TRUE(pChecker->NotifyRequestUpdateAudioRendererBegin());
    EXPECT_FALSE(pChecker->NotifyAccessAudioRendererConfigBegin());
    EXPECT_FALSE(pChecker->NotifyAccessAudioRendererConfigEnd());
    EXPECT_TRUE(pChecker->NotifyRequestUpdateAudioRendererEnd());

    nn::audio::detail::ReleaseResourceExclusionChecker(pChecker);
}
