﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>
#include <nn/audio/audio_Applet.h>
#include <nn/audio/audio_Debugger.h>
#include <nn/applet/applet_Apis.h>
#include <nn/audio/audio_AudioOut.h>
#include <nn/audio/audio_FinalOutputRecorder.h>
#include "testAudio_ScopedGameRecord.h"
#include <nn/mem.h>
#include <nn/nn_SdkLog.h>

#if !defined(NN_SDK_BUILD_RELEASE)
#define NN_TEST_EXPECT_DEATH_OR_RESULT(func, expectedResult) \
    EXPECT_DEATH_IF_SUPPORTED(func, "")
#else
#define NN_TEST_EXPECT_DEATH_OR_RESULT(func, expectedResult) \
    NNT_EXPECT_RESULT_FAILURE(expectedResult, func)
#endif

namespace {

const size_t g_DataSize = 4096;
const size_t g_BufferSize = 4 * nn::audio::FinalOutputRecorderBuffer::SizeGranularity;
char g_TestBuffer[1024 * 1024];
const int g_audioFramesPerSecond = 200;
void GenerateSquareWaveInt16(void* buffer, int channelCount, int sampleRate, int sampleCount, int amplitude)
{
    static int s_TotalSampleCount = 0;

    const int WAVE_SAMPLE_RATE = 440;
    const int WAVE_LENGTH = sampleRate / WAVE_SAMPLE_RATE;
    int16_t* buf = reinterpret_cast<int16_t*>(buffer);
    for (int sample = 0; sample < sampleCount; sample++)
    {
        int16_t value = static_cast<int16_t>(s_TotalSampleCount < (WAVE_LENGTH / 2) ? amplitude : -amplitude);
        for (int ch = 0; ch < channelCount; ch++)
        {
            buf[sample*channelCount + ch] = value;
        }
        s_TotalSampleCount++;
        if (s_TotalSampleCount == WAVE_LENGTH)
        {
            s_TotalSampleCount = 0;
        }
    }
}

NN_AUDIO_ALIGNAS_AUDIO_IN_BUFFER_ALIGN char g_Buffer[g_BufferSize];
}

TEST(InitializeFinalOutputRecorderParameter, Success)
{
    nn::audio::FinalOutputRecorderParameter parameter;

    nn::audio::InitializeFinalOutputRecorderParameter(&parameter);
    EXPECT_EQ(parameter.sampleRate, 0);
    EXPECT_EQ(parameter.channelCount, 0);
}

TEST(OpenFinalOutputRecorder, Success)
{
    int sampleRate;
    int channelCount;
    // With default
    {
        nn::audio::FinalOutputRecorderParameter parameter = { 0 };
        ScopedGameRecord gameRecord;
        NNT_EXPECT_RESULT_SUCCESS(
            nn::audio::OpenFinalOutputRecorder(gameRecord.Get(), parameter));
        sampleRate = GetFinalOutputRecorderSampleRate(gameRecord.Get());
        channelCount = GetFinalOutputRecorderChannelCount(gameRecord.Get());
        auto state = nn::audio::GetFinalOutputRecorderState(gameRecord.Get());
        EXPECT_EQ(
            state, nn::audio::FinalOutputRecorderState_Stopped);
    }

    // With non-default sample rate
    {
        nn::audio::FinalOutputRecorderParameter parameter = { sampleRate };
        ScopedGameRecord gameRecord;
        NNT_EXPECT_RESULT_SUCCESS(
            nn::audio::OpenFinalOutputRecorder(gameRecord.Get(), parameter));
        EXPECT_EQ(
            nn::audio::GetFinalOutputRecorderState(gameRecord.Get()), nn::audio::FinalOutputRecorderState_Stopped);
    }

    // With non-default channel count
    {
        nn::audio::FinalOutputRecorderParameter parameter = { 0 };
        parameter.channelCount = channelCount;
        ScopedGameRecord gameRecord;
        NNT_EXPECT_RESULT_SUCCESS(
            nn::audio::OpenFinalOutputRecorder(gameRecord.Get(), parameter));
        EXPECT_EQ(
            nn::audio::GetFinalOutputRecorderState(gameRecord.Get()), nn::audio::FinalOutputRecorderState_Stopped);
    }

    {
        nn::audio::FinalOutputRecorderParameter parameter0 = { 0 };
        nn::audio::FinalOutputRecorderParameter parameter1 = { sampleRate };
        parameter1.channelCount = channelCount;
        ScopedGameRecord gameRecord0, gameRecord1;
        NNT_EXPECT_RESULT_SUCCESS(
            nn::audio::OpenFinalOutputRecorder(gameRecord0.Get(), parameter0));
        NNT_EXPECT_RESULT_SUCCESS(
            nn::audio::OpenFinalOutputRecorder(gameRecord1.Get(), parameter1));
    }
}

TEST(OpenFinalOutputRecorder, Error)
{

    const int InvalidSampleRate = 1;
    const int InvalidChannelCount = 3;
    nn::audio::FinalOutputRecorder gameRecord;
    {
        nn::audio::FinalOutputRecorderParameter parameter1 = { InvalidSampleRate };
        NNT_EXPECT_RESULT_FAILURE(
            nn::audio::ResultInvalidSampleRate, nn::audio::OpenFinalOutputRecorder(&gameRecord, parameter1));

        nn::audio::FinalOutputRecorderParameter parameter2 = { 0 };
        parameter2.channelCount = InvalidChannelCount;
        NNT_EXPECT_RESULT_FAILURE(
            nn::audio::ResultInvalidSampleRate, nn::audio::OpenFinalOutputRecorder(&gameRecord, parameter1));
    }
}

TEST(CloseFinalOutputRecorder, Success)
{
    nn::audio::FinalOutputRecorderParameter parameter = { 0 };
    ScopedGameRecord gameRecord;
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::OpenFinalOutputRecorder(gameRecord.Get(), parameter));
    nn::audio::CloseFinalOutputRecorder(gameRecord.Get());
}

TEST(StartFinalOutputRecorder, Success)
{
    nn::audio::FinalOutputRecorderParameter parameter = { 0 };
    ScopedGameRecord gameRecord;
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::OpenFinalOutputRecorder(gameRecord.Get(), parameter));
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::StartFinalOutputRecorder(gameRecord.Get()));
    EXPECT_EQ(
        nn::audio::GetFinalOutputRecorderState(gameRecord.Get()), nn::audio::FinalOutputRecorderState_Started);
}

TEST(StartFinalOutputRecorder, Precondition)
{
    nn::audio::FinalOutputRecorderParameter parameter = { 0 };
    nn::audio::InitializeFinalOutputRecorderParameter(&parameter);
    ScopedGameRecord gameRecord;
#if defined(NN_SDK_BUILD_RELEASE)
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::OpenFinalOutputRecorder(gameRecord.Get(), parameter));
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::StartFinalOutputRecorder(gameRecord.Get()));
    EXPECT_EQ(
        nn::audio::GetFinalOutputRecorderState(gameRecord.Get()), nn::audio::FinalOutputRecorderState_Started);
    // Precondition violation - starting FinalOutputRecorder when already started
    NNT_EXPECT_RESULT_FAILURE(
        nn::audio::ResultOperationFailed, nn::audio::StartFinalOutputRecorder(gameRecord.Get()));
    #else
    // nullptr
    EXPECT_DEATH_IF_SUPPORTED(
        nn::audio::StartFinalOutputRecorder(nullptr), "");
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::OpenFinalOutputRecorder(gameRecord.Get(), parameter));
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::StartFinalOutputRecorder(gameRecord.Get()));
    EXPECT_EQ(
        nn::audio::GetFinalOutputRecorderState(gameRecord.Get()), nn::audio::FinalOutputRecorderState_Started);
    EXPECT_DEATH_IF_SUPPORTED(
        nn::audio::StartFinalOutputRecorder(gameRecord.Get()), "");
#endif
}

TEST(MultipleStartStopFinalOutputRecorder, Success)
{
    nn::audio::FinalOutputRecorderParameter parameter = { 0 };
    ScopedGameRecord gameRecord;
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::OpenFinalOutputRecorder(gameRecord.Get(), parameter));
    for(int i = 0; i < 16; ++i)
    {
        NNT_EXPECT_RESULT_SUCCESS(
            nn::audio::StartFinalOutputRecorder(gameRecord.Get()));
        nn::audio::StopFinalOutputRecorder(gameRecord.Get());
    }
    EXPECT_EQ(
        nn::audio::GetFinalOutputRecorderState(gameRecord.Get()), nn::audio::FinalOutputRecorderState_Stopped);
}

TEST(StopFinalOutputRecorder, Success)
{
    nn::audio::FinalOutputRecorderParameter parameter = { 0 };
    ScopedGameRecord gameRecord;
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::OpenFinalOutputRecorder(gameRecord.Get(), parameter));
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::StartFinalOutputRecorder(gameRecord.Get()));
    nn::audio::StopFinalOutputRecorder(gameRecord.Get());
    EXPECT_EQ(
        nn::audio::GetFinalOutputRecorderState(gameRecord.Get()), nn::audio::FinalOutputRecorderState_Stopped);
}

TEST(StopFinalOutputRecorder, Precondition)
{
    nn::audio::FinalOutputRecorderParameter parameter = { 0 };
    nn::audio::InitializeFinalOutputRecorderParameter(&parameter);
    ScopedGameRecord gameRecord;
#if defined(NN_SDK_BUILD_RELEASE)
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::OpenFinalOutputRecorder(gameRecord.Get(), parameter));
    EXPECT_EQ(
        nn::audio::GetFinalOutputRecorderState(gameRecord.Get()), nn::audio::FinalOutputRecorderState_Stopped);

    //precondition violation - Stop FinalOutputRecorder when already stopped
    nn::audio::StopFinalOutputRecorder(gameRecord.Get());

    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::StartFinalOutputRecorder(gameRecord.Get()));
    nn::audio::StopFinalOutputRecorder(gameRecord.Get());

    //precondition violation
    nn::audio::StopFinalOutputRecorder(gameRecord.Get());
#else
    EXPECT_DEATH_IF_SUPPORTED(
        nn::audio::StopFinalOutputRecorder(gameRecord.Get()), "");
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::OpenFinalOutputRecorder(gameRecord.Get(), parameter));
    EXPECT_EQ(
        nn::audio::GetFinalOutputRecorderState(gameRecord.Get()), nn::audio::FinalOutputRecorderState_Stopped);
    EXPECT_DEATH_IF_SUPPORTED(
        nn::audio::StopFinalOutputRecorder(gameRecord.Get()), "");
#endif
}

TEST(GetFinalOutputRecorderSampleRate, Success)
{
    nn::audio::FinalOutputRecorderParameter parameter = { 0 };
    ScopedGameRecord gameRecord;
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::OpenFinalOutputRecorder(gameRecord.Get(), parameter));
    EXPECT_GT(
        nn::audio::GetFinalOutputRecorderSampleRate(gameRecord.Get()), 0);
}

TEST(GetFinalOutputRecorderChannelCount, Success)
{
    nn::audio::FinalOutputRecorderParameter parameter = { 0 };
    ScopedGameRecord gameRecord;
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::OpenFinalOutputRecorder(gameRecord.Get(), parameter));
    EXPECT_GT(
        nn::audio::GetFinalOutputRecorderChannelCount(gameRecord.Get()), 0);
}

TEST(GetFinalOutputRecorderSampleFormat, Success)
{
    nn::audio::FinalOutputRecorderParameter parameter = { 0 };
    ScopedGameRecord gameRecord;
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::OpenFinalOutputRecorder(gameRecord.Get(), parameter));
    EXPECT_NE(
        nn::audio::GetFinalOutputRecorderSampleFormat(gameRecord.Get()), nn::audio::SampleFormat_Invalid);
}

TEST(AppendFinalOutputRecorderBuffer, Success)
{
    nn::audio::FinalOutputRecorderParameter parameter = { 0 };
    ScopedGameRecord gameRecord;
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::OpenFinalOutputRecorder(gameRecord.Get(), parameter));

    nn::audio::FinalOutputRecorderBuffer gameRecordBuffer;
    nn::audio::SetFinalOutputRecorderBufferInfo(&gameRecordBuffer, g_Buffer, g_BufferSize, g_DataSize);
    nn::audio::AppendFinalOutputRecorderBuffer(gameRecord.Get(), &gameRecordBuffer);
    EXPECT_TRUE(nn::audio::ContainsFinalOutputRecorderBuffer(gameRecord.Get(), &gameRecordBuffer));
}

TEST(AppendAfterStartStopTest, Success)
{
    nn::audio::FinalOutputRecorderParameter parameter;
    nn::audio::InitializeFinalOutputRecorderParameter(&parameter);
    ScopedGameRecord gameRecord;
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::OpenFinalOutputRecorder(gameRecord.Get(), parameter));
    nn::audio::FinalOutputRecorderBuffer gameRecordBuffer;
    nn::audio::SetFinalOutputRecorderBufferInfo(&gameRecordBuffer, g_Buffer, sizeof(g_Buffer), sizeof(g_Buffer));

    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::StartFinalOutputRecorder(gameRecord.Get()));

    nn::audio::StopFinalOutputRecorder(gameRecord.Get());

    for(int i = 0; i < nn::audio::AudioInBufferCountMax; ++i)
    {
        nn::audio::AppendFinalOutputRecorderBuffer(gameRecord.Get(), &gameRecordBuffer);
    }

    nn::audio::CloseFinalOutputRecorder(gameRecord.Get());

    //Replaced thread creation with os call to make sure g_Buffer is not mapped
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    EXPECT_FALSE(nn::os::IsMemoryLocked(g_Buffer, sizeof(g_Buffer)));
#endif
}

TEST(GetReleasedFinalOutputRecorderBufferTest, Success)
{
    nn::audio::FinalOutputRecorderParameter parameter = { 0 };
    ScopedGameRecord gameRecord;
    nn::os::SystemEvent gameRecordBufferEvent;
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::OpenFinalOutputRecorder(gameRecord.Get(), &gameRecordBufferEvent, parameter));

    EXPECT_EQ(nn::audio::GetReleasedFinalOutputRecorderBuffer(gameRecord.Get()), nullptr);

    nn::audio::FinalOutputRecorderBuffer gameRecordBuffer;
    nn::audio::SetFinalOutputRecorderBufferInfo(&gameRecordBuffer, g_Buffer, g_BufferSize, g_DataSize);
    nn::audio::AppendFinalOutputRecorderBuffer(gameRecord.Get(), &gameRecordBuffer);

    EXPECT_EQ(nn::audio::GetReleasedFinalOutputRecorderBuffer(gameRecord.Get()), nullptr);

    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::StartFinalOutputRecorder(gameRecord.Get()));

    nn::audio::FinalOutputRecorderBuffer* pReleased = nullptr;
    while (pReleased == nullptr)
    {
        gameRecordBufferEvent.Wait();
        pReleased = nn::audio::GetReleasedFinalOutputRecorderBuffer(gameRecord.Get());
    }

    EXPECT_EQ(pReleased, &gameRecordBuffer);
    EXPECT_EQ(nn::audio::GetReleasedFinalOutputRecorderBuffer(gameRecord.Get()), nullptr);
}

TEST(MaxFinalOutputRecorderBufferAppendTest, Success)
{
    nn::audio::FinalOutputRecorderParameter parameter = { 0 };
    ScopedGameRecord gameRecord;
    nn::os::SystemEvent gameRecordBufferEvent;
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::OpenFinalOutputRecorder(gameRecord.Get(), &gameRecordBufferEvent, parameter));

    EXPECT_EQ(nn::audio::GetReleasedFinalOutputRecorderBuffer(gameRecord.Get()), nullptr);
    const int numBuffers = 16;
    nn::audio::FinalOutputRecorderBuffer gameRecordBuffer[numBuffers];
    for(int i = 0; i < numBuffers; ++i)
    {
        nn::audio::SetFinalOutputRecorderBufferInfo(&gameRecordBuffer[i], g_Buffer, g_BufferSize, g_DataSize);
        nn::audio::AppendFinalOutputRecorderBuffer(gameRecord.Get(), &gameRecordBuffer[i]);
    }

    EXPECT_EQ(nn::audio::GetReleasedFinalOutputRecorderBuffer(gameRecord.Get()), nullptr);

    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::StartFinalOutputRecorder(gameRecord.Get()));

    nn::audio::FinalOutputRecorderBuffer* pReleased = nullptr;
    unsigned int pReleasedCount = 0;
    while(pReleasedCount < numBuffers)
    {
        while (pReleased == nullptr)
        {
            gameRecordBufferEvent.Wait();
            pReleased = nn::audio::GetReleasedFinalOutputRecorderBuffer(gameRecord.Get());
        }
        ++pReleasedCount;
    }
}

TEST(GetReleasedFinalOutputRecorderBufferTimeTest, Success)
{
    nn::audio::FinalOutputRecorderParameter parameter = { 0 };
    ScopedGameRecord gameRecord;
    nn::os::SystemEvent gameRecordBufferEvent;
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::OpenFinalOutputRecorder(gameRecord.Get(), &gameRecordBufferEvent, parameter));

    EXPECT_EQ(nn::audio::GetReleasedFinalOutputRecorderBuffer(gameRecord.Get()), nullptr);

    nn::audio::FinalOutputRecorderBuffer gameRecordBuffer;
    nn::audio::SetFinalOutputRecorderBufferInfo(&gameRecordBuffer, g_Buffer, g_BufferSize, g_DataSize);
    int64_t releaseTime = nn::audio::GetFinalOutputRecorderBufferEndTime(&gameRecordBuffer).GetMilliSeconds();
    EXPECT_EQ(releaseTime, 0);
    nn::audio::AppendFinalOutputRecorderBuffer(gameRecord.Get(), &gameRecordBuffer);

    EXPECT_EQ(nn::audio::GetReleasedFinalOutputRecorderBuffer(gameRecord.Get()), nullptr);
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::StartFinalOutputRecorder(gameRecord.Get()));

    nn::audio::FinalOutputRecorderBuffer* pReleased = nullptr;
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(50));
    while (pReleased == nullptr)
    {
        gameRecordBufferEvent.Wait();
        pReleased = nn::audio::GetReleasedFinalOutputRecorderBuffer(gameRecord.Get());
    }
    releaseTime = nn::audio::GetFinalOutputRecorderBufferEndTime(&gameRecordBuffer).GetMilliSeconds();
    NN_SDK_LOG("Release Time: %lld\n", releaseTime);
    EXPECT_GT(releaseTime, 0);
    EXPECT_EQ(pReleased, &gameRecordBuffer);
    EXPECT_EQ(nn::audio::GetReleasedFinalOutputRecorderBuffer(gameRecord.Get()), nullptr);
    nn::audio::AppendFinalOutputRecorderBuffer(gameRecord.Get(), &gameRecordBuffer);
    releaseTime = nn::audio::GetFinalOutputRecorderBufferEndTime(&gameRecordBuffer).GetMilliSeconds();
    EXPECT_EQ(releaseTime, 0);
}

TEST(ContainsFinalOutputRecorderBufferTest, Success)
{
    nn::audio::FinalOutputRecorderParameter parameter = { 0 };
    ScopedGameRecord gameRecord;
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::OpenFinalOutputRecorder(gameRecord.Get(), parameter));
    nn::audio::FinalOutputRecorderBuffer gameRecordBuffer;
    nn::audio::SetFinalOutputRecorderBufferInfo(&gameRecordBuffer, g_Buffer, g_BufferSize, g_DataSize);

    EXPECT_FALSE(nn::audio::ContainsFinalOutputRecorderBuffer(gameRecord.Get(), &gameRecordBuffer));

    nn::audio::AppendFinalOutputRecorderBuffer(gameRecord.Get(), &gameRecordBuffer);

    EXPECT_TRUE(nn::audio::ContainsFinalOutputRecorderBuffer(gameRecord.Get(), &gameRecordBuffer));

    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::StartFinalOutputRecorder(gameRecord.Get()));

    nn::audio::FinalOutputRecorderBuffer* pReleased = nullptr;
    while (pReleased == nullptr)
    {
        pReleased = nn::audio::GetReleasedFinalOutputRecorderBuffer(gameRecord.Get());
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));
    }

    EXPECT_FALSE(nn::audio::ContainsFinalOutputRecorderBuffer(gameRecord.Get(), &gameRecordBuffer));
}

TEST(RegisterApplet, Success)
{
    auto appletId = nn::applet::GetAppletResourceUserId();
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RegisterAppletResourceUserId(appletId));
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::UnregisterAppletResourceUserId(appletId));
}

TEST(RegisterApplet, Failure)
{
    auto appletId = nn::applet::AppletResourceUserId::GetInvalidId();
    NNT_EXPECT_RESULT_FAILURE(nn::audio::ResultAppletResourceUserIdNotFound, nn::audio::UnregisterAppletResourceUserId(appletId));
}

TEST(RecordedBufferComparisonTest, Success)
{
    nn::mem::StandardAllocator alloc(g_TestBuffer, sizeof(g_TestBuffer));
    nn::audio::FinalOutputRecorderParameter parameter = { 0 };
    nn::audio::AudioOutParameter paramAudioOut = { 0 };
    const size_t dataBufferSize = 96000;
    int16_t* gameRecordData = reinterpret_cast<int16_t*>(alloc.Allocate((dataBufferSize) * sizeof(int16_t)));
    int16_t* audioOutData = reinterpret_cast<int16_t*>(alloc.Allocate((dataBufferSize) * sizeof(int16_t)));
    memset(gameRecordData, 0, dataBufferSize * sizeof(int16_t));
    memset(audioOutData, 0, dataBufferSize * sizeof(int16_t));
    intptr_t gameRecordDataPosition = reinterpret_cast<intptr_t>(gameRecordData);
    intptr_t audioOutDataPosition = reinterpret_cast<intptr_t>(audioOutData);
    uint64_t gameRecordDataOffset = 0;
    uint64_t audioOutDataOffset = 0;
    const int bufferCount = 3;
    int bufferSize = g_BufferSize;
    void* grBuffers[bufferCount];
    void* aoBuffers[bufferCount];
    ScopedGameRecord gameRecord;
    nn::audio::AudioOut audioOut;
    nn::os::SystemEvent audioOutBufferEvent;
    nn::os::SystemEvent gameRecordBufferEvent;
    nn::audio::InitializeAudioOutParameter(&paramAudioOut);
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::OpenFinalOutputRecorder(gameRecord.Get(), &gameRecordBufferEvent, parameter));
    if (nn::audio::OpenDefaultAudioOut(&audioOut, &audioOutBufferEvent, paramAudioOut).IsFailure())
    {
        paramAudioOut.sampleRate = 0;
        NN_ABORT_UNLESS(
            nn::audio::OpenDefaultAudioOut(&audioOut, &audioOutBufferEvent, paramAudioOut).IsSuccess(),
            "Failed to open AudioOut."
        );
    }
    nn::audio::FinalOutputRecorderBuffer gameRecordBuffer[bufferCount];
    nn::audio::AudioOutBuffer audioOutBuffer[bufferCount];
    nn::audio::SampleFormat sampleFormat = nn::audio::GetAudioOutSampleFormat(&audioOut);
    int sampleRate = nn::audio::GetAudioOutSampleRate(&audioOut);
    const int frameRate = 20;
    const int frameSampleCount = sampleRate / frameRate;
    const int amplitude = std::numeric_limits<int16_t>::max() / 16;
    int channelCount = nn::audio::GetAudioOutChannelCount(&audioOut);
    const size_t dataSize = frameSampleCount * channelCount * nn::audio::GetSampleByteSize(sampleFormat);
    for(int i = 0; i < bufferCount; ++i)
    {
        grBuffers[i] = alloc.Allocate((bufferSize));
        aoBuffers[i] = alloc.Allocate((bufferSize));

        NN_ABORT_UNLESS(grBuffers[i] != nullptr, "Failed to allocate game record buffer\n");
        NN_ABORT_UNLESS(aoBuffers[i] != nullptr, "Failed to allocate audio out buffer\n");

        nn::audio::SetFinalOutputRecorderBufferInfo(&gameRecordBuffer[i], grBuffers[i], bufferSize, dataSize);
        nn::audio::SetAudioOutBufferInfo(&audioOutBuffer[i], aoBuffers[i], bufferSize, dataSize);

        GenerateSquareWaveInt16(aoBuffers[i], channelCount, sampleRate, frameSampleCount, amplitude);
        nn::audio::AppendAudioOutBuffer(&audioOut, &audioOutBuffer[i]);
        nn::audio::AppendFinalOutputRecorderBuffer(gameRecord.Get(), &gameRecordBuffer[i]);
    }

    NNT_EXPECT_RESULT_SUCCESS( nn::audio::StartFinalOutputRecorder(gameRecord.Get()));
    NNT_EXPECT_RESULT_SUCCESS( nn::audio::StartAudioOut(&audioOut) );
    nn::audio::FinalOutputRecorderBuffer* pGameRecordBuffer = nullptr;
    nn::audio::AudioOutBuffer* pAudioOutBuffer = nullptr;

    while (gameRecordDataOffset < dataBufferSize && audioOutDataOffset < dataBufferSize)
    {
        gameRecordBufferEvent.Wait();
        while((pGameRecordBuffer = nn::audio::GetReleasedFinalOutputRecorderBuffer(gameRecord.Get())) != nullptr)
        {
            size_t gameRecordDataSize = nn::audio::GetFinalOutputRecorderBufferDataSize(pGameRecordBuffer);
            void* gameRecordDataBuffer = nn::audio::GetFinalOutputRecorderBufferDataPointer(pGameRecordBuffer);
            uint64_t gameRecordDataCurrentPosition = gameRecordDataPosition + gameRecordDataOffset;
            void* gameRecordCopyPosition = reinterpret_cast<void*>( gameRecordDataCurrentPosition);
            memcpy(gameRecordCopyPosition, gameRecordDataBuffer, gameRecordDataSize);
            gameRecordDataOffset += gameRecordDataSize;
            nn::audio::AppendFinalOutputRecorderBuffer(gameRecord.Get(), pGameRecordBuffer);
        }

        while((pAudioOutBuffer = nn::audio::GetReleasedAudioOutBuffer(&audioOut)) != nullptr)
        {
            size_t audioOutDataSize = nn::audio::GetAudioOutBufferDataSize(pAudioOutBuffer);
            void* audioOutDataBuffer = nn::audio::GetAudioOutBufferDataPointer(pAudioOutBuffer);
            uint64_t audioOutDataCurrentPosition = audioOutDataPosition + audioOutDataOffset;
            void* audioOutCopyPosition = reinterpret_cast<void*>(audioOutDataCurrentPosition);
            memcpy(audioOutCopyPosition, audioOutDataBuffer, audioOutDataSize);
            audioOutDataOffset += audioOutDataSize;
            GenerateSquareWaveInt16(reinterpret_cast<void*>(audioOutDataBuffer), channelCount, sampleRate, frameSampleCount, amplitude);
            nn::audio::SetAudioOutBufferInfo(pAudioOutBuffer, audioOutDataBuffer, bufferSize, audioOutDataSize);
            nn::audio::AppendAudioOutBuffer(&audioOut, pAudioOutBuffer);
        }
    }

    for(int i = 0; i < bufferCount; ++i)
    {
        if(nn::audio::ContainsAudioOutBuffer(&audioOut, &audioOutBuffer[i]))
        {
            pAudioOutBuffer = nn::audio::GetReleasedAudioOutBuffer(&audioOut);
            while(pAudioOutBuffer == nullptr)
            {
                audioOutBufferEvent.Wait();
                pAudioOutBuffer = nn::audio::GetReleasedAudioOutBuffer(&audioOut);
            }
            if(audioOutDataOffset < dataBufferSize)
            {
                size_t audioOutDataSize = nn::audio::GetAudioOutBufferDataSize(pAudioOutBuffer);
                void* audioOutDataBuffer = nn::audio::GetAudioOutBufferDataPointer(pAudioOutBuffer);
                intptr_t audioOutDataCurrentPosition = audioOutDataPosition + audioOutDataOffset;
                void* audioOutCopyPosition = reinterpret_cast<void*>(audioOutDataCurrentPosition);
                memcpy(audioOutCopyPosition, audioOutDataBuffer, audioOutDataSize);
                audioOutDataOffset += audioOutDataSize;
            }
            else
            {
                break;
            }
        }
        if(nn::audio::ContainsFinalOutputRecorderBuffer(gameRecord.Get(), &gameRecordBuffer[i]))
        {
            pGameRecordBuffer = nn::audio::GetReleasedFinalOutputRecorderBuffer(gameRecord.Get());
            while(pGameRecordBuffer == nullptr)
            {
                gameRecordBufferEvent.Wait();
                pGameRecordBuffer = nn::audio::GetReleasedFinalOutputRecorderBuffer(gameRecord.Get());
            }
            if(gameRecordDataOffset < dataBufferSize)
            {
                size_t gameRecordDataSize = nn::audio::GetFinalOutputRecorderBufferDataSize(pGameRecordBuffer);
                void* gameRecordDataBuffer = nn::audio::GetFinalOutputRecorderBufferDataPointer(pGameRecordBuffer);
                intptr_t gameRecordDataCurrentPosition = gameRecordDataPosition + gameRecordDataOffset;
                void* gameRecordCopyPosition = reinterpret_cast<void*>(gameRecordDataCurrentPosition);
                memcpy(gameRecordCopyPosition, gameRecordDataBuffer, gameRecordDataSize);
                gameRecordDataOffset += gameRecordDataSize;
            }
            else
            {
                break;
            }
        }
    }
    int offset = -1;
    for (int i = 0; i < dataBufferSize / sizeof(int16_t); ++i)
    {
        if (offset == -1 && gameRecordData[i] != 0)
        {
            offset = i;
        }
        if (offset != -1)
        {
            EXPECT_EQ(gameRecordData[i], audioOutData[i - offset]);
            if(gameRecordData[i] != audioOutData[i - offset])
            {
                NN_SDK_LOG("i: %d, gr offset: %d, ao offset: %d, data offset: %d\n", i, gameRecordDataOffset, audioOutDataOffset, offset);
                break;
            }
        }
    }
    EXPECT_LE(offset, channelCount * (sampleRate / g_audioFramesPerSecond));
    for(int i = 0; i < bufferCount; ++i)
    {
        if(grBuffers[i] != nullptr)
        {
            alloc.Free(grBuffers[i]);
        }
        if(aoBuffers[i] != nullptr)
        {
            alloc.Free(aoBuffers[i]);
        }
    }
    alloc.Free(gameRecordData);
    alloc.Free(audioOutData);
    nn::audio::StopAudioOut(&audioOut);
    nn::audio::CloseAudioOut(&audioOut);
}//NOLINT(impl/function_size)

TEST(RecordedBufferComparisonTest6ch, Success)
{
    nn::mem::StandardAllocator alloc(g_TestBuffer, sizeof(g_TestBuffer));
    nn::audio::FinalOutputRecorderParameter parameter = { 0 };
    nn::audio::AudioOutParameter paramAudioOut = { 0 };
    const size_t dataBufferSize = 96000;
    int16_t* gameRecordData = reinterpret_cast<int16_t*>(alloc.Allocate((dataBufferSize) * sizeof(int16_t)));
    int16_t* audioOutData = reinterpret_cast<int16_t*>(alloc.Allocate((dataBufferSize) * sizeof(int16_t)));
    memset(gameRecordData, 0, dataBufferSize * sizeof(int16_t));
    memset(audioOutData, 0, dataBufferSize * sizeof(int16_t));
    intptr_t gameRecordDataPosition = reinterpret_cast<intptr_t>(gameRecordData);
    intptr_t audioOutDataPosition = reinterpret_cast<intptr_t>(audioOutData);
    uint64_t gameRecordDataOffset = 0;
    uint64_t audioOutDataOffset = 0;
    const int bufferCount = 3;
    void* grBuffers[bufferCount];
    void* aoBuffers[bufferCount];
    ScopedGameRecord gameRecord;
    nn::audio::AudioOut audioOut;
    nn::os::SystemEvent audioOutBufferEvent;
    nn::os::SystemEvent gameRecordBufferEvent;
    nn::audio::InitializeAudioOutParameter(&paramAudioOut);
    nn::audio::InitializeFinalOutputRecorderParameter(&parameter);
    int channelCount = 6;
    paramAudioOut.channelCount = channelCount;
    parameter.channelCount = channelCount;
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::OpenFinalOutputRecorder(gameRecord.Get(), &gameRecordBufferEvent, parameter));
    if (nn::audio::OpenDefaultAudioOut(&audioOut, &audioOutBufferEvent, paramAudioOut).IsFailure())
    {
        paramAudioOut.sampleRate = 0;
        NN_ABORT_UNLESS(
            nn::audio::OpenDefaultAudioOut(&audioOut, &audioOutBufferEvent, paramAudioOut).IsSuccess(),
            "Failed to open AudioOut."
        );
    }

    nn::audio::SampleFormat sampleFormat = nn::audio::GetAudioOutSampleFormat(&audioOut);
    int sampleRate = nn::audio::GetAudioOutSampleRate(&audioOut);
    const int frameRate = 20;
    const int frameSampleCount = sampleRate / frameRate;
    const int amplitude = std::numeric_limits<int16_t>::max() / 16;
    const size_t dataSize = frameSampleCount * channelCount * nn::audio::GetSampleByteSize(sampleFormat);
    const size_t bufferSize = nn::util::align_up(dataSize, nn::audio::FinalOutputRecorderBuffer::SizeGranularity);

    nn::audio::FinalOutputRecorderBuffer gameRecordBuffer[bufferCount];
    nn::audio::AudioOutBuffer audioOutBuffer[bufferCount];
    for(int i = 0; i < bufferCount; ++i)
    {
        grBuffers[i] = alloc.Allocate((bufferSize));
        aoBuffers[i] = alloc.Allocate((bufferSize));

        NN_ABORT_UNLESS(grBuffers[i] != nullptr, "Failed to allocate game record buffer\n");
        NN_ABORT_UNLESS(aoBuffers[i] != nullptr, "Failed to allocate audio out buffer\n");

        nn::audio::SetFinalOutputRecorderBufferInfo(&gameRecordBuffer[i], grBuffers[i], bufferSize, dataSize);
        nn::audio::SetAudioOutBufferInfo(&audioOutBuffer[i], aoBuffers[i], bufferSize, dataSize);

        GenerateSquareWaveInt16(aoBuffers[i], channelCount, sampleRate, frameSampleCount, amplitude);
        nn::audio::AppendAudioOutBuffer(&audioOut, &audioOutBuffer[i]);
        nn::audio::AppendFinalOutputRecorderBuffer(gameRecord.Get(), &gameRecordBuffer[i]);
    }

    NNT_EXPECT_RESULT_SUCCESS(nn::audio::StartFinalOutputRecorder(gameRecord.Get()));
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::StartAudioOut(&audioOut));
    nn::audio::FinalOutputRecorderBuffer* pGameRecordBuffer = nullptr;
    nn::audio::AudioOutBuffer* pAudioOutBuffer = nullptr;

    while (gameRecordDataOffset < dataBufferSize && audioOutDataOffset < dataBufferSize)
    {
        gameRecordBufferEvent.Wait();
        while((pGameRecordBuffer = nn::audio::GetReleasedFinalOutputRecorderBuffer(gameRecord.Get())) != nullptr)
        {
            if(gameRecordDataOffset < dataBufferSize)
            {
                size_t gameRecordDataSize = nn::audio::GetFinalOutputRecorderBufferDataSize(pGameRecordBuffer);
                void* gameRecordDataBuffer = nn::audio::GetFinalOutputRecorderBufferDataPointer(pGameRecordBuffer);
                intptr_t gameRecordDataCurrentPosition = gameRecordDataPosition + gameRecordDataOffset;
                void* gameRecordCopyPosition = reinterpret_cast<void*>(gameRecordDataCurrentPosition);
                memcpy(gameRecordCopyPosition, gameRecordDataBuffer, gameRecordDataSize);
                gameRecordDataOffset += gameRecordDataSize;
                nn::audio::AppendFinalOutputRecorderBuffer(gameRecord.Get(), pGameRecordBuffer);
            }
        }
        audioOutBufferEvent.Wait();
        while((pAudioOutBuffer = nn::audio::GetReleasedAudioOutBuffer(&audioOut)) != nullptr)
        {
            if(audioOutDataOffset < dataBufferSize)
            {
                size_t audioOutDataSize = nn::audio::GetAudioOutBufferDataSize(pAudioOutBuffer);
                void* audioOutDataBuffer = nn::audio::GetAudioOutBufferDataPointer(pAudioOutBuffer);
                intptr_t audioOutDataCurrentPosition = audioOutDataPosition + audioOutDataOffset;
                void* audioOutCopyPosition = reinterpret_cast<void*>(audioOutDataCurrentPosition);
                memcpy(audioOutCopyPosition, audioOutDataBuffer, audioOutDataSize);
                audioOutDataOffset += audioOutDataSize;
                GenerateSquareWaveInt16(reinterpret_cast<void*>(audioOutDataBuffer), channelCount, sampleRate, frameSampleCount, amplitude);
                nn::audio::SetAudioOutBufferInfo(pAudioOutBuffer, audioOutDataBuffer, bufferSize, audioOutDataSize);
                nn::audio::AppendAudioOutBuffer(&audioOut, pAudioOutBuffer);
            }
        }
    }
    for(int i = 0; i < bufferCount; ++i)
    {
        if(nn::audio::ContainsAudioOutBuffer(&audioOut, &audioOutBuffer[i]))
        {
            pAudioOutBuffer = nn::audio::GetReleasedAudioOutBuffer(&audioOut);
            while(pAudioOutBuffer == nullptr)
            {
                audioOutBufferEvent.Wait();
                pAudioOutBuffer = nn::audio::GetReleasedAudioOutBuffer(&audioOut);
            }
            if(audioOutDataOffset < dataBufferSize)
            {
                size_t audioOutDataSize = nn::audio::GetAudioOutBufferDataSize(pAudioOutBuffer);
                void* audioOutDataBuffer = nn::audio::GetAudioOutBufferDataPointer(pAudioOutBuffer);
                intptr_t audioOutDataCurrentPosition = audioOutDataPosition + audioOutDataOffset;
                void* audioOutCopyPosition = reinterpret_cast<void*>(audioOutDataCurrentPosition);
                memcpy(audioOutCopyPosition, audioOutDataBuffer, audioOutDataSize);
                audioOutDataOffset += audioOutDataSize;
            }
            else
            {
                break;
            }
        }
        if(nn::audio::ContainsFinalOutputRecorderBuffer(gameRecord.Get(), &gameRecordBuffer[i]))
        {
            pGameRecordBuffer = nn::audio::GetReleasedFinalOutputRecorderBuffer(gameRecord.Get());
            while(pGameRecordBuffer == nullptr)
            {
                gameRecordBufferEvent.Wait();
                pGameRecordBuffer = nn::audio::GetReleasedFinalOutputRecorderBuffer(gameRecord.Get());
            }
            if(gameRecordDataOffset < dataBufferSize)
            {
                size_t gameRecordDataSize = nn::audio::GetFinalOutputRecorderBufferDataSize(pGameRecordBuffer);
                void* gameRecordDataBuffer = nn::audio::GetFinalOutputRecorderBufferDataPointer(pGameRecordBuffer);
                intptr_t gameRecordDataCurrentPosition = gameRecordDataPosition + gameRecordDataOffset;
                void* gameRecordCopyPosition = reinterpret_cast<void*>(gameRecordDataCurrentPosition);
                memcpy(gameRecordCopyPosition, gameRecordDataBuffer, gameRecordDataSize);
                gameRecordDataOffset += gameRecordDataSize;
            }
            else
            {
                break;
            }
        }
    }

    int offset = -1;
    for (int i = 0; i < dataBufferSize / sizeof(int16_t); ++i)
    {
        if (offset == -1 && gameRecordData[i] != 0)
        {
            offset = i;
        }
        if (offset != -1)
        {
            EXPECT_EQ(gameRecordData[i], audioOutData[i - offset]);
            if(gameRecordData[i] != audioOutData[i - offset])
            {
                NN_SDK_LOG("i: %d, gr offset: %d, ao offset: %d\n", i, gameRecordDataOffset, audioOutDataOffset);
                break;
            }
        }
    }
    EXPECT_LE(offset, channelCount * (sampleRate / g_audioFramesPerSecond));
    for(int i = 0; i < bufferCount; ++i)
    {
        if(grBuffers[i] != nullptr)
        {
            alloc.Free(grBuffers[i]);
        }
        if(aoBuffers[i] != nullptr)
        {
            alloc.Free(aoBuffers[i]);
        }
    }
    alloc.Free(gameRecordData);
    alloc.Free(audioOutData);
    nn::audio::StopAudioOut(&audioOut);
    nn::audio::CloseAudioOut(&audioOut);
}//NOLINT(impl/function_size)

TEST(RecordAfterStartStop, Success)
{
    nn::mem::StandardAllocator alloc(g_TestBuffer, sizeof(g_TestBuffer));
    nn::audio::FinalOutputRecorderParameter parameter = { 0 };
    nn::audio::AudioOutParameter paramAudioOut = { 0 };
    const size_t dataBufferSize = 96000;
    int16_t* gameRecordData = reinterpret_cast<int16_t*>(alloc.Allocate((dataBufferSize) * sizeof(int16_t)));
    int16_t* audioOutData = reinterpret_cast<int16_t*>(alloc.Allocate((dataBufferSize) * sizeof(int16_t)));
    memset(gameRecordData, 0, dataBufferSize * sizeof(int16_t));
    memset(audioOutData, 0, dataBufferSize * sizeof(int16_t));
    intptr_t gameRecordDataPosition = reinterpret_cast<intptr_t>(gameRecordData);
    intptr_t audioOutDataPosition = reinterpret_cast<intptr_t>(audioOutData);
    uint64_t gameRecordDataOffset = 0;
    uint64_t audioOutDataOffset = 0;
    const int bufferCount = 3;
    void* grBuffers[bufferCount];
    void* aoBuffers[bufferCount];
    ScopedGameRecord gameRecord;
    nn::audio::AudioOut audioOut;
    nn::os::SystemEvent audioOutBufferEvent;
    nn::os::SystemEvent gameRecordBufferEvent;
    int channelCount = 6;
    nn::audio::InitializeFinalOutputRecorderParameter(&parameter);
    parameter.channelCount = channelCount;
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::OpenFinalOutputRecorder(gameRecord.Get(), &gameRecordBufferEvent, parameter));
    for(int i = 0; i < 16; ++i)
    {
        NNT_EXPECT_RESULT_SUCCESS(
            nn::audio::StartFinalOutputRecorder(gameRecord.Get()));
        nn::audio::StopFinalOutputRecorder(gameRecord.Get());
    }
    EXPECT_EQ(
        nn::audio::GetFinalOutputRecorderState(gameRecord.Get()), nn::audio::FinalOutputRecorderState_Stopped);
    nn::audio::InitializeAudioOutParameter(&paramAudioOut);
    paramAudioOut.channelCount = channelCount;
    if (nn::audio::OpenDefaultAudioOut(&audioOut, &audioOutBufferEvent, paramAudioOut).IsFailure())
    {
        paramAudioOut.sampleRate = 0;
        NN_ABORT_UNLESS(
            nn::audio::OpenDefaultAudioOut(&audioOut, &audioOutBufferEvent, paramAudioOut).IsSuccess(),
            "Failed to open AudioOut."
        );
    }

    nn::audio::SampleFormat sampleFormat = nn::audio::GetAudioOutSampleFormat(&audioOut);
    int sampleRate = nn::audio::GetAudioOutSampleRate(&audioOut);
    const int frameRate = 20;
    const int frameSampleCount = sampleRate / frameRate;
    const int amplitude = std::numeric_limits<int16_t>::max() / 16;
    const size_t dataSize = frameSampleCount * channelCount * nn::audio::GetSampleByteSize(sampleFormat);
    const size_t bufferSize = nn::util::align_up(dataSize, nn::audio::FinalOutputRecorderBuffer::SizeGranularity);
    nn::audio::FinalOutputRecorderBuffer gameRecordBuffer[bufferCount];
    nn::audio::AudioOutBuffer audioOutBuffer[bufferCount];

    for(int i = 0; i < bufferCount; ++i)
    {
        grBuffers[i] = alloc.Allocate((bufferSize));
        aoBuffers[i] = alloc.Allocate((bufferSize));

        NN_ABORT_UNLESS(grBuffers[i] != nullptr, "Failed to allocate game record buffer\n");
        NN_ABORT_UNLESS(aoBuffers[i] != nullptr, "Failed to allocate audio out buffer\n");

        nn::audio::SetFinalOutputRecorderBufferInfo(&gameRecordBuffer[i], grBuffers[i], bufferSize, dataSize);
        nn::audio::SetAudioOutBufferInfo(&audioOutBuffer[i], aoBuffers[i], bufferSize, dataSize);

        GenerateSquareWaveInt16(aoBuffers[i], channelCount, sampleRate, frameSampleCount, amplitude);
        nn::audio::AppendAudioOutBuffer(&audioOut, &audioOutBuffer[i]);
        nn::audio::AppendFinalOutputRecorderBuffer(gameRecord.Get(), &gameRecordBuffer[i]);
    }
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::StartFinalOutputRecorder(gameRecord.Get()));
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::StartAudioOut(&audioOut));
    nn::audio::AudioOutBuffer* pAudioOutBuffer = nullptr;
    nn::audio::FinalOutputRecorderBuffer* pGameRecordBuffer = nullptr;
    while (gameRecordDataOffset < dataBufferSize && audioOutDataOffset < dataBufferSize)
    {
        gameRecordBufferEvent.Wait();
        if((pGameRecordBuffer = nn::audio::GetReleasedFinalOutputRecorderBuffer(gameRecord.Get())) != nullptr)
        {
            size_t gameRecordDataSize = nn::audio::GetFinalOutputRecorderBufferDataSize(pGameRecordBuffer);
            void* gameRecordDataBuffer = nn::audio::GetFinalOutputRecorderBufferDataPointer(pGameRecordBuffer);
            uint64_t gameRecordDataCurrentPosition = gameRecordDataPosition + gameRecordDataOffset;
            void* gameRecordCopyPosition = reinterpret_cast<void*>(gameRecordDataCurrentPosition);
            memcpy(gameRecordCopyPosition, gameRecordDataBuffer, gameRecordDataSize);
            gameRecordDataOffset += gameRecordDataSize;
            nn::audio::AppendFinalOutputRecorderBuffer(gameRecord.Get(), pGameRecordBuffer);
        }
        audioOutBufferEvent.Wait();
        if((pAudioOutBuffer = nn::audio::GetReleasedAudioOutBuffer(&audioOut)) != nullptr)
        {
            size_t audioOutDataSize = nn::audio::GetAudioOutBufferDataSize(pAudioOutBuffer);
            void* audioOutDataBuffer = nn::audio::GetAudioOutBufferDataPointer(pAudioOutBuffer);
            uint64_t audioOutDataCurrentPosition = audioOutDataPosition + audioOutDataOffset;
            void* audioOutCopyPosition = reinterpret_cast<void*>(audioOutDataCurrentPosition);
            memcpy(audioOutCopyPosition, audioOutDataBuffer, audioOutDataSize);
            audioOutDataOffset += audioOutDataSize;
            GenerateSquareWaveInt16(reinterpret_cast<void*>(audioOutDataBuffer), channelCount, sampleRate, frameSampleCount, amplitude);
            nn::audio::SetAudioOutBufferInfo(pAudioOutBuffer, audioOutDataBuffer, bufferSize, audioOutDataSize);
            nn::audio::AppendAudioOutBuffer(&audioOut, pAudioOutBuffer);
        }
    }
    for(int i = 0; i < bufferCount; ++i)
    {
        if(nn::audio::ContainsAudioOutBuffer(&audioOut, &audioOutBuffer[i]))
        {
            pAudioOutBuffer = nn::audio::GetReleasedAudioOutBuffer(&audioOut);
            while(pAudioOutBuffer == nullptr)
            {
                audioOutBufferEvent.Wait();
                pAudioOutBuffer = nn::audio::GetReleasedAudioOutBuffer(&audioOut);
            }
            if(audioOutDataOffset < dataBufferSize)
            {
                size_t audioOutDataSize = nn::audio::GetAudioOutBufferDataSize(pAudioOutBuffer);
                void* audioOutDataBuffer = nn::audio::GetAudioOutBufferDataPointer(pAudioOutBuffer);
                intptr_t audioOutDataCurrentPosition = audioOutDataPosition + audioOutDataOffset;
                void* audioOutCopyPosition = reinterpret_cast<void*>(audioOutDataCurrentPosition);
                memcpy(audioOutCopyPosition, audioOutDataBuffer, audioOutDataSize);
                audioOutDataOffset += audioOutDataSize;
            }
            else
            {
                break;
            }
        }
        if(nn::audio::ContainsFinalOutputRecorderBuffer(gameRecord.Get(), &gameRecordBuffer[i]))
        {
            pGameRecordBuffer = nn::audio::GetReleasedFinalOutputRecorderBuffer(gameRecord.Get());
            while(pGameRecordBuffer == nullptr)
            {
                gameRecordBufferEvent.Wait();
                pGameRecordBuffer = nn::audio::GetReleasedFinalOutputRecorderBuffer(gameRecord.Get());
            }
            if(gameRecordDataOffset < dataBufferSize)
            {
                size_t gameRecordDataSize = nn::audio::GetFinalOutputRecorderBufferDataSize(pGameRecordBuffer);
                void* gameRecordDataBuffer = nn::audio::GetFinalOutputRecorderBufferDataPointer(pGameRecordBuffer);
                intptr_t gameRecordDataCurrentPosition = gameRecordDataPosition + gameRecordDataOffset;
                void* gameRecordCopyPosition = reinterpret_cast<void*>(gameRecordDataCurrentPosition);
                memcpy(gameRecordCopyPosition, gameRecordDataBuffer, gameRecordDataSize);
                gameRecordDataOffset += gameRecordDataSize;
            }
            else
            {
                break;
            }
        }
    }

    int offset = -1;
    for (int i = 0; i < dataBufferSize / sizeof(int16_t); ++i)
    {
        if (offset == -1 && gameRecordData[i] != 0)
        {
            offset = i;
        }
        if (offset != -1)
        {
            EXPECT_EQ(gameRecordData[i], audioOutData[i - offset]);
            if(gameRecordData[i] != audioOutData[i - offset])
            {
                NN_SDK_LOG("i: %d, gr offset: %d, ao offset: %d\n", i, gameRecordDataOffset, audioOutDataOffset);
                break;
            }
        }
    }
    EXPECT_LE(offset, channelCount * (sampleRate / g_audioFramesPerSecond));
    for(int i = 0; i < bufferCount; ++i)
    {
        if(grBuffers[i] != nullptr)
        {
            alloc.Free(grBuffers[i]);
        }
        if(aoBuffers[i] != nullptr)
        {
            alloc.Free(aoBuffers[i]);
        }
    }
    alloc.Free(gameRecordData);
    alloc.Free(audioOutData);
    nn::audio::StopAudioOut(&audioOut);
    nn::audio::CloseAudioOut(&audioOut);
} //NOLINT(impl/function_size)

TEST(OpenDuplicateFinalOutputRecorderTest, Error)
{
    nn::audio::FinalOutputRecorderParameter parameter = { 0 };
    nn::audio::InitializeFinalOutputRecorderParameter(&parameter);

    {
        ScopedGameRecord gameRecord;
        NNT_EXPECT_RESULT_SUCCESS(
            nn::audio::OpenFinalOutputRecorder(gameRecord.Get(), parameter));
        NN_TEST_EXPECT_DEATH_OR_RESULT(
            nn::audio::OpenFinalOutputRecorder(gameRecord.Get(), parameter), nn::audio::ResultAlreadyOpen);
    }
    {
        ScopedGameRecord gameRecord;
        nn::os::SystemEvent event;
        NNT_EXPECT_RESULT_SUCCESS(
            nn::audio::OpenFinalOutputRecorder(gameRecord.Get(), &event, parameter));
        NN_TEST_EXPECT_DEATH_OR_RESULT(
            nn::audio::OpenFinalOutputRecorder(gameRecord.Get(), &event, parameter), nn::audio::ResultAlreadyOpen);
        nn::os::DestroySystemEvent(event.GetBase());
    }
    {
        ScopedGameRecord gameRecord;
        NNT_EXPECT_RESULT_SUCCESS(
            nn::audio::OpenFinalOutputRecorder(gameRecord.Get(), parameter));
        NN_TEST_EXPECT_DEATH_OR_RESULT(
            nn::audio::OpenFinalOutputRecorder(gameRecord.Get(), parameter), nn::audio::ResultAlreadyOpen);
    }
    {
        ScopedGameRecord gameRecord;
        nn::os::SystemEvent event;
        NNT_EXPECT_RESULT_SUCCESS(
            nn::audio::OpenFinalOutputRecorder(gameRecord.Get(), &event, parameter));
        NN_TEST_EXPECT_DEATH_OR_RESULT(
            nn::audio::OpenFinalOutputRecorder(gameRecord.Get(), &event, parameter), nn::audio::ResultAlreadyOpen);
        nn::os::DestroySystemEvent(event.GetBase());
    }
}
