﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testAudio_Effect.h"

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/audio.h>
#include <nnt/audioUtil/testAudio_Util.h>
#include <nnt/audioUtil/testAudio_Constants.h>
#include <nn/fs/fs_Result.h>

#include <memory> // std::unique_ptr

#include "../../Programs/Eris/Sources/Libraries/audio/dsp/audio_Qf.h" // QF_FRACTIONAL_BIT_COUNT
#include <cmath> //pow


namespace
{
    const std::string g_AudioTestSourceFileName = "/shutter.wav";
    const std::string g_AudioTestDelayed1chFileName = "/testAudio_Effect_Reverb1chShutter.wav";
    const std::string g_AudioTestDelayed2chFileName = "/testAudio_Effect_Reverb2chShutter.wav";
    const std::string g_AudioTestDelayed4chFileName = "/testAudio_Effect_Reverb4chShutter.wav";
    const std::string g_AudioTestDelayed6chFileName = "/testAudio_Effect_Reverb6chShutter.wav";
    const float g_AudioTestQfPrecision = 1.0f / std::powf(2, static_cast<float>(nn::audio::QF_FRACTIONAL_BIT_COUNT));

    NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN char g_WorkBuffer[2 * 1024 * 1024];
    NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN char g_WorkBuffer2[1024 * 1024];

    char g_RecorderBuffer[16 * 1024 * 1024];
}
// ========================================================================================
// Test cases
// ========================================================================================

namespace nnt {
namespace audio {

using namespace nn::audio;


#if !defined(NN_SDK_BUILD_RELEASE)
namespace {
void ReverbDeathTests(ReverbType* pReverb, ReverbType::EarlyMode earlyMode, ReverbType::LateMode lateMode, float gain, float ratio, float value, nn::TimeSpan time, bool enable) NN_NOEXCEPT
{
    { // Deatch Check
        EXPECT_DEATH_IF_SUPPORTED(SetReverbEarlyMode(pReverb, earlyMode), "");
        EXPECT_DEATH_IF_SUPPORTED(GetReverbEarlyMode(pReverb), "");
        EXPECT_DEATH_IF_SUPPORTED(GetReverbEarlyGain(pReverb), "");
        EXPECT_DEATH_IF_SUPPORTED(SetReverbEarlyGain(pReverb, gain) ,"");
        EXPECT_DEATH_IF_SUPPORTED(GetReverbPredelayTime(pReverb) ,"");
        EXPECT_DEATH_IF_SUPPORTED(SetReverbPredelayTime(pReverb, time) ,"");
        EXPECT_DEATH_IF_SUPPORTED(GetReverbLateMode(pReverb) ,"");
        EXPECT_DEATH_IF_SUPPORTED(SetReverbLateMode(pReverb, lateMode) ,"");
        EXPECT_DEATH_IF_SUPPORTED(GetReverbLateGain(pReverb) ,"");
        EXPECT_DEATH_IF_SUPPORTED(SetReverbLateGain(pReverb, gain) ,"");
        EXPECT_DEATH_IF_SUPPORTED(GetReverbDecayTime(pReverb) ,"");
        EXPECT_DEATH_IF_SUPPORTED(SetReverbDecayTime(pReverb, time) ,"");
        EXPECT_DEATH_IF_SUPPORTED(GetReverbHighFrequencyDecayRatio(pReverb) ,"");
        EXPECT_DEATH_IF_SUPPORTED(SetReverbHighFrequencyDecayRatio(pReverb, ratio) ,"");
        EXPECT_DEATH_IF_SUPPORTED(GetReverbColoration(pReverb) ,"");
        EXPECT_DEATH_IF_SUPPORTED(SetReverbColoration(pReverb, value) ,"");
        EXPECT_DEATH_IF_SUPPORTED(GetReverbReverbGain(pReverb) ,"");
        EXPECT_DEATH_IF_SUPPORTED(SetReverbReverbGain(pReverb, gain) ,"");
        EXPECT_DEATH_IF_SUPPORTED(GetReverbOutGain(pReverb) ,"");
        EXPECT_DEATH_IF_SUPPORTED(SetReverbOutGain(pReverb, gain) ,"");
        EXPECT_DEATH_IF_SUPPORTED(GetReverbDryGain(pReverb) ,"");
        EXPECT_DEATH_IF_SUPPORTED(SetReverbDryGain(pReverb, gain) ,"");
        EXPECT_DEATH_IF_SUPPORTED(IsReverbEnabled(pReverb) ,"");
        EXPECT_DEATH_IF_SUPPORTED(SetReverbEnabled(pReverb, enable) ,"");
        EXPECT_DEATH_IF_SUPPORTED(GetReverbChannelCountMax(pReverb) ,"");
    }
}
}; // namespace

TEST(TestReverbParameters, Preconditions)
{
    {
        ReverbType::EarlyMode earlyMode = ReverbType::EarlyMode_Hall;
        ReverbType::LateMode lateMode = ReverbType::LateMode_Hall;
        float gain = 0.0f;
        float ratio = 0.0f;
        float value = 0.0f;
        nn::TimeSpan time = nn::TimeSpan::FromMilliSeconds(0);
        bool enable = false;

        // specifying nullptr to ReverbType*
        ReverbType* pReverb = nullptr;
        ReverbDeathTests(pReverb, earlyMode, lateMode, gain, ratio, value, time, enable);

        // specifying uninitialized ReverbType to ReverbType*
        ReverbType reverb;
        ::std::memset(&reverb, 0xff, sizeof(reverb));
        ReverbDeathTests(&reverb, earlyMode, lateMode, gain, ratio, value, time, enable);
    }

    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));
    nn::mem::StandardAllocator effectAllocator(g_WorkBuffer2, sizeof(g_WorkBuffer2));
    EffectScopedRenderer renderer(allocator);

    int channels[] = {1, 2, 4, 6};
    int8_t input[] = {0, 1, 2, 3, 4, 5};
    int8_t output[] = {0, 1, 2, 3, 4, 5};

    for (auto j = 0; j < sizeof(channels) / sizeof(int);  ++j)
    {
        auto param = nnt::audio::util::GetAudioRendererParameterForWaveComparison();
        param.mixBufferCount = 1 + channels[j]; // SubMix(1) + FinalMix(channels[j])
        renderer.RendererSetup(&param, 1, channels[j]);
        nn::audio::MemoryPoolType effectPool;
        nn::audio::AcquireMemoryPool(renderer.GetConfig(), &effectPool, g_WorkBuffer2, sizeof(g_WorkBuffer2));
        nn::audio::RequestAttachMemoryPool(&effectPool);

        ReverbType reverb;
        ReverbType* pReverb = &reverb;
        size_t reverbSize = GetRequiredBufferSizeForReverb(param.sampleRate, channels[j]);
        void* reverbBuffer = effectAllocator.Allocate(reverbSize);
        NN_ASSERT_NOT_NULL(reverbBuffer);
        NNT_ASSERT_RESULT_SUCCESS(AddReverb(renderer.GetConfig(), &reverb, reverbBuffer, reverbSize, renderer.GetFinalMix(), channels[j]));
        nn::audio::SetReverbInputOutput(&reverb, input, output, channels[j]);

        float invalidValues[] = { -0.1f,  1.1f };
        for (auto i = 0; i < sizeof(invalidValues) / sizeof(float); ++i)
        {
            float val = invalidValues[i];
            EXPECT_DEATH_IF_SUPPORTED(SetReverbEarlyGain(pReverb, val) ,"");
            EXPECT_DEATH_IF_SUPPORTED(SetReverbLateGain(pReverb, val) ,"");
            EXPECT_DEATH_IF_SUPPORTED(SetReverbReverbGain(pReverb, val) ,"");
            EXPECT_DEATH_IF_SUPPORTED(SetReverbOutGain(pReverb, val) ,"");
            EXPECT_DEATH_IF_SUPPORTED(SetReverbDryGain(pReverb, val) ,"");
            EXPECT_DEATH_IF_SUPPORTED(SetReverbColoration(pReverb, val) ,"");
        }

        {
            ReverbType::EarlyMode earlyMode = ReverbType::EarlyMode_Count;
            EXPECT_DEATH_IF_SUPPORTED(SetReverbEarlyMode(pReverb, earlyMode), "");

            ReverbType::LateMode lateMode = ReverbType::LateMode_Count;
            EXPECT_DEATH_IF_SUPPORTED(SetReverbLateMode(pReverb, lateMode) ,"");

            nn::TimeSpan invalidTime;
            invalidTime = nn::TimeSpan::FromMilliSeconds(301);
            EXPECT_DEATH_IF_SUPPORTED(SetReverbPredelayTime(pReverb, invalidTime) ,"");
            invalidTime = nn::TimeSpan::FromMilliSeconds(99);
            EXPECT_DEATH_IF_SUPPORTED(SetReverbDecayTime(pReverb, invalidTime) ,"");
            invalidTime = nn::TimeSpan::FromMilliSeconds(20001);
            EXPECT_DEATH_IF_SUPPORTED(SetReverbDecayTime(pReverb, invalidTime) ,"");

            float val = -0.0f;
            EXPECT_DEATH_IF_SUPPORTED(SetReverbHighFrequencyDecayRatio(pReverb, val) ,"");
            val = 1.1f;
            EXPECT_DEATH_IF_SUPPORTED(SetReverbHighFrequencyDecayRatio(pReverb, val) ,"");
        }

        renderer.Start();
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
        renderer.Stop();
        renderer.RendererShutdown();

        effectAllocator.Free(reverbBuffer);
    }
}

TEST(TestReverbParameters, Success)
{
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));
    nn::mem::StandardAllocator effectAllocator(g_WorkBuffer2, sizeof(g_WorkBuffer2));
    EffectScopedRenderer renderer(allocator);
    int channels[] = {1, 2, 4, 6};
    int8_t input[] = {0, 1, 2, 3, 4, 5};
    int8_t output[] = {0, 1, 2, 3, 4, 5};

    for (auto j = 0; j < sizeof(channels) / sizeof(int);  ++j)
    {
        AudioRendererParameter param = nnt::audio::util::GetAudioRendererParameterForWaveComparison();
        param.mixBufferCount = 1 + channels[j]; // SubMix(1) + FinalMix(channels[j])
        renderer.RendererSetup(&param, 1, channels[j]);
        nn::audio::MemoryPoolType effectPool;
        nn::audio::AcquireMemoryPool(renderer.GetConfig(), &effectPool, g_WorkBuffer2, sizeof(g_WorkBuffer2));
        nn::audio::RequestAttachMemoryPool(&effectPool);

        ReverbType reverb;
        ReverbType* pReverb = &reverb;
        size_t reverbSize = GetRequiredBufferSizeForReverb(param.sampleRate, channels[j]);
        void* reverbBuffer = effectAllocator.Allocate(reverbSize);
        NN_ASSERT_NOT_NULL(reverbBuffer);
        NNT_ASSERT_RESULT_SUCCESS(AddReverb(renderer.GetConfig(), &reverb, reverbBuffer, reverbSize, renderer.GetFinalMix(), channels[j]));
        EXPECT_EQ(channels[j], nn::audio::GetReverbChannelCountMax(&reverb));

        nn::audio::SetReverbInputOutput(&reverb, input, output, channels[j]);

        float invalidValues[] = { 0.0f,  1.0f };
        for (auto i = 0; i < sizeof(invalidValues) / sizeof(float); ++i)
        {
            float val = invalidValues[i];

            SetReverbEarlyGain(pReverb, val);
            SetReverbLateGain(pReverb, val);
            SetReverbReverbGain(pReverb, val);
            SetReverbOutGain(pReverb, val);
            SetReverbDryGain(pReverb, val);
            SetReverbColoration(pReverb, val);

            EXPECT_NEAR(val, GetReverbEarlyGain(pReverb), g_AudioTestQfPrecision);
            EXPECT_NEAR(val, GetReverbLateGain(pReverb), g_AudioTestQfPrecision);
            EXPECT_NEAR(val, GetReverbReverbGain(pReverb), g_AudioTestQfPrecision);
            EXPECT_NEAR(val, GetReverbOutGain(pReverb), g_AudioTestQfPrecision);
            EXPECT_NEAR(val, GetReverbDryGain(pReverb), g_AudioTestQfPrecision);
            EXPECT_NEAR(val, GetReverbColoration(pReverb), g_AudioTestQfPrecision);
        }

        {
            nn::TimeSpan validTime;
            validTime = nn::TimeSpan::FromMilliSeconds(300);
            SetReverbPredelayTime(pReverb, validTime);
            EXPECT_NEAR(static_cast<float>(validTime.GetMilliSeconds()), static_cast<float>(GetReverbPredelayTime(&reverb).GetMilliSeconds()), g_AudioTestQfPrecision);

            validTime = nn::TimeSpan::FromMilliSeconds(100);
            SetReverbDecayTime(pReverb, validTime);
            EXPECT_NEAR(static_cast<float>(validTime.GetMilliSeconds()), static_cast<float>(GetReverbDecayTime(&reverb).GetMilliSeconds()), 1); // 内部で[秒]管理している都合、ここの精度は落ちる

            validTime = nn::TimeSpan::FromSeconds(20);
            SetReverbDecayTime(pReverb, validTime);
            EXPECT_NEAR(static_cast<float>(validTime.GetMilliSeconds()), static_cast<float>(GetReverbDecayTime(&reverb).GetMilliSeconds()), g_AudioTestQfPrecision);

            float val = 0.1f;
            SetReverbHighFrequencyDecayRatio(pReverb, val);
            EXPECT_NEAR(val, GetReverbHighFrequencyDecayRatio(pReverb), g_AudioTestQfPrecision);
        }

        renderer.Start();
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
        renderer.Stop();
        renderer.RendererShutdown();

        effectAllocator.Free(reverbBuffer);
    }
}
#endif // !defined(NN_SDK_BUILD_RELEASE)

TEST(TestReverb6chWaveComp, Success)
{
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));

    std::string mountPath = "";
    nnt::audio::util::GetMountPath(mountPath);

    // Test environment
    auto params = nnt::audio::util::GetAudioRendererParameterForWaveComparison();

    const std::string SourceFile = g_SourceDataFolder + g_AudioTestSourceFileName;
    const std::string TargetFile = nnt::audio::util::GetResultDataDirectory() + g_AudioTestDelayed6chFileName;
    const std::string GoldenFile = g_RefernerceDataFolder + g_AudioTestDelayed6chFileName;
    const nn::TimeSpan CheckDuration = nn::TimeSpan::FromSeconds(1);
    const int32_t RecorderBufferSize = 100;
    const int32_t ChannelCount = 6;
    const int32_t BitRate = 16;
    const size_t SendReturnBufferSize = GetRequiredBufferSizeForAuxSendReturnBuffer(&params, RecorderBufferSize, ChannelCount);

    // Prepare Test Components
    /////////////////////////////////////////////////////////////////////////////
    nnt::audio::util::SimpleFsUtility fsUtil;
    nnt::audio::util::MinimalRenderer renderer(g_WorkBuffer2, sizeof(g_WorkBuffer2));
    nnt::audio::util::Recorder recorder;
    nnt::audio::util::WaveFile testWav;

    //fsUtil->InitializeFileSystem(c_MountName, c_MountPath);
    ASSERT_TRUE(fsUtil.InitializeFileSystem(g_MountName.c_str(), mountPath.c_str()).IsSuccess());
    renderer.Initialize(params);
    nn::audio::MemoryPoolType effectBufferPool;
    nn::audio::AcquireMemoryPool(renderer.GetConfig(), &effectBufferPool, g_WorkBuffer, sizeof(g_WorkBuffer));
    nn::audio::RequestAttachMemoryPool(&effectBufferPool);

    int64_t estimatedFileSize = nnt::audio::util::WaveFile::GetRequiredFileSize(params.sampleRate, ChannelCount, BitRate, CheckDuration);
    nn::Result rt = fsUtil.PrepareNewFile(TargetFile.c_str(), estimatedFileSize);
    ASSERT_TRUE(rt.IsSuccess()) << "Failed to create new file.";
    testWav.Open(TargetFile.c_str(), params.sampleRate, ChannelCount, BitRate);

    // Recorder Setup
    //////////////////////////////////////////////////////////////////////////

    nn::audio::ReverbType reverb;
    size_t reverbSize = GetRequiredBufferSizeForReverb(params.sampleRate, ChannelCount);
    void* reverbBuffer = allocator.Allocate(reverbSize);
    NN_ASSERT_NOT_NULL(reverbBuffer);

    nn::Result result = nn::audio::AddReverb(renderer.GetConfig(), &reverb, reverbBuffer, reverbSize, renderer.GetFinalMix(), ChannelCount);
    ASSERT_TRUE(result.IsSuccess());
    int8_t bus[] = { 0, 1, 2, 3, 4, 5 };
    nn::audio::SetReverbColoration(&reverb, 0.0f);
    nn::TimeSpan decayTime = nn::TimeSpan::FromSeconds(2);
    nn::audio::SetReverbDecayTime(&reverb, decayTime);
    nn::audio::SetReverbDryGain(&reverb, 0.5f);
    nn::audio::SetReverbEarlyGain(&reverb, 0.8f);
    nn::audio::SetReverbEarlyMode(&reverb, nn::audio::ReverbType::EarlyMode_Hall);
    nn::audio::SetReverbHighFrequencyDecayRatio(&reverb, 0.5f);
    nn::audio::SetReverbLateGain(&reverb, 0.5f);
    nn::audio::SetReverbLateMode(&reverb, nn::audio::ReverbType::LateMode_Hall);
    nn::audio::SetReverbOutGain(&reverb, 1.0f);
    nn::TimeSpan predelayTime = nn::TimeSpan::FromMilliSeconds(10);
    nn::audio::SetReverbPredelayTime(&reverb, predelayTime);
    nn::audio::SetReverbReverbGain(&reverb, 0.8f);
    nn::audio::SetReverbInputOutput(&reverb, bus, bus, sizeof(bus));
    nn::audio::SetReverbEnabled(&reverb, true);

    // Recorder Setup
    //////////////////////////////////////////////////////////////////////////

    // Setup AuxType
    nn::audio::AuxType aux;
    auto bufferSize = nn::audio::GetRequiredBufferSizeForAuxSendReturnBuffer(&params, RecorderBufferSize, ChannelCount);
    auto sendBuffer = static_cast<int32_t*>(allocator.Allocate(bufferSize, nn::audio::BufferAlignSize));
    auto returnBuffer = static_cast<int32_t*>(allocator.Allocate(bufferSize, nn::audio::BufferAlignSize));
    nn::audio::AddAux(renderer.GetConfig(), &aux, renderer.GetFinalMix(), sendBuffer, returnBuffer, bufferSize);
    int8_t inputIndex[ChannelCount] = { 0, 1, 2, 3, 4, 5 };
    int8_t outputIndex[ChannelCount] = { 0, 1, 2, 3, 4, 5 };
    nn::audio::SetAuxInputOutput(&aux, inputIndex, outputIndex, ChannelCount);

    // Attach AuxType to Recorder & WaveFile.
    ASSERT_TRUE(recorder.Prepare(&params, &aux, &testWav, SendReturnBufferSize, CheckDuration, g_RecorderBuffer, sizeof(g_RecorderBuffer)).IsSuccess());

    // Prepare Target Voice.
    //////////////////////////////////////////////////////////////////////////
    nn::audio::VoiceType voiceShutter;
    nn::audio::WaveBuffer waveBufferShutter;
    size_t dataShutterSize = nnt::audio::util::GetWaveSampleSize(SourceFile);
    void* dataShutter = allocator.Allocate(dataShutterSize, nn::audio::BufferAlignSize);
    {
        int sampleRate = 0;

        size_t dataSize = nnt::audio::util::LoadSourceSamples(SourceFile, dataShutter, dataShutterSize, &sampleRate);

        nn::audio::AcquireVoiceSlot(renderer.GetConfig(),
            &voiceShutter,
            sampleRate,
            1,
            nn::audio::SampleFormat_PcmInt16,
            nn::audio::VoiceType::PriorityHighest,
            nullptr,
            0);
        nn::audio::SetVoiceDestination(renderer.GetConfig(),
            &voiceShutter,
            renderer.GetFinalMix());
        waveBufferShutter.buffer = dataShutter;
        waveBufferShutter.size = dataSize;
        waveBufferShutter.startSampleOffset = 0;
        waveBufferShutter.endSampleOffset = static_cast<int32_t>(dataSize / sizeof(int16_t));
        waveBufferShutter.loop = true;
        waveBufferShutter.isEndOfStream = false;
        waveBufferShutter.pContext = nullptr;
        waveBufferShutter.contextSize = 0;

        nn::audio::AppendWaveBuffer(&voiceShutter, &waveBufferShutter);
        nn::audio::SetVoicePlayState(&voiceShutter, nn::audio::VoiceType::PlayState_Play);
        nn::audio::SetVoiceMixVolume(&voiceShutter, renderer.GetFinalMix(), 0.707f / 2, 0, 0);
        nn::audio::SetVoiceMixVolume(&voiceShutter, renderer.GetFinalMix(), 0.707f / 2, 0, 1);
        nn::audio::SetVoiceMixVolume(&voiceShutter, renderer.GetFinalMix(), 0.707f / 2, 0, 2);
        nn::audio::SetVoiceMixVolume(&voiceShutter, renderer.GetFinalMix(), 0.707f / 2, 0, 3);
        nn::audio::SetVoiceMixVolume(&voiceShutter, renderer.GetFinalMix(), 0.707f / 2, 0, 4);
        nn::audio::SetVoiceMixVolume(&voiceShutter, renderer.GetFinalMix(), 0.707f / 2, 0, 5);
    }

    // start running
    //////////////////////////////////////////////////////////////////////////
    recorder.Start();
    renderer.Start();

    // keep playing at least for recording period.
    while (recorder.Record())
    {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16)); // yet-another Wait for Vsync
        renderer.Update();
    }

    // finish running
    //////////////////////////////////////////////////////////////////////////
    renderer.Stop();
    recorder.Stop();
    testWav.Close();

    allocator.Free(dataShutter);
    allocator.Free(sendBuffer);
    allocator.Free(returnBuffer);

    nnt::audio::util::SimpleVerify(TargetFile, GoldenFile);

} // NOLINT(impl/function_size)

TEST(TestReverb4chWaveComp, Success)
{
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));

    std::string mountPath = "";
    nnt::audio::util::GetMountPath(mountPath);

    // Test environment
    auto params = nnt::audio::util::GetAudioRendererParameterForWaveComparison();

    const std::string SourceFile = g_SourceDataFolder + g_AudioTestSourceFileName;
    const std::string TargetFile = nnt::audio::util::GetResultDataDirectory() + g_AudioTestDelayed4chFileName;
    const std::string GoldenFile = g_RefernerceDataFolder + g_AudioTestDelayed4chFileName;
    const nn::TimeSpan CheckDuration = nn::TimeSpan::FromSeconds(1);
    const int32_t RecorderBufferSize = 100;
    const int32_t ChannelCount = 4;
    const int32_t BitRate = 16;
    const size_t SendReturnBufferSize = GetRequiredBufferSizeForAuxSendReturnBuffer(&params, RecorderBufferSize, ChannelCount);

    // Prepare Test Components
    /////////////////////////////////////////////////////////////////////////////
    nnt::audio::util::SimpleFsUtility fsUtil;
    nnt::audio::util::MinimalRenderer renderer(g_WorkBuffer2, sizeof(g_WorkBuffer2));
    nnt::audio::util::Recorder recorder;
    nnt::audio::util::WaveFile testWav;

    //fsUtil->InitializeFileSystem(c_MountName, c_MountPath);
    ASSERT_TRUE(fsUtil.InitializeFileSystem(g_MountName.c_str(), mountPath.c_str()).IsSuccess());
    renderer.Initialize(params);
    nn::audio::MemoryPoolType effectBufferPool;
    nn::audio::AcquireMemoryPool(renderer.GetConfig(), &effectBufferPool, g_WorkBuffer, sizeof(g_WorkBuffer));
    nn::audio::RequestAttachMemoryPool(&effectBufferPool);

    int64_t estimatedFileSize = nnt::audio::util::WaveFile::GetRequiredFileSize(params.sampleRate, ChannelCount, BitRate, CheckDuration);
    nn::Result rt = fsUtil.PrepareNewFile(TargetFile.c_str(), estimatedFileSize);
    ASSERT_TRUE(rt.IsSuccess()) << "Failed to create new file.";
    testWav.Open(TargetFile.c_str(), params.sampleRate, ChannelCount, BitRate);

    // Recorder Setup
    //////////////////////////////////////////////////////////////////////////

    nn::audio::ReverbType reverb;
    size_t reverbSize = GetRequiredBufferSizeForReverb(params.sampleRate, ChannelCount);
    void* reverbBuffer = allocator.Allocate(reverbSize);
    NN_ASSERT_NOT_NULL(reverbBuffer);

    nn::Result result = nn::audio::AddReverb(renderer.GetConfig(), &reverb, reverbBuffer, reverbSize, renderer.GetFinalMix(), ChannelCount);
    ASSERT_TRUE(result.IsSuccess());
    int8_t bus[] = { 0, 1, 2, 3 };
    nn::audio::SetReverbColoration(&reverb, 0.0f);
    nn::TimeSpan decayTime = nn::TimeSpan::FromSeconds(2);
    nn::audio::SetReverbDecayTime(&reverb, decayTime);
    nn::audio::SetReverbDryGain(&reverb, 0.5f);
    nn::audio::SetReverbEarlyGain(&reverb, 0.8f);
    nn::audio::SetReverbEarlyMode(&reverb, nn::audio::ReverbType::EarlyMode_Hall);
    nn::audio::SetReverbHighFrequencyDecayRatio(&reverb, 0.5f);
    nn::audio::SetReverbLateGain(&reverb, 0.5f);
    nn::audio::SetReverbLateMode(&reverb, nn::audio::ReverbType::LateMode_Hall);
    nn::audio::SetReverbOutGain(&reverb, 1.0f);
    nn::TimeSpan predelayTime = nn::TimeSpan::FromMilliSeconds(10);
    nn::audio::SetReverbPredelayTime(&reverb, predelayTime);
    nn::audio::SetReverbReverbGain(&reverb, 0.8f);
    nn::audio::SetReverbInputOutput(&reverb, bus, bus, sizeof(bus));
    nn::audio::SetReverbEnabled(&reverb, true);

    // Recorder Setup
    //////////////////////////////////////////////////////////////////////////

    // Setup AuxType
    nn::audio::AuxType aux;
    auto bufferSize = nn::audio::GetRequiredBufferSizeForAuxSendReturnBuffer(&params, RecorderBufferSize, ChannelCount);
    auto sendBuffer = static_cast<int32_t*>(allocator.Allocate(bufferSize, nn::audio::BufferAlignSize));
    auto returnBuffer = static_cast<int32_t*>(allocator.Allocate(bufferSize, nn::audio::BufferAlignSize));
    nn::audio::AddAux(renderer.GetConfig(), &aux, renderer.GetFinalMix(), sendBuffer, returnBuffer, bufferSize);
    int8_t inputIndex[ChannelCount] = { 0, 1, 2, 3 };
    int8_t outputIndex[ChannelCount] = { 0, 1, 2, 3 };
    nn::audio::SetAuxInputOutput(&aux, inputIndex, outputIndex, ChannelCount);

    // Attach AuxType to Recorder & WaveFile.
    ASSERT_TRUE(recorder.Prepare(&params, &aux, &testWav, SendReturnBufferSize, CheckDuration, g_RecorderBuffer, sizeof(g_RecorderBuffer)).IsSuccess());

    // Prepare Target Voice.
    //////////////////////////////////////////////////////////////////////////
    nn::audio::VoiceType voiceShutter;
    nn::audio::WaveBuffer waveBufferShutter;
    size_t dataShutterSize = nnt::audio::util::GetWaveSampleSize(SourceFile);
    void* dataShutter = allocator.Allocate(dataShutterSize, nn::audio::BufferAlignSize);
    {
        int sampleRate = 0;

        size_t dataSize = nnt::audio::util::LoadSourceSamples(SourceFile, dataShutter, dataShutterSize, &sampleRate);

        nn::audio::AcquireVoiceSlot(renderer.GetConfig(),
            &voiceShutter,
            sampleRate,
            1,
            nn::audio::SampleFormat_PcmInt16,
            nn::audio::VoiceType::PriorityHighest,
            nullptr,
            0);
        nn::audio::SetVoiceDestination(renderer.GetConfig(),
            &voiceShutter,
            renderer.GetFinalMix());
        waveBufferShutter.buffer = dataShutter;
        waveBufferShutter.size = dataSize;
        waveBufferShutter.startSampleOffset = 0;
        waveBufferShutter.endSampleOffset = static_cast<int32_t>(dataSize / sizeof(int16_t));
        waveBufferShutter.loop = true;
        waveBufferShutter.isEndOfStream = false;
        waveBufferShutter.pContext = nullptr;
        waveBufferShutter.contextSize = 0;

        nn::audio::AppendWaveBuffer(&voiceShutter, &waveBufferShutter);
        nn::audio::SetVoicePlayState(&voiceShutter, nn::audio::VoiceType::PlayState_Play);
        nn::audio::SetVoiceMixVolume(&voiceShutter, renderer.GetFinalMix(), 0.707f / 2, 0, 0);
        nn::audio::SetVoiceMixVolume(&voiceShutter, renderer.GetFinalMix(), 0.707f / 2, 0, 1);
        nn::audio::SetVoiceMixVolume(&voiceShutter, renderer.GetFinalMix(), 0.707f / 2, 0, 2);
        nn::audio::SetVoiceMixVolume(&voiceShutter, renderer.GetFinalMix(), 0.707f / 2, 0, 3);
    }

    // start running
    //////////////////////////////////////////////////////////////////////////
    recorder.Start();
    renderer.Start();

    // keep playing at least for recording period.
    while (recorder.Record())
    {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16)); // yet-another Wait for Vsync
        renderer.Update();
    }

    // finish running
    //////////////////////////////////////////////////////////////////////////
    renderer.Stop();
    recorder.Stop();
    testWav.Close();

    allocator.Free(dataShutter);
    allocator.Free(sendBuffer);
    allocator.Free(returnBuffer);

    nnt::audio::util::SimpleVerify(TargetFile, GoldenFile);

} // NOLINT(impl/function_size)

TEST(TestReverb2chWaveComp, Success)
{
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));

    std::string mountPath = "";
    nnt::audio::util::GetMountPath(mountPath);

    // Test environment
    auto params = nnt::audio::util::GetAudioRendererParameterForWaveComparison();

    const std::string SourceFile = g_SourceDataFolder + g_AudioTestSourceFileName;
    const std::string TargetFile = nnt::audio::util::GetResultDataDirectory() + g_AudioTestDelayed2chFileName;
    const std::string GoldenFile = g_RefernerceDataFolder + g_AudioTestDelayed2chFileName;
    const nn::TimeSpan CheckDuration = nn::TimeSpan::FromSeconds(1);
    const int32_t RecorderBufferSize = 100;
    const int32_t ChannelCount = 2;
    const int32_t BitRate = 16;
    const size_t SendReturnBufferSize = GetRequiredBufferSizeForAuxSendReturnBuffer(&params, RecorderBufferSize, ChannelCount);

    // Prepare Test Components
    /////////////////////////////////////////////////////////////////////////////
    nnt::audio::util::SimpleFsUtility fsUtil;
    nnt::audio::util::MinimalRenderer renderer(g_WorkBuffer2, sizeof(g_WorkBuffer2));
    nnt::audio::util::Recorder recorder;
    nnt::audio::util::WaveFile testWav;

    //fsUtil->InitializeFileSystem(c_MountName, c_MountPath);
    ASSERT_TRUE(fsUtil.InitializeFileSystem(g_MountName.c_str(), mountPath.c_str()).IsSuccess());
    renderer.Initialize(params);
    nn::audio::MemoryPoolType effectBufferPool;
    nn::audio::AcquireMemoryPool(renderer.GetConfig(), &effectBufferPool, g_WorkBuffer, sizeof(g_WorkBuffer));
    nn::audio::RequestAttachMemoryPool(&effectBufferPool);

    int64_t estimatedFileSize = nnt::audio::util::WaveFile::GetRequiredFileSize(params.sampleRate, ChannelCount, BitRate, CheckDuration);
    nn::Result rt = fsUtil.PrepareNewFile(TargetFile.c_str(), estimatedFileSize);
    ASSERT_TRUE(rt.IsSuccess()) << "Failed to create new file.";
    testWav.Open(TargetFile.c_str(), params.sampleRate, ChannelCount, BitRate);

    // Recorder Setup
    //////////////////////////////////////////////////////////////////////////

    nn::audio::ReverbType reverb;
    size_t reverbSize = GetRequiredBufferSizeForReverb(params.sampleRate, ChannelCount);
    void* reverbBuffer = allocator.Allocate(reverbSize);
    NN_ASSERT_NOT_NULL(reverbBuffer);

    nn::Result result = nn::audio::AddReverb(renderer.GetConfig(), &reverb, reverbBuffer, reverbSize, renderer.GetFinalMix(), 2);
    ASSERT_TRUE(result.IsSuccess());
    int8_t bus[] = {0, 1};
    nn::audio::SetReverbColoration(&reverb, 0.0f);
    nn::TimeSpan decayTime = nn::TimeSpan::FromSeconds(2);
    nn::audio::SetReverbDecayTime(&reverb, decayTime);
    nn::audio::SetReverbDryGain(&reverb, 0.5f);
    nn::audio::SetReverbEarlyGain(&reverb, 0.8f);
    nn::audio::SetReverbEarlyMode(&reverb, nn::audio::ReverbType::EarlyMode_Hall);
    nn::audio::SetReverbHighFrequencyDecayRatio(&reverb, 0.5f);
    nn::audio::SetReverbLateGain(&reverb, 0.5f);
    nn::audio::SetReverbLateMode(&reverb, nn::audio::ReverbType::LateMode_Hall);
    nn::audio::SetReverbOutGain(&reverb, 1.0f);
    nn::TimeSpan predelayTime = nn::TimeSpan::FromMilliSeconds(10);
    nn::audio::SetReverbPredelayTime(&reverb, predelayTime);
    nn::audio::SetReverbReverbGain(&reverb, 0.8f);
    nn::audio::SetReverbInputOutput(&reverb, bus, bus, sizeof(bus));
    nn::audio::SetReverbEnabled(&reverb, true);

    // Recorder Setup
    //////////////////////////////////////////////////////////////////////////

    // Setup AuxType
    nn::audio::AuxType aux;
    auto bufferSize = nn::audio::GetRequiredBufferSizeForAuxSendReturnBuffer(&params, RecorderBufferSize, ChannelCount);
    auto sendBuffer = static_cast<int32_t*>(allocator.Allocate(bufferSize, nn::audio::BufferAlignSize));
    auto returnBuffer = static_cast<int32_t*>(allocator.Allocate(bufferSize, nn::audio::BufferAlignSize));
    nn::audio::AddAux(renderer.GetConfig(), &aux, renderer.GetFinalMix(), sendBuffer,  returnBuffer, bufferSize);
    int8_t inputIndex[ChannelCount] = {0, 1};
    int8_t outputIndex[ChannelCount] = {0, 1};
    nn::audio::SetAuxInputOutput(&aux, inputIndex, outputIndex, ChannelCount);

    // Attach AuxType to Recorder & WaveFile.
    ASSERT_TRUE(recorder.Prepare(&params, &aux, &testWav, SendReturnBufferSize, CheckDuration, g_RecorderBuffer, sizeof(g_RecorderBuffer)).IsSuccess());

    // Prepare Target Voice.
    //////////////////////////////////////////////////////////////////////////
    nn::audio::VoiceType voiceShutter;
    nn::audio::WaveBuffer waveBufferShutter;
    size_t dataShutterSize = nnt::audio::util::GetWaveSampleSize(SourceFile);
    void* dataShutter = allocator.Allocate(dataShutterSize, nn::audio::BufferAlignSize);
    {
        int sampleRate = 0;

        size_t dataSize = nnt::audio::util::LoadSourceSamples(SourceFile, dataShutter, dataShutterSize, &sampleRate);

        nn::audio::AcquireVoiceSlot(renderer.GetConfig(),
            &voiceShutter,
            sampleRate,
            1,
            nn::audio::SampleFormat_PcmInt16,
            nn::audio::VoiceType::PriorityHighest,
            nullptr,
            0);
        nn::audio::SetVoiceDestination(renderer.GetConfig(),
            &voiceShutter,
            renderer.GetFinalMix());
        waveBufferShutter.buffer = dataShutter;
        waveBufferShutter.size = dataSize;
        waveBufferShutter.startSampleOffset = 0;
        waveBufferShutter.endSampleOffset = static_cast<int32_t>(dataSize / sizeof(int16_t));
        waveBufferShutter.loop = true;
        waveBufferShutter.isEndOfStream = false;
        waveBufferShutter.pContext = nullptr;
        waveBufferShutter.contextSize = 0;

        nn::audio::AppendWaveBuffer(&voiceShutter, &waveBufferShutter);
        nn::audio::SetVoicePlayState(&voiceShutter, nn::audio::VoiceType::PlayState_Play);
        nn::audio::SetVoiceMixVolume(&voiceShutter, renderer.GetFinalMix(), 0.707f / 2, 0, 0);
        nn::audio::SetVoiceMixVolume(&voiceShutter, renderer.GetFinalMix(), 0.707f / 2, 0, 1);
    }

    // start running
    //////////////////////////////////////////////////////////////////////////
    recorder.Start();
    renderer.Start();

    // keep playing at least for recording period.
    while (recorder.Record())
    {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16)); // yet-another Wait for Vsync
        renderer.Update();
    }

    // finish running
    //////////////////////////////////////////////////////////////////////////
    renderer.Stop();
    recorder.Stop();
    testWav.Close();

    allocator.Free(dataShutter);
    allocator.Free(sendBuffer);
    allocator.Free(returnBuffer);

    nnt::audio::util::SimpleVerify(TargetFile, GoldenFile);

} // NOLINT(impl/function_size)

TEST(TestReverb1chWaveComp, Success)
{
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));

    std::string mountPath = "";
    nnt::audio::util::GetMountPath(mountPath);

    // Test environment
    auto params = nnt::audio::util::GetAudioRendererParameterForWaveComparison();

    const std::string SourceFile = g_SourceDataFolder + g_AudioTestSourceFileName;
    const std::string TargetFile = nnt::audio::util::GetResultDataDirectory() + g_AudioTestDelayed1chFileName;
    const std::string GoldenFile = g_RefernerceDataFolder + g_AudioTestDelayed1chFileName;
    const nn::TimeSpan CheckDuration = nn::TimeSpan::FromSeconds(1);
    const int32_t RecorderBufferSize = 100;
    const int32_t ChannelCount = 1;
    const int32_t BitRate = 16;
    const size_t SendReturnBufferSize = GetRequiredBufferSizeForAuxSendReturnBuffer(&params, RecorderBufferSize, ChannelCount);

    // Prepare Test Components
    /////////////////////////////////////////////////////////////////////////////
    nnt::audio::util::SimpleFsUtility fsUtil;
    nnt::audio::util::MinimalRenderer renderer(g_WorkBuffer2, sizeof(g_WorkBuffer2));
    nnt::audio::util::Recorder recorder;
    nnt::audio::util::WaveFile testWav;

    //fsUtil->InitializeFileSystem(c_MountName, c_MountPath);
    ASSERT_TRUE(fsUtil.InitializeFileSystem(g_MountName.c_str(), mountPath.c_str()).IsSuccess());
    renderer.Initialize(params);
    nn::audio::MemoryPoolType effectBufferPool;
    nn::audio::AcquireMemoryPool(renderer.GetConfig(), &effectBufferPool, g_WorkBuffer, sizeof(g_WorkBuffer));
    nn::audio::RequestAttachMemoryPool(&effectBufferPool);

    int64_t estimatedFileSize = nnt::audio::util::WaveFile::GetRequiredFileSize(params.sampleRate, ChannelCount, BitRate, CheckDuration);
    nn::Result rt = fsUtil.PrepareNewFile(TargetFile.c_str(), estimatedFileSize);
    ASSERT_TRUE(rt.IsSuccess()) << "Failed to create new file.";
    testWav.Open(TargetFile.c_str(), params.sampleRate, ChannelCount, BitRate);

    // Recorder Setup
    //////////////////////////////////////////////////////////////////////////

    nn::audio::ReverbType reverb;
    size_t reverbSize = GetRequiredBufferSizeForReverb(params.sampleRate, ChannelCount);
    void* reverbBuffer = allocator.Allocate(reverbSize);
    NN_ASSERT_NOT_NULL(reverbBuffer);

    nn::Result result = nn::audio::AddReverb(renderer.GetConfig(), &reverb, reverbBuffer, reverbSize, renderer.GetFinalMix(), ChannelCount);
    ASSERT_TRUE(result.IsSuccess());
    int8_t bus[] = { 0 };
    nn::audio::SetReverbColoration(&reverb, 0.0f);
    nn::TimeSpan decayTime = nn::TimeSpan::FromSeconds(2);
    nn::audio::SetReverbDecayTime(&reverb, decayTime);
    nn::audio::SetReverbDryGain(&reverb, 0.5f);
    nn::audio::SetReverbEarlyGain(&reverb, 0.8f);
    nn::audio::SetReverbEarlyMode(&reverb, nn::audio::ReverbType::EarlyMode_Hall);
    nn::audio::SetReverbHighFrequencyDecayRatio(&reverb, 0.5f);
    nn::audio::SetReverbLateGain(&reverb, 0.5f);
    nn::audio::SetReverbLateMode(&reverb, nn::audio::ReverbType::LateMode_Hall);
    nn::audio::SetReverbOutGain(&reverb, 1.0f);
    nn::TimeSpan predelayTime = nn::TimeSpan::FromMilliSeconds(10);
    nn::audio::SetReverbPredelayTime(&reverb, predelayTime);
    nn::audio::SetReverbReverbGain(&reverb, 0.8f);
    nn::audio::SetReverbInputOutput(&reverb, bus, bus, sizeof(bus));
    nn::audio::SetReverbEnabled(&reverb, true);

    // Recorder Setup
    //////////////////////////////////////////////////////////////////////////

    // Setup AuxType
    nn::audio::AuxType aux;
    auto bufferSize = nn::audio::GetRequiredBufferSizeForAuxSendReturnBuffer(&params, RecorderBufferSize, ChannelCount);
    auto sendBuffer = static_cast<int32_t*>(allocator.Allocate(bufferSize, nn::audio::BufferAlignSize));
    auto returnBuffer = static_cast<int32_t*>(allocator.Allocate(bufferSize, nn::audio::BufferAlignSize));
    nn::audio::AddAux(renderer.GetConfig(), &aux, renderer.GetFinalMix(), sendBuffer, returnBuffer, bufferSize);
    int8_t inputIndex[ChannelCount] = { 0 };
    int8_t outputIndex[ChannelCount] = { 0 };
    nn::audio::SetAuxInputOutput(&aux, inputIndex, outputIndex, ChannelCount);

    // Attach AuxType to Recorder & WaveFile.
    ASSERT_TRUE(recorder.Prepare(&params, &aux, &testWav, SendReturnBufferSize, CheckDuration, g_RecorderBuffer, sizeof(g_RecorderBuffer)).IsSuccess());

    // Prepare Target Voice.
    //////////////////////////////////////////////////////////////////////////
    nn::audio::VoiceType voiceShutter;
    nn::audio::WaveBuffer waveBufferShutter;
    size_t dataShutterSize = nnt::audio::util::GetWaveSampleSize(SourceFile);
    void* dataShutter = allocator.Allocate(dataShutterSize, nn::audio::BufferAlignSize);
    {
        int sampleRate = 0;

        size_t dataSize = nnt::audio::util::LoadSourceSamples(SourceFile, dataShutter, dataShutterSize, &sampleRate);

        nn::audio::AcquireVoiceSlot(renderer.GetConfig(),
            &voiceShutter,
            sampleRate,
            1,
            nn::audio::SampleFormat_PcmInt16,
            nn::audio::VoiceType::PriorityHighest,
            nullptr,
            0);
        nn::audio::SetVoiceDestination(renderer.GetConfig(),
            &voiceShutter,
            renderer.GetFinalMix());
        waveBufferShutter.buffer = dataShutter;
        waveBufferShutter.size = dataSize;
        waveBufferShutter.startSampleOffset = 0;
        waveBufferShutter.endSampleOffset = static_cast<int32_t>(dataSize / sizeof(int16_t));
        waveBufferShutter.loop = true;
        waveBufferShutter.isEndOfStream = false;
        waveBufferShutter.pContext = nullptr;
        waveBufferShutter.contextSize = 0;

        nn::audio::AppendWaveBuffer(&voiceShutter, &waveBufferShutter);
        nn::audio::SetVoicePlayState(&voiceShutter, nn::audio::VoiceType::PlayState_Play);
        nn::audio::SetVoiceMixVolume(&voiceShutter, renderer.GetFinalMix(), 0.707f / 2, 0, 0);
    }

    // start running
    //////////////////////////////////////////////////////////////////////////
    recorder.Start();
    renderer.Start();

    // keep playing at least for recording period.
    while (recorder.Record())
    {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16)); // yet-another Wait for Vsync
        renderer.Update();
    }

    // finish running
    //////////////////////////////////////////////////////////////////////////
    renderer.Stop();
    recorder.Stop();
    testWav.Close();

    allocator.Free(dataShutter);
    allocator.Free(sendBuffer);
    allocator.Free(returnBuffer);

    nnt::audio::util::SimpleVerify(TargetFile, GoldenFile);

} // NOLINT(impl/function_size)

} // namespace audio
} // namespace nnt

TEST(TestReverb, ResourceLimit)
{
    const int EffectCount = 4;
    const int EffectChannelCount = 6;

    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));

    nn::audio::AudioRendererParameter parameter;
    nn::audio::InitializeAudioRendererParameter(&parameter);
    parameter.effectCount = EffectCount;
    parameter.mixBufferCount = EffectChannelCount;

    nnt::audio::util::ScopedAudioRenderer sar(parameter);
    nn::audio::FinalMixType finalMix;
    nn::audio::AcquireFinalMix(&sar.GetConfig(), &finalMix, EffectChannelCount);

    nn::audio::ReverbType reverb[EffectCount + 1];

    auto size = nn::audio::GetRequiredBufferSizeForReverb(parameter.sampleRate, EffectChannelCount);
    auto buffer = allocator.Allocate(size, nn::audio::BufferAlignSize);
    EXPECT_TRUE(buffer != nullptr);

    for (auto i = 0; i < EffectCount; ++i)
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::AddReverb(&sar.GetConfig(), &reverb[i], buffer, size, &finalMix, EffectChannelCount));
    }
    NNT_EXPECT_RESULT_FAILURE(nn::audio::ResultOutOfResource, nn::audio::AddReverb(&sar.GetConfig(), &reverb[EffectCount], buffer, size, &finalMix, EffectChannelCount));

    for (auto i = 0; i < EffectCount; ++i)
    {
        nn::audio::SetReverbEnabled(&reverb[i], false);
        EXPECT_TRUE(buffer == nn::audio::RemoveReverb(&sar.GetConfig(), &reverb[i], &finalMix));
    }

    allocator.Free(buffer);
}

TEST(TestReverb, SuccessAddRemoveManyTimes)
{
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));
    nn::mem::StandardAllocator effectAllocator(g_WorkBuffer2, sizeof(g_WorkBuffer2));

    const auto parameter = nnt::audio::util::GetAudioRendererParameterForWaveComparison();
    nnt::audio::EffectScopedRenderer renderer(allocator);
    const int channelCountMax = 6;
    auto reverbSize = nn::audio::GetRequiredBufferSizeForReverb(parameter.sampleRate, channelCountMax);

    renderer.RendererSetup(&parameter);
    nn::audio::MemoryPoolType effectPool;
    nn::audio::AcquireMemoryPool(renderer.GetConfig(), &effectPool, g_WorkBuffer2, sizeof(g_WorkBuffer2));
    nn::audio::RequestAttachMemoryPool(&effectPool);

    nn::audio::ReverbType reverb;
    void* reverbBuffer = effectAllocator.Allocate(reverbSize);
    NN_ASSERT_NOT_NULL(reverbBuffer);
    // Renderer を開始
    renderer.Start();
    renderer.Update();
    renderer.Wait();

    for (auto i = 0; i < 50; ++i) // 50 は適当な数。深い意味はない。
    {
        const int channelCountTable[] = { 1, 2, 4, 6};
        const int channelCount = channelCountTable[i % (sizeof(channelCountTable) / sizeof(channelCountTable[0]))];

        // reverb 用バッファに適当な値を書き込んで、ダーティな状態にしておく
        memset(reverbBuffer, i, reverbSize);

        // reverb を renderer に追加し Update を呼び出すことで、reverbBuffer が指すメモリを DSP が触っている状態にする
        NNT_ASSERT_RESULT_SUCCESS(nn::audio::AddReverb(renderer.GetConfig(), &reverb, reverbBuffer, reverbSize, renderer.GetSubMix(), channelCount));
        renderer.Update();
        renderer.Wait();

        // あえて workBuffer を破壊しても停止することなく動作を続ける
        memset(reverbBuffer, i, reverbSize);
        nn::os::FlushDataCache(reverbBuffer, reverbSize);

        // 外すために無効化
        nn::audio::SetReverbEnabled(&reverb, false);

        // 外れるまで待機
        while(IsReverbRemovable(&reverb) == false)
        {
            renderer.Update();
            renderer.Wait();
        }

        // delay を外す
        ASSERT_TRUE(nn::audio::RemoveReverb(renderer.GetConfig(), &reverb, renderer.GetSubMix()) == reverbBuffer);
    }

    renderer.RendererShutdown();
    effectAllocator.Free(reverbBuffer);
}

TEST(TestReverb, UpdateAudioRendererOnEachStep)
{
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));
    nn::mem::StandardAllocator effectAllocator(g_WorkBuffer2, sizeof(g_WorkBuffer2));

    const auto parameter = nnt::audio::util::GetAudioRendererParameterForWaveComparison();
    nnt::audio::EffectScopedRenderer renderer(allocator);
    const int channelCountMax = 6;
    auto reverbSize = nn::audio::GetRequiredBufferSizeForReverb(parameter.sampleRate, channelCountMax);

    renderer.RendererSetup(&parameter);
    nn::audio::MemoryPoolType effectPool;
    nn::audio::AcquireMemoryPool(renderer.GetConfig(), &effectPool, g_WorkBuffer2, sizeof(g_WorkBuffer2));
    nn::audio::RequestAttachMemoryPool(&effectPool);

    nn::audio::ReverbType reverb;
    void* reverbBuffer = effectAllocator.Allocate(reverbSize);
    NN_ASSERT_NOT_NULL(reverbBuffer);
    // Renderer を開始
    renderer.Start();
    renderer.Update();
    renderer.Wait();

    for (auto i = 0; i < 10; ++i) // 10 は適当な数。深い意味はない。
    {
        const int channelCountTable[] = { 1, 2, 4, 6 };
        const int channelCount = channelCountTable[i % (sizeof(channelCountTable) / sizeof(channelCountTable[0]))];
        EXPECT_LE(channelCount, channelCountMax);
        NNT_ASSERT_RESULT_SUCCESS(nn::audio::AddReverb(renderer.GetConfig(), &reverb, reverbBuffer, reverbSize, renderer.GetSubMix(), channelCount));

        renderer.Update();
        renderer.Wait();

        int8_t bus[] = { 0, 1, 2, 3, 4, 5 };
        EXPECT_LE(static_cast<unsigned int>(channelCount), sizeof(bus) / sizeof(bus[0]));
        nn::audio::SetReverbInputOutput(&reverb, bus, bus, channelCount);
        renderer.Update();
        renderer.Wait();

        nn::audio::SetReverbEarlyMode(&reverb, nn::audio::ReverbType::EarlyMode_Hall);
        renderer.Update();
        renderer.Wait();

        nn::audio::SetReverbEarlyGain(&reverb, 0.8f);
        renderer.Update();
        renderer.Wait();

        nn::audio::SetReverbLateMode(&reverb, nn::audio::ReverbType::LateMode_Hall);
        renderer.Update();
        renderer.Wait();

        nn::audio::SetReverbDecayTime(&reverb, nn::TimeSpan::FromSeconds(2));
        renderer.Update();
        renderer.Wait();

        nn::audio::SetReverbHighFrequencyDecayRatio(&reverb, 0.5f);
        renderer.Update();
        renderer.Wait();

        nn::audio::SetReverbColoration(&reverb, 0.0f);
        renderer.Update();
        renderer.Wait();

        nn::audio::SetReverbReverbGain(&reverb, 0.8f);
        renderer.Update();
        renderer.Wait();

        nn::audio::SetReverbOutGain(&reverb, 1.0f);
        renderer.Update();
        renderer.Wait();

        nn::audio::SetReverbDryGain(&reverb, 0.5f);
        renderer.Update();
        renderer.Wait();

        nn::audio::SetReverbLateGain(&reverb, 0.5f);
        renderer.Update();
        renderer.Wait();

        nn::audio::SetReverbPredelayTime(&reverb, nn::TimeSpan::FromMilliSeconds(10));
        renderer.Update();
        renderer.Wait();

        // 外すために無効化
        nn::audio::SetReverbEnabled(&reverb, false);

        // 外れるまで待機
        while(IsReverbRemovable(&reverb) == false)
        {
            renderer.Update();
            renderer.Wait();
        }

        // reverb を外す
        ASSERT_TRUE(nn::audio::RemoveReverb(renderer.GetConfig(), &reverb, renderer.GetSubMix()) == reverbBuffer);

        renderer.Update();
        renderer.Wait();
    }

    renderer.RendererShutdown();
    effectAllocator.Free(reverbBuffer);
} // NOLINT(readability/fn_size)
