﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/



#include "testAudio_Effect.h"

#include "../../Programs/Eris/Sources/Libraries/audio/audio_AuxTypes.h"
#include "../../Programs/Eris/Sources/Libraries/audio/audio_EffectManager.h"
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/os.h>
#include <nn/nn_TimeSpan.h>
#include <nn/audio/audio_AudioRendererApi-os.win32.h>

#include <limits> // std::numeric_limits


namespace nnt {
namespace audio {

namespace {

NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN char g_WorkBuffer[1024 * 1024 * 2];  // 4096 == nn::os::MemoryPageSize
NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN char g_WorkBuffer2[1024 * 1024];  // 4096 == nn::os::MemoryPageSize
}

#if !defined(NN_SDK_BUILD_RELEASE)
TEST(TestGetRequiredBufferSizeForAux, Precondition)
{
    nn::audio::AudioRendererParameter param = nnt::audio::util::GetAudioRendererParameterForDefault();
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetRequiredBufferSizeForAuxSendReturnBuffer(nullptr, 10, 1), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetRequiredBufferSizeForAuxSendReturnBuffer(&param, 1, 1), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetRequiredBufferSizeForAuxSendReturnBuffer(&param, 0, 1), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetRequiredBufferSizeForAuxSendReturnBuffer(&param, 10, 0), "");
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

TEST(TestSetGetAuxEnable, Success)
{
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));
    EffectScopedRenderer renderer(allocator);
    const auto param = nnt::audio::util::GetAudioRendererParameterForWaveComparison();
    const size_t extBufSize = nn::audio::GetRequiredBufferSizeForAuxSendReturnBuffer(&param, 10, 1);
    int32_t* pSendBuffer = reinterpret_cast<int32_t*>(allocator.Allocate(extBufSize, nn::audio::BufferAlignSize));
    int32_t* pReturnBuffer = reinterpret_cast<int32_t*>(allocator.Allocate(extBufSize, nn::audio::BufferAlignSize));
    renderer.RendererSetup(&param);
    nn::audio::MemoryPoolType effectPool;
    nn::audio::AcquireMemoryPool(renderer.GetConfig(), &effectPool, g_WorkBuffer, sizeof(g_WorkBuffer));
    nn::audio::RequestAttachMemoryPool(&effectPool);

    nn::audio::AuxType exEffect;
    EXPECT_TRUE(nn::audio::AddAux(renderer.GetConfig(), &exEffect, renderer.GetFinalMix(), pSendBuffer, pReturnBuffer, extBufSize).IsSuccess());

    nn::audio::SetAuxEnabled(&exEffect, true);
    EXPECT_TRUE(nn::audio::IsAuxEnabled(&exEffect));
    nn::audio::SetAuxEnabled(&exEffect, false);
    EXPECT_FALSE(nn::audio::IsAuxEnabled(&exEffect));

    renderer.RendererShutdown();
}

TEST(TestAddAux, Success)
{
    //ScopedAllocator allocator;
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));
    EffectScopedRenderer renderer(allocator);

    const nn::audio::AudioRendererParameter param = { 48000, 240, 6, 24, 1, 1, 1, 0 };
    const size_t extBufSize = nn::audio::GetRequiredBufferSizeForAuxSendReturnBuffer(&param, 10, 1);
    renderer.RendererSetup(&param);
    nn::audio::MemoryPoolType effectPool;
    nn::audio::AcquireMemoryPool(renderer.GetConfig(), &effectPool, g_WorkBuffer, sizeof(g_WorkBuffer));
    nn::audio::RequestAttachMemoryPool(&effectPool);

    nn::audio::AudioRendererConfig* pConfig = renderer.GetConfig();
    nn::audio::AuxType exEffect;
    int32_t* pSendBuffer = reinterpret_cast<int32_t*>(allocator.Allocate(extBufSize));
    ASSERT_NE(pSendBuffer, nullptr);

    int32_t* pReturnBuffer = reinterpret_cast<int32_t*>(allocator.Allocate(extBufSize));
    ASSERT_NE(pReturnBuffer, nullptr);

    EXPECT_TRUE(nn::audio::AddAux(pConfig, &exEffect, renderer.GetSubMix(), pSendBuffer, pReturnBuffer, extBufSize).IsSuccess());
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::RemoveAux(pConfig, &exEffect, renderer.GetSubMix()), "");
    nn::audio::SetAuxEnabled(&exEffect, false);
    nn::audio::RemoveAux(pConfig, &exEffect, renderer.GetSubMix());

    EXPECT_TRUE(nn::audio::AddAux(pConfig, &exEffect, renderer.GetFinalMix(), pSendBuffer, pReturnBuffer, extBufSize).IsSuccess());
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::RemoveAux(pConfig, &exEffect, renderer.GetFinalMix()), "");
    nn::audio::SetAuxEnabled(&exEffect, false);
    nn::audio::RemoveAux(pConfig, &exEffect, renderer.GetFinalMix());

    // TODO: Verify if really added.

    renderer.RendererShutdown();

    allocator.Free(pSendBuffer);
    allocator.Free(pReturnBuffer);
}

#if !defined(NN_SDK_BUILD_RELEASE)
template<typename T>
void TestAddAux_PreconditionImpl(nn::audio::AudioRendererConfig* pConfig, T* pMix, int32_t* pSendBuffer, int32_t* pReturnBuffer, size_t extBufSize)
{
    nn::audio::AuxType exEffect;
    T* pDummyMix = nullptr;

    EXPECT_DEATH_IF_SUPPORTED(nn::audio::AddAux(nullptr, &exEffect, pMix,      pSendBuffer, pReturnBuffer, extBufSize), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::AddAux(pConfig, nullptr,   pMix,      pSendBuffer, pReturnBuffer, extBufSize), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::AddAux(pConfig, &exEffect, pDummyMix, pSendBuffer, pReturnBuffer, extBufSize), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::AddAux(pConfig, &exEffect, pMix,      nullptr,     pReturnBuffer, extBufSize), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::AddAux(pConfig, &exEffect, pMix,      pSendBuffer, nullptr,       extBufSize), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::AddAux(pConfig, &exEffect, pMix,      pSendBuffer, pReturnBuffer, 0),          "");
}

TEST(TestAddAux, Precondition)
{
    //ScopedAllocator allocator;
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));
    EffectScopedRenderer renderer(allocator);

    const auto param = nnt::audio::util::GetAudioRendererParameterForWaveComparison();
    const size_t extBufSize = nn::audio::GetRequiredBufferSizeForAuxSendReturnBuffer(&param, 10, 1);
    renderer.RendererSetup(&param);
    nn::audio::MemoryPoolType effectPool;
    nn::audio::AcquireMemoryPool(renderer.GetConfig(), &effectPool, g_WorkBuffer, sizeof(g_WorkBuffer));
    nn::audio::RequestAttachMemoryPool(&effectPool);

    int32_t* pSendBuffer = reinterpret_cast<int32_t*>(allocator.Allocate(extBufSize, nn::audio::BufferAlignSize));
    int32_t* pReturnBuffer = reinterpret_cast<int32_t*>(allocator.Allocate(extBufSize, nn::audio::BufferAlignSize));

    TestAddAux_PreconditionImpl(renderer.GetConfig(), renderer.GetSubMix(), pSendBuffer, pReturnBuffer, extBufSize);
    TestAddAux_PreconditionImpl(renderer.GetConfig(), renderer.GetFinalMix(), pSendBuffer, pReturnBuffer, extBufSize);

    renderer.RendererShutdown();

    allocator.Free(pSendBuffer);
    allocator.Free(pReturnBuffer);
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

template<typename T>
void TestRemoveAux_SuccessImpl(nn::audio::AudioRendererConfig* pConfig, T* pMix, int32_t* pSendBuffer, int32_t* pReturnBuffer, size_t extBufSize)
{
    nn::audio::AuxType aux;

    // effect count = 1 なので、ひとつだけ add 可能
    EXPECT_TRUE(nn::audio::AddAux(pConfig, &aux, pMix, pSendBuffer, pReturnBuffer, extBufSize).IsSuccess());
    // 二つ目は失敗する
    EXPECT_TRUE(nn::audio::AddAux(pConfig, &aux, pMix, pSendBuffer, pReturnBuffer, extBufSize).IsFailure());
    // remove すれば、再度 Add 可能
    nn::audio::SetAuxEnabled(&aux, false);
    RemoveAux(pConfig, &aux, pMix);
    EXPECT_TRUE(nn::audio::AddAux(pConfig, &aux, pMix, pSendBuffer, pReturnBuffer, extBufSize).IsSuccess());
    nn::audio::SetAuxEnabled(&aux, false);
    RemoveAux(pConfig, &aux, pMix);
}

TEST(TestRemoveAux, Success)
{
    //ScopedAllocator allocater;
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));
    EffectScopedRenderer renderer(allocator);
    const nn::audio::AudioRendererParameter param = { 48000, 240, 6, 24, 1, 1, 1 };
    const size_t extBufSize = nn::audio::GetRequiredBufferSizeForAuxSendReturnBuffer(&param, 10, 1);
    renderer.RendererSetup(&param);
    nn::audio::MemoryPoolType effectPool;
    nn::audio::AcquireMemoryPool(renderer.GetConfig(), &effectPool, g_WorkBuffer, sizeof(g_WorkBuffer));
    nn::audio::RequestAttachMemoryPool(&effectPool);

    int32_t* pSendBuffer = reinterpret_cast<int32_t*>(allocator.Allocate(extBufSize, nn::audio::BufferAlignSize));
    int32_t* pReturnBuffer = reinterpret_cast<int32_t*>(allocator.Allocate(extBufSize, nn::audio::BufferAlignSize));

    TestRemoveAux_SuccessImpl(renderer.GetConfig(), renderer.GetSubMix(), pSendBuffer, pReturnBuffer, extBufSize);
    TestRemoveAux_SuccessImpl(renderer.GetConfig(), renderer.GetFinalMix(), pSendBuffer, pReturnBuffer, extBufSize);

    renderer.RendererShutdown();

    allocator.Free(pSendBuffer);
    allocator.Free(pReturnBuffer);
}

#if !defined(NN_SDK_BUILD_RELEASE)
template<typename T>
void TestRemoveAux_PreconditionImpl(nn::audio::AudioRendererConfig* pConfig, T* pMix, int32_t* pSendBuffer, int32_t* pReturnBuffer, size_t extBufSize)
{
    nn::audio::AuxType aux;
    EXPECT_TRUE(nn::audio::AddAux(pConfig, &aux, pMix, pSendBuffer, pReturnBuffer, extBufSize).IsSuccess());

    EXPECT_DEATH_IF_SUPPORTED(nn::audio::RemoveAux(nullptr, &aux, pMix), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::RemoveAux(pConfig, nullptr, pMix), "");
    nn::audio::FinalMixType* pNullMix = nullptr;
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::RemoveAux(pConfig, &aux, pNullMix), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::RemoveAux(pConfig, &aux, pMix), "");
    nn::audio::SetAuxEnabled(&aux, false);
    nn::audio::RemoveAux(pConfig, &aux, pMix);
}

TEST(TestRemoveAux, Precondition)
{
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));
    EffectScopedRenderer renderer(allocator);
    const nn::audio::AudioRendererParameter param = { 48000, 240, 6, 24, 1, 1, 1 };
    const size_t extBufSize = nn::audio::GetRequiredBufferSizeForAuxSendReturnBuffer(&param, 10, 1);
    renderer.RendererSetup(&param);
    nn::audio::MemoryPoolType effectPool;
    nn::audio::AcquireMemoryPool(renderer.GetConfig(), &effectPool, g_WorkBuffer, sizeof(g_WorkBuffer));
    nn::audio::RequestAttachMemoryPool(&effectPool);

    int32_t* pSendBuffer = reinterpret_cast<int32_t*>(allocator.Allocate(extBufSize, nn::audio::BufferAlignSize));
    int32_t* pReturnBuffer = reinterpret_cast<int32_t*>(allocator.Allocate(extBufSize, nn::audio::BufferAlignSize));

    TestRemoveAux_PreconditionImpl(renderer.GetConfig(), renderer.GetSubMix(), pSendBuffer, pReturnBuffer, extBufSize);
    TestRemoveAux_PreconditionImpl(renderer.GetConfig(), renderer.GetFinalMix(), pSendBuffer, pReturnBuffer, extBufSize);

    renderer.RendererShutdown();

    allocator.Free(pSendBuffer);
    allocator.Free(pReturnBuffer);
}

TEST(TestGetAuxChannelCountMax, Precondition)
{
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetAuxChannelCountMax(nullptr), "");
}

#endif  // !defined(NN_SDK_BUILD_RELEASE)

TEST(TestGetAuxChannelCountMax, Success)
{
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));
    auto param = nnt::audio::util::GetAudioRendererParameterForDefault();
    param.mixBufferCount = nn::audio::MixBufferCountMax + 1;

    const size_t extBufSize = nn::audio::GetRequiredBufferSizeForAuxSendReturnBuffer(&param, 10, 1);
    int32_t* pSendBuffer = reinterpret_cast<int32_t*>(allocator.Allocate(extBufSize, nn::audio::BufferAlignSize));
    int32_t* pReturnBuffer = reinterpret_cast<int32_t*>(allocator.Allocate(extBufSize, nn::audio::BufferAlignSize));

    const int mixBufferCounts[] = { 1, 2, nn::audio::MixBufferCountMax - 1,  nn::audio::MixBufferCountMax };

    for(const auto mixBufferCountForFinalMix : mixBufferCounts)
    {
        EffectScopedRenderer renderer(allocator);
        nn::audio::AuxType exEffect;
        const auto mixBufferCountForSubMix = 1;
        renderer.RendererSetup(&param, mixBufferCountForSubMix, mixBufferCountForFinalMix);

        nn::audio::MemoryPoolType effectPool;
        nn::audio::AcquireMemoryPool(renderer.GetConfig(), &effectPool, g_WorkBuffer, sizeof(g_WorkBuffer));
        nn::audio::RequestAttachMemoryPool(&effectPool);

        EXPECT_TRUE(nn::audio::AddAux(renderer.GetConfig(), &exEffect, renderer.GetFinalMix(), pSendBuffer, pReturnBuffer, extBufSize).IsSuccess());
        EXPECT_EQ(nn::audio::GetAuxChannelCountMax(&exEffect), mixBufferCountForFinalMix);

        renderer.RendererShutdown();
    }
}

TEST(TestGetSetAuxInputOutput, Success)
{
    nn::audio::AuxType aux;
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));
    EffectScopedRenderer renderer(allocator);

    const auto param = nnt::audio::util::GetAudioRendererParameterForWaveComparison();
    const size_t extBufSize = nn::audio::GetRequiredBufferSizeForAuxSendReturnBuffer(&param, 10, 1);
    int32_t* pSendBuffer = reinterpret_cast<int32_t*>(allocator.Allocate(extBufSize, nn::audio::BufferAlignSize));
    int32_t* pReturnBuffer = reinterpret_cast<int32_t*>(allocator.Allocate(extBufSize, nn::audio::BufferAlignSize));
    renderer.RendererSetup(&param, 1, 10);
    EXPECT_TRUE(nn::audio::AddAux(renderer.GetConfig(), &aux, renderer.GetFinalMix(), pSendBuffer, pReturnBuffer, extBufSize).IsSuccess());
    nn::audio::MemoryPoolType effectPool;
    nn::audio::AcquireMemoryPool(renderer.GetConfig(), &effectPool, g_WorkBuffer, sizeof(g_WorkBuffer));
    nn::audio::RequestAttachMemoryPool(&effectPool);

    const int32_t chNum = 10;
    int8_t inputChannels[chNum] =  { 10, 11, 12, 13, 14, 0, 0, 0, 0, 0 };
    int8_t outputChannels[chNum] = { 15, 16, 17, 18, 19, 0, 0, 0, 0, 0 };

    int32_t verifyChNum = 0;
    int8_t verifyInBuf[chNum] = { 0 };
    int8_t verifyOutBuf[chNum] = { 0 };
    int8_t zeros[chNum] = { 0 };

    nn::audio::SetAuxInputOutput(&aux, inputChannels, outputChannels, 1);
    nn::audio::GetAuxInputOutput(&aux, verifyInBuf, verifyOutBuf, &verifyChNum, nn::audio::MixBufferCountMax);
    EXPECT_EQ(verifyChNum, 1);
    EXPECT_EQ(verifyInBuf[0], 10);
    EXPECT_EQ(verifyOutBuf[0], 15);
    EXPECT_EQ(verifyInBuf[1], 0); // 境界チェック
    EXPECT_EQ(verifyOutBuf[1], 0);


    nn::audio::SetAuxInputOutput(&aux, inputChannels, outputChannels, chNum);

    verifyChNum = 0;
    memset(verifyInBuf, 0, chNum);
    memset(verifyOutBuf, 0, chNum);

    nn::audio::GetAuxInputOutput(&aux, verifyInBuf, verifyOutBuf, &verifyChNum, 0);
    EXPECT_EQ(verifyChNum, 0);
    EXPECT_EQ(memcmp(verifyInBuf, zeros, chNum), 0);
    EXPECT_EQ(memcmp(verifyOutBuf, zeros, chNum), 0);

    verifyChNum = 0;
    memset(verifyInBuf, 0, chNum);
    memset(verifyOutBuf, 0, chNum);

    nn::audio::GetAuxInputOutput(&aux, verifyInBuf, verifyOutBuf, &verifyChNum, 1);
    EXPECT_EQ(verifyChNum, 1);
    EXPECT_EQ(verifyInBuf[0], 10);
    EXPECT_EQ(verifyOutBuf[0], 15);
    EXPECT_EQ(verifyInBuf[1], 0);
    EXPECT_EQ(verifyOutBuf[1], 0);

    verifyChNum = 0;
    memset(verifyInBuf, 0, chNum);
    memset(verifyOutBuf, 0, chNum);

    nn::audio::GetAuxInputOutput(&aux, verifyInBuf, verifyOutBuf, &verifyChNum, nn::audio::MixBufferCountMax);
    EXPECT_EQ(verifyChNum, chNum);
    EXPECT_EQ(memcmp(verifyInBuf, inputChannels, chNum), 0);
    EXPECT_EQ(memcmp(verifyOutBuf, outputChannels, chNum), 0);

    renderer.RendererShutdown();
}

#if !defined(NN_SDK_BUILD_RELEASE)
TEST(TestGetSetAuxInputOutput, Precondition)
{
    nn::audio::AuxType aux;
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));
    EffectScopedRenderer renderer(allocator);

    const auto param = nnt::audio::util::GetAudioRendererParameterForWaveComparison();
    const size_t extBufSize = nn::audio::GetRequiredBufferSizeForAuxSendReturnBuffer(&param, 10, 1);
    int32_t* pSendBuffer = reinterpret_cast<int32_t*>(allocator.Allocate(extBufSize, nn::audio::BufferAlignSize));
    int32_t* pReturnBuffer = reinterpret_cast<int32_t*>(allocator.Allocate(extBufSize, nn::audio::BufferAlignSize));
    const int chNum = 10;
    int8_t inputChannels[chNum];
    int8_t outputChannels[chNum];

    renderer.RendererSetup(&param);
    EXPECT_TRUE(nn::audio::AddAux(renderer.GetConfig(), &aux, renderer.GetFinalMix(), pSendBuffer, pReturnBuffer, extBufSize).IsSuccess());
    nn::audio::MemoryPoolType effectPool;
    nn::audio::AcquireMemoryPool(renderer.GetConfig(), &effectPool, g_WorkBuffer, sizeof(g_WorkBuffer));
    nn::audio::RequestAttachMemoryPool(&effectPool);

    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetAuxInputOutput(nullptr, inputChannels, outputChannels, chNum), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetAuxInputOutput(&aux, nullptr, outputChannels, chNum), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetAuxInputOutput(&aux, inputChannels, nullptr, chNum), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetAuxInputOutput(&aux, inputChannels, outputChannels, -1), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetAuxInputOutput(&aux, inputChannels, outputChannels, nn::audio::MixBufferCountMax + 1), "");

    renderer.RendererShutdown();
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

TEST(TestGetAuxSampleRateCount, GetAuxSampleRateSuccess)
{
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));
    EffectScopedRenderer renderer(allocator);

    const auto param = nnt::audio::util::GetAudioRendererParameterForWaveComparison();
    const int frameCount = 10;
    const size_t extBufSize = nn::audio::GetRequiredBufferSizeForAuxSendReturnBuffer(&param, frameCount, 1);
    renderer.RendererSetup(&param);
    nn::audio::MemoryPoolType effectPool;
    nn::audio::AcquireMemoryPool(renderer.GetConfig(), &effectPool, g_WorkBuffer, sizeof(g_WorkBuffer));
    nn::audio::RequestAttachMemoryPool(&effectPool);

    nn::audio::AuxType exEffect;

    int32_t* pSendBuffer = reinterpret_cast<int32_t*>(allocator.Allocate(extBufSize, nn::audio::BufferAlignSize));
    int32_t* pReturnBuffer = reinterpret_cast<int32_t*>(allocator.Allocate(extBufSize, nn::audio::BufferAlignSize));

    EXPECT_TRUE(nn::audio::AddAux(renderer.GetConfig(), &exEffect, renderer.GetFinalMix(), pSendBuffer, pReturnBuffer, extBufSize).IsSuccess());
    EXPECT_TRUE(static_cast<int>(param.sampleRate) == nn::audio::GetAuxSampleRate(&exEffect));
    EXPECT_TRUE(static_cast<int>(param.sampleCount * frameCount) == nn::audio::GetAuxSampleCount(&exEffect));

    renderer.RendererShutdown();

    allocator.Free(pSendBuffer);
    allocator.Free(pReturnBuffer);
}

#if !defined(NN_SDK_BUILD_RELEASE)
TEST(TestGetAuxSampleRateCount, Precondition)
{
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetAuxSampleRate(nullptr), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetAuxSampleCount(nullptr), "");
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

TEST(TestAuxAddRemoveManyTimes, Success)
{
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));
    nn::mem::StandardAllocator effectAllocator(g_WorkBuffer2, sizeof(g_WorkBuffer2));

    const auto parameter = nnt::audio::util::GetAudioRendererParameterForWaveComparison();
    nnt::audio::EffectScopedRenderer renderer(allocator);
    auto sendReturnBufferSize = nn::audio::GetRequiredBufferSizeForAuxSendReturnBuffer(&parameter, 10, 2);

    renderer.RendererSetup(&parameter);

    nn::audio::AuxType aux;
    void* sendBuffer = effectAllocator.Allocate(sendReturnBufferSize);
    void* returnBuffer = effectAllocator.Allocate(sendReturnBufferSize);
    NN_ASSERT_NOT_NULL(sendBuffer);
    NN_ASSERT_NOT_NULL(returnBuffer);

    nn::audio::MemoryPoolType memoryPool;
    ASSERT_TRUE(nn::audio::AcquireMemoryPool(renderer.GetConfig(), &memoryPool, g_WorkBuffer2, sizeof(g_WorkBuffer2)));
    ASSERT_TRUE(nn::audio::RequestAttachMemoryPool(&memoryPool));

    // Renderer を開始
    renderer.Start();
    renderer.Update();
    renderer.Wait();

    for (auto i = 0; i < 50; ++i) // 50 は適当な数。深い意味はない。
    {
        // reverb 用バッファに適当な値を書き込んで、ダーティな状態にしておく
        memset(sendBuffer, i, sendReturnBufferSize);
        memset(returnBuffer, i, sendReturnBufferSize);

        // reverb を renderer に追加し Update を呼び出すことで、reverbBuffer が指すメモリを DSP が触っている状態にする
        NNT_ASSERT_RESULT_SUCCESS(nn::audio::AddAux(renderer.GetConfig(), &aux, renderer.GetSubMix(), sendBuffer, returnBuffer, sendReturnBufferSize));
        renderer.Update();
        renderer.Wait();

        // あえて workBuffer を破壊しても停止することなく動作を続ける
        memset(sendBuffer, i, sendReturnBufferSize);
        memset(returnBuffer, i, sendReturnBufferSize);
        nn::os::FlushDataCache(sendBuffer, sendReturnBufferSize);
        nn::os::FlushDataCache(returnBuffer, sendReturnBufferSize);

        // 外すために無効化
        nn::audio::SetAuxEnabled(&aux, false);

        // 外れるまで待機
        while (IsAuxRemovable(&aux) == false)
        {
            renderer.Update();
            renderer.Wait();
        }

        // aux を外す
        nn::audio::RemoveAux(renderer.GetConfig(), &aux, renderer.GetSubMix());
    }

    renderer.RendererShutdown();
    effectAllocator.Free(sendBuffer);
    effectAllocator.Free(returnBuffer);
}

TEST(TestAux, UpdateAudioRendererOnEachStep)
{
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));
    nn::mem::StandardAllocator effectAllocator(g_WorkBuffer2, sizeof(g_WorkBuffer2));

    nnt::audio::EffectScopedRenderer renderer(allocator);

    const auto param = nnt::audio::util::GetAudioRendererParameterForWaveComparison();
    const int channelCountMax = 4; // 適当なチャンネル数 (Reverb, Delay のように最大チャンネル数の制約がないため)

    renderer.RendererSetup(&param, 1, 10);
    nn::audio::MemoryPoolType effectPool;
    nn::audio::AcquireMemoryPool(renderer.GetConfig(), &effectPool, g_WorkBuffer2, sizeof(g_WorkBuffer2));
    nn::audio::RequestAttachMemoryPool(&effectPool);

    nn::audio::AuxType aux;
    const size_t extBufSize = nn::audio::GetRequiredBufferSizeForAuxSendReturnBuffer(&param, 10, channelCountMax);
    int32_t* pSendBuffer = reinterpret_cast<int32_t*>(effectAllocator.Allocate(extBufSize, nn::audio::BufferAlignSize));
    ASSERT_NE(pSendBuffer, nullptr);
    int32_t* pReturnBuffer = reinterpret_cast<int32_t*>(effectAllocator.Allocate(extBufSize, nn::audio::BufferAlignSize));
    ASSERT_NE(pReturnBuffer, nullptr);

    // Renderer を開始
    renderer.Start();
    renderer.Update();
    renderer.Wait();

    for (auto i = 0; i < 10; ++i) // 10 は適当な数。深い意味はない。
    {
        EXPECT_TRUE(nn::audio::AddAux(renderer.GetConfig(), &aux, renderer.GetFinalMix(), pSendBuffer, pReturnBuffer, extBufSize).IsSuccess());

        renderer.Update();
        renderer.Wait();

        const int8_t inputChannels[]  = {0, 1, 2, 3};
        const int8_t outputChannels[] = {3, 2, 1, 0};
        const int channelCountTable[] = {1, 2, 4};
        const int channelCount = channelCountTable[i % (sizeof(channelCountTable) / sizeof(channelCountTable[0]))];

        EXPECT_LE(channelCount, channelCountMax);
        nn::audio::SetAuxInputOutput(&aux, inputChannels, outputChannels, channelCount);

        renderer.Update();
        renderer.Wait();

        int32_t buffer[16]; // 適当なバッファサイズ
        nn::audio::ReadAuxSendBuffer(&aux, buffer, sizeof(buffer) / sizeof(buffer[0]));
        renderer.Update();
        renderer.Wait();

        nn::audio::WriteAuxReturnBuffer(&aux, buffer, sizeof(buffer) / sizeof(buffer[0]));
        renderer.Update();
        renderer.Wait();

        // 外すために無効化
        nn::audio::SetAuxEnabled(&aux, false);

        // 外れるまで待機
        while(IsAuxRemovable(&aux) == false)
        {
            renderer.Update();
            renderer.Wait();
        }

        // Aux を外す
        nn::audio::RemoveAux(renderer.GetConfig(), &aux, renderer.GetFinalMix());
    }

    renderer.RendererShutdown();
    effectAllocator.Free(pSendBuffer);
    effectAllocator.Free(pReturnBuffer);
} // NOLINT(readability/fn_size)

namespace {

inline void Wait(nnt::audio::EffectScopedRenderer& renderer, nn::os::Event& fastRenderingEvent)
{
#ifdef NN_BUILD_CONFIG_OS_WIN
    nn::audio::TriggerRendering();
    fastRenderingEvent.Wait();
#endif
    renderer.Wait();
}
void CheckValue(void* workBuffer, int count, int answer, int line)
{
    auto buffer = reinterpret_cast<int32_t*>(workBuffer);
    int32_t maxVal = -1;
    for (auto i = 0; i < count; ++i)
    {
        maxVal = std::max(maxVal, buffer[i]);
    }
    if (count > 0)
    {
        ASSERT_EQ(maxVal, answer) << "Failed in Line:" << line;
    }
}

void AddAuxEffect(nnt::audio::EffectScopedRenderer& renderer, nn::audio::AuxType& aux, void* sendBuffer, void* returnBuffer, size_t bufferSize)
{
    NNT_ASSERT_RESULT_SUCCESS(
        nn::audio::AddAux(
            renderer.GetConfig(),
            &aux,
            renderer.GetSubMix(),
            reinterpret_cast<int32_t*>(sendBuffer),
            reinterpret_cast<int32_t*>(returnBuffer),
            bufferSize));
    int8_t index[1] = { 0 };
    nn::audio::SetAuxInputOutput(&aux, index, index, 1);
    renderer.Update();
}

void RemoveAuxEffect(nnt::audio::EffectScopedRenderer& renderer, nn::os::Event &fastRenderingEvent, nn::audio::AuxType& aux) NN_NOEXCEPT
{
    nn::audio::SetAuxEnabled(&aux, false);
    const int TimeOutCount = 1000;
    auto i = 0;
    for (; i < TimeOutCount; ++i)
    {
        if (nn::audio::IsAuxRemovable(&aux))
        {
            break;
        }
        renderer.Update();
        Wait(renderer, fastRenderingEvent);
    }
    ASSERT_LT(i, TimeOutCount);
    nn::audio::RemoveAux(renderer.GetConfig(), &aux, renderer.GetSubMix());
}

int ProcessAux(nn::audio::AuxType& aux, void* workBuffer, size_t bufferSize, int appendValue) NN_NOEXCEPT
{
    memset(workBuffer, 0, bufferSize);
    auto buffer = reinterpret_cast<int32_t*>(workBuffer);
    auto readCount = nn::audio::ReadAuxSendBuffer(&aux, buffer, static_cast<int>(bufferSize / sizeof(int32_t)));
    for (auto i = 0; i < readCount; ++i)
    {
        buffer[i] += appendValue;
    }
    return nn::audio::WriteAuxReturnBuffer(&aux, buffer, readCount);
}

} // namespace

TEST(TestAuxAddRemove, CheckProcessingOrder)
{
    nn::mem::StandardAllocator allocator(g_WorkBuffer, sizeof(g_WorkBuffer));
    nn::mem::StandardAllocator effectAllocator(g_WorkBuffer2, sizeof(g_WorkBuffer2));

    nn::audio::AudioRendererParameter parameter(nnt::audio::util::GetAudioRendererParameterForWaveComparison());
    parameter.effectCount = 3;
    nnt::audio::EffectScopedRenderer renderer(allocator);
    const int AuxBufferFrameCount = 3;
    auto sendReturnBufferSize = nn::audio::GetRequiredBufferSizeForAuxSendReturnBuffer(&parameter, AuxBufferFrameCount, 1);

    renderer.RendererSetup(&parameter);
    nn::audio::MemoryPoolType memoryPool;
    ASSERT_TRUE(nn::audio::AcquireMemoryPool(renderer.GetConfig(), &memoryPool, g_WorkBuffer2, sizeof(g_WorkBuffer2)));
    ASSERT_TRUE(nn::audio::RequestAttachMemoryPool(&memoryPool));

    nn::audio::AuxType aux1;
    void* sendBuffer1 = effectAllocator.Allocate(sendReturnBufferSize);
    void* returnBuffer1 = effectAllocator.Allocate(sendReturnBufferSize);
    NN_ASSERT_NOT_NULL(sendBuffer1);
    NN_ASSERT_NOT_NULL(returnBuffer1);

    nn::audio::AuxType aux2;
    void* sendBuffer2 = effectAllocator.Allocate(sendReturnBufferSize);
    void* returnBuffer2 = effectAllocator.Allocate(sendReturnBufferSize);
    NN_ASSERT_NOT_NULL(sendBuffer2);
    NN_ASSERT_NOT_NULL(returnBuffer2);

    nn::audio::AuxType aux3;
    void* sendBuffer3 = effectAllocator.Allocate(sendReturnBufferSize);
    void* returnBuffer3 = effectAllocator.Allocate(sendReturnBufferSize);
    NN_ASSERT_NOT_NULL(sendBuffer3);
    NN_ASSERT_NOT_NULL(returnBuffer3);

    void* workBuffer1 = effectAllocator.Allocate(sendReturnBufferSize);
    NN_ASSERT_NOT_NULL(workBuffer1);
    void* workBuffer2 = effectAllocator.Allocate(sendReturnBufferSize);
    NN_ASSERT_NOT_NULL(workBuffer2);
    void* workBuffer3 = effectAllocator.Allocate(sendReturnBufferSize);
    NN_ASSERT_NOT_NULL(workBuffer3);

// Watch 用 デバッグ用途に残す
//     int32_t& send1((reinterpret_cast<int32_t*>(sendBuffer1))[0x80]);
//     int32_t& send2((reinterpret_cast<int32_t*>(sendBuffer2))[0x80]);
//     int32_t& send3((reinterpret_cast<int32_t*>(sendBuffer3))[0x80]);
//     int32_t& return1((reinterpret_cast<int32_t*>(returnBuffer1))[0x80]);
//     int32_t& return2((reinterpret_cast<int32_t*>(returnBuffer2))[0x80]);
//     int32_t& return3((reinterpret_cast<int32_t*>(returnBuffer3))[0x80]);

    int writeCount1 = 0;
    int writeCount2 = 0;
    int writeCount3 = 0;

    // Renderer を開始
    nn::os::Event renderingEvent(nn::os::EventClearMode_AutoClear);
#ifdef NN_BUILD_CONFIG_OS_WIN
    nn::audio::SetFastRenderingMode(true, &renderingEvent);
#endif
    renderer.Start();
    Wait(renderer, renderingEvent);

    const int SkipCount = AuxBufferFrameCount * 2; // AuxBuffer の中身を入れ替えるために行う空読の回数
    const int TestLoopCount = SkipCount * 3;

    // aux1
    AddAuxEffect(renderer, aux1, sendBuffer1, returnBuffer1, sendReturnBufferSize);
    for (auto i = 0; i < TestLoopCount; ++i)
    {
        Wait(renderer, renderingEvent);

        writeCount1 = ProcessAux(aux1, workBuffer1, sendReturnBufferSize, 1);
        if (i > SkipCount)
        {
            CheckValue(workBuffer1, writeCount1, 1, __LINE__);
        }
    }

    // aux1 -> aux2
    AddAuxEffect(renderer, aux2, sendBuffer2, returnBuffer2, sendReturnBufferSize);
    for (auto i = 0; i < TestLoopCount; ++i)
    {
        Wait(renderer, renderingEvent);

        writeCount1 = ProcessAux(aux1, workBuffer1, sendReturnBufferSize, 1);
        writeCount2 = ProcessAux(aux2, workBuffer2, sendReturnBufferSize, 2);
        if (i > SkipCount)
        {
            CheckValue(workBuffer1, writeCount1, 1, __LINE__);
            CheckValue(workBuffer2, writeCount2, 1 + 2, __LINE__);
        }
    }

    // aux1 -> aux2 -> aux3
    AddAuxEffect(renderer, aux3, sendBuffer3, returnBuffer3, sendReturnBufferSize);
    for (auto i = 0; i < TestLoopCount; ++i)
    {
        Wait(renderer, renderingEvent);

        writeCount1 = ProcessAux(aux1, workBuffer1, sendReturnBufferSize, 1);
        writeCount2 = ProcessAux(aux2, workBuffer2, sendReturnBufferSize, 2);
        writeCount3 = ProcessAux(aux3, workBuffer3, sendReturnBufferSize, 3);
        if (i > SkipCount)
        {
            CheckValue(workBuffer1, writeCount1, 1, __LINE__);
            CheckValue(workBuffer2, writeCount2, 1 + 2, __LINE__);
            CheckValue(workBuffer3, writeCount3, 1 + 2 + 3, __LINE__);
        }
    }

    // aux1 -> aux2
    RemoveAuxEffect(renderer, renderingEvent, aux3);
    for (auto i = 0; i < TestLoopCount; ++i)
    {
        Wait(renderer, renderingEvent);

        writeCount1 = ProcessAux(aux1, workBuffer1, sendReturnBufferSize, 1);
        writeCount2 = ProcessAux(aux2, workBuffer2, sendReturnBufferSize, 2);
        if (i > SkipCount)
        {
            CheckValue(workBuffer1, writeCount1, 1, __LINE__);
            CheckValue(workBuffer2, writeCount2, 1 + 2, __LINE__);
        }
    }

    // aux1 -> aux2 -> aux3
    AddAuxEffect(renderer, aux3, sendBuffer3, returnBuffer3, sendReturnBufferSize);
    for (auto i = 0; i < TestLoopCount; ++i)
    {
        Wait(renderer, renderingEvent);

        writeCount1 = ProcessAux(aux1, workBuffer1, sendReturnBufferSize, 1);
        writeCount2 = ProcessAux(aux2, workBuffer2, sendReturnBufferSize, 2);
        writeCount3 = ProcessAux(aux3, workBuffer3, sendReturnBufferSize, 3);
        if (i > SkipCount)
        {
            CheckValue(workBuffer1, writeCount1, 1, __LINE__);
            CheckValue(workBuffer2, writeCount2, 1 + 2, __LINE__);
            CheckValue(workBuffer3, writeCount3, 1 + 2 + 3, __LINE__);
        }
    }

    // aux1 -> aux3
    RemoveAuxEffect(renderer, renderingEvent, aux2);
    for (auto i = 0; i < TestLoopCount; ++i)
    {
        Wait(renderer, renderingEvent);

        writeCount1 = ProcessAux(aux1, workBuffer1, sendReturnBufferSize, 1);
        writeCount3 = ProcessAux(aux3, workBuffer3, sendReturnBufferSize, 3);
        if (i > SkipCount)
        {
            CheckValue(workBuffer1, writeCount1, 1, __LINE__);
            CheckValue(workBuffer3, writeCount3, 1 + 3, __LINE__);
        }
    }

    // aux1 -> aux3 -> aux2
    AddAuxEffect(renderer, aux2, sendBuffer2, returnBuffer2, sendReturnBufferSize);
    for (auto i = 0; i < TestLoopCount; ++i)
    {
        Wait(renderer, renderingEvent);

        writeCount1 = ProcessAux(aux1, workBuffer1, sendReturnBufferSize, 1);
        writeCount3 = ProcessAux(aux3, workBuffer3, sendReturnBufferSize, 3);
        writeCount2 = ProcessAux(aux2, workBuffer2, sendReturnBufferSize, 2);
        if (i > SkipCount)
        {
            CheckValue(workBuffer1, writeCount1, 1, __LINE__);
            CheckValue(workBuffer3, writeCount3, 1 + 3, __LINE__);
            CheckValue(workBuffer2, writeCount2, 1 + 3 + 2, __LINE__);
        }
    }

    // aux3 -> aux2
    RemoveAuxEffect(renderer, renderingEvent, aux1);
    for (auto i = 0; i < TestLoopCount; ++i)
    {
        Wait(renderer, renderingEvent);

        writeCount3 = ProcessAux(aux3, workBuffer3, sendReturnBufferSize, 3);
        writeCount2 = ProcessAux(aux2, workBuffer2, sendReturnBufferSize, 2);
        if (i > SkipCount)
        {
            CheckValue(workBuffer3, writeCount3, 3, __LINE__);
            CheckValue(workBuffer2, writeCount2, 3 + 2, __LINE__);
        }
    }

    // aux3 -> aux2 -> aux1
    AddAuxEffect(renderer, aux1, sendBuffer1, returnBuffer1, sendReturnBufferSize);
    for (auto i = 0; i < TestLoopCount; ++i)
    {
        Wait(renderer, renderingEvent);

        writeCount3 = ProcessAux(aux3, workBuffer3, sendReturnBufferSize, 3);
        writeCount2 = ProcessAux(aux2, workBuffer2, sendReturnBufferSize, 2);
        writeCount1 = ProcessAux(aux1, workBuffer1, sendReturnBufferSize, 1);
        if (i > SkipCount)
        {
            CheckValue(workBuffer3, writeCount3, 3, __LINE__);
            CheckValue(workBuffer2, writeCount2, 3 + 2, __LINE__);
            CheckValue(workBuffer1, writeCount1, 3 + 2 + 1, __LINE__);
        }
    }

    renderer.RendererShutdown();
    effectAllocator.Free(sendBuffer1);
    effectAllocator.Free(returnBuffer1);
    effectAllocator.Free(sendBuffer2);
    effectAllocator.Free(returnBuffer2);
    effectAllocator.Free(sendBuffer3);
    effectAllocator.Free(returnBuffer3);
    effectAllocator.Free(workBuffer1);
    effectAllocator.Free(workBuffer2);
    effectAllocator.Free(workBuffer3);

    effectAllocator.Finalize();
    allocator.Finalize();
} // NOLINT(readability/fn_size)

TEST(TestAuxAddRemove, CheckProcessingOrderCleared)
{
    NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN int8_t buffer[4096 * 2];

    nn::os::SystemEvent systemEvent;
    nn::audio::AudioRendererParameter parameter;
    nn::audio::InitializeAudioRendererParameter(&parameter);
    parameter.sampleRate = 48000;
    parameter.sampleCount = 240;
    parameter.voiceCount = 1;
    parameter.effectCount = 2;
    parameter.mixBufferCount = 2;
    parameter.subMixCount = 0;


    util::ScopedAudioRenderer sar(parameter, &systemEvent);
    auto* pConfig = &sar.GetConfig();
    nn::audio::FinalMixType finalMix;
    ASSERT_TRUE(nn::audio::AcquireFinalMix(pConfig, &finalMix, 2));

    nn::audio::MemoryPoolType memoryPool;
    ASSERT_TRUE(nn::audio::AcquireMemoryPool(pConfig, &memoryPool, buffer, sizeof(buffer)));
    ASSERT_TRUE(nn::audio::RequestAttachMemoryPool(&memoryPool));

    sar.Start();

    nn::audio::AuxType aux;
    const int8_t routing[2] = { 0, 1 };
    NNT_ASSERT_RESULT_SUCCESS(nn::audio::AddAux(pConfig, &aux, &finalMix, buffer, &buffer[4096], 4096));
    nn::audio::SetAuxInputOutput(&aux, routing, routing, sizeof(routing));
    nn::audio::SetAuxEnabled(&aux, true);
    systemEvent.Wait();
    sar.Update();

    nn::audio::ReleaseFinalMix(pConfig, &finalMix);
    ASSERT_TRUE(nn::audio::RequestDetachMemoryPool(&memoryPool));
    nn::audio::SetAuxEnabled(&aux, false);

    systemEvent.Wait();
    sar.Update();

    // 未修正時は、このループ内でプロセス側に invalid access が発生していた。
    for (auto i = 0; i < 10; i++)
    {
        systemEvent.Wait();
        sar.Update();
    }
}

} // namespace audio
} // namespace nnt
