﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>  // std::max
#include <limits>     // std::numeric_limits

#include <nnt.h>

#include <nn/audio.h>
#include <nn/mem.h>
#include <nn/nn_Log.h>
#include <nn/os.h>

#include "../../programs/Eris/Sources/Libraries/audio/common/audio_NodeIdManager.h"
#include "../../Programs/Eris/Sources/Libraries/audio/audio_MixManager.h"

#include "testAudio_SimpleAudioRenderer.h"

namespace {

const nn::audio::AudioRendererParameter GetDefaultParameter()
{
    nn::audio::AudioRendererParameter param;
    nn::audio::InitializeAudioRendererParameter(&param);
    param.sampleRate = 48000;
    param.sampleCount = 240;
    param.mixBufferCount = 16;
    param.subMixCount = 8;
    param.voiceCount = 24;
    param.sinkCount = 1;
    param.effectCount = 0;
    param.performanceFrameCount = 0;
    return param;
}

NN_ALIGNAS(4096) char g_WorkBuffer[1024 * 1024];  // 4096 == nn::os::MemoryPageSize

}

class ScopedAudioRenderer
{
private:
    void* m_WorkBuffer;
    void* m_ConfigBuffer;
    nn::audio::AudioRendererHandle m_Handle;
    nn::audio::AudioRendererConfig m_Config;

    nn::mem::StandardAllocator m_Allocator;

public:
    NN_IMPLICIT ScopedAudioRenderer(const nn::audio::AudioRendererParameter& parameter = GetDefaultParameter())
    {
        m_Allocator.Initialize(g_WorkBuffer, sizeof(g_WorkBuffer));

        size_t workBufferSize = nn::audio::GetAudioRendererWorkBufferSize(parameter);
        m_WorkBuffer = m_Allocator.Allocate(workBufferSize, nn::os::MemoryPageSize);
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenAudioRenderer(&m_Handle, parameter, m_WorkBuffer, workBufferSize));

        size_t configBufferSize = nn::audio::GetAudioRendererConfigWorkBufferSize(parameter);
        m_ConfigBuffer = m_Allocator.Allocate(configBufferSize);
        nn::audio::InitializeAudioRendererConfig(&m_Config, parameter, m_ConfigBuffer, configBufferSize);

        NNT_EXPECT_RESULT_SUCCESS(nn::audio::StartAudioRenderer(m_Handle));
    }
    ~ScopedAudioRenderer()
    {
        nn::audio::StopAudioRenderer(m_Handle);
        nn::audio::CloseAudioRenderer(m_Handle);

        m_Allocator.Free(m_ConfigBuffer);
        m_Allocator.Free(m_WorkBuffer);
        m_Allocator.Finalize();
    }
    nn::audio::AudioRendererHandle GetHandle() const
    {
        return m_Handle;
    }
    nn::audio::AudioRendererConfig& GetConfig()
    {
        return m_Config;
    }
};

#if !defined(NN_SDK_BUILD_RELEASE)
TEST(ReleaseSubMixPrivate, PreCondition)
{
    ScopedAudioRenderer sar;
    nn::audio::SubMixType subMix;
    nn::audio::MixInfo invalidMixInfo;
    nn::audio::SubMixType invalidSubMix = { reinterpret_cast<nn::audio::MixInfo*>(&invalidMixInfo) };

    EXPECT_DEATH_IF_SUPPORTED(nn::audio::ReleaseSubMix(nullptr, &subMix), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::ReleaseSubMix(&sar.GetConfig(), nullptr), "");

    EXPECT_DEATH_IF_SUPPORTED(nn::audio::ReleaseSubMix(&sar.GetConfig(), &invalidSubMix), "");
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

#if !defined(NN_SDK_BUILD_RELEASE)
TEST(SetSubMixDestinationPrivate, PreCondition)
{
    ScopedAudioRenderer sar;
    nn::audio::SubMixType subMix[2];
    nn::audio::MixInfo invalidMixInfo;
    nn::audio::SubMixType invalidSubMix = { &invalidMixInfo } ;
    nn::audio::FinalMixType finalMix;
    nn::audio::FinalMixType invalidFinalMix = { &invalidMixInfo };

    EXPECT_TRUE(nn::audio::AcquireSubMix(&sar.GetConfig(), &subMix[0], 48000, 1));
    EXPECT_TRUE(nn::audio::AcquireSubMix(&sar.GetConfig(), &subMix[1], 48000, 1));
    EXPECT_TRUE(nn::audio::AcquireFinalMix(&sar.GetConfig(), &finalMix, 1));

    // Destination が SubMix の場合
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetSubMixDestination(nullptr, &subMix[0], &subMix[1]), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetSubMixDestination(&sar.GetConfig(), nullptr, &subMix[1]), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetSubMixDestination(&sar.GetConfig(), &subMix[0], static_cast<nn::audio::SubMixType*>(nullptr)), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetSubMixDestination(&sar.GetConfig(), &invalidSubMix, &subMix[1]), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetSubMixDestination(&sar.GetConfig(), &subMix[0], &invalidSubMix), "");

    // Destination が FinalMix の場合
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetSubMixDestination(nullptr, &subMix[0], &finalMix), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetSubMixDestination(&sar.GetConfig(), nullptr, &finalMix), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetSubMixDestination(&sar.GetConfig(), &subMix[0], static_cast<nn::audio::FinalMixType*>(nullptr)), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetSubMixDestination(&sar.GetConfig(), &invalidSubMix, &finalMix), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetSubMixDestination(&sar.GetConfig(), &subMix[0], &invalidFinalMix), "");
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

TEST(GetSubMixNodeId, Success)
{
    ScopedAudioRenderer sar;
    nn::audio::SubMixType subMix;

    EXPECT_TRUE(nn::audio::AcquireSubMix(&sar.GetConfig(), &subMix, 48000, 1));

    EXPECT_EQ(nn::audio::GetSubMixNodeId(&subMix), (static_cast<uint32_t>(nn::audio::common::NodeIdManager::NodeIdType::Mix) << nn::audio::common::NodeIdManager::TypeShift | subMix._pMixInfo->mixId << nn::audio::common::NodeIdManager::BaseShift));
}
