﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>

#include <cstdlib>
#include <memory>
#include <array>

#include <nn/audio.h>
#include <nn/mem.h>
#include <nn/os.h>

#include <nnt/audioUtil/testAudio_Util.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>

#if defined(NN_BUILD_CONFIG_OS_WIN)
#include <nn/audio/audio_AudioRendererApi-os.win32.h>
#endif

namespace {

NN_ALIGNAS(4096) int8_t g_WorkBuffer[1024 * 1024 * 8];
nn::mem::StandardAllocator g_Allocator(g_WorkBuffer, sizeof(g_WorkBuffer));
NN_ALIGNAS(4096) int16_t g_PcmBuffer[240];
NN_ALIGNAS(4096) int8_t g_EffectBuffer[1024 * 1024 * 4];

void* Alloc(size_t size)
{
    return g_Allocator.Allocate(size);
}

void Free(void* ptr)
{
    g_Allocator.Free(ptr);
}

enum TestState
{
    NoDetailTarget,
    DetailTargetVoice,
    DetailTargetSubMix,
    DetailTargetFinalMix,
    Finish,
};

nn::audio::PerformanceEntryType entriesAnswer[] = {
    nn::audio::PerformanceEntryType_Voice,
    nn::audio::PerformanceEntryType_SubMix,
    nn::audio::PerformanceEntryType_FinalMix,
    nn::audio::PerformanceEntryType_Sink,
};

nn::audio::PerformanceDetailType pcmDetailAnswer[] = {
    nn::audio::PerformanceDetailType_PcmInt16DataSource,
    nn::audio::PerformanceDetailType_BiquadFilter,
    nn::audio::PerformanceDetailType_Volume,
    nn::audio::PerformanceDetailType_Mix,
};

nn::audio::PerformanceDetailType subMixDetailAnswer[] = {
    nn::audio::PerformanceDetailType_Mix, // bufferMixer1
    nn::audio::PerformanceDetailType_Delay,
    nn::audio::PerformanceDetailType_Mix,
};

nn::audio::PerformanceDetailType finalMixDetailAnswer[] = {
    nn::audio::PerformanceDetailType_Mix,    // bufferMixer2
    nn::audio::PerformanceDetailType_Reverb,
    nn::audio::PerformanceDetailType_Volume,
    nn::audio::PerformanceDetailType_Volume,
};

TestState GetNextState(int& index) NN_NOEXCEPT
{
    static std::array<TestState, 8> testSequence = {{
        TestState::NoDetailTarget,
        TestState::DetailTargetVoice,
        TestState::NoDetailTarget,
        TestState::DetailTargetSubMix,
        TestState::NoDetailTarget,
        TestState::DetailTargetFinalMix,
        TestState::NoDetailTarget,
        TestState::Finish
    }};
    EXPECT_LE(index, static_cast<int>(testSequence.size()));
    return testSequence[index++];
}

void SetupBasics(nnt::audio::util::ScopedAudioRenderer& sar,
    nn::audio::VoiceType& voice,
    nn::audio::SubMixType& subMix,
    nn::audio::FinalMixType& finalMix,
    nn::audio::MemoryPoolType& memoryPool) NN_NOEXCEPT
{
    ASSERT_TRUE(nn::audio::AcquireFinalMix(&sar.GetConfig(), &finalMix, 2));
    ASSERT_TRUE(nn::audio::AcquireSubMix(&sar.GetConfig(), &subMix, 48000, nn::audio::MixBufferCountMax));
    nn::audio::SetSubMixDestination(&sar.GetConfig(), &subMix, &finalMix);
    EXPECT_TRUE(nn::audio::AcquireVoiceSlot(&sar.GetConfig(), &voice, 48000, 1, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest, nullptr, 0));
    nn::audio::SetVoiceDestination(&sar.GetConfig(), &voice, &subMix);
    nn::audio::WaveBuffer waveBuffer;
    waveBuffer.buffer = g_PcmBuffer;
    waveBuffer.size = sizeof(g_PcmBuffer);
    waveBuffer.startSampleOffset = 0;
    waveBuffer.endSampleOffset = static_cast<int32_t>(waveBuffer.size / sizeof(g_PcmBuffer[0]));
    waveBuffer.loop = true;
    waveBuffer.isEndOfStream = false;
    nn::audio::AppendWaveBuffer(&voice, &waveBuffer);
    nn::audio::SetVoicePlayState(&voice, nn::audio::VoiceType::PlayState_Play);
    nn::audio::AcquireMemoryPool(&sar.GetConfig(), &memoryPool, g_PcmBuffer, nn::util::align_up(sizeof(g_PcmBuffer), nn::audio::MemoryPoolType::SizeGranularity));
    nn::audio::RequestAttachMemoryPool(&memoryPool);

    sar.Update();
}

void CheckEntries(nn::audio::PerformanceInfo& perfInfo, nn::audio::PerformanceEntryType types[], int typeCount,
                 nn::audio::VoiceType* voice, nn::audio::SubMixType* subMix, nn::audio::FinalMixType* finalMix, nn::audio::DeviceSinkType* deviceSink) NN_NOEXCEPT
{
    int count = 0;
    auto entries = perfInfo.GetEntries(&count);
    EXPECT_EQ(count, typeCount);
    for (auto i = 0; i < count; ++i)
    {
        EXPECT_EQ(entries[i].entryType, types[i]);
        switch (types[i])
        {
        case nn::audio::PerformanceEntryType_Voice:
            EXPECT_EQ(nn::audio::GetVoiceNodeId(voice), entries[i].id);
            break;
        case nn::audio::PerformanceEntryType_SubMix:
            EXPECT_EQ(nn::audio::GetSubMixNodeId(subMix), entries[i].id);
            break;
        case nn::audio::PerformanceEntryType_FinalMix:
            EXPECT_EQ(nn::audio::GetFinalMixNodeId(finalMix), entries[i].id);
            break;
        case nn::audio::PerformanceEntryType_Sink:
            EXPECT_EQ(nn::audio::GetSinkNodeId(deviceSink), entries[i].id);
            break;
        case nn::audio::PerformanceEntryType_Unknown:
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
}

void CheckDetails(nn::audio::PerformanceInfo& perfInfo, nn::audio::PerformanceDetailType types[], int typeCount,
    nn::audio::PerformanceEntryType parentType,
    nn::audio::NodeId parentId) NN_NOEXCEPT
{
    int count = 0;
    auto details = perfInfo.GetDetails(&count);
    EXPECT_EQ(count, typeCount);
    for (auto i = 0; i < count; ++i)
    {
        EXPECT_EQ(details[i].detailType, types[i]);
        EXPECT_EQ(details[i].parentEntryType, parentType);
        EXPECT_EQ(details[i].parentId, parentId);
    }

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    if (parentType == nn::audio::PerformanceEntryType_Voice)
    {
        // The first command in command list shoule be started sooner
        const int StartTimeThreshold = 1000;  // [us]
        EXPECT_LE(details[0].startTime, StartTimeThreshold);
    }
#endif
}

}

TEST(PerformanceMetricsApis, Success)
{
    const int bufferCount = 2;
    nn::os::SystemEvent systemEvent;

    nn::audio::AudioRendererParameter parameter = nnt::audio::util::GetAudioRendererParameterForDefault();
    parameter.voiceCount = 1;
    parameter.mixBufferCount = nn::audio::MixBufferCountMax * 2;
    parameter.performanceFrameCount = 10;
    nnt::audio::util::ScopedAudioRenderer sar(parameter, &systemEvent);

    nn::audio::VoiceType voice;
    nn::audio::SubMixType subMix;
    nn::audio::FinalMixType finalMix;
    nn::audio::DeviceSinkType deviceSink;
    nn::audio::MemoryPoolType memoryPool;

    SetupBasics(sar, voice, subMix, finalMix, memoryPool);

    int8_t indices[] = { 0, 1 };
    nn::audio::AddDeviceSink(&sar.GetConfig(), &deviceSink, &finalMix, indices, 2, "MainAudioOut");

    nn::mem::StandardAllocator effectAllocator(g_EffectBuffer, sizeof(g_EffectBuffer));
    nn::audio::MemoryPoolType effectMemoryPool;
    nn::audio::AcquireMemoryPool(&sar.GetConfig(), &effectMemoryPool, g_EffectBuffer, nn::util::align_up(sizeof(g_EffectBuffer), nn::audio::MemoryPoolType::SizeGranularity));
    nn::audio::RequestAttachMemoryPool(&effectMemoryPool);

    // Add effects to SubMix
    nn::audio::BufferMixerType bufferMixer1;
    nn::audio::AddBufferMixer(&sar.GetConfig(), &bufferMixer1, &subMix);
    nn::audio::DelayType delay;
    auto delayBufferSize = nn::audio::GetRequiredBufferSizeForDelay(nn::TimeSpan::FromMilliSeconds(1), parameter.sampleRate, 1);
    auto delayBuffer = effectAllocator.Allocate(delayBufferSize);
    ASSERT_NE(delayBuffer, nullptr);
    nn::audio::AddDelay(&sar.GetConfig(), &delay, delayBuffer, delayBufferSize, &subMix, nn::TimeSpan::FromMilliSeconds(1), 1);

    // Add effects to FinalMix
    nn::audio::BufferMixerType bufferMixer2;
    nn::audio::AddBufferMixer(&sar.GetConfig(), &bufferMixer2, &finalMix);
    nn::audio::ReverbType reverb;
    auto reverbBufferSize = nn::audio::GetRequiredBufferSizeForReverb(parameter.sampleRate, 2);
    auto reverbBuffer = effectAllocator.Allocate(reverbBufferSize);
    ASSERT_NE(reverbBuffer, nullptr);
    nn::audio::AddReverb(&sar.GetConfig(), &reverb, reverbBuffer, reverbBufferSize, &finalMix, 2);

    // Graph:
    // Voice(1) -> SubMix -> FinalMix -> DeviceSink
    //               |          |
    //             Delay      Reverb
    //               |          |
    //          BufferMixer BufferMixer

    // Performance buffer settings
    auto perfSize = nn::audio::GetRequiredBufferSizeForPerformanceFrames(parameter);
    void* perfBuffer[bufferCount];
    void* prevBuffer = nullptr;
    int index = 0;
    for (auto i = 0; i < bufferCount; ++i)
    {
        perfBuffer[i] = Alloc(perfSize);
        NN_ASSERT_NOT_NULL(perfBuffer[i]);
    }
    prevBuffer = perfBuffer[index];
    ASSERT_EQ(nn::audio::SetPerformanceFrameBuffer(&sar.GetConfig(), perfBuffer[index], perfSize), nullptr);
    index = ++index % bufferCount;

    // get performance log.
    nn::audio::PerformanceInfo perfInfo;
    bool testResult = false;
    const int timeoutCountMax = 1000; // about 5 sec.
    int timeoutCount = 0;

    int stateIndex = 0;
    auto state = GetNextState(stateIndex);

#if defined(NN_BUILD_CONFIG_OS_WIN)
    nn::os::Event event(nn::os::EventClearMode_AutoClear);
    nn::audio::SetFastRenderingMode(true, &event);
    EXPECT_TRUE(nn::audio::GetFastRenderingMode());
#endif

    sar.Update();
    sar.Start();

#if defined(NN_BUILD_CONFIG_OS_WIN)
    int64_t frameIndex = -1;
#endif
    for (;;)
    {
#if defined(NN_BUILD_CONFIG_OS_WIN)
        nn::audio::TriggerRendering();
        event.Wait();
#endif
        systemEvent.Wait();
        sar.Update();

        auto consumedBuffer = nn::audio::SetPerformanceFrameBuffer(&sar.GetConfig(), perfBuffer[index], perfSize);
        EXPECT_EQ(consumedBuffer, prevBuffer);
        prevBuffer = perfBuffer[index];
        index = ++index % bufferCount;

        if (testResult || ++timeoutCount > timeoutCountMax)
        {
            break;
        }

        if (perfInfo.SetBuffer(consumedBuffer, perfSize) == false ||
            perfInfo.GetTotalProcessingTime() == 0)
        {
            continue;
        }

        do
        {
#if defined(NN_BUILD_CONFIG_OS_WIN)
            // DSP スレッドとの同期が安定しないとチェックに失敗するため Win のみでチェック
            if (frameIndex > 0)
            {
                EXPECT_EQ(perfInfo.GetFrameIndex(), frameIndex);
            }
            frameIndex = perfInfo.GetFrameIndex();
            ++frameIndex;
#endif

            // StartTime が現在時刻と乖離しすぎていないかチェック
            // 基準値はとりあえず 1 秒にしておく
            EXPECT_LT(nn::os::ConvertToTimeSpan(nn::os::GetSystemTick() - perfInfo.GetStartTime()), nn::TimeSpan::FromSeconds(1));

            switch (state)
            {
            case TestState::NoDetailTarget:
            {
                SCOPED_TRACE("TestState::NoDetailTarget");
                int count = 0;
                if (perfInfo.GetDetails(&count) == nullptr && count == 0)
                {
                    CheckEntries(perfInfo, entriesAnswer, sizeof(entriesAnswer) / sizeof(entriesAnswer[0]),
                                 &voice, &subMix, &finalMix, &deviceSink);
                    state = GetNextState(stateIndex);
                }
                else
                {
                    nn::audio::ClearPerformanceDetailTarget(&sar.GetConfig());
                }
                break;
            }
            case TestState::DetailTargetVoice:
            {
                SCOPED_TRACE("TestState::DetailTargetVoice");
                int count = 0;
                if (perfInfo.GetDetails(&count) && count > 0)
                {
                    CheckEntries(perfInfo, entriesAnswer, sizeof(entriesAnswer) / sizeof(entriesAnswer[0]),
                                 &voice, &subMix, &finalMix, &deviceSink);
                    CheckDetails(perfInfo, pcmDetailAnswer, sizeof(pcmDetailAnswer) / sizeof(pcmDetailAnswer[0]),
                        nn::audio::PerformanceEntryType_Voice, nn::audio::GetVoiceNodeId(&voice));
                    state = GetNextState(stateIndex);
                }
                else
                {
                    nn::audio::SetPerformanceDetailTarget(&sar.GetConfig(), &voice);
                }
                break;
            }
            case TestState::DetailTargetSubMix:
            {
                SCOPED_TRACE("TestState::DetailTargetSubMix");
                int count = 0;
                if (perfInfo.GetDetails(&count) && count > 0)
                {
                    CheckEntries(perfInfo, entriesAnswer, sizeof(entriesAnswer) / sizeof(entriesAnswer[0]),
                                 &voice, &subMix, &finalMix, &deviceSink);
                    CheckDetails(perfInfo, subMixDetailAnswer, sizeof(subMixDetailAnswer) / sizeof(subMixDetailAnswer[0]),
                        nn::audio::PerformanceEntryType_SubMix, nn::audio::GetSubMixNodeId(&subMix));
                    state = GetNextState(stateIndex);
                }
                else
                {
                    nn::audio::SetPerformanceDetailTarget(&sar.GetConfig(), &subMix);
                }
                break;
            }
            case TestState::DetailTargetFinalMix:
            {
                SCOPED_TRACE("TestState::DetailTargetFinalMix");
                int count = 0;
                if (perfInfo.GetDetails(&count) && count > 0)
                {
                    CheckEntries(perfInfo, entriesAnswer, sizeof(entriesAnswer) / sizeof(entriesAnswer[0]),
                                 &voice, &subMix, &finalMix, &deviceSink);
                    CheckDetails(perfInfo, finalMixDetailAnswer, sizeof(finalMixDetailAnswer) / sizeof(finalMixDetailAnswer[0]),
                        nn::audio::PerformanceEntryType_FinalMix, nn::audio::GetFinalMixNodeId(&finalMix));
                    state = GetNextState(stateIndex);
                }
                else
                {
                    nn::audio::SetPerformanceDetailTarget(&sar.GetConfig(), &finalMix);
                }
                break;
            }
            case TestState::Finish:
                testResult = true;
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
            }

        } while (perfInfo.MoveToNextFrame());

    }
    EXPECT_TRUE(testResult) << "*** Execution timeout. current state: " << state;

    nn::audio::StopAudioRenderer(sar.GetHandle());

    for (auto i = 0; i < bufferCount; ++i)
    {
        if (perfBuffer[i])
        {
            Free(perfBuffer[i]);
            perfBuffer[i] = nullptr;
        }
    }

    if (delayBuffer)
    {
        effectAllocator.Free(delayBuffer);
        delayBuffer = nullptr;
    }
    if (reverbBuffer)
    {
        effectAllocator.Free(reverbBuffer);
        reverbBuffer = nullptr;
    }
} // NOLINT(readability/fn_size)

TEST(PerformanceMetricsApis, Precondition)
{
    nn::audio::AudioRendererParameter parameter = nnt::audio::util::GetAudioRendererParameterForDefault();
    nn::os::SystemEvent systemEvent;
    parameter.performanceFrameCount = 0;
    nnt::audio::util::ScopedAudioRenderer sar(parameter, &systemEvent);

    // Append a FinalMix only.
    nn::audio::FinalMixType finalMix;
    ASSERT_TRUE(nn::audio::AcquireFinalMix(&sar.GetConfig(), &finalMix, 2));

    // Performance buffer settings
    EXPECT_EQ(nn::audio::GetRequiredBufferSizeForPerformanceFrames(parameter), 0);
    const size_t perfSize = 1024;
    void* perfBuffer = Alloc(perfSize);
    ASSERT_EQ(nn::audio::SetPerformanceFrameBuffer(&sar.GetConfig(), perfBuffer, perfSize), nullptr);

    sar.Update();
    sar.Start();

    for (auto i = 0; i < 5; ++i)
    {
        systemEvent.Wait(); // wait few frames.
    }

    sar.Update();
    void* returnedBuffer = nullptr;
    ASSERT_EQ(returnedBuffer = nn::audio::SetPerformanceFrameBuffer(&sar.GetConfig(), perfBuffer, perfSize), perfBuffer);

    // get performance log.
    //nn::audio::PerformanceInfo perfInfo;
    // EXPECT_FALSE(perfInfo.SetBuffer(returnedBuffer, perfSize)); enable after SIGLO-19077 fixed

    if (perfBuffer)
    {
        Free(perfBuffer);
        perfBuffer = nullptr;
    }
}

// 処理負荷制御・ボイスドロップは実機環境でしか動作しないため実装分岐する
#if defined(NN_BUILD_CONFIG_HARDWARE_NX) || defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2)
TEST(GetVoiceDropCount, Success)
{
    const auto voiceCount = 300;
    nn::audio::AudioRendererParameter parameter = nnt::audio::util::GetAudioRendererParameterForDefault();
    parameter.voiceCount = voiceCount;
    parameter.isVoiceDropEnabled = true;
    parameter.performanceFrameCount = 10;
    nn::os::SystemEvent systemEvent;
    nnt::audio::util::ScopedAudioRenderer sar(parameter, &systemEvent);

    // Performance buffer settings
    const int bufferCount = 2;
    auto perfSize = nn::audio::GetRequiredBufferSizeForPerformanceFrames(parameter);
    void* perfBuffer[bufferCount];
    void* prevBuffer = nullptr;
    int index = 0;
    for (auto i = 0; i < bufferCount; ++i)
    {
        perfBuffer[i] = Alloc(perfSize);
        NN_ASSERT_NOT_NULL(perfBuffer[i]);
    }
    prevBuffer = perfBuffer[index];
    ASSERT_EQ(nn::audio::SetPerformanceFrameBuffer(&sar.GetConfig(), perfBuffer[index], perfSize), nullptr);
    index = ++index % bufferCount;

    // Setup FinalMix
    nn::audio::FinalMixType finalMix;
    EXPECT_TRUE(nn::audio::AcquireFinalMix(&sar.GetConfig(), &finalMix, 1));

    // Setup MemoryPool
    nn::audio::MemoryPoolType memoryPool;
    NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN int16_t pcmBuffer[nn::audio::MemoryPoolType::SizeGranularity];
    ASSERT_TRUE(nn::audio::AcquireMemoryPool(&sar.GetConfig(), &memoryPool, pcmBuffer, sizeof(pcmBuffer)));
    ASSERT_TRUE(nn::audio::RequestAttachMemoryPool(&memoryPool));

    // Setup Voice
    nn::audio::VoiceType voices[voiceCount];
    for(auto& voice : voices)
    {
        EXPECT_TRUE(nn::audio::AcquireVoiceSlot(&sar.GetConfig(), &voice, 48000, 1, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityLowest, nullptr, 0));
        nn::audio::SetVoiceDestination(&sar.GetConfig(), &voice, &finalMix);

        nn::audio::SetVoicePlayState(&voice, nn::audio::VoiceType::PlayState_Play);

        nn::audio::WaveBuffer waveBuffer;
        waveBuffer.buffer = pcmBuffer;
        waveBuffer.size = parameter.sampleCount * sizeof(pcmBuffer[0]);
        waveBuffer.startSampleOffset = 0;
        waveBuffer.endSampleOffset = static_cast<int32_t>(waveBuffer.size / sizeof(pcmBuffer[0]));
        waveBuffer.loop = true;
        waveBuffer.isEndOfStream = false;

        nn::audio::AppendWaveBuffer(&voice, &waveBuffer);
        nn::audio::SetVoicePitch(&voice, nn::audio::VoiceType::GetPitchMax());
    }

    sar.Start();

    auto amountVoiceDropCountByPerfMetrics = 0;
    // Wait for 5 audio frames...
    for(int i = 0; i < 5; ++i)
    {
        // Sync with AudioRenderer
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));
        systemEvent.Wait();

        auto consumedBuffer = nn::audio::SetPerformanceFrameBuffer(&sar.GetConfig(), perfBuffer[index], perfSize);
        EXPECT_EQ(consumedBuffer, prevBuffer);
        prevBuffer = perfBuffer[index];
        index = ++index % bufferCount;

        nn::audio::PerformanceInfo perfInfo;
        if(perfInfo.SetBuffer(consumedBuffer, perfSize))
        {
            do
            {
                amountVoiceDropCountByPerfMetrics += perfInfo.GetVoiceDropCount();
            } while(perfInfo.MoveToNextFrame());
        }

    }

    // Check the amount of dropped voice
    auto amountVoiceDropCountByDroppedFlag = 0;
    for(auto& voice : voices)
    {
        if(nn::audio::IsVoiceDroppedFlagOn(&voice))
        {
            ++amountVoiceDropCountByDroppedFlag;
        }
    }

    EXPECT_EQ(amountVoiceDropCountByPerfMetrics, amountVoiceDropCountByDroppedFlag);
    for (auto i = 0; i < bufferCount; ++i)
    {
        if (perfBuffer[i])
        {
            Free(perfBuffer[i]);
            perfBuffer[i] = nullptr;
        }
    }
}

TEST(IsRenderingTimeLimitExceeded, Success)
{
    const int bufferCount = 2;
    const auto voiceCount = 300;
    nn::audio::AudioRendererParameter parameter = nnt::audio::util::GetAudioRendererParameterForDefault();
    parameter.voiceCount = voiceCount;
    parameter.performanceFrameCount = 10;
    nn::os::SystemEvent systemEvent;
    nnt::audio::util::ScopedAudioRenderer sar(parameter, &systemEvent);

    // Performance buffer settings
    auto perfSize = nn::audio::GetRequiredBufferSizeForPerformanceFrames(parameter);
    void* perfBuffer[bufferCount];
    void* prevBuffer = nullptr;
    int index = 0;
    for (auto i = 0; i < bufferCount; ++i)
    {
        perfBuffer[i] = Alloc(perfSize);
        NN_ASSERT_NOT_NULL(perfBuffer[i]);
    }
    prevBuffer = perfBuffer[index];
    ASSERT_EQ(nn::audio::SetPerformanceFrameBuffer(&sar.GetConfig(), perfBuffer[index], perfSize), nullptr);
    index = ++index % bufferCount;

    // Setup FinalMix
    nn::audio::FinalMixType finalMix;
    EXPECT_TRUE(nn::audio::AcquireFinalMix(&sar.GetConfig(), &finalMix, 1));

    // Setup MemoryPool
    nn::audio::MemoryPoolType memoryPool;
    NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN int16_t pcmBuffer[nn::audio::MemoryPoolType::SizeGranularity];
    ASSERT_TRUE(nn::audio::AcquireMemoryPool(&sar.GetConfig(), &memoryPool, pcmBuffer, sizeof(pcmBuffer)));
    ASSERT_TRUE(nn::audio::RequestAttachMemoryPool(&memoryPool));

    // Setup Voice
    nn::audio::VoiceType voices[voiceCount];
    for(auto& voice : voices)
    {
        EXPECT_TRUE(nn::audio::AcquireVoiceSlot(&sar.GetConfig(), &voice, 48000, 1, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest, nullptr, 0));
        nn::audio::SetVoiceDestination(&sar.GetConfig(), &voice, &finalMix);

        nn::audio::SetVoicePlayState(&voice, nn::audio::VoiceType::PlayState_Play);

        nn::audio::WaveBuffer waveBuffer;
        waveBuffer.buffer = pcmBuffer;
        waveBuffer.size = parameter.sampleCount * sizeof(pcmBuffer[0]);
        waveBuffer.startSampleOffset = 0;
        waveBuffer.endSampleOffset = static_cast<int32_t>(waveBuffer.size / sizeof(pcmBuffer[0]));
        waveBuffer.loop = true;
        waveBuffer.isEndOfStream = false;

        nn::audio::AppendWaveBuffer(&voice, &waveBuffer);
        nn::audio::SetVoicePitch(&voice, nn::audio::VoiceType::GetPitchMax());
    }

    sar.Start();

    // Wait for 5 audio frames...
    auto isRenderingTimeLimitExceeded = false;
    for(int i = 0; i < 5; ++i)
    {
        // Sync with AudioRenderer
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));
        systemEvent.Wait();

        auto consumedBuffer = nn::audio::SetPerformanceFrameBuffer(&sar.GetConfig(), perfBuffer[index], perfSize);
        EXPECT_EQ(consumedBuffer, prevBuffer);
        prevBuffer = perfBuffer[index];
        index = ++index % bufferCount;

        nn::audio::PerformanceInfo perfInfo;
        if(perfInfo.SetBuffer(consumedBuffer, perfSize))
        {
            do
            {
                if(perfInfo.IsRenderingTimeLimitExceeded())
                {
                    isRenderingTimeLimitExceeded = true;
                }
            } while(perfInfo.MoveToNextFrame());
        }
    }

    EXPECT_TRUE(isRenderingTimeLimitExceeded);

    for (auto i = 0; i < bufferCount; ++i)
    {
        if (perfBuffer[i])
        {
            Free(perfBuffer[i]);
            perfBuffer[i] = nullptr;
        }
    }
}
#endif // defined(NN_BUILD_CONFIG_HARDWARE_NX) || defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2)
