﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>  // std::max
#include <cstdlib>    // std::aligned_alloc
#include <cstring>    // std::memcmp
#include <memory>
#include <string>
#include <vector>
#include <array>

#include <nnt.h>
#include <nnt/audioUtil/testAudio_Constants.h>
#include <nnt/audioUtil/testAudio_Util.h>

#include <nn/audio.h>
#include <nn/mem.h>



#include "../../../../Programs/Eris/Sources/Libraries/audio/common/audio_IndexAllocator.h"

using namespace nn::audio;


TEST(IndexAllocator, normal)
{
    using IndexType = uint32_t;
    using Allocator = common::IndexAllocator<IndexType>;

    auto indexCnt = 32;
    auto bufSize = Allocator::CalcBufferSize(indexCnt);
    auto buf = new uint8_t[bufSize];
    Allocator allocator;
    allocator.Initialize(indexCnt, buf, bufSize);

    auto totalFreeSize = allocator.GetTotalFreeSize();
    int allocatedSize = 0;
    const int IndCount = 10;
    std::array<IndexType, IndCount> indices;
    std::array<int, IndCount> length = { 3, 2, 1, 4, 1, 2, 3, 4, 5, 7 };

    for (auto i = 0; i < IndCount; i++)
    {
        EXPECT_TRUE(allocator.Alloc(&indices[i], length[i]));
        allocatedSize += length[i];
        EXPECT_EQ(allocator.GetTotalFreeSize(), totalFreeSize - allocatedSize);
    }

    for (auto i = 0; i < IndCount; i++)
    {
        allocator.Free(indices[i], length[i]);
        allocatedSize -= length[i];
        EXPECT_EQ(allocator.GetTotalFreeSize(), totalFreeSize - allocatedSize);
    }

    delete[] buf;
}

TEST(IndexAllocator, boundary)
{
    using IndexType = uint32_t;
    using Allocator = common::IndexAllocator<IndexType>;

    auto indexCnt = 34;
    auto bufsize = Allocator::CalcBufferSize(indexCnt);
    auto buf = new uint8_t[bufsize];
    Allocator allocator;
    allocator.Initialize(indexCnt, buf, bufsize);

    auto size = static_cast<int>(allocator.GetTotalFreeSize());
    ASSERT_TRUE(size == indexCnt);
    IndexType ind1;
    IndexType ind2;
    EXPECT_TRUE(allocator.Alloc(&ind1, size / 2));
    EXPECT_TRUE(allocator.Alloc(&ind2, size / 2));

    EXPECT_EQ(allocator.GetTotalFreeSize(), 0);
    auto before = ind2;
    EXPECT_FALSE(allocator.Alloc(&ind2, 1));
    EXPECT_EQ(ind2, before);

    delete[] buf;
}

TEST(IndexAllocator, loop)
{
    using IndexType = uint8_t;
    using Allocator = common::IndexAllocator<IndexType>;

    auto indexCnt = 8;
    auto bufsize = Allocator::CalcBufferSize(indexCnt);
    auto buf = new uint8_t[bufsize];
    Allocator allocator;
    allocator.Initialize(indexCnt, buf, bufsize);

    auto size = static_cast<int>(allocator.GetTotalFreeSize());
    auto allocationUnit = 3;
    ASSERT_TRUE(size % allocationUnit); // アロケーションに端数が出る状況のテストである

    IndexType ind[8];

    EXPECT_TRUE(allocator.Alloc(&ind[0], 3));
    EXPECT_TRUE(allocator.Alloc(&ind[1], 3));
    EXPECT_FALSE(allocator.Alloc(&ind[2], 3)); // 8 を超過して失敗する
    EXPECT_TRUE(allocator.Alloc(&ind[2], 2));

    allocator.Free(ind[1], 3);
    EXPECT_FALSE(allocator.Alloc(&ind[1], 4));
    EXPECT_TRUE(allocator.Alloc(&ind[1], 3));

    delete[] buf;
}


TEST(InstanceAllocator, Success)
{
    struct DataType
    {
        uint64_t v64;
        uint32_t v32;
        // contains padding;
    };
    NN_STATIC_ASSERT(sizeof(DataType) == 16);

    using ArrayAllocator = common::InstanceAllocator<DataType>;
    auto instanceCount = 16;
    auto bufSize = ArrayAllocator::GetWorkBufferSize(instanceCount);
    auto buf = alloca(bufSize + ArrayAllocator::GetWorkBufferAlignSize() - 1);
    buf = reinterpret_cast<void*>(nn::util::align_up(reinterpret_cast<uintptr_t>(buf), ArrayAllocator::GetWorkBufferAlignSize()));

    ArrayAllocator allocator;
    allocator.Initialize(instanceCount, buf, bufSize);

    DataType* ptrs[10];
    memset(ptrs, 0, sizeof(ptrs));

    ptrs[0] = allocator.Alloc(1);
    ptrs[1] = allocator.Alloc(2);
    ptrs[2] = allocator.Alloc(3);
    ptrs[3] = allocator.Alloc(4);
    ptrs[4] = allocator.Alloc(5);
    EXPECT_TRUE(allocator.HasInstances(1));
    EXPECT_FALSE(allocator.HasInstances(2));
    EXPECT_EQ(allocator.Alloc(6), nullptr);

    EXPECT_NE(ptrs[0], nullptr);
    EXPECT_NE(ptrs[1], nullptr);
    EXPECT_NE(ptrs[2], nullptr);
    EXPECT_NE(ptrs[3], nullptr);
    EXPECT_NE(ptrs[4], nullptr);

    allocator.Free(ptrs[1], 2); ptrs[1] = nullptr;
    EXPECT_TRUE(allocator.HasInstances(3));
    EXPECT_FALSE(allocator.HasInstances(4));
    allocator.Free(ptrs[2], 3); ptrs[2] = nullptr;
    EXPECT_TRUE(allocator.HasInstances(6));
    EXPECT_FALSE(allocator.HasInstances(7));

    EXPECT_EQ(allocator.Alloc(6), nullptr); // 空は ６ あるが、断片化しており、アサイン不可能

    allocator.Free(ptrs[3], 4); ptrs[4] = nullptr;
    ptrs[5] = allocator.Alloc(6);
    EXPECT_NE(ptrs[5], nullptr);
}
