﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>
#include <cstdlib>
#include <cstring>

#include <nn/os.h>
#include <nn/audio.h>
#include <nn/audio/audio_AudioOutApi.private.h>
#include <nn/audio/audio_AudioOutTypes.private.h>
#include <nn/audio/audio_Applet.h>
#include <nn/audio/audio_Debugger.h>
#include <nn/applet/applet_Apis.h>
#include <nn/util/util_BitUtil.h>
#include <nnt/audioUtil/testAudio_Util.h>
#include "../../../Programs/Eris/Sources/Libraries/audio/common/audio_Util.h" // common::AudioOutInitializedMagic

#define NN_TEST_AUDIO_EXPECT_RANGE(value, begin, end)       \
    {                                                       \
        auto nn_test_audio_expect_range_value = (value);    \
        EXPECT_GE(nn_test_audio_expect_range_value, begin); \
        EXPECT_LT(nn_test_audio_expect_range_value, end);   \
    }

namespace {

NN_AUDIO_ALIGNAS_AUDIO_OUT_BUFFER_ALIGN char g_Buffer[nn::audio::AudioOutBuffer::SizeGranularity];

}

class ScopedAudioOut
{
public:
    ScopedAudioOut() : m_AudioOut()
    {
    }
    ~ScopedAudioOut()
    {
        if (nn::audio::GetAudioOutState(&m_AudioOut) == nn::audio::AudioOutState_Started)
        {
            nn::audio::StopAudioOut(&m_AudioOut);
        }
        nn::audio::CloseAudioOut(&m_AudioOut);
    }
    nn::audio::AudioOut* operator->()
    {
        return &m_AudioOut;
    }
    nn::audio::AudioOut* Get()
    {
        return &m_AudioOut;
    }
private:
    nn::audio::AudioOut m_AudioOut;
};

/**
 * @brief       InitializeAudioOutParameter() の正常系テストです。
 */
TEST(InitializeAudioOutParameterPrivate, Success)
{
    nn::audio::AudioOutParameter parameter;

    nn::audio::InitializeAudioOutParameter(&parameter);
    EXPECT_EQ(parameter.sampleRate, 0);
    EXPECT_EQ(parameter.channelCount, 0);
    EXPECT_EQ(parameter._reserved, nn::audio::common::AudioOutInitializedMagic);
}

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
/**
 * @brief       Debugger 向け API の正常系テストです
 */
TEST(SuspendAndResumeForDebug, Success)
{
    auto appletResourceUserId = nn::applet::GetAppletResourceUserId();

    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestSuspendAudioOutsForDebug(appletResourceUserId));
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestResumeAudioOutsForDebug(appletResourceUserId));
}

/**
 * @brief       API call test in suspended status
 */
TEST(SuspendAndResumeForDebug, Suspended)
{
    auto appletResourceUserId = nn::applet::GetAppletResourceUserId();

    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RegisterAppletResourceUserId(appletResourceUserId));

    nn::audio::AudioOutParameter parameter;
    nn::audio::InitializeAudioOutParameter(&parameter);

    {  // Open, Start, Suspend
        ScopedAudioOut aout;
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenDefaultAudioOut(aout.Get(), parameter));
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::StartAudioOut(aout.Get()));
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestSuspendAudioOutsForDebug(appletResourceUserId));

        nn::audio::AudioOutBuffer audioOutBuffer;
        nn::audio::SetAudioOutBufferInfo(&audioOutBuffer, g_Buffer, sizeof(g_Buffer), sizeof(g_Buffer));
        nn::audio::AppendAudioOutBuffer(aout.Get(), &audioOutBuffer);

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(500));

        if (nnt::audio::util::IsFirmwareSettingEnabled("suspend_for_debugger_enabled"))
        {
            EXPECT_EQ(nn::audio::GetReleasedAudioOutBuffer(aout.Get()), nullptr);
        }
        else
        {
            EXPECT_EQ(nn::audio::GetReleasedAudioOutBuffer(aout.Get()), &audioOutBuffer);
        }

        NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestResumeAudioOutsForDebug(appletResourceUserId));
    }
    {  // Open, Suspend, Start
        ScopedAudioOut aout;
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenDefaultAudioOut(aout.Get(), parameter));
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestSuspendAudioOutsForDebug(appletResourceUserId));
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::StartAudioOut(aout.Get()));

        nn::audio::AudioOutBuffer audioOutBuffer;
        nn::audio::SetAudioOutBufferInfo(&audioOutBuffer, g_Buffer, sizeof(g_Buffer), sizeof(g_Buffer));
        nn::audio::AppendAudioOutBuffer(aout.Get(), &audioOutBuffer);

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(500));

        if (nnt::audio::util::IsFirmwareSettingEnabled("suspend_for_debugger_enabled"))
        {
            EXPECT_EQ(nn::audio::GetReleasedAudioOutBuffer(aout.Get()), nullptr);
        }
        else
        {
            EXPECT_EQ(nn::audio::GetReleasedAudioOutBuffer(aout.Get()), &audioOutBuffer);
        }

        NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestResumeAudioOutsForDebug(appletResourceUserId));
    }
    {  // Suspend, Open, Start
        ScopedAudioOut aout;
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestSuspendAudioOutsForDebug(appletResourceUserId));
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenDefaultAudioOut(aout.Get(), parameter));
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::StartAudioOut(aout.Get()));

        nn::audio::AudioOutBuffer audioOutBuffer;
        nn::audio::SetAudioOutBufferInfo(&audioOutBuffer, g_Buffer, sizeof(g_Buffer), sizeof(g_Buffer));
        nn::audio::AppendAudioOutBuffer(aout.Get(), &audioOutBuffer);

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(500));

        if (nnt::audio::util::IsFirmwareSettingEnabled("suspend_for_debugger_enabled"))
        {
            EXPECT_EQ(nn::audio::GetReleasedAudioOutBuffer(aout.Get()), nullptr);
        }
        else
        {
            EXPECT_EQ(nn::audio::GetReleasedAudioOutBuffer(aout.Get()), &audioOutBuffer);
        }

        NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestResumeAudioOutsForDebug(appletResourceUserId));
    }
    {  // AudioOut is stopped and closed without resuming
        ScopedAudioOut aout;
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenDefaultAudioOut(aout.Get(), parameter));
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::StartAudioOut(aout.Get()));
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestSuspendAudioOutsForDebug(appletResourceUserId));
    }
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestResumeAudioOutsForDebug(appletResourceUserId));

    NNT_EXPECT_RESULT_SUCCESS(nn::audio::UnregisterAppletResourceUserId(appletResourceUserId));
}

/**
 * @brief       AppletManager 向け API の正常系テストです
 */
TEST(SuspendAndResume, Success)
{
    auto appletResourceUserId = nn::applet::GetAppletResourceUserId();
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RegisterAppletResourceUserId(appletResourceUserId));
    {
        // TODO: Playback by AudioOut.
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));
    }

    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestSuspendAudioOuts(appletResourceUserId, nn::TimeSpan::FromMilliSeconds(1)));
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestResumeAudioOuts(appletResourceUserId, nn::TimeSpan::FromMilliSeconds(1)));

    {
        float volume;
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::GetAudioOutsProcessMasterVolume(&volume, appletResourceUserId));
    }
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::SetAudioOutsProcessMasterVolume(appletResourceUserId, 1.0f, nn::TimeSpan::FromMilliSeconds(1)));
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::UnregisterAppletResourceUserId(appletResourceUserId));
}

/**
 * @brief       API call test in suspended status
 */
TEST(Suspend, Suspended)
{
    auto appletResourceUserId = nn::applet::GetAppletResourceUserId();

    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RegisterAppletResourceUserId(appletResourceUserId));

    nn::audio::AudioOutParameter parameter;
    nn::audio::InitializeAudioOutParameter(&parameter);

    {  // Open, Start, Suspend
        ScopedAudioOut aout;
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenDefaultAudioOut(aout.Get(), parameter));
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::StartAudioOut(aout.Get()));
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestSuspendAudioOuts(appletResourceUserId, nn::TimeSpan::FromSeconds(0)));

        // wait for AudioOut suspended without fail (TODO: this should be removed)
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(50));

        nn::audio::AudioOutBuffer audioOutBuffer;
        nn::audio::SetAudioOutBufferInfo(&audioOutBuffer, g_Buffer, sizeof(g_Buffer), sizeof(g_Buffer));
        nn::audio::AppendAudioOutBuffer(aout.Get(), &audioOutBuffer);

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(500));

        EXPECT_EQ(nn::audio::GetReleasedAudioOutBuffer(aout.Get()), nullptr);

        NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestResumeAudioOuts(appletResourceUserId, nn::TimeSpan::FromSeconds(0)));
    }
    {  // Open, Suspend, Start
        ScopedAudioOut aout;
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenDefaultAudioOut(aout.Get(), parameter));
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestSuspendAudioOuts(appletResourceUserId, nn::TimeSpan::FromSeconds(0)));
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::StartAudioOut(aout.Get()));

        // wait for AudioOut suspended without fail (TODO: this should be removed)
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(50));

        nn::audio::AudioOutBuffer audioOutBuffer;
        nn::audio::SetAudioOutBufferInfo(&audioOutBuffer, g_Buffer, sizeof(g_Buffer), sizeof(g_Buffer));
        nn::audio::AppendAudioOutBuffer(aout.Get(), &audioOutBuffer);

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(500));

        EXPECT_EQ(nn::audio::GetReleasedAudioOutBuffer(aout.Get()), nullptr);

        NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestResumeAudioOuts(appletResourceUserId, nn::TimeSpan::FromSeconds(0)));
    }
    {  // Suspend, Open, Start
        ScopedAudioOut aout;
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestSuspendAudioOuts(appletResourceUserId, nn::TimeSpan::FromSeconds(0)));
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenDefaultAudioOut(aout.Get(), parameter));
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::StartAudioOut(aout.Get()));

        // wait for AudioOut suspended without fail (TODO: this should be removed)
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(50));

        nn::audio::AudioOutBuffer audioOutBuffer;
        nn::audio::SetAudioOutBufferInfo(&audioOutBuffer, g_Buffer, sizeof(g_Buffer), sizeof(g_Buffer));
        nn::audio::AppendAudioOutBuffer(aout.Get(), &audioOutBuffer);

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(500));

        EXPECT_EQ(nn::audio::GetReleasedAudioOutBuffer(aout.Get()), nullptr);

        NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestResumeAudioOuts(appletResourceUserId, nn::TimeSpan::FromSeconds(0)));
    }
    {  // AudioOut is stopped and closed without resuming
        ScopedAudioOut aout;
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenDefaultAudioOut(aout.Get(), parameter));
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::StartAudioOut(aout.Get()));
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestSuspendAudioOuts(appletResourceUserId, nn::TimeSpan::FromSeconds(0)));
    }
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestResumeAudioOuts(appletResourceUserId, nn::TimeSpan::FromSeconds(0)));

    NNT_EXPECT_RESULT_SUCCESS(nn::audio::UnregisterAppletResourceUserId(appletResourceUserId));
}

/**
 * @brief       AppletManager 向け API の事前条件テストです。
 */
TEST(SuspendAndResume, Precondition)
{
    auto id = nn::applet::GetAppletResourceUserId();

    EXPECT_DEATH_IF_SUPPORTED(nn::audio::RequestSuspendAudioOuts(id, nn::TimeSpan::FromMilliSeconds(0 - 1)) , "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::RequestSuspendAudioOuts(id, nn::TimeSpan::FromMilliSeconds(10 + 1)) , "");
}

TEST(RegisterApplet, Success)
{
    auto id = nn::applet::GetAppletResourceUserId();
    float volume = 1.0f;
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RegisterAppletResourceUserId(id)); // Actually register. (Lookup Count = 1)
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RegisterAppletResourceUserId(id)); // Only the lookup count goes up. (Lookup Count = 2)
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::UnregisterAppletResourceUserId(id)); // Only the lookup count goes down. (Lookup Count = 1)

    nn::audio::SetAudioOutsProcessMasterVolume(id, 0.5f, 0);
    nn::audio::GetAudioOutsProcessMasterVolume(&volume, id);

    EXPECT_TRUE(volume == 0.5f);
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::UnregisterAppletResourceUserId(id)); // Actually cancel registration. (Lookup Count = 0)
}


TEST(RegisterApplet, Failure)
{
    auto id = nn::applet::AppletResourceUserId::GetInvalidId();
    NNT_EXPECT_RESULT_FAILURE(nn::audio::ResultAppletResourceUserIdNotFound, nn::audio::UnregisterAppletResourceUserId(id));

    float volume = 0.0f;
    NNT_EXPECT_RESULT_FAILURE(nn::audio::ResultAppletResourceUserIdNotFound, nn::audio::GetAudioOutsProcessMasterVolume(&volume, id));
    NNT_EXPECT_RESULT_FAILURE(nn::audio::ResultAppletResourceUserIdNotFound, nn::audio::SetAudioOutsProcessMasterVolume(id, 0.5f, 0));
    NNT_EXPECT_RESULT_FAILURE(nn::audio::ResultAppletResourceUserIdNotFound, nn::audio::RequestSuspendAudioOuts(id, nn::TimeSpan::FromMilliSeconds(10)));
    NNT_EXPECT_RESULT_FAILURE(nn::audio::ResultAppletResourceUserIdNotFound, nn::audio::RequestResumeAudioOuts(id, nn::TimeSpan::FromMilliSeconds(10)));
}
#endif // defined(NN_BUILD_CONFIG_OS_HORIZON)

TEST(SetAudioOutCountMaxForLibraryApplet, Precondition)
{
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetAudioOutCountMaxForLibraryApplet(-1), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetAudioOutCountMaxForLibraryApplet(nn::audio::AudioOutCountMaxForLibraryApplet + 1), "");
}

TEST(SetAudioOutCountMaxForLibraryApplet, Success)
{
    nn::audio::AudioOut audioOuts[nn::audio::AudioOutCountMaxForLibraryApplet];

    // Check default value
    EXPECT_EQ(nn::audio::AudioOutCountMax, nn::audio::GetAudioOutCountMaxForLibraryApplet());

    // Check min value
    nn::audio::SetAudioOutCountMaxForLibraryApplet(0);
    EXPECT_EQ(0, nn::audio::GetAudioOutCountMaxForLibraryApplet());

    // Check no audioout available
    {
        auto& audioOut = audioOuts[0];
        nn::audio::AudioOutParameter parameter;
        nn::audio::InitializeAudioOutParameter(&parameter);
        EXPECT_DEATH_IF_SUPPORTED(nn::audio::OpenDefaultAudioOut(&audioOut, parameter), "");
    }

    // Check max value
    nn::audio::SetAudioOutCountMaxForLibraryApplet(nn::audio::AudioOutCountMaxForLibraryApplet);
    EXPECT_EQ(nn::audio::AudioOutCountMaxForLibraryApplet, nn::audio::GetAudioOutCountMaxForLibraryApplet());

    // Use AudioOutCountMaxForLibraryApplet AudioOuts
    static NN_AUDIO_ALIGNAS_AUDIO_OUT_BUFFER_ALIGN char buffers[nn::audio::AudioOutCountMaxForLibraryApplet][nn::audio::AudioOutBuffer::SizeGranularity];
    nn::audio::AudioOutBuffer audioOutBuffers[nn::audio::AudioOutCountMaxForLibraryApplet];
    for(int i = 0; i < nn::audio::AudioOutCountMaxForLibraryApplet; ++i)
    {
        auto& audioOut = audioOuts[i];
        nn::audio::AudioOutParameter parameter;
        nn::audio::InitializeAudioOutParameter(&parameter);
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenDefaultAudioOut(&audioOut, parameter));
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::StartAudioOut(&audioOut));
        nn::audio::SetAudioOutBufferInfo(&audioOutBuffers[i], buffers, sizeof(buffers[0]), sizeof(buffers[0])); // 43 ms
        nn::audio::AppendAudioOutBuffer(&audioOut, &audioOutBuffers[i]);
    }

    int playbackCompletedAudioOutsCount = 0;
    const auto SleepTime = nn::TimeSpan::FromMilliSeconds(50);
    const auto TryCountMax = 10; // 500 ms
    for(int i = 0; i < TryCountMax && playbackCompletedAudioOutsCount < nn::audio::AudioOutCountMaxForLibraryApplet; ++i)
    {
        nn::os::SleepThread(SleepTime);

        for(auto& audioOut : audioOuts)
        {
            if(nn::audio::GetReleasedAudioOutBuffer(&audioOut) != nullptr)
            {
                ++playbackCompletedAudioOutsCount;
            }
        }
    }

    if(playbackCompletedAudioOutsCount != nn::audio::AudioOutCountMaxForLibraryApplet)
    {
        EXPECT_TRUE(false) << "Fail to get released audioout buffer.";
    }

    for(auto& audioOut : audioOuts)
    {
        nn::audio::StopAudioOut(&audioOut);
        nn::audio::CloseAudioOut(&audioOut);
    }

    // Set defaul value
    nn::audio::SetAudioOutCountMaxForLibraryApplet(nn::audio::AudioOutCountMax);
}

// A test process aborts if this test case is success.
#if 0
TEST(OpenAudioOut, SystemEventLeak)
{
    nn::os::SystemEvent systemEvent;
    nn::audio::AudioOutParameter parameter;
    nn::audio::InitializeAudioOutParameter(&parameter);

    for(int i = 0; i < 10000; ++i)
    {
        ScopedAudioOut aout;
        NNT_EXPECT_RESULT_SUCCESS(
            nn::audio::OpenDefaultAudioOut(aout.Get(), &systemEvent, parameter));
    }
}
#endif
