﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nnt.h>
#include <nn/os.h>
#include <nn/util/util_BitUtil.h>
#include <nn/nn_Common.h>
#include <nn/audio/audio_Result.h>

#include "../../../../../Programs/Eris/Sources/Libraries/audio/server/audio_AppletVolumeManager.h"

// ログ出力による負荷によってテストが失敗するケースがある事に注意
#define NNT_AUDIO_APPLETVOLUMEMANAGER_LOG(...)
#if !defined(NNT_AUDIO_APPLETVOLUMEMANAGER_LOG)
#include <nn/nn_Log.h>
#define NNT_AUDIO_APPLETVOLUMEMANAGER_LOG(...) NN_LOG(__VA_ARGS__)
#endif

namespace  {

class SessionStubManager
{
public:
    struct SessionStub
    {
        enum class State
        {
            Resumed,
            Suspended
        };

        float volume;
        uint32_t durationUsec;
        State state;
    };

protected:
    nn::os::Mutex m_Mutex;

    SessionStub m_InSessionStubs[nn::audio::AppletVolumeManager::MaxInSessions];
    SessionStub m_OutSessionStubs[nn::audio::AppletVolumeManager::MaxOutSessions];
    SessionStub m_RendererSessionStubs[nn::audio::AppletVolumeManager::MaxRendererSessions];
    SessionStub m_RecorderSessionStubs[nn::audio::AppletVolumeManager::MaxRecorderSessions];

public:
    SessionStubManager()
        : m_Mutex(false)
    {

    }

    SessionStub& GetSessionStub(nn::audio::AppletVolumeManager::SessionType sessionType, int index) NN_NOEXCEPT
    {
        switch (sessionType)
        {
        case nn::audio::AppletVolumeManager::SessionType_AudioIn:
            return m_InSessionStubs[index];
        case nn::audio::AppletVolumeManager::SessionType_AudioOut:
            return m_OutSessionStubs[index];
        case nn::audio::AppletVolumeManager::SessionType_AudioRenderer:
            return m_RendererSessionStubs[index];
        case nn::audio::AppletVolumeManager::SessionType_FinalOutputRecorder:
            return m_RecorderSessionStubs[index];
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    void SetVolume(nn::audio::AppletVolumeManager::SessionType sessionType, int index, float volume, uint32_t durationUsec) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        GetSessionStub(sessionType, index).volume = volume;
        GetSessionStub(sessionType, index).durationUsec = durationUsec;
    }

    float GetVolume(nn::audio::AppletVolumeManager::SessionType sessionType, int index) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        return GetSessionStub(sessionType, index).volume;
    }

    uint32_t GetDuration(nn::audio::AppletVolumeManager::SessionType sessionType, int index) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        return GetSessionStub(sessionType, index).durationUsec;
    }

    void SetState(nn::audio::AppletVolumeManager::SessionType sessionType, int index, SessionStub::State state) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        GetSessionStub(sessionType, index).state = state;
    }

    SessionStub::State GetState(nn::audio::AppletVolumeManager::SessionType sessionType, int index) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        return GetSessionStub(sessionType, index).state;
    }
};

SessionStubManager g_SessionStubManager;

} // anonymous namespace

#define NNT_AUDIO_CHECK_SESSION_STATE_SUSPENDED(sessionType, index) EXPECT_EQ(g_SessionStubManager.GetState((sessionType), (index)), SessionStubManager::SessionStub::State::Suspended)
#define NNT_AUDIO_CHECK_SESSION_STATE_RESUMED(sessionType, index) EXPECT_EQ(g_SessionStubManager.GetState((sessionType), (index)), SessionStubManager::SessionStub::State::Resumed)
#define NNT_AUDIO_CHECK_SESSION_VOLUME(sessionType, index, expected, threshold) EXPECT_NEAR(g_SessionStubManager.GetVolume((sessionType), (index)), (expected), (threshold))

// Stub
namespace nn { namespace audio { namespace dsp {

void SessionSuspend(AppletVolumeManager::SessionType type, int index) NN_NOEXCEPT
{
    NNT_AUDIO_APPLETVOLUMEMANAGER_LOG("[SessionSuspend] type(%d) index(%d)\n", type, index);
    g_SessionStubManager.SetState(type, index, SessionStubManager::SessionStub::State::Suspended);
}

void SessionResume(AppletVolumeManager::SessionType type, int index) NN_NOEXCEPT
{
    NNT_AUDIO_APPLETVOLUMEMANAGER_LOG("[SessionResume] type(%d) index(%d)\n", type, index);
    g_SessionStubManager.SetState(type, index, SessionStubManager::SessionStub::State::Resumed);
}

void SetVolume(AppletVolumeManager::SessionType type, int index, float volume, uint32_t durationUsec) NN_NOEXCEPT
{
    NNT_AUDIO_APPLETVOLUMEMANAGER_LOG("[SetVolume] type(%d) index(%d) volume(%.2f) durationUsec(%u)\n",
            type,
            index,
            volume,
            durationUsec);
    g_SessionStubManager.SetVolume(type, index, volume, durationUsec);
}

void SetRecordVolume(AppletVolumeManager::SessionType type, int index, float volume, uint32_t durationUsec) NN_NOEXCEPT
{
    NNT_AUDIO_APPLETVOLUMEMANAGER_LOG("[SetVolume] type(%d) index(%d) volume(%.2f) durationUsec(%u)\n",
            type,
            index,
            volume,
            durationUsec);
    NN_UNUSED(type);
    NN_UNUSED(index);
    NN_UNUSED(volume);
    NN_UNUSED(durationUsec);
}

}}} // namespace nn::audio::dsp

TEST(AppletVolumeManager, InitializeAndFinalize)
{
    nn::audio::AppletVolumeManager::Initialize();
    nn::audio::AppletVolumeManager::Finalize();
}

TEST(AppletVolumeManager, FinalizeWithSuspending)
{
    nn::audio::AppletVolumeManager::Initialize();

    nn::applet::AppletResourceUserId aruid = { 1 };
    const auto sessionType = nn::audio::AppletVolumeManager::SessionType_AudioOut;
    const auto index = 0;
    const auto durationNano = 1000 * 1000 * 1000; // 1000 ms

    NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::RegisterAppletResourceUserId(aruid));
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::OpenSession(sessionType, index, aruid, true));
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::Suspend(sessionType, durationNano, aruid));

    // Suspending...

    nn::audio::AppletVolumeManager::Finalize();
}

TEST(AppletVolumeManager, SleepAndWake)
{
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::Initialize());

    // Sleep -> Wake
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::Sleep());
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::Wake());

    // Wake -> Wake
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::Wake());
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::Wake());

    // Sleep -> Sleep
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::Sleep());
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::Sleep());

    NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::Finalize());
}

TEST(AppletVolumeManager, Dump)
{
    nn::audio::AppletVolumeManager::Initialize();
    nn::audio::AppletVolumeManager::Dump();
    nn::audio::AppletVolumeManager::Finalize();
}

TEST(AppletVolumeManager, RegisterAndUnregisterAppletResourceUserId)
{

    NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::Initialize());

    {
        nn::applet::AppletResourceUserId aruid = { 1 };

        // Register -> Unregster
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::RegisterAppletResourceUserId(aruid));
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::UnregisterAppletResourceUserId(aruid));

        // Register -> Register -> Unregster -> Unregster
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::RegisterAppletResourceUserId(aruid));
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::RegisterAppletResourceUserId(aruid));
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::UnregisterAppletResourceUserId(aruid));
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::UnregisterAppletResourceUserId(aruid));

        // Register -> Register -> Unregster -> Register-> Unregster -> Unregster
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::RegisterAppletResourceUserId(aruid));
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::RegisterAppletResourceUserId(aruid));
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::UnregisterAppletResourceUserId(aruid));
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::RegisterAppletResourceUserId(aruid));
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::UnregisterAppletResourceUserId(aruid));
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::UnregisterAppletResourceUserId(aruid));
    }

    // Register ARUID max count
    {
        nn::applet::AppletResourceUserId aruids[nn::audio::AppletVolumeManager::MaxRegisteredApplets];

        // Set ARUID
        for(int i = 0; i < sizeof(aruids) / sizeof(aruids[0]); ++i)
        {
            // 0 は Invalid な ARUID なので + 1
            aruids[i].lower = i + 1;
        }

        for(const auto aruid : aruids)
        {
            NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::RegisterAppletResourceUserId(aruid));
        }

        EXPECT_TRUE(nn::audio::ResultMaxAppletResourceUserId::Includes(nn::audio::AppletVolumeManager::RegisterAppletResourceUserId(aruids[0])));

        for(const auto aruid : aruids)
        {
            NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::UnregisterAppletResourceUserId(aruid));
        }

        EXPECT_TRUE(nn::audio::ResultAppletResourceUserIdNotFound::Includes(nn::audio::AppletVolumeManager::UnregisterAppletResourceUserId(aruids[0])));
    }

    NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::Finalize());
}


TEST(AppletVolumeManager, RegisterAndUnregisterAppletResourceUserIdLeakTest)
{

    NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::Initialize());

    // Register ARUID max count
    static int TestCount = 128;
    for(int testNum = 0; testNum < TestCount; ++testNum)
    {
        {
            nn::applet::AppletResourceUserId aruids[nn::audio::AppletVolumeManager::MaxRegisteredApplets];

            // Set ARUID
            for(int i = 0; i < sizeof(aruids) / sizeof(aruids[0]); ++i)
            {
                // 0 は Invalid な ARUID なので + 1
                aruids[i].lower = i + 1;
            }

            for(const auto aruid : aruids)
            {
                NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::RegisterAppletResourceUserId(aruid));
            }

            EXPECT_TRUE(nn::audio::ResultMaxAppletResourceUserId::Includes(nn::audio::AppletVolumeManager::RegisterAppletResourceUserId(aruids[0])));

            for(const auto aruid : aruids)
            {
                NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::UnregisterAppletResourceUserId(aruid));
            }

            EXPECT_TRUE(nn::audio::ResultAppletResourceUserIdNotFound::Includes(nn::audio::AppletVolumeManager::UnregisterAppletResourceUserId(aruids[0])));
        }
    }

    NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::Finalize());
}

TEST(AppletVolumeManager, SingleAppletOpenAndCloseSession)
{
    nn::applet::AppletResourceUserId aruid = { 1 };

    NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::Initialize());
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::RegisterAppletResourceUserId(aruid));

    // 1 session per 1 type
    for(int i = 0; i < nn::audio::AppletVolumeManager::NumSessionTypes; ++i)
    {
        const auto sessionType = static_cast<nn::audio::AppletVolumeManager::SessionType>(i);
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::OpenSession(sessionType, 0, aruid, true));
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::CloseSession(sessionType, 0));
    }

    NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::UnregisterAppletResourceUserId(aruid));
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::Finalize());
}

TEST(AppletVolumeManager, MultiAppletOpenAndCloseSession)
{
    // 1 applet / 1 audiorenderer でテスト
    const int appletCount = nn::audio::AppletVolumeManager::MaxRendererSessions;
    const auto sessionType = nn::audio::AppletVolumeManager::SessionType_AudioRenderer;
    nn::applet::AppletResourceUserId aruids[appletCount];

    NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::Initialize());

    // Set ARUID
    for(int i = 0; i < sizeof(aruids) / sizeof(aruids[0]); ++i)
    {
        // 0 は Invalid な ARUID なので + 1
        aruids[i].lower = i + 1;
    }

    for(int i = 0; i < sizeof(aruids) / sizeof(aruids[0]); ++i)
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::RegisterAppletResourceUserId(aruids[i]));
    }

    for(int i = 0; i < sizeof(aruids) / sizeof(aruids[0]); ++i)
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::OpenSession(sessionType, i, aruids[i], true));
    }

    for(int i = 0; i < sizeof(aruids) / sizeof(aruids[0]); ++i)
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::CloseSession(sessionType, i));
    }

    for(int i = 0; i < sizeof(aruids) / sizeof(aruids[0]); ++i)
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::UnregisterAppletResourceUserId(aruids[i]));
    }

    NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::Finalize());
}

TEST(AppletVolumeManager, SuspendAndResumeSession)
{
    nn::applet::AppletResourceUserId aruid = { 1 };
    const auto durationNano = 1000 * 1000; // 1 ms
    const auto waitTime = nn::TimeSpan::FromNanoSeconds(durationNano * 10);

    NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::Initialize());
    nn::audio::AppletVolumeManager::Dump();
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::RegisterAppletResourceUserId(aruid));

    const auto sessionType = nn::audio::AppletVolumeManager::SessionType_AudioOut;
    const auto index = 0;

    // Suspend and Resume with no session
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::OpenSession(sessionType, index, aruid, true));
        NNT_AUDIO_CHECK_SESSION_VOLUME(sessionType, index, 1.0f, 0.1f);

        NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::Suspend(sessionType, durationNano, aruid));
        nn::os::SleepThread(waitTime);
        NNT_AUDIO_CHECK_SESSION_VOLUME(sessionType, index, 0.0f, 0.1f);

        NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::Resume(sessionType, durationNano, aruid));
        nn::os::SleepThread(waitTime);
        NNT_AUDIO_CHECK_SESSION_VOLUME(sessionType, index, 1.0f, 0.1f);

        NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::CloseSession(sessionType, index));
    }

    // Suspend and Resume with 2 session
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::OpenSession(sessionType, index, aruid, true));
        NNT_AUDIO_CHECK_SESSION_VOLUME(sessionType, index, 1.0f, 0.1f);

        NNT_AUDIO_CHECK_SESSION_STATE_RESUMED(sessionType, index);
        nn::os::SleepThread(waitTime);
        NNT_AUDIO_CHECK_SESSION_VOLUME(sessionType, index, 1.0f, 0.1f);

        NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::Suspend(sessionType, durationNano, aruid));
        nn::os::SleepThread(waitTime);
        NNT_AUDIO_CHECK_SESSION_STATE_SUSPENDED(sessionType, index);
        NNT_AUDIO_CHECK_SESSION_VOLUME(sessionType, index, 0.0f, 0.1f);

        NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::Resume(sessionType, durationNano, aruid));
        nn::os::SleepThread(waitTime);
        NNT_AUDIO_CHECK_SESSION_STATE_RESUMED(sessionType, index);
        NNT_AUDIO_CHECK_SESSION_VOLUME(sessionType, index, 1.0f, 0.1f);

        NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::CloseSession(sessionType, index));
    }

    // Suspend -> Suspend -> Resume -> Resume
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::OpenSession(sessionType, index, aruid, true));
        NNT_AUDIO_CHECK_SESSION_VOLUME(sessionType, index, 1.0f, 0.1f);

        NNT_AUDIO_CHECK_SESSION_STATE_RESUMED(sessionType, index);
        nn::os::SleepThread(waitTime);
        NNT_AUDIO_CHECK_SESSION_VOLUME(sessionType, index, 1.0f, 0.1f);

        NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::Suspend(sessionType, durationNano, aruid));
        nn::os::SleepThread(waitTime);
        NNT_AUDIO_CHECK_SESSION_STATE_SUSPENDED(sessionType, index);
        NNT_AUDIO_CHECK_SESSION_VOLUME(sessionType, index, 0.0f, 0.1f);

        NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::Suspend(sessionType, durationNano, aruid));
        nn::os::SleepThread(waitTime);
        NNT_AUDIO_CHECK_SESSION_STATE_SUSPENDED(sessionType, index);
        NNT_AUDIO_CHECK_SESSION_VOLUME(sessionType, index, 0.0f, 0.1f);

        NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::Resume(sessionType, durationNano, aruid));
        nn::os::SleepThread(waitTime);
        NNT_AUDIO_CHECK_SESSION_STATE_RESUMED(sessionType, index);
        NNT_AUDIO_CHECK_SESSION_VOLUME(sessionType, index, 1.0f, 0.1f);

        NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::Resume(sessionType, durationNano, aruid));
        nn::os::SleepThread(waitTime);
        NNT_AUDIO_CHECK_SESSION_STATE_RESUMED(sessionType, index);
        NNT_AUDIO_CHECK_SESSION_VOLUME(sessionType, index, 1.0f, 0.1f);

        NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::CloseSession(sessionType, index));
    }

    const auto durationLongNano = 500 * 1000 * 1000; // 500 ms
    // Supending 中に Resume 要求
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::OpenSession(sessionType, index, aruid, true));
        NNT_AUDIO_CHECK_SESSION_VOLUME(sessionType, index, 1.0f, 0.1f);

        // Suspend
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::Suspend(sessionType, durationLongNano, aruid));
        nn::os::SleepThread(waitTime);
        // Suspend 状態になるまでは Lerp のために Resumed になるはず
        NNT_AUDIO_CHECK_SESSION_STATE_RESUMED(sessionType, index);

        // Suspending...

        // Resume
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::Resume(sessionType, durationNano, aruid));
        nn::os::SleepThread(waitTime);
        NNT_AUDIO_CHECK_SESSION_STATE_RESUMED(sessionType, index);
        NNT_AUDIO_CHECK_SESSION_VOLUME(sessionType, index, 1.0f, 0.1f);

        NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::CloseSession(sessionType, index));
    }

    // Resuming 中に Suspend 要求
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::OpenSession(sessionType, index, aruid, true));
        NNT_AUDIO_CHECK_SESSION_VOLUME(sessionType, index, 1.0f, 0.1f);

        // Resume
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::Resume(sessionType, durationLongNano, aruid));
        nn::os::SleepThread(waitTime);
        NNT_AUDIO_CHECK_SESSION_STATE_RESUMED(sessionType, index);
        NNT_AUDIO_CHECK_SESSION_VOLUME(sessionType, index, 1.0f, 0.1f);

        // Resuming...

        // Suspend
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::Suspend(sessionType, durationNano, aruid));
        nn::os::SleepThread(waitTime);
        NNT_AUDIO_CHECK_SESSION_STATE_SUSPENDED(sessionType, index);
        NNT_AUDIO_CHECK_SESSION_VOLUME(sessionType, index, 0.0f, 0.1f);

        NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::CloseSession(sessionType, index));
    }

    // Quick Suspend and Resume
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::Suspend(sessionType, durationNano, aruid));
        nn::os::SleepThread(waitTime);
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::OpenSession(sessionType, index, aruid, true));
        NNT_AUDIO_CHECK_SESSION_VOLUME(sessionType, index, 0.0f, 0.1f);

        NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::Resume(sessionType, durationNano, aruid));
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::Suspend(sessionType, durationNano, aruid));
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::Resume(sessionType, durationNano, aruid));
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
        NNT_AUDIO_CHECK_SESSION_VOLUME(sessionType, index, 1.0f, 0.1f);

        NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::CloseSession(sessionType, index));
    }


    NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::UnregisterAppletResourceUserId(aruid));

    // Degradation check for SIGLO-75312 (m_SavedVolume is not set when Session::Open() is called.)
    // Note: m_SavedVolume is renamed m_SavedAppletVolume. (SIGLO-74822)
    {
        // Set 0.0f to m_XXXSessions[0].m_SavedVolume.
        {
            NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::RegisterAppletResourceUserId(aruid));
            NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::OpenSession(sessionType, index, aruid, true));
            NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::SetAppletVolume(sessionType, 0.0f, durationNano, aruid)); // Set 0.0f to m_SavedVolume
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
            NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::Suspend(sessionType, durationNano, aruid));
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
            NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::CloseSession(sessionType, index));
            NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::UnregisterAppletResourceUserId(aruid));
        }

        // Re-use m_XXXSessions[0].
        {
            NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::RegisterAppletResourceUserId(aruid));

            NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::OpenSession(sessionType, index, aruid, true));
            NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::Suspend(sessionType, durationNano, aruid));
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));

            NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::Resume(sessionType, durationNano, aruid)); // m_SavedVolume is used here.
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
            NNT_AUDIO_CHECK_SESSION_VOLUME(sessionType, index, 1.0f, 0.1f);

            NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::CloseSession(sessionType, index));
            NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::UnregisterAppletResourceUserId(aruid));
        }
    }

    NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::Finalize());
} // NOLINT(readability/fn_size)

TEST(AppletVolumeManager, IsRunning)
{
    nn::applet::AppletResourceUserId aruid = { 1 };
    const auto durationNano = 1000 * 1000; // 1 ms
    const auto waitTime = nn::TimeSpan::FromNanoSeconds(durationNano * 10);
    const auto sessionType = nn::audio::AppletVolumeManager::SessionType_AudioRenderer;
    const auto index = 0;

    NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::Initialize());
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::RegisterAppletResourceUserId(aruid));
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::OpenSession(sessionType, 0, aruid, true));

    EXPECT_TRUE(nn::audio::AppletVolumeManager::IsRunning(sessionType, index));
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::Suspend(sessionType, durationNano, aruid));
    nn::os::SleepThread(waitTime);
    EXPECT_FALSE(nn::audio::AppletVolumeManager::IsRunning(sessionType, index));
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::Resume(sessionType, durationNano, aruid));
    nn::os::SleepThread(waitTime);
    EXPECT_TRUE(nn::audio::AppletVolumeManager::IsRunning(sessionType, index));

    NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::CloseSession(sessionType, 0));
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::Resume(sessionType, durationNano, aruid));
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::Finalize());
}

TEST(AppletVolumeManager, SetAndGetAppletVolume)
{
    nn::applet::AppletResourceUserId aruid = { 1 };

    NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::Initialize());
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::RegisterAppletResourceUserId(aruid));

    const auto sessionType = nn::audio::AppletVolumeManager::SessionType_AudioRenderer;
    const auto endVolume = 1.0f;
    const auto durationNano = 1000 * 1000; // 1 ms
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::SetAppletVolume(sessionType, endVolume, durationNano, aruid));

    float resultEndVolume;
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::GetAppletVolume(&resultEndVolume, sessionType, aruid));
    EXPECT_EQ(resultEndVolume, endVolume);

    NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::UnregisterAppletResourceUserId(aruid));
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::Finalize());
}

TEST(AppletVolumeManager, SetAndGetVolume)
{
    nn::applet::AppletResourceUserId aruid = { 1 };

    NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::Initialize());
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::RegisterAppletResourceUserId(aruid));

    const auto sessionType = nn::audio::AppletVolumeManager::SessionType_AudioRenderer;
    const auto index = 0;

    // Check default volume.
    const auto defaultVolume = 1.0f;
    float volume = 0.0f;
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::GetVolume(&volume, sessionType, index, aruid));
    EXPECT_EQ(defaultVolume, volume);

    // Set volume and check it
    const float updateVolume = 0.5f;
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::SetVolume(sessionType, index, updateVolume, aruid));
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::GetVolume(&volume, sessionType, index, aruid));
    EXPECT_EQ(updateVolume, volume);

    NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::UnregisterAppletResourceUserId(aruid));
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::Finalize());
}

TEST(AppletVolumeManager, SetAndGetDevice)
{
    nn::applet::AppletResourceUserId aruid = { 1 };

    NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::Initialize());
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::RegisterAppletResourceUserId(aruid));

    const auto deviceType = nn::audio::AppletVolumeManager::DeviceType_AhubSpeaker;
    const auto channelCount = 2; // Stereo

    NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::SetDevice(deviceType, channelCount));

    nn::audio::AppletVolumeManager::DeviceType resultDeviceType;
    nn::audio::AppletVolumeManager::GetActiveDevice(&resultDeviceType);
    EXPECT_EQ(resultDeviceType, deviceType);

    int resultChannelCount = nn::audio::AppletVolumeManager::GetDeviceChannelCount();
    EXPECT_EQ(resultChannelCount, channelCount);

    NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::UnregisterAppletResourceUserId(aruid));
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::Finalize());
}

TEST(AppletVolumeManager, StartDeviceFadeInAndOut)
{
    nn::applet::AppletResourceUserId aruid = { 1 };
    nn::os::Event event(nn::os::EventClearMode_AutoClear);
    const auto durationNano = 1000 * 1000; // 1 ms
    const auto waitTime = nn::TimeSpan::FromNanoSeconds(durationNano * 100);

    NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::Initialize());
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::RegisterAppletResourceUserId(aruid));

    EXPECT_TRUE(nn::audio::AppletVolumeManager::StartDeviceFadeOut(&event, durationNano));
#if defined(NN_BUILD_CONFIG_SPEC_NX)
    // Return FALSE right after fading started.
    EXPECT_FALSE(nn::audio::AppletVolumeManager::StartDeviceFadeIn(&event, durationNano));
#endif
    nn::os::SleepThread(waitTime);
    EXPECT_TRUE(nn::audio::AppletVolumeManager::StartDeviceFadeIn(&event, durationNano));
    nn::os::SleepThread(waitTime);

    NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::UnregisterAppletResourceUserId(aruid));
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::Finalize());
}

TEST(AppletVolumeManager, QueryDeviceEvent)
{
    nn::os::NativeHandle nativeHandle;
    nn::applet::AppletResourceUserId aruid = { 1 };
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::Initialize());
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::RegisterAppletResourceUserId(aruid));

    // SIGLO-53208 以降、
    // QueryDeviceEvent() は AppletId が registered な状態であることを事前条件とする
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::QueryDeviceEvent(&nativeHandle, aruid));

    NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::UnregisterAppletResourceUserId(aruid));
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::Finalize());
}

TEST(AppletVolumeManager, SetAndGetDeviceVolume)
{
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::Initialize());

    nn::applet::AppletResourceUserId aruid = { 1 };
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::RegisterAppletResourceUserId(aruid));

    for(int i = 0; i < nn::audio::AppletVolumeManager::DeviceType_Count; ++i)
    {

        for(int j = 0; j < nn::audio::AppletVolumeManager::NumSessionTypes; ++j)
        {
            const auto sessionType = static_cast<nn::audio::AppletVolumeManager::SessionType>(j);
            const auto index = 0;
            float deviceVolume;
            const auto deviceType = static_cast<nn::audio::AppletVolumeManager::DeviceType>(i);
            const auto initialDeviceVolume = 1.0f;
            const auto testDeviceVolume = 0.3f; // 初期値以外の値

            NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::OpenSession(sessionType, index, aruid, true));

            // 初期値チェック
            nn::audio::AppletVolumeManager::GetDeviceVolume(&deviceVolume, deviceType, sessionType, aruid);
            EXPECT_NEAR(deviceVolume, initialDeviceVolume, 0.1f);

            nn::audio::AppletVolumeManager::SetDeviceVolume(testDeviceVolume, deviceType, sessionType, aruid);
            nn::audio::AppletVolumeManager::GetDeviceVolume(&deviceVolume, deviceType, sessionType, aruid);
            EXPECT_NEAR(deviceVolume, testDeviceVolume, 0.1f);

            NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::CloseSession(sessionType, index));
        }
    }

    NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::UnregisterAppletResourceUserId(aruid));
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::AppletVolumeManager::Finalize());
}
