﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "common.fsid"

#include <nn/atk.h>
#include <nn/mem.h>
#include <nn/nn_Log.h>

#include <nnt.h>
#include <nnt/atkUtil/testAtk_Util.h>
#include <nnt/atkUtil/testAtk_CommonSetup.h>


namespace {

const int MemoryHeapSize = 32 * 1024 * 1024;

// Create 以外の SoundHeap テストを追加する際に使うと思われるので、一旦これらをコメントアウト
// nnt::atk::util::FsCommonSetup   g_FsSetup;
// nnt::atk::util::AtkCommonSetup   g_AtkSetup;

class SoundHeapSetup : public nnt::atk::util::AtkCommonSetup
{
public:
    void Initialize(nn::mem::StandardAllocator& allocator) NN_NOEXCEPT NN_OVERRIDE
    {
        nnt::atk::util::AtkCommonSetup::InitializeParam param;
        this->Initialize(param, allocator);
    }

    // テスト対象である SoundHeap の初期化はクラス外で行う
    void Initialize(nnt::atk::util::AtkCommonSetup::InitializeParam param, nn::mem::StandardAllocator& allocator) NN_NOEXCEPT NN_OVERRIDE
    {
        InitializeSoundSystem(param.GetSoundSystemParam(), allocator);
    }

    virtual void Finalize(nn::mem::StandardAllocator& allocator) NN_NOEXCEPT NN_OVERRIDE
    {
        FinalizeSoundSystem(allocator);
    }
};

SoundHeapSetup g_SoundHeapSetup;

static char                 g_HeapMemory[MemoryHeapSize];
nn::mem::StandardAllocator  g_Allocator;
}


#if !defined(NN_SDK_BUILD_RELEASE)
TEST(SoundHeap, CreateDeathTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_SoundHeapSetup.Initialize(g_Allocator);

    nn::atk::SoundHeap soundHeap;
    const int SoundHeapSize = 4 * 1024 * 1024;
    EXPECT_DEATH_IF_SUPPORTED(soundHeap.Create(nullptr, SoundHeapSize), "");
    soundHeap.Destroy();

    g_SoundHeapSetup.Finalize(g_Allocator);
    g_Allocator.Finalize();
}
#endif

TEST(SoundHeap, CreateTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_SoundHeapSetup.Initialize(g_Allocator);

    nn::atk::SoundHeap soundHeap;
    const int SoundHeapSize = 4 * 1024 * 1024;
    void* pMemoryForSoundHeap = g_Allocator.Allocate(SoundHeapSize, nn::audio::MemoryPoolType::AddressAlignment);
    ASSERT_NE(pMemoryForSoundHeap, nullptr);

    // 正常系
    EXPECT_TRUE(soundHeap.Create(pMemoryForSoundHeap, SoundHeapSize));
    soundHeap.Destroy();

    // アラインを 1 ずらしても Create に成功するかをチェックする (For SIGLO-47289)
    const int AlignDifference = 1;
    EXPECT_TRUE(soundHeap.Create(reinterpret_cast<char*>(pMemoryForSoundHeap) + AlignDifference, SoundHeapSize - AlignDifference));
    soundHeap.Destroy();

    g_Allocator.Free(pMemoryForSoundHeap);
    g_SoundHeapSetup.Finalize(g_Allocator);
    g_Allocator.Finalize();
}

TEST(SoundHeap, CreateWithShortHeapSizeTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_SoundHeapSetup.Initialize(g_Allocator);

    nn::atk::SoundHeap soundHeap;
    const int SoundHeapSize = nn::audio::MemoryPoolType::AddressAlignment;
    void* pMemoryForSoundHeap = g_Allocator.Allocate(SoundHeapSize, nn::audio::MemoryPoolType::AddressAlignment);
    ASSERT_NE(pMemoryForSoundHeap, nullptr);

    // サイズ 0 で確保しようとした場合は失敗
    EXPECT_FALSE(soundHeap.Create(pMemoryForSoundHeap, 0));
    soundHeap.Destroy();

    // nn::audio::MemoryPoolType::AddressAlignment のアラインに沿って確保する場合は成功
    EXPECT_TRUE(soundHeap.Create(pMemoryForSoundHeap, SoundHeapSize));
    soundHeap.Destroy();

    // nn::audio::MemoryPoolType::AddressAlignment のアラインに沿わずアライン以下のサイズで確保する場合は失敗
    const int AlignDifference = 1;
    EXPECT_FALSE(soundHeap.Create(reinterpret_cast<char*>(pMemoryForSoundHeap) + AlignDifference, SoundHeapSize - AlignDifference));
    soundHeap.Destroy();

    g_Allocator.Free(pMemoryForSoundHeap);
    g_SoundHeapSetup.Finalize(g_Allocator);
    g_Allocator.Finalize();
}
