﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "common.fsid"

#include <nnt.h>
#include <nnt/atkUtil/testAtk_Util.h>
#include <nnt/atkUtil/testAtk_CommonSetup.h>
#include <nn/atk.h>
#include <nn/nn_Log.h>
#include <nn/mem.h>

namespace {

const int MemoryHeapSize = 32 * 1024 * 1024;
const auto WaitTime = nn::TimeSpan::FromMilliSeconds(16);
const auto TimeOut = nn::TimeSpan::FromMilliSeconds(5000);

nnt::atk::util::FsCommonSetup   g_FsSetup;
nnt::atk::util::AtkCommonSetup  g_AtkSetup;

static char g_HeapMemory[MemoryHeapSize];
nn::mem::StandardAllocator  g_Allocator;

#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
//  UpdateAndWaitForStrmPrepare 関数で VoiceCommand の確認ができているか
bool g_IsCheckedVoiceCommandInFunctionOfUpdateAndWaitForStrmPrepare = false;
//  UpdateAndWaitForStrmStart 関数で VoiceCommand の確認ができているか
bool g_IsCheckedVoiceCommandInFunctionOfUpdateAndWaitForStrmStart = false;
#endif

void UpdateAndWaitForStrmPrepare(nn::atk::StreamSoundHandle& streamSoundHandle)
{
    // ストリームサウンドの場合は、サウンドアーカイブプレイヤーで再生・再生準備をしても、すぐに再生準備が完了しない。
    // 一旦サウンドアーカイブプレイヤーを Update() して再生準備のコマンドを送り、再生準備が完了するのを待つ必要がある。
    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    soundArchivePlayer.Update();

#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
    if( g_IsCheckedVoiceCommandInFunctionOfUpdateAndWaitForStrmPrepare == false )
    {
        //  VoiceCommandProcess を呼ばなければ Prepare されないことを確認
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(2000));
        EXPECT_FALSE(streamSoundHandle.IsPrepared()) << "Stream sound was prepared without voice commands.";

        //  2 回目のチェックはしなくてもいい
        g_IsCheckedVoiceCommandInFunctionOfUpdateAndWaitForStrmPrepare = true;
    }
#endif

    for(auto elapsed = nn::TimeSpan::FromMilliSeconds(0); elapsed < TimeOut; elapsed += WaitTime)
    {
#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
        nn::atk::SoundSystem::VoiceCommandProcess(6);
#endif
        nn::os::SleepThread(WaitTime);
        if(streamSoundHandle.IsPrepared())
        {
            break;
        }
    }
    EXPECT_TRUE(streamSoundHandle.IsPrepared()) << "Preparing stream sound timed out.";
}

void UpdateAndWaitForStrmStart(nn::atk::StreamSoundHandle& streamSoundHandle)
{
    // 実際に再生が行われていることをパラメータ上で確認できるまで待つ
    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    soundArchivePlayer.Update();

#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
    if( g_IsCheckedVoiceCommandInFunctionOfUpdateAndWaitForStrmStart == false )
    {
        //  VoiceCommandProcess を呼ばなければ再生されないことを確認
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(2000));
        EXPECT_EQ( 0, streamSoundHandle.GetPlaySamplePosition()) << "Stream sound was started without voice commands.";

        //  2 回目のチェックはしなくてもいい
        g_IsCheckedVoiceCommandInFunctionOfUpdateAndWaitForStrmStart = true;
    }
#endif

    for(auto elapsed = nn::TimeSpan::FromMilliSeconds(0); elapsed < TimeOut; elapsed += WaitTime)
    {
#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
        nn::atk::SoundSystem::VoiceCommandProcess(6);
#endif
        nn::os::SleepThread(WaitTime);
        if(streamSoundHandle.GetPlaySamplePosition() != 0)
        {
            break;
        }
    }
    EXPECT_TRUE(streamSoundHandle.GetPlaySamplePosition() != 0) << "Start stream sound timed out.";
}

}

TEST(StreamSoundHandle, PlaySamplePositionTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();
    g_AtkSetup.Initialize(g_Allocator);

    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundHandle soundHandle;

    nn::atk::StreamSoundHandle invalidStreamSoundHandle(&soundHandle);
    // IsSuspendByLoadingDelay() は基本的には常に false
    EXPECT_FALSE(invalidStreamSoundHandle.IsSuspendByLoadingDelay());

    EXPECT_TRUE(soundArchivePlayer.PrepareSound(&soundHandle, STRM_MARIOKART).IsSuccess());
    nn::atk::StreamSoundHandle streamSoundHandle(&soundHandle);
    UpdateAndWaitForStrmPrepare(streamSoundHandle);

    EXPECT_FALSE(streamSoundHandle.IsSuspendByLoadingDelay());

    int64_t currentPosition = streamSoundHandle.GetPlaySamplePosition();
    EXPECT_EQ(0, currentPosition);
    EXPECT_EQ(0, streamSoundHandle.GetPlayLoopCount());
    streamSoundHandle.StartPrepared();
    UpdateAndWaitForStrmStart(streamSoundHandle);

    uint32_t commandTag = 0;
    while(!nnt::atk::util::SendDummyCommand(&commandTag))
    {
        soundArchivePlayer.Update();
#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
        nn::atk::SoundSystem::VoiceCommandProcess(6);
#endif
        nn::os::SleepThread(WaitTime);
    }

    int64_t previousPosition = currentPosition;
    bool isCommandSend = true;
    int testCount = 0;
    const int TestCountMax = 10;

    while(testCount < TestCountMax)
    {
        // 音源の再生が終了していた場合はそのまま終了
        if(!soundHandle.IsAttachedSound())
        {
            break;
        }

        if(isCommandSend)
        {
            if(nnt::atk::util::IsCommandFinished(commandTag))
            {
                currentPosition = streamSoundHandle.GetPlaySamplePosition();

#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
                //  vcmd 版では進まないこともあるため >= で判定します。(SIGLO-52881)
                EXPECT_GE(currentPosition, previousPosition);
#else
                EXPECT_GT(currentPosition, previousPosition);
#endif

                previousPosition = currentPosition;
                ++testCount;
                isCommandSend = nnt::atk::util::SendDummyCommand(&commandTag);
            }
        }
        else
        {
            isCommandSend = nnt::atk::util::SendDummyCommand(&commandTag);
        }

        soundArchivePlayer.Update();
#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
        nn::atk::SoundSystem::VoiceCommandProcess(6);
#endif
        nn::os::SleepThread(WaitTime);
    }

    EXPECT_FALSE(streamSoundHandle.IsSuspendByLoadingDelay());

    g_AtkSetup.Finalize(g_Allocator);
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}

TEST(StreamSoundHandle, PlaySamplePositionTestWithNoLoop)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();
    g_AtkSetup.Initialize(g_Allocator);

    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundHandle soundHandle;

    nn::atk::StreamSoundHandle invalidStreamSoundHandle(&soundHandle);
    // IsSuspendByLoadingDelay() は基本的には常に false
    EXPECT_FALSE(invalidStreamSoundHandle.IsSuspendByLoadingDelay());

    EXPECT_TRUE(soundArchivePlayer.PrepareSound(&soundHandle, STRM_MARIOKART_INTRO).IsSuccess());
    nn::atk::StreamSoundHandle streamSoundHandle(&soundHandle);
    UpdateAndWaitForStrmPrepare(streamSoundHandle);

    EXPECT_FALSE(streamSoundHandle.IsSuspendByLoadingDelay());

    int64_t currentPosition = streamSoundHandle.GetPlaySamplePosition();
    EXPECT_EQ(0, currentPosition);
    EXPECT_EQ(0, streamSoundHandle.GetPlayLoopCount());
    streamSoundHandle.StartPrepared();
    UpdateAndWaitForStrmStart(streamSoundHandle);

    int64_t previousPosition = currentPosition;
    int testFrameCount = 0;
    const int TestFrameCountMax = 1000;

    while (testFrameCount < TestFrameCountMax)
    {
        // 再生位置取得後に音源の再生が終わる可能性があることを考慮して、最初に再生位置を保持しておく
        currentPosition = streamSoundHandle.GetPlaySamplePosition();
        // 音源の再生が終了していた場合はそのまま終了
        if (!soundHandle.IsAttachedSound())
        {
            break;
        }

        EXPECT_GE(currentPosition, previousPosition);
        previousPosition = currentPosition;
        ++testFrameCount;

        soundArchivePlayer.Update();
#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
        nn::atk::SoundSystem::VoiceCommandProcess(6);
#endif
        nn::os::SleepThread(WaitTime);
    }

    EXPECT_FALSE(soundHandle.IsAttachedSound()) << "StreamSound playing timed out.";

    g_AtkSetup.Finalize(g_Allocator);
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}

TEST(StreamSoundHandle, StreamSoundDataInfoTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();
    g_AtkSetup.Initialize(g_Allocator);

    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundHandle soundHandle;
    nn::atk::StreamSoundDataInfo info;

    nn::atk::StreamSoundHandle invalidStreamSoundHandle(&soundHandle);
    EXPECT_FALSE(invalidStreamSoundHandle.ReadStreamSoundDataInfo(&info));

    EXPECT_TRUE(soundArchivePlayer.PrepareSound(&soundHandle, STRM_MARIOKART).IsSuccess());
    nn::atk::StreamSoundHandle streamSoundHandle(&soundHandle);
    UpdateAndWaitForStrmPrepare(streamSoundHandle);
    EXPECT_TRUE(streamSoundHandle.ReadStreamSoundDataInfo(&info));

    EXPECT_NE(0, info.loopEnd);
    // テストデータである STRM_MARIOKART は 32000 Hz のサンプルレート
    EXPECT_EQ(32000, info.sampleRate);

    g_AtkSetup.Finalize(g_Allocator);
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}

#ifndef NN_SDK_BUILD_RELEASE
TEST(StreamSoundHandle, InvalidMixVolumeChannelTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();
    g_AtkSetup.Initialize(g_Allocator);

    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundHandle soundHandle;

    EXPECT_TRUE(soundArchivePlayer.PrepareSound(&soundHandle, STRM_MARIOKART).IsSuccess());
    nn::atk::StreamSoundHandle streamSoundHandle(&soundHandle);
    soundArchivePlayer.Update();
#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
    while ( !streamSoundHandle.IsPrepared() )
    {
        nn::atk::SoundSystem::VoiceCommandProcess(4);
        nn::os::SleepThread(WaitTime);
    }
#endif

    // 5.1ch 音量設定のチェック
    nn::atk::MixVolume mixVolume;
    nn::atk::StreamSoundHandle::TrackBitFlagSet bitFlag;
    bitFlag.Reset();
    bitFlag[0] = true;
    EXPECT_DEATH_IF_SUPPORTED(streamSoundHandle.SetMixVolume(-1, mixVolume), ".*");
    EXPECT_DEATH_IF_SUPPORTED(streamSoundHandle.SetMixVolume(nn::atk::WaveChannelMax, mixVolume), ".*");
    EXPECT_DEATH_IF_SUPPORTED(streamSoundHandle.SetTrackMixVolume(bitFlag, -1, mixVolume), ".*");
    EXPECT_DEATH_IF_SUPPORTED(streamSoundHandle.SetTrackMixVolume(bitFlag, nn::atk::WaveChannelMax, mixVolume), ".*");

    g_AtkSetup.Finalize(g_Allocator);
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}

nn::atk::StreamRegionCallbackResult StreamInvalidRegionIndexCallback(nn::atk::StreamRegionCallbackParam* param, void* /*arg*/)
{
    // リージョン数以上のリージョン番号を指定してアサートを発生させる
    param->regionNo = param->regionCount;
    return nn::atk::StreamRegionCallbackResult_Continue;
}
void ProcessInvalidRegionIndexCallbackForDeathTest(nn::atk::SoundArchivePlayer& soundArchivePlayer) NN_NOEXCEPT
{
    // 不正なリージョン番号を指定するリージョンコールバックを設定
    nn::atk::SoundHandle soundHandle;
    nn::atk::SoundStartable::StartInfo info;
    info.enableFlag |= nn::atk::SoundStartable::StartInfo::EnableFlagBit_StreamSoundInfo;
    info.streamSoundInfo.regionCallback = StreamInvalidRegionIndexCallback;
    info.streamSoundInfo.regionCallbackArg = nullptr;

    // StreamSoundPlayer のプリペア待ち(StreamSoundLoader のタスクが TaskThread にアペンドされる)
    EXPECT_TRUE(soundArchivePlayer.PrepareSound(&soundHandle, STRM_NAMEDREGION_TEST, &info).IsSuccess());
    nn::atk::StreamSoundHandle streamSoundHandle(&soundHandle);
    UpdateAndWaitForStrmPrepare(streamSoundHandle);

    // SoundArchivePlayer::Finalize() の StreamSoundLoader:WaitFinalize() で ChangeRegion() のアサートを発生させる
    g_AtkSetup.Finalize(g_Allocator);
}
TEST(StreamSoundHandle, InvalidRegionIndexDeathTest)
{
    nnt::atk::util::OnPreAtkTest();
    // StreamSoundHandle を使ったテストではないが、STRM の再生リクエスト関連のテストのためここで行う
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();
    g_AtkSetup.Initialize(g_Allocator);

    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    EXPECT_DEATH_IF_SUPPORTED(ProcessInvalidRegionIndexCallbackForDeathTest(soundArchivePlayer), ".*");

    g_AtkSetup.Finalize(g_Allocator);
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}
#endif
