﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "common.fsid"

#include <nnt.h>
#include <nnt/atkUtil/testAtk_Util.h>
#include <nnt/atkUtil/testAtk_CommonSetup.h>
#include <nn/atk.h>
#include <nn/nn_Log.h>
#include <nn/mem.h>

namespace {

const int MemoryHeapSize = 32 * 1024 * 1024;
const unsigned int TestData[] = {SEQ_MARIOKART, WSD_SNARE, STRM_MARIOKART};
const int FadeTime[] = {0, 5};

nnt::atk::util::FsCommonSetup   g_FsSetup;
nnt::atk::util::AtkCommonSetup  g_AtkSetup;

static char g_HeapMemory[MemoryHeapSize];
nn::mem::StandardAllocator  g_Allocator;

void LoadData()
{
    // 音源データの読み込み
    g_AtkSetup.LoadData(SEQ_MARIOKART, "SEQ_MARIOKART");
    g_AtkSetup.LoadData(WSD_SNARE, "WSD_SNARE");
}

template<class T>
void LoadData(T& setup)
{
    setup.LoadData(SEQ_MARIOKART, "SEQ_MARIOKART");
    setup.LoadData(WSD_SNARE, "WSD_SNARE");
}

void UpdateAndWaitForStrmPrepare(nn::atk::SoundHandle& soundHandle)
{
    const auto WaitTime = nn::TimeSpan::FromMilliSeconds(16);
    const auto TimeOut = nn::TimeSpan::FromMilliSeconds(5000);

    // ストリームサウンドの場合は、サウンドアーカイブプレイヤーで再生・再生準備をしても、すぐに再生準備が完了しない。
    // 一旦サウンドアーカイブプレイヤーを Update() して再生準備のコマンドを送り、再生準備が完了するのを待つ必要がある。
    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    soundArchivePlayer.Update();

    for(auto elapsed = nn::TimeSpan::FromMilliSeconds(0); elapsed < TimeOut; elapsed += WaitTime)
    {
#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
        nn::atk::SoundSystem::VoiceCommandProcess(6);
#endif
        nn::os::SleepThread(WaitTime);
        if(soundHandle.IsPrepared())
        {
            break;
        }
    }
    EXPECT_TRUE(soundHandle.IsPrepared()) << "Preparing stream sound timed out.";
}

void UpdateAndWaitAudioFrame()
{
    const auto WaitTime = nn::TimeSpan::FromMilliSeconds(5);
    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();

    soundArchivePlayer.Update();
#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
    nn::atk::SoundSystem::VoiceCommandProcess(1);
#endif
    nn::os::SleepThread(WaitTime);
}

void UpdateAndWaitAudioFrame(nn::atk::SoundArchivePlayer& player)
{
    const auto WaitTime = nn::TimeSpan::FromMilliSeconds(5);

    player.Update();
#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
    nn::atk::SoundSystem::VoiceCommandProcess(1);
#endif
    nn::os::SleepThread(WaitTime);
}

void ForceStopSound(nn::atk::SoundHandle& handle, nn::atk::SoundArchivePlayer& player)
{
    handle.Stop(1);
    player.Update();
    player.Update();
#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
    nn::atk::SoundSystem::VoiceCommandProcess(2);
#endif
}

void UpdateSoundArchivePlayer(nn::atk::SoundArchivePlayer& soundArchivePlayer)
{
    soundArchivePlayer.Update();
#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
    //  VoiceCommand 版では VoiceCommandProcess の呼び出しも行う
    //  Wait を行わないため、引数の frameCount には 1 を与える
    nn::atk::SoundSystem::VoiceCommandProcess(1);
#endif
}

void PauseSoundHandle(nn::atk::SoundHandle& soundHandle, bool flag, int fadeFrames, nn::atk::PauseMode pauseMode)
{
    soundHandle.Pause(flag, fadeFrames, pauseMode);
#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
    //  VoiceCommand 版では VoiceCommandProcess の呼び出しも行う
    //  Wait を行わないため、引数の frameCount には 1 を与える
    nn::atk::SoundSystem::VoiceCommandProcess(1);
#endif
}

void PauseSoundHandle(nn::atk::SoundHandle& soundHandle, bool flag, int fadeFrames)
{
    soundHandle.Pause( flag, fadeFrames );
#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
    //  VoiceCommand 版では VoiceCommandProcess の呼び出しも行う
    //  Wait を行わないため、引数の frameCount には 1 を与える
    nn::atk::SoundSystem::VoiceCommandProcess(1);
#endif
}

void MuteSoundHandle(nn::atk::SoundHandle& soundHandle, bool flag, int fadeFrames)
{
    soundHandle.Mute( flag, fadeFrames );
#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
    //  VoiceCommand 版では VoiceCommandProcess の呼び出しも行う
    //  Wait を行わないため、引数の frameCount には 1 を与える
    nn::atk::SoundSystem::VoiceCommandProcess(1);
#endif
}

class CustomSubMixSetup : public nnt::atk::util::AtkCommonSetup
{
public:
    struct SoundSystemCondition
    {
        int busCountMax;
        int totalChannelCount;
        bool isBusMixVolumeEnabled;
        bool isVolumeThroughModeEnabled;
    };

    struct SubMixCondition
    {
        int srcBusCount;
        int srcChannelCount;
    };

    void Initialize(nn::mem::StandardAllocator& allocator) NN_NOEXCEPT NN_OVERRIDE
    {
        AtkCommonSetup::Initialize(allocator);
    }
    void Initialize(nnt::atk::util::AtkCommonSetup::InitializeParam param, nn::mem::StandardAllocator& allocator) NN_NOEXCEPT NN_OVERRIDE
    {
        AtkCommonSetup::Initialize(param, allocator);
    }
    void Initialize(const SoundSystemCondition& soundSystemCondition, const SubMixCondition& subMixCondition, nn::mem::StandardAllocator& allocator) NN_NOEXCEPT
    {
        nn::atk::SoundSystem::SoundSystemParam actualParam;
        actualParam.enableCustomSubMix = true;
        actualParam.subMixCount = 1;
        actualParam.busCountMax = soundSystemCondition.busCountMax;
        actualParam.subMixTotalChannelCount = soundSystemCondition.totalChannelCount;
        actualParam.enableBusMixVolume = soundSystemCondition.isBusMixVolumeEnabled;
        actualParam.enableVolumeThroughMode = soundSystemCondition.isVolumeThroughModeEnabled;
        InitializeSoundSystem(actualParam, allocator);

        m_pSubMix = reinterpret_cast<nn::atk::SubMix*>( allocator.Allocate(sizeof(nn::atk::SubMix)) );
        new(m_pSubMix) nn::atk::SubMix();
        NN_ABORT_UNLESS_NOT_NULL(m_pSubMix);
        nn::atk::FinalMix& finalMix = nn::atk::SoundSystem::GetFinalMix();
        size_t subMixBufferSize = m_pSubMix->GetRequiredMemorySize(subMixCondition.srcBusCount, subMixCondition.srcChannelCount, &finalMix);
        m_SubMixBuffer = allocator.Allocate(subMixBufferSize);
        NN_ABORT_UNLESS(m_pSubMix->Initialize(subMixCondition.srcBusCount, subMixCondition.srcChannelCount, &finalMix, m_SubMixBuffer, subMixBufferSize));
        m_pSubMix->SetDestination(&finalMix);

        nnt::atk::util::AtkCommonSetup::InitializeParam initializeParam;
        InitializeSoundHeap(initializeParam.GetSoundHeapSize(), allocator);
        InitializeSoundArchive(initializeParam.GetSoundArchivePath(), allocator);
        InitializeSoundDataManager(allocator);
        InitializeSoundArchivePlayer(allocator);

        GetSoundArchivePlayer().SetDefaultOutputReceiver(m_pSubMix);
    }
    void Finalize(nn::mem::StandardAllocator& allocator) NN_NOEXCEPT NN_OVERRIDE
    {
        if(m_pSubMix == nullptr)
        {
            AtkCommonSetup::Finalize(allocator);
            return;
        }

        FinalizeSoundArchivePlayer(allocator);
        FinalizeSoundDataManager(allocator);
        FinalizeSoundArchive(allocator);
        FinalizeSoundHeap(allocator);

        while(!m_pSubMix->IsFinalizable())
        {
            UpdateAndWaitAudioFrame(GetSoundArchivePlayer());
        }
        m_pSubMix->Finalize();


        allocator.Free(m_SubMixBuffer);
        allocator.Free(m_pSubMix);
        m_SubMixBuffer = nullptr;
        m_pSubMix = nullptr;

        FinalizeSoundSystem(allocator);
    }
    void LoadData(nn::atk::SoundArchive::ItemId itemId, const char* itemLabel) NN_NOEXCEPT NN_OVERRIDE
    {
        AtkCommonSetup::LoadData(itemId, itemLabel);
    }
    void LoadData(const char* itemLabel) NN_NOEXCEPT NN_OVERRIDE
    {
        AtkCommonSetup::LoadData(itemLabel);
    }

private:
    nn::atk::SubMix* m_pSubMix;
    void* m_SubMixBuffer;
};

CustomSubMixSetup  g_CustomSubMixSetup;

}

TEST(SoundHandle, InitialAttachTest)
{
    nnt::atk::util::OnPreAtkTest();
    // 初期状態の確認
    nn::atk::SoundHandle soundHandle;
    EXPECT_FALSE(soundHandle.IsAttachedSound());
    EXPECT_TRUE(soundHandle.GetId() == nn::atk::InvalidSoundId);

    // StartSound の関連付けテスト
    for(int i = 0; i < static_cast<int>(sizeof(TestData) / sizeof(TestData[0])); i++)
    {
        g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
        g_FsSetup.Initialize();
        g_AtkSetup.Initialize(g_Allocator);
        LoadData();

        nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
        EXPECT_TRUE(soundArchivePlayer.StartSound(&soundHandle, TestData[i]).IsSuccess());
        if(TestData[i] == STRM_MARIOKART)
        {
            UpdateAndWaitForStrmPrepare(soundHandle);
        }
        EXPECT_TRUE(soundHandle.IsAttachedSound());
        EXPECT_TRUE(soundHandle.GetId() == TestData[i]);

        soundHandle.DetachSound();
        EXPECT_FALSE(soundHandle.IsAttachedSound());
        EXPECT_TRUE(soundHandle.GetId() == nn::atk::InvalidSoundId);

        g_AtkSetup.Finalize(g_Allocator);
        g_FsSetup.Finalize();
        g_Allocator.Finalize();
    }

    // HoldSound の関連付けテスト
    for(int i = 0; i < static_cast<int>(sizeof(TestData) / sizeof(TestData[0])); i++)
    {
        g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
        g_FsSetup.Initialize();
        g_AtkSetup.Initialize(g_Allocator);
        LoadData();

        nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
        EXPECT_TRUE(soundArchivePlayer.HoldSound(&soundHandle, TestData[i]).IsSuccess());
        if(TestData[i] == STRM_MARIOKART)
        {
            UpdateAndWaitForStrmPrepare(soundHandle);
        }
        EXPECT_TRUE(soundHandle.IsAttachedSound());
        EXPECT_TRUE(soundHandle.GetId() == TestData[i]);
        soundHandle.DetachSound();
        EXPECT_FALSE(soundHandle.IsAttachedSound());
        EXPECT_TRUE(soundHandle.GetId() == nn::atk::InvalidSoundId);

        g_AtkSetup.Finalize(g_Allocator);
        g_FsSetup.Finalize();
        g_Allocator.Finalize();
    }

    // PrepareSound の関連付けテスト
    for(int i = 0; i < static_cast<int>(sizeof(TestData) / sizeof(TestData[0])); i++)
    {
        g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
        g_FsSetup.Initialize();
        g_AtkSetup.Initialize(g_Allocator);
        LoadData();

        nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
        EXPECT_TRUE(soundArchivePlayer.PrepareSound(&soundHandle, TestData[i]).IsSuccess());
        if(TestData[i] == STRM_MARIOKART)
        {
            UpdateAndWaitForStrmPrepare(soundHandle);
        }
        EXPECT_TRUE(soundHandle.IsAttachedSound());
        EXPECT_TRUE(soundHandle.GetId() == TestData[i]);
        soundHandle.DetachSound();
        EXPECT_FALSE(soundHandle.IsAttachedSound());
        EXPECT_TRUE(soundHandle.GetId() == nn::atk::InvalidSoundId);

        g_AtkSetup.Finalize(g_Allocator);
        g_FsSetup.Finalize();
        g_Allocator.Finalize();
    }
}

// NwRenderer のコードが残っていてもビルドエラーにならないことを確認するテストです。
TEST( SoundHandle, IsNwRendererCompatibleTest )
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize( g_HeapMemory, MemoryHeapSize );
    g_FsSetup.Initialize();
    g_AtkSetup.Initialize( g_Allocator );
    LoadData();

    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundHandle soundHandle;

    nn::atk::SoundStartable::StartInfo startInfo;
    startInfo.enableFlag = nn::atk::SoundStartable::StartInfo::EnableFlagBit_VoiceRendererType;
    startInfo.voiceRendererType = nn::atk::VoiceRendererType_Nw;

    EXPECT_TRUE( soundArchivePlayer.StartSound( &soundHandle, WSD_SNARE, &startInfo ).IsSuccess() );
    soundHandle.Stop( 0 );
    EXPECT_TRUE( soundArchivePlayer.StartSound( &soundHandle, SEQ_MARIOKART, &startInfo ).IsSuccess() );
    soundHandle.Stop( 0 );
    EXPECT_TRUE( soundArchivePlayer.StartSound( &soundHandle, STRM_MARIOKART, &startInfo ).IsSuccess() );
    soundHandle.Stop( 0 );

    g_AtkSetup.Finalize( g_Allocator );
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}

TEST(SoundHandle, SoundHandlingTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();
    g_AtkSetup.Initialize(g_Allocator);
    LoadData();

    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundHandle soundHandle;

    for(int i = 0; i < static_cast<int>(sizeof(TestData) / sizeof(TestData[0])); i++)
    {
        for(int j = 0; j < static_cast<int>(sizeof(FadeTime) / sizeof(FadeTime[0])); j++)
        {
            // 準備状態のテスト
            EXPECT_TRUE(soundArchivePlayer.PrepareSound(&soundHandle, TestData[i]).IsSuccess());
            if(TestData[i] == STRM_MARIOKART)
            {
                UpdateAndWaitForStrmPrepare(soundHandle);
            }
            EXPECT_TRUE(soundHandle.IsAttachedSound());
            EXPECT_TRUE(soundHandle.IsPrepared());

            soundHandle.FadeIn(FadeTime[j]);
            soundHandle.StartPrepared();
            EXPECT_TRUE(soundHandle.IsPrepared());
            EXPECT_FALSE(soundHandle.IsPause());
            EXPECT_FALSE(soundHandle.IsMute());
            EXPECT_EQ(FadeTime[j], soundHandle.GetRemainingFadeFrames());
            for(int t = 1; t <= FadeTime[j]; t++)
            {
                UpdateSoundArchivePlayer( soundArchivePlayer );
                EXPECT_EQ(FadeTime[j] - t, soundHandle.GetRemainingFadeFrames());
            }

            EXPECT_EQ(FadeTime[j], soundHandle.GetPlayFrameCount());

            // Pause 状態のテスト
            soundHandle.Pause(true, FadeTime[j]);
            EXPECT_TRUE(soundHandle.IsPrepared());
            EXPECT_TRUE(soundHandle.IsPause());
            EXPECT_FALSE(soundHandle.IsMute());
            // 0 以下をfadeTimeとして指定した場合 soundHandle.GetRemainingPauseFadeFrames() は 1 となる
            const int ExpectFadeFrames = FadeTime[j] > 0 ? FadeTime[j] : 1;
            EXPECT_EQ(ExpectFadeFrames, soundHandle.GetRemainingPauseFadeFrames());
            for(int t = 1; t <= FadeTime[j]; t++)
            {
                UpdateSoundArchivePlayer( soundArchivePlayer );
                EXPECT_EQ(FadeTime[j] - t, soundHandle.GetRemainingPauseFadeFrames());
            }

            soundHandle.Pause(false, FadeTime[j]);
            EXPECT_TRUE(soundHandle.IsPrepared());
            EXPECT_FALSE(soundHandle.IsPause());
            EXPECT_FALSE(soundHandle.IsMute());
            // 0 以下をfadeTimeとして指定した場合 soundHandle.GetRemainingPauseFadeFrames() は 1 となる
            EXPECT_EQ(ExpectFadeFrames, soundHandle.GetRemainingPauseFadeFrames());
            for(int t = 1; t <= FadeTime[j]; t++)
            {
                UpdateSoundArchivePlayer( soundArchivePlayer );
                EXPECT_EQ(FadeTime[j] - t, soundHandle.GetRemainingPauseFadeFrames());
            }

            // Mute 状態のテスト
            soundHandle.Mute(true, FadeTime[j]);
            EXPECT_TRUE(soundHandle.IsPrepared());
            EXPECT_FALSE(soundHandle.IsPause());
            EXPECT_TRUE(soundHandle.IsMute());
            // 0 以下をfadeTimeとして指定した場合 soundHandle.GetRemainingMuteFadeFrames() は 1 となる
            EXPECT_EQ(ExpectFadeFrames, soundHandle.GetRemainingMuteFadeFrames());
            for(int t = 1; t <= FadeTime[j]; t++)
            {
                UpdateSoundArchivePlayer( soundArchivePlayer );
                EXPECT_EQ(FadeTime[j] - t, soundHandle.GetRemainingMuteFadeFrames());
            }

            soundHandle.Mute(false, FadeTime[j]);
            EXPECT_TRUE(soundHandle.IsPrepared());
            EXPECT_FALSE(soundHandle.IsPause());
            EXPECT_FALSE(soundHandle.IsMute());
            // 0 以下をfadeTimeとして指定した場合 soundHandle.GetRemainingMuteFadeFrames() は 1 となる
            EXPECT_EQ(ExpectFadeFrames, soundHandle.GetRemainingMuteFadeFrames());
            for(int t = 1; t <= FadeTime[j]; t++)
            {
                UpdateSoundArchivePlayer( soundArchivePlayer );
                EXPECT_EQ(FadeTime[j] - t, soundHandle.GetRemainingMuteFadeFrames());
            }

            // Stop 状態のテスト
            soundHandle.Stop(FadeTime[j]);
            EXPECT_EQ(FadeTime[j], soundHandle.GetRemainingFadeFrames());
            for(int t = 1; t <= FadeTime[j]; t++)
            {
                UpdateSoundArchivePlayer( soundArchivePlayer );
                EXPECT_EQ(FadeTime[j] - t, soundHandle.GetRemainingFadeFrames());
            }
            EXPECT_FALSE(soundHandle.IsAttachedSound());
            EXPECT_FALSE(soundHandle.IsPrepared());

            // TODO: 待つ処理を加えないと STRM_MARIOKART の再生準備がタイムアウトで失敗する可能性がある原因について調査する
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(35));
        }
    }

    g_AtkSetup.Finalize(g_Allocator);
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}

TEST(SoundHandle, PauseStateTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();
    g_AtkSetup.Initialize(g_Allocator);
    LoadData();

    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundHandle soundHandle;

    for(int i = 0; i < static_cast<int>(sizeof(TestData) / sizeof(TestData[0])); i++)
    {
        for(int j = 0; j < static_cast<int>(sizeof(FadeTime) / sizeof(FadeTime[0])); j++)
        {
            EXPECT_EQ(nn::atk::SoundHandle::PauseState::PauseState_Invalid, soundHandle.GetPauseState());
            EXPECT_TRUE(soundArchivePlayer.PrepareSound( &soundHandle, TestData[i]).IsSuccess());
            if(TestData[i] == STRM_MARIOKART)
            {
                UpdateAndWaitForStrmPrepare(soundHandle);
            }
            EXPECT_EQ(nn::atk::SoundHandle::PauseState::PauseState_Normal, soundHandle.GetPauseState());

            soundHandle.StartPrepared();
            EXPECT_EQ(nn::atk::SoundHandle::PauseState::PauseState_Normal, soundHandle.GetPauseState());

            PauseSoundHandle( soundHandle, true, FadeTime[j] );
            EXPECT_EQ(nn::atk::SoundHandle::PauseState::PauseState_Pausing, soundHandle.GetPauseState());
            PauseSoundHandle( soundHandle, true, FadeTime[j] );
            EXPECT_EQ(nn::atk::SoundHandle::PauseState::PauseState_Pausing, soundHandle.GetPauseState());
            PauseSoundHandle( soundHandle, false, FadeTime[j] );
            EXPECT_EQ(nn::atk::SoundHandle::PauseState::PauseState_Unpausing, soundHandle.GetPauseState());
            PauseSoundHandle( soundHandle, false, FadeTime[j] );
            EXPECT_EQ(nn::atk::SoundHandle::PauseState::PauseState_Unpausing, soundHandle.GetPauseState());
            PauseSoundHandle( soundHandle, true, FadeTime[j] );
            EXPECT_EQ(nn::atk::SoundHandle::PauseState::PauseState_Pausing, soundHandle.GetPauseState());
            while(soundHandle.GetRemainingPauseFadeFrames() != 0)
            {
                UpdateSoundArchivePlayer( soundArchivePlayer );
            }
            // 互換性との関係上、 PauseState_Paused へ移行する際には 1 回多く Update を呼ぶ
            EXPECT_EQ(nn::atk::SoundHandle::PauseState::PauseState_Pausing, soundHandle.GetPauseState());
            UpdateSoundArchivePlayer( soundArchivePlayer );
            EXPECT_EQ(nn::atk::SoundHandle::PauseState::PauseState_Paused, soundHandle.GetPauseState());

            PauseSoundHandle( soundHandle, false, FadeTime[j] );
            EXPECT_EQ(nn::atk::SoundHandle::PauseState::PauseState_Unpausing, soundHandle.GetPauseState());
            while(soundHandle.GetRemainingPauseFadeFrames() != 0)
            {
                UpdateSoundArchivePlayer( soundArchivePlayer );
            }
            EXPECT_EQ(nn::atk::SoundHandle::PauseState::PauseState_Normal, soundHandle.GetPauseState());

            soundHandle.Stop(0);
            // TODO: 待つ処理を加えないと STRM_MARIOKART の再生準備がタイムアウトで失敗する可能性がある原因について調査する
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(35));
        }
    }

    g_AtkSetup.Finalize(g_Allocator);
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}

TEST(SoundHandle, ImmediatePauseStateTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();
    g_AtkSetup.Initialize(g_Allocator);
    LoadData();

    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundHandle soundHandle;

    for (int i = 0; i < static_cast<int>(sizeof(TestData) / sizeof(TestData[0])); i++)
    {
        for (int j = 0; j < static_cast<int>(sizeof(FadeTime) / sizeof(FadeTime[0])); j++)
        {
            EXPECT_EQ(nn::atk::SoundHandle::PauseState::PauseState_Invalid, soundHandle.GetPauseState());
            EXPECT_TRUE(soundArchivePlayer.PrepareSound(&soundHandle, TestData[i]).IsSuccess());
            if (TestData[i] == STRM_MARIOKART)
            {
                UpdateAndWaitForStrmPrepare(soundHandle);
            }
            EXPECT_EQ(nn::atk::SoundHandle::PauseState::PauseState_Normal, soundHandle.GetPauseState());

            soundHandle.StartPrepared();
            EXPECT_EQ(nn::atk::SoundHandle::PauseState::PauseState_Normal, soundHandle.GetPauseState());

            PauseSoundHandle(soundHandle, true, FadeTime[j], nn::atk::PauseMode_PauseImmediately);
            EXPECT_EQ(nn::atk::SoundHandle::PauseState::PauseState_Pausing, soundHandle.GetPauseState());
            PauseSoundHandle(soundHandle, true, FadeTime[j], nn::atk::PauseMode_PauseImmediately);
            EXPECT_EQ(nn::atk::SoundHandle::PauseState::PauseState_Pausing, soundHandle.GetPauseState());
            PauseSoundHandle(soundHandle, false, FadeTime[j], nn::atk::PauseMode_PauseImmediately);
            EXPECT_EQ(nn::atk::SoundHandle::PauseState::PauseState_Unpausing, soundHandle.GetPauseState());
            PauseSoundHandle(soundHandle, false, FadeTime[j], nn::atk::PauseMode_PauseImmediately);
            EXPECT_EQ(nn::atk::SoundHandle::PauseState::PauseState_Unpausing, soundHandle.GetPauseState());
            PauseSoundHandle(soundHandle, true, FadeTime[j], nn::atk::PauseMode_PauseImmediately);
            EXPECT_EQ(nn::atk::SoundHandle::PauseState::PauseState_Pausing, soundHandle.GetPauseState());
            while (soundHandle.GetRemainingPauseFadeFrames() != 0)
            {
                UpdateSoundArchivePlayer(soundArchivePlayer);
            }
            // Update を呼ぶ回数は FadeTime で指定した回数で十分
            EXPECT_EQ(nn::atk::SoundHandle::PauseState::PauseState_Paused, soundHandle.GetPauseState());

            PauseSoundHandle(soundHandle, false, FadeTime[j], nn::atk::PauseMode_PauseImmediately);
            EXPECT_EQ(nn::atk::SoundHandle::PauseState::PauseState_Unpausing, soundHandle.GetPauseState());
            while (soundHandle.GetRemainingPauseFadeFrames() != 0)
            {
                UpdateSoundArchivePlayer(soundArchivePlayer);
            }
            EXPECT_EQ(nn::atk::SoundHandle::PauseState::PauseState_Normal, soundHandle.GetPauseState());

            soundHandle.Stop(0);
            // TODO: 待つ処理を加えないと STRM_MARIOKART の再生準備がタイムアウトで失敗する可能性がある原因について調査する
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(35));
        }
    }

    g_AtkSetup.Finalize(g_Allocator);
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}

TEST(SoundHandle, MuteStateTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();
    g_AtkSetup.Initialize(g_Allocator);
    LoadData();


    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundHandle soundHandle;

    for(int i = 0; i < static_cast<int>(sizeof(TestData) / sizeof(TestData[0])); i++)
    {
        for(int j = 0; j < static_cast<int>(sizeof(FadeTime) / sizeof(FadeTime[0])); j++)
        {
            EXPECT_EQ(nn::atk::SoundHandle::MuteState::MuteState_Invalid, soundHandle.GetMuteState());
            EXPECT_TRUE(soundArchivePlayer.PrepareSound(&soundHandle, TestData[i]).IsSuccess());
            if(TestData[i] == STRM_MARIOKART)
            {
                UpdateAndWaitForStrmPrepare(soundHandle);
            }
            EXPECT_EQ(nn::atk::SoundHandle::MuteState::MuteState_Normal, soundHandle.GetMuteState());

            soundHandle.StartPrepared();
            EXPECT_EQ(nn::atk::SoundHandle::MuteState::MuteState_Normal, soundHandle.GetMuteState());

            MuteSoundHandle(soundHandle, true, FadeTime[j]);
            EXPECT_EQ(nn::atk::SoundHandle::MuteState::MuteState_Muting, soundHandle.GetMuteState());
            MuteSoundHandle(soundHandle, true, FadeTime[j]);
            EXPECT_EQ(nn::atk::SoundHandle::MuteState::MuteState_Muting, soundHandle.GetMuteState());
            MuteSoundHandle(soundHandle, false, FadeTime[j]);
            EXPECT_EQ(nn::atk::SoundHandle::MuteState::MuteState_Unmuting, soundHandle.GetMuteState());
            MuteSoundHandle(soundHandle, false, FadeTime[j]);
            EXPECT_EQ(nn::atk::SoundHandle::MuteState::MuteState_Unmuting, soundHandle.GetMuteState());
            MuteSoundHandle(soundHandle, true, FadeTime[j]);
            EXPECT_EQ(nn::atk::SoundHandle::MuteState::MuteState_Muting, soundHandle.GetMuteState());
            while(soundHandle.GetRemainingMuteFadeFrames() != 0)
            {
                UpdateSoundArchivePlayer( soundArchivePlayer );
            }
            EXPECT_EQ(nn::atk::SoundHandle::MuteState::MuteState_Muted, soundHandle.GetMuteState());

            MuteSoundHandle(soundHandle, false, FadeTime[j]);
            EXPECT_EQ(nn::atk::SoundHandle::MuteState::MuteState_Unmuting, soundHandle.GetMuteState());
            while(soundHandle.GetRemainingMuteFadeFrames() != 0)
            {
                UpdateSoundArchivePlayer( soundArchivePlayer );
            }
            EXPECT_EQ(nn::atk::SoundHandle::MuteState::MuteState_Normal, soundHandle.GetMuteState());

            soundHandle.Stop(0);
            // TODO: 待つ処理を加えないと STRM_MARIOKART の再生準備がタイムアウトで失敗する可能性がある原因について調査する
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(35));
        }
    }

    g_AtkSetup.Finalize(g_Allocator);
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}

TEST(SoundHandle, IdChangeTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();
    g_AtkSetup.Initialize(g_Allocator);
    LoadData();

    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundHandle soundHandle;

    for(int i = 0; i < static_cast<int>(sizeof(TestData) / sizeof(TestData[0])); i++)
    {
        EXPECT_EQ(nn::atk::InvalidSoundId, soundHandle.GetId());
        EXPECT_TRUE(soundArchivePlayer.PrepareSound(&soundHandle, TestData[i]).IsSuccess());
        if(TestData[i] == STRM_MARIOKART)
        {
            UpdateAndWaitForStrmPrepare(soundHandle);
        }
        EXPECT_EQ(TestData[i], soundHandle.GetId());
        EXPECT_TRUE(soundHandle.IsAttachedSound());
        EXPECT_TRUE(soundHandle.IsPrepared());

        for(int j = 0; j < static_cast<int>(sizeof(TestData) / sizeof(TestData[0])); j++)
        {
            if(i == j) continue;
            soundHandle.SetId(TestData[j]);
            EXPECT_EQ(TestData[j], soundHandle.GetId());
        }

        // Stop 状態のテスト
        soundHandle.Stop(0);
        EXPECT_FALSE(soundHandle.IsAttachedSound());
        EXPECT_FALSE(soundHandle.IsPrepared());

        // TODO: 待つ処理を加えないと STRM_MARIOKART の再生準備がタイムアウトで失敗する可能性がある原因について調査する
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(35));
    }

    g_AtkSetup.Finalize(g_Allocator);
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}

TEST(SoundHandle, GetUserParamTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();
    g_AtkSetup.Initialize(g_Allocator);
    LoadData();

    nn::atk::SoundHandle soundHandle;
    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundArchive& soundArchive = g_AtkSetup.GetSoundArchive();
    nn::atk::SoundDataManager& soundDataManager = g_AtkSetup.GetSoundDataManager();

    // UserParamSize には 4 の倍数でない適当な値を設定
    const size_t UserParamSize = 10;

    // 初期化項目の設定
    size_t memSizeForSoundArchivePlayer = soundArchivePlayer.GetRequiredMemSize(&soundArchive, UserParamSize);
    void* pMemoryForSoundArchivePlayer = nnt::atk::util::AllocateUninitializedMemory(g_Allocator, memSizeForSoundArchivePlayer, nn::atk::SoundArchivePlayer::BufferAlignSize );
    size_t memSizeForStreamBuffer = soundArchivePlayer.GetRequiredStreamBufferSize(&soundArchive);
    void* memoryForStreamBuffer = nnt::atk::util::AllocateUninitializedMemory(g_Allocator, memSizeForStreamBuffer, nn::atk::FsSoundArchive::BufferAlignSize);
    size_t memSizeForCacheBuffer = soundArchivePlayer.GetRequiredStreamCacheSize(&soundArchive, 16 * 1024);
    void* pMemoryForCacheBuffer = nnt::atk::util::AllocateUninitializedMemory(g_Allocator, memSizeForCacheBuffer, nn::atk::FsSoundArchive::BufferAlignSize);

    nn::atk::SoundArchivePlayer::InitializeParam initializeParam;
    initializeParam.pSoundArchive = &soundArchive;
    initializeParam.pSoundDataManager = &soundDataManager;
    initializeParam.pSetupBuffer = pMemoryForSoundArchivePlayer;
    initializeParam.setupBufferSize = memSizeForSoundArchivePlayer;
    initializeParam.pStreamBuffer = memoryForStreamBuffer;
    initializeParam.streamBufferSize = memSizeForStreamBuffer;
    initializeParam.pStreamCacheBuffer = pMemoryForCacheBuffer;
    initializeParam.streamCacheSize = memSizeForCacheBuffer;
    initializeParam.userParamSizePerSound = UserParamSize;

    nn::atk::SoundArchivePlayer soundArchivePlayerWithParam;

    EXPECT_TRUE(soundArchivePlayerWithParam.Initialize(initializeParam));

    for(int i = 0; i < static_cast<int>(sizeof(TestData) / sizeof(TestData[0])); i++)
    {
        EXPECT_TRUE(soundArchivePlayer.PrepareSound(&soundHandle, TestData[i]).IsSuccess());
        EXPECT_EQ(nullptr, soundHandle.GetUserParam());
        soundHandle.Stop(0);

        EXPECT_TRUE(soundArchivePlayerWithParam.PrepareSound(&soundHandle, TestData[i]).IsSuccess());
        if(TestData[i] == STRM_MARIOKART)
        {
            UpdateAndWaitForStrmPrepare(soundHandle);
        }
        EXPECT_NE(nullptr, soundHandle.GetUserParam());
        soundHandle.Stop(0);

        // TODO: 待つ処理を加えないと STRM_MARIOKART の再生準備がタイムアウトで失敗する可能性がある原因について調査する
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(35));
    }

    soundArchivePlayerWithParam.Finalize();
    g_AtkSetup.Finalize(g_Allocator);
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}

void SetMixVolumeToAllChannels(nn::atk::MixVolume* pOutMixVolume, float volume) NN_NOEXCEPT
{
    for( int i = 0; i < nn::atk::ChannelIndex_Count; i++ )
    {
        pOutMixVolume->channel[i] = volume;
    }
}

TEST(SoundHandle, SetSoundParameterTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();
    g_AtkSetup.Initialize(g_Allocator);
    LoadData();

    nn::atk::SoundSystem::SetOutputMode(nn::atk::OutputMode_Surround);
    nn::atk::SoundHandle soundHandle;
    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();

    for(int i = 0; i < static_cast<int>(sizeof(TestData) / sizeof(TestData[0])); i++)
    {
        EXPECT_TRUE(soundArchivePlayer.StartSound(&soundHandle, TestData[i]).IsSuccess());
        if(TestData[i] == STRM_MARIOKART)
        {
            UpdateAndWaitForStrmPrepare(soundHandle);
        }

        // サウンド共通パラメータの境界値、 API リファレンスに記されている値、および初期値を与えます。
        // 境界値がない場合は、いくつかのパラメータを適当に選び与えます。
        //
        // SoundHandle には Get する関数がないため値の検査はせず、値のセット後一回 Update をし、
        // アサーション等が発生しテストが落ちないかを確認するテストとなっています。
        // TODO: SoundHandle の元となる BasicSound の単体テストを追加し、こちらのテストを削除するかどうか検討する

        // ボリュームは BasicSound::SetVolume で NN_SDK_ASSERT( volume >= 0.0f ) を行っている
        soundHandle.SetVolume(0.0f);
        UpdateAndWaitAudioFrame();
        soundHandle.SetVolume(2.0f);
        UpdateAndWaitAudioFrame();
        soundHandle.SetVolume(4.0f);
        UpdateAndWaitAudioFrame();
        soundHandle.SetVolume(1.0f);
        UpdateAndWaitAudioFrame();
        soundHandle.SetVolume(std::numeric_limits<float>::infinity());
        UpdateAndWaitAudioFrame();

        soundHandle.SetPan(-2.0f);
        UpdateAndWaitAudioFrame();
        soundHandle.SetPan(-1.0f);
        UpdateAndWaitAudioFrame();
        soundHandle.SetPan(1.0f);
        UpdateAndWaitAudioFrame();
        soundHandle.SetPan(2.0f);
        UpdateAndWaitAudioFrame();
        soundHandle.SetPan(0.0f);
        UpdateAndWaitAudioFrame();
        soundHandle.SetPan(std::numeric_limits<float>::infinity());
        UpdateAndWaitAudioFrame();
        soundHandle.SetPan(-std::numeric_limits<float>::infinity());
        UpdateAndWaitAudioFrame();
        soundHandle.SetPan(std::numeric_limits<float>::quiet_NaN());
        UpdateAndWaitAudioFrame();
        soundHandle.SetPan(std::numeric_limits<float>::signaling_NaN());
        UpdateAndWaitAudioFrame();

        soundHandle.SetSurroundPan(-2.0f);
        UpdateAndWaitAudioFrame();
        soundHandle.SetSurroundPan(-1.0f);
        UpdateAndWaitAudioFrame();
        soundHandle.SetSurroundPan(1.0f);
        UpdateAndWaitAudioFrame();
        soundHandle.SetSurroundPan(2.0f);
        UpdateAndWaitAudioFrame();
        soundHandle.SetSurroundPan(0.0f);
        UpdateAndWaitAudioFrame();
        soundHandle.SetSurroundPan(std::numeric_limits<float>::infinity());
        UpdateAndWaitAudioFrame();
        soundHandle.SetSurroundPan(-std::numeric_limits<float>::infinity());
        UpdateAndWaitAudioFrame();
        soundHandle.SetSurroundPan(std::numeric_limits<float>::quiet_NaN());
        UpdateAndWaitAudioFrame();
        soundHandle.SetSurroundPan(std::numeric_limits<float>::signaling_NaN());
        UpdateAndWaitAudioFrame();

        soundHandle.SetMixMode(nn::atk::MixMode_MixVolume);
        nn::atk::MixVolume mixVolume;
        SetMixVolumeToAllChannels(&mixVolume, 0.0f);
        soundHandle.SetMixVolume(mixVolume);
        UpdateAndWaitAudioFrame();
        SetMixVolumeToAllChannels(&mixVolume, 1.0f);
        soundHandle.SetMixVolume(mixVolume);
        UpdateAndWaitAudioFrame();
        SetMixVolumeToAllChannels(&mixVolume, -1.0f);
        soundHandle.SetMixVolume(mixVolume);
        UpdateAndWaitAudioFrame();
        SetMixVolumeToAllChannels(&mixVolume, std::numeric_limits<float>::infinity());
        soundHandle.SetMixVolume(mixVolume);
        UpdateAndWaitAudioFrame();
        SetMixVolumeToAllChannels(&mixVolume, -std::numeric_limits<float>::infinity());
        soundHandle.SetMixVolume(mixVolume);
        UpdateAndWaitAudioFrame();
        SetMixVolumeToAllChannels(&mixVolume, std::numeric_limits<float>::quiet_NaN());
        soundHandle.SetMixVolume(mixVolume);
        UpdateAndWaitAudioFrame();
        SetMixVolumeToAllChannels(&mixVolume, std::numeric_limits<float>::signaling_NaN());
        soundHandle.SetMixVolume(mixVolume);
        UpdateAndWaitAudioFrame();

        soundHandle.SetMainSend(-1.0f);
        UpdateAndWaitAudioFrame();
        soundHandle.SetMainSend(0.0f);
        UpdateAndWaitAudioFrame();
        soundHandle.SetMainSend(1.0f);
        UpdateAndWaitAudioFrame();
        soundHandle.SetMainSend(std::numeric_limits<float>::infinity());
        UpdateAndWaitAudioFrame();
        soundHandle.SetMainSend(-std::numeric_limits<float>::infinity());
        UpdateAndWaitAudioFrame();
        soundHandle.SetMainSend(std::numeric_limits<float>::quiet_NaN());
        UpdateAndWaitAudioFrame();
        soundHandle.SetMainSend(std::numeric_limits<float>::signaling_NaN());
        UpdateAndWaitAudioFrame();

        for (int j = 0; j < nn::atk::AuxBus_Count; j++)
        {
            soundHandle.SetEffectSend(static_cast<nn::atk::AuxBus>(j), -1.0f);
            UpdateAndWaitAudioFrame();
            soundHandle.SetEffectSend(static_cast<nn::atk::AuxBus>(j), 0.0f);
            UpdateAndWaitAudioFrame();
            soundHandle.SetEffectSend(static_cast<nn::atk::AuxBus>(j), std::numeric_limits<float>::infinity());
            UpdateAndWaitAudioFrame();
            soundHandle.SetEffectSend(static_cast<nn::atk::AuxBus>(j), -std::numeric_limits<float>::infinity());
            UpdateAndWaitAudioFrame();
            soundHandle.SetEffectSend(static_cast<nn::atk::AuxBus>(j), std::numeric_limits<float>::quiet_NaN());
            UpdateAndWaitAudioFrame();
            soundHandle.SetEffectSend(static_cast<nn::atk::AuxBus>(j), std::numeric_limits<float>::signaling_NaN());
            UpdateAndWaitAudioFrame();
        }

        // ピッチは BasicSound::SetPitch で NN_SDK_ASSERT( pitch >= 0.0f ) を行っている
        soundHandle.SetPitch(0.5f);
        UpdateAndWaitAudioFrame();
        soundHandle.SetPitch(2.0f);
        UpdateAndWaitAudioFrame();
        soundHandle.SetPitch(1.0f);
        UpdateAndWaitAudioFrame();
        soundHandle.SetPitch(std::numeric_limits<float>::infinity());
        UpdateAndWaitAudioFrame();

        soundHandle.SetLowPassFilterFrequency(-1.0f);
        UpdateAndWaitAudioFrame();
        soundHandle.SetLowPassFilterFrequency(0.0f);
        UpdateAndWaitAudioFrame();
        soundHandle.SetLowPassFilterFrequency(1.0f);
        UpdateAndWaitAudioFrame();
        soundHandle.SetLowPassFilterFrequency(std::numeric_limits<float>::infinity());
        UpdateAndWaitAudioFrame();
        soundHandle.SetLowPassFilterFrequency(-std::numeric_limits<float>::infinity());
        UpdateAndWaitAudioFrame();
        soundHandle.SetLowPassFilterFrequency(std::numeric_limits<float>::quiet_NaN());
        UpdateAndWaitAudioFrame();
        soundHandle.SetLowPassFilterFrequency(std::numeric_limits<float>::signaling_NaN());
        UpdateAndWaitAudioFrame();

        soundHandle.SetBiquadFilter(nn::atk::BiquadFilterType_LowPassFilter, 1.0f);
        UpdateAndWaitAudioFrame();
        soundHandle.SetBiquadFilter(nn::atk::BiquadFilterType_LowPassFilter, 0.0f);
        UpdateAndWaitAudioFrame();

        soundHandle.SetBiquadFilter(nn::atk::BiquadFilterType_LowPassFilterNw4fCompatible48k, 1.0f);
        UpdateAndWaitAudioFrame();
        soundHandle.SetBiquadFilter(nn::atk::BiquadFilterType_LowPassFilterNw4fCompatible48k, 0.0f);
        UpdateAndWaitAudioFrame();

        soundHandle.SetBiquadFilter(nn::atk::BiquadFilterType_HighPassFilterNw4fCompatible48k, 1.0f);
        UpdateAndWaitAudioFrame();
        soundHandle.SetBiquadFilter(nn::atk::BiquadFilterType_HighPassFilterNw4fCompatible48k, 0.0f);
        UpdateAndWaitAudioFrame();

        soundHandle.SetBiquadFilter(nn::atk::BiquadFilterType_BandPassFilter512Nw4fCompatible48k, 1.0f);
        UpdateAndWaitAudioFrame();
        soundHandle.SetBiquadFilter(nn::atk::BiquadFilterType_BandPassFilter512Nw4fCompatible48k, 0.0f);
        UpdateAndWaitAudioFrame();

        soundHandle.SetBiquadFilter(nn::atk::BiquadFilterType_BandPassFilter1024Nw4fCompatible48k, 1.0f);
        UpdateAndWaitAudioFrame();
        soundHandle.SetBiquadFilter(nn::atk::BiquadFilterType_BandPassFilter1024Nw4fCompatible48k, 0.0f);
        UpdateAndWaitAudioFrame();

        soundHandle.SetBiquadFilter(nn::atk::BiquadFilterType_BandPassFilter2048Nw4fCompatible48k, 1.0f);
        UpdateAndWaitAudioFrame();
        soundHandle.SetBiquadFilter(nn::atk::BiquadFilterType_BandPassFilter2048Nw4fCompatible48k, 0.0f);
        UpdateAndWaitAudioFrame();

        soundHandle.SetPlayerPriority(nn::atk::PlayerPriorityMin);
        UpdateAndWaitAudioFrame();
        soundHandle.SetPlayerPriority(nn::atk::PlayerPriorityMax);
        UpdateAndWaitAudioFrame();

        soundHandle.SetOutputLine(0);
        UpdateAndWaitAudioFrame();
        soundHandle.ResetOutputLine();
        UpdateAndWaitAudioFrame();

        // TODO: 待つ処理を加えないと STRM_MARIOKART の再生準備がタイムアウトで失敗する可能性がある原因について調査する
        soundHandle.Stop(0);
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(35));
    }

    g_AtkSetup.Finalize(g_Allocator);
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
} // NOLINT(impl/function_size)

#ifndef NN_SDK_BUILD_RELEASE
TEST(SoundHandle, InvalidMixVolumeChannelTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();
    g_AtkSetup.Initialize(g_Allocator);
    g_AtkSetup.LoadData(SE_WIHAHO, "SE_WIHAHO");

    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundHandle soundHandle;

    EXPECT_TRUE(soundArchivePlayer.StartSound(&soundHandle, SE_WIHAHO).IsSuccess());

    // 5.1ch 音量設定のチェック
    nn::atk::MixVolume mixVolume;
    EXPECT_DEATH_IF_SUPPORTED(soundHandle.SetMixVolume(-1, mixVolume), ".*");
    EXPECT_DEATH_IF_SUPPORTED(soundHandle.SetMixVolume(nn::atk::WaveChannelMax, mixVolume), ".*");

    g_AtkSetup.Finalize(g_Allocator);
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}

void UpdateAndWaitForProcessCommand(nn::atk::SoundArchivePlayer& archivePlayer) NN_NOEXCEPT
{
    archivePlayer.Update();
    nnt::atk::util::WaitForProcessCommand();
}

void SetVolumeForDeathTest(
    nn::atk::SoundArchivePlayer& archivePlayer,
    nn::atk::SoundHandle& handle,
    float volume) NN_NOEXCEPT
{
    handle.SetVolume(volume);
    UpdateAndWaitForProcessCommand(archivePlayer);
}

void SetPitchForDeathTest(
    nn::atk::SoundArchivePlayer& archivePlayer,
    nn::atk::SoundHandle& handle,
    float pitch) NN_NOEXCEPT
{
    handle.SetPitch(pitch);
    UpdateAndWaitForProcessCommand(archivePlayer);
}

void SetBiquadFilterForDeathTest(
    nn::atk::SoundArchivePlayer& archivePlayer,
    nn::atk::SoundHandle& handle,
    int type,
    float value) NN_NOEXCEPT
{
    handle.SetBiquadFilter(type, value);
    UpdateAndWaitForProcessCommand(archivePlayer);
}

TEST(SoundHandle, SetInvalidHandleParameterTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();
    g_AtkSetup.Initialize(g_Allocator);
    LoadData();

    nn::atk::SoundSystem::SetOutputMode(nn::atk::OutputMode_Surround);
    nn::atk::SoundHandle soundHandle;
    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();

    for(int i = 0; i < static_cast<int>(sizeof(TestData) / sizeof(TestData[0])); i++)
    {
        EXPECT_TRUE(soundArchivePlayer.StartSound(&soundHandle, TestData[i]).IsSuccess());
        if(TestData[i] == STRM_MARIOKART)
        {
            UpdateAndWaitForStrmPrepare(soundHandle);
        }

        // ボリュームは BasicSound::SetVolume で NN_SDK_ASSERT( volume >= 0.0f ) を行っている
        EXPECT_DEATH_IF_SUPPORTED(SetVolumeForDeathTest(soundArchivePlayer, soundHandle, -1.0f), ".*");
        EXPECT_DEATH_IF_SUPPORTED(SetVolumeForDeathTest(soundArchivePlayer, soundHandle, -std::numeric_limits<float>::infinity()), ".*");
        EXPECT_DEATH_IF_SUPPORTED(SetVolumeForDeathTest(soundArchivePlayer, soundHandle, std::numeric_limits<float>::quiet_NaN()), ".*");
        EXPECT_DEATH_IF_SUPPORTED(SetVolumeForDeathTest(soundArchivePlayer, soundHandle, std::numeric_limits<float>::signaling_NaN()), ".*");

        // ピッチは BasicSound::SetPitch で NN_SDK_ASSERT( pitch >= 0.0f ) を行っている
        EXPECT_DEATH_IF_SUPPORTED(SetPitchForDeathTest(soundArchivePlayer, soundHandle, -1.0f), ".*");
        EXPECT_DEATH_IF_SUPPORTED(SetPitchForDeathTest(soundArchivePlayer, soundHandle, -std::numeric_limits<float>::infinity()), ".*");
        EXPECT_DEATH_IF_SUPPORTED(SetPitchForDeathTest(soundArchivePlayer, soundHandle, std::numeric_limits<float>::quiet_NaN()), ".*");
        EXPECT_DEATH_IF_SUPPORTED(SetPitchForDeathTest(soundArchivePlayer, soundHandle, std::numeric_limits<float>::signaling_NaN()), ".*");

        // BiquadFilter は BasicSoundPlayer::SetBiquadFilter で NN_SDK_ASSERT( value >= 0.0f && value <= 1.0f ); を行っている
        EXPECT_DEATH_IF_SUPPORTED(SetBiquadFilterForDeathTest(soundArchivePlayer, soundHandle, nn::atk::BiquadFilterType_LowPassFilter, -1.0f), ".*");
        EXPECT_DEATH_IF_SUPPORTED(SetBiquadFilterForDeathTest(soundArchivePlayer, soundHandle, nn::atk::BiquadFilterType_LowPassFilter, std::numeric_limits<float>::infinity()), ".*");
        EXPECT_DEATH_IF_SUPPORTED(SetBiquadFilterForDeathTest(soundArchivePlayer, soundHandle, nn::atk::BiquadFilterType_LowPassFilter, -std::numeric_limits<float>::infinity()), ".*");
        EXPECT_DEATH_IF_SUPPORTED(SetBiquadFilterForDeathTest(soundArchivePlayer, soundHandle, nn::atk::BiquadFilterType_LowPassFilter, std::numeric_limits<float>::quiet_NaN()), ".*");
        EXPECT_DEATH_IF_SUPPORTED(SetBiquadFilterForDeathTest(soundArchivePlayer, soundHandle, nn::atk::BiquadFilterType_LowPassFilter, std::numeric_limits<float>::signaling_NaN()), ".*");
        EXPECT_DEATH_IF_SUPPORTED(SetBiquadFilterForDeathTest(soundArchivePlayer, soundHandle, nn::atk::BiquadFilterType_LowPassFilterNw4fCompatible48k, -1.0f), ".*");
        EXPECT_DEATH_IF_SUPPORTED(SetBiquadFilterForDeathTest(soundArchivePlayer, soundHandle, nn::atk::BiquadFilterType_LowPassFilterNw4fCompatible48k, std::numeric_limits<float>::infinity()), ".*");
        EXPECT_DEATH_IF_SUPPORTED(SetBiquadFilterForDeathTest(soundArchivePlayer, soundHandle, nn::atk::BiquadFilterType_LowPassFilterNw4fCompatible48k, -std::numeric_limits<float>::infinity()), ".*");
        EXPECT_DEATH_IF_SUPPORTED(SetBiquadFilterForDeathTest(soundArchivePlayer, soundHandle, nn::atk::BiquadFilterType_LowPassFilterNw4fCompatible48k, std::numeric_limits<float>::quiet_NaN()), ".*");
        EXPECT_DEATH_IF_SUPPORTED(SetBiquadFilterForDeathTest(soundArchivePlayer, soundHandle, nn::atk::BiquadFilterType_LowPassFilterNw4fCompatible48k, std::numeric_limits<float>::signaling_NaN()), ".*");
        EXPECT_DEATH_IF_SUPPORTED(SetBiquadFilterForDeathTest(soundArchivePlayer, soundHandle, nn::atk::BiquadFilterType_HighPassFilterNw4fCompatible48k, -1.0f), ".*");
        EXPECT_DEATH_IF_SUPPORTED(SetBiquadFilterForDeathTest(soundArchivePlayer, soundHandle, nn::atk::BiquadFilterType_HighPassFilterNw4fCompatible48k, std::numeric_limits<float>::infinity()), ".*");
        EXPECT_DEATH_IF_SUPPORTED(SetBiquadFilterForDeathTest(soundArchivePlayer, soundHandle, nn::atk::BiquadFilterType_HighPassFilterNw4fCompatible48k, -std::numeric_limits<float>::infinity()), ".*");
        EXPECT_DEATH_IF_SUPPORTED(SetBiquadFilterForDeathTest(soundArchivePlayer, soundHandle, nn::atk::BiquadFilterType_HighPassFilterNw4fCompatible48k, std::numeric_limits<float>::quiet_NaN()), ".*");
        EXPECT_DEATH_IF_SUPPORTED(SetBiquadFilterForDeathTest(soundArchivePlayer, soundHandle, nn::atk::BiquadFilterType_HighPassFilterNw4fCompatible48k, std::numeric_limits<float>::signaling_NaN()), ".*");
        EXPECT_DEATH_IF_SUPPORTED(SetBiquadFilterForDeathTest(soundArchivePlayer, soundHandle, nn::atk::BiquadFilterType_BandPassFilter512Nw4fCompatible48k, -1.0f), ".*");
        EXPECT_DEATH_IF_SUPPORTED(SetBiquadFilterForDeathTest(soundArchivePlayer, soundHandle, nn::atk::BiquadFilterType_BandPassFilter512Nw4fCompatible48k, std::numeric_limits<float>::infinity()), ".*");
        EXPECT_DEATH_IF_SUPPORTED(SetBiquadFilterForDeathTest(soundArchivePlayer, soundHandle, nn::atk::BiquadFilterType_BandPassFilter512Nw4fCompatible48k, -std::numeric_limits<float>::infinity()), ".*");
        EXPECT_DEATH_IF_SUPPORTED(SetBiquadFilterForDeathTest(soundArchivePlayer, soundHandle, nn::atk::BiquadFilterType_BandPassFilter512Nw4fCompatible48k, std::numeric_limits<float>::quiet_NaN()), ".*");
        EXPECT_DEATH_IF_SUPPORTED(SetBiquadFilterForDeathTest(soundArchivePlayer, soundHandle, nn::atk::BiquadFilterType_BandPassFilter512Nw4fCompatible48k, std::numeric_limits<float>::signaling_NaN()), ".*");
        EXPECT_DEATH_IF_SUPPORTED(SetBiquadFilterForDeathTest(soundArchivePlayer, soundHandle, nn::atk::BiquadFilterType_BandPassFilter1024Nw4fCompatible48k, -1.0f), ".*");
        EXPECT_DEATH_IF_SUPPORTED(SetBiquadFilterForDeathTest(soundArchivePlayer, soundHandle, nn::atk::BiquadFilterType_BandPassFilter1024Nw4fCompatible48k, std::numeric_limits<float>::infinity()), ".*");
        EXPECT_DEATH_IF_SUPPORTED(SetBiquadFilterForDeathTest(soundArchivePlayer, soundHandle, nn::atk::BiquadFilterType_BandPassFilter1024Nw4fCompatible48k, -std::numeric_limits<float>::infinity()), ".*");
        EXPECT_DEATH_IF_SUPPORTED(SetBiquadFilterForDeathTest(soundArchivePlayer, soundHandle, nn::atk::BiquadFilterType_BandPassFilter1024Nw4fCompatible48k, std::numeric_limits<float>::quiet_NaN()), ".*");
        EXPECT_DEATH_IF_SUPPORTED(SetBiquadFilterForDeathTest(soundArchivePlayer, soundHandle, nn::atk::BiquadFilterType_BandPassFilter1024Nw4fCompatible48k, std::numeric_limits<float>::signaling_NaN()), ".*");
        EXPECT_DEATH_IF_SUPPORTED(SetBiquadFilterForDeathTest(soundArchivePlayer, soundHandle, nn::atk::BiquadFilterType_BandPassFilter2048Nw4fCompatible48k, -1.0f), ".*");
        EXPECT_DEATH_IF_SUPPORTED(SetBiquadFilterForDeathTest(soundArchivePlayer, soundHandle, nn::atk::BiquadFilterType_BandPassFilter2048Nw4fCompatible48k, std::numeric_limits<float>::infinity()), ".*");
        EXPECT_DEATH_IF_SUPPORTED(SetBiquadFilterForDeathTest(soundArchivePlayer, soundHandle, nn::atk::BiquadFilterType_BandPassFilter2048Nw4fCompatible48k, -std::numeric_limits<float>::infinity()), ".*");
        EXPECT_DEATH_IF_SUPPORTED(SetBiquadFilterForDeathTest(soundArchivePlayer, soundHandle, nn::atk::BiquadFilterType_BandPassFilter2048Nw4fCompatible48k, std::numeric_limits<float>::quiet_NaN()), ".*");
        EXPECT_DEATH_IF_SUPPORTED(SetBiquadFilterForDeathTest(soundArchivePlayer, soundHandle, nn::atk::BiquadFilterType_BandPassFilter2048Nw4fCompatible48k, std::numeric_limits<float>::signaling_NaN()), ".*");

        // TODO: 待つ処理を加えないと STRM_MARIOKART の再生準備がタイムアウトで失敗する可能性がある原因について調査する
        soundHandle.Stop(0);
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(35));
    }

    g_AtkSetup.Finalize(g_Allocator);
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}
#endif

TEST(SoundHandle, SetSendTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();

    const bool CustomSubMixCondition[2] = {false, true};
    for(bool isUsedCustomSubMix : CustomSubMixCondition)
    {
        const int SoundSystemBusCountMax = isUsedCustomSubMix ? nn::atk::OutputReceiver::BusCountMax : nn::atk::AuxBus_Count + 1;
        // SoundSystem に設定した最大バス数よりも小さい値でサブミックスを定義します
        const int CustomSubMixBusCountMax = SoundSystemBusCountMax - 1;
        if(isUsedCustomSubMix)
        {
            CustomSubMixSetup::SoundSystemCondition soundSystemCondition;
            soundSystemCondition.busCountMax = SoundSystemBusCountMax;
            soundSystemCondition.totalChannelCount = nn::atk::OutputReceiver::BusCountMax;
            soundSystemCondition.isBusMixVolumeEnabled = false;
            soundSystemCondition.isVolumeThroughModeEnabled = false;
            CustomSubMixSetup::SubMixCondition subMixCondition;
            subMixCondition.srcBusCount = CustomSubMixBusCountMax;
            subMixCondition.srcChannelCount = 1;
            g_CustomSubMixSetup.Initialize(soundSystemCondition, subMixCondition, g_Allocator);
        }
        else
        {
            g_CustomSubMixSetup.Initialize(g_Allocator);
        }

        LoadData(g_CustomSubMixSetup);

        nn::atk::SoundHandle soundHandle;
        nn::atk::SoundArchivePlayer& soundArchivePlayer = g_CustomSubMixSetup.GetSoundArchivePlayer();

        for(int i = 0; i < static_cast<int>(sizeof(TestData) / sizeof(TestData[0])); i++)
        {
            EXPECT_TRUE(soundArchivePlayer.StartSound(&soundHandle, TestData[i]).IsSuccess());
            if(TestData[i] == STRM_MARIOKART)
            {
                UpdateAndWaitForStrmPrepare(soundHandle);
            }

            // カスタムサブミックスを使用しない時は SoundSystem の最大バス数までのインデックス、
            // 使用する場合はカスタムサブミックスの持つバス数までで値が代入できるか確認します。
            const int TestBusCount = isUsedCustomSubMix ? CustomSubMixBusCountMax : SoundSystemBusCountMax;
            for (int busIndex = 0; busIndex < TestBusCount; busIndex++)
            {
                // サウンド共通パラメータの境界値、 API リファレンスに記されている値、および初期値を与えます。
                // 境界値がない場合は、いくつかのパラメータを適当に選び与えます。
                soundHandle.SetSend(busIndex, -1.0f);
                UpdateAndWaitAudioFrame(soundArchivePlayer);
                soundHandle.SetSend(busIndex, 0.0f);
                UpdateAndWaitAudioFrame(soundArchivePlayer);
                soundHandle.SetSend(busIndex, 1.0f);
                UpdateAndWaitAudioFrame(soundArchivePlayer);
            }

            ForceStopSound(soundHandle, soundArchivePlayer);
        }

        g_CustomSubMixSetup.Finalize(g_Allocator);
    }

    g_FsSetup.Finalize();
    g_Allocator.Finalize();
} // NOLINT(impl/function_size)

#ifndef NN_SDK_BUILD_RELEASE
TEST(SoundHandle, SetInvalidSendTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();

    const bool CustomSubMixCondition[2] = {false, true};
    for(bool isUsedCustomSubMix : CustomSubMixCondition)
    {
        const int SoundSystemBusCountMax = isUsedCustomSubMix ? nn::atk::OutputReceiver::BusCountMax : nn::atk::AuxBus_Count + 1;
        // SoundSystem に設定した最大バス数よりも小さい値でサブミックスを定義します
        const int CustomSubMixBusCountMax = SoundSystemBusCountMax - 1;
        if(isUsedCustomSubMix)
        {
            CustomSubMixSetup::SoundSystemCondition soundSystemCondition;
            soundSystemCondition.busCountMax = SoundSystemBusCountMax;
            soundSystemCondition.totalChannelCount = nn::atk::OutputReceiver::BusCountMax;
            soundSystemCondition.isBusMixVolumeEnabled = false;
            soundSystemCondition.isVolumeThroughModeEnabled = false;
            CustomSubMixSetup::SubMixCondition subMixCondition;
            subMixCondition.srcBusCount = CustomSubMixBusCountMax;
            subMixCondition.srcChannelCount = 1;
            g_CustomSubMixSetup.Initialize(soundSystemCondition, subMixCondition, g_Allocator);
        }
        else
        {
            g_CustomSubMixSetup.Initialize(g_Allocator);
        }

        LoadData(g_CustomSubMixSetup);

        nn::atk::SoundHandle soundHandle;
        nn::atk::SoundArchivePlayer& soundArchivePlayer = g_CustomSubMixSetup.GetSoundArchivePlayer();

        for(int i = 0; i < static_cast<int>(sizeof(TestData) / sizeof(TestData[0])); i++)
        {
            EXPECT_TRUE(soundArchivePlayer.StartSound(&soundHandle, TestData[i]).IsSuccess());
            if(TestData[i] == STRM_MARIOKART)
            {
                UpdateAndWaitForStrmPrepare(soundHandle);
            }

            EXPECT_DEATH_IF_SUPPORTED(soundHandle.SetSend(-1, 0.0f), ".*");
            EXPECT_DEATH_IF_SUPPORTED(soundHandle.SetSend(SoundSystemBusCountMax, 0.0f), ".*");
            if(isUsedCustomSubMix)
            {
                EXPECT_DEATH_IF_SUPPORTED(soundHandle.SetSend(CustomSubMixBusCountMax, 0.0f), ".*");
            }

            ForceStopSound(soundHandle, soundArchivePlayer);
        }

        g_CustomSubMixSetup.Finalize(g_Allocator);
    }

    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}
#endif

TEST(SoundHandle, SetBusMixVolumeTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();

    const bool CustomSubMixCondition[2] = {false, true};
    for(bool isUsedCustomSubMix : CustomSubMixCondition)
    {
        const int SoundSystemBusCountMax = isUsedCustomSubMix ? nn::atk::OutputReceiver::BusCountMax : nn::atk::AuxBus_Count + 1;
        // SoundSystem に設定した最大バス数よりも小さい値でサブミックスを定義します
        const int CustomSubMixBusCountMax = SoundSystemBusCountMax - 1;
        if(isUsedCustomSubMix)
        {
            CustomSubMixSetup::SoundSystemCondition soundSystemCondition;
            soundSystemCondition.busCountMax = SoundSystemBusCountMax;
            soundSystemCondition.totalChannelCount = nn::atk::OutputReceiver::BusCountMax;
            soundSystemCondition.isBusMixVolumeEnabled = true;
            soundSystemCondition.isVolumeThroughModeEnabled = false;
            CustomSubMixSetup::SubMixCondition subMixCondition;
            subMixCondition.srcBusCount = CustomSubMixBusCountMax;
            subMixCondition.srcChannelCount = 1;
            g_CustomSubMixSetup.Initialize(soundSystemCondition, subMixCondition, g_Allocator);
        }
        else
        {
            nnt::atk::util::AtkCommonSetup::InitializeParam initializeParam;
            nn::atk::SoundSystem::SoundSystemParam soundSystemParam;
            soundSystemParam.enableBusMixVolume = true;
            initializeParam.SetSoundSystemParam(soundSystemParam);
            g_CustomSubMixSetup.Initialize(initializeParam, g_Allocator);
        }

        LoadData(g_CustomSubMixSetup);

        nn::atk::SoundHandle soundHandle;
        nn::atk::SoundArchivePlayer& soundArchivePlayer = g_CustomSubMixSetup.GetSoundArchivePlayer();

        for(int i = 0; i < static_cast<int>(sizeof(TestData) / sizeof(TestData[0])); i++)
        {
            EXPECT_TRUE(soundArchivePlayer.StartSound(&soundHandle, TestData[i]).IsSuccess());
            if(TestData[i] == STRM_MARIOKART)
            {
                UpdateAndWaitForStrmPrepare(soundHandle);
            }

            // カスタムサブミックスを使用しない時は SoundSystem の最大バス数までのインデックス、
            // 使用する場合はカスタムサブミックスの持つバス数までで値が代入できるか確認します。
            const int TestBusCount = isUsedCustomSubMix ? CustomSubMixBusCountMax : SoundSystemBusCountMax;
            for (int busIndex = 0; busIndex < TestBusCount; busIndex++)
            {
                soundHandle.SetBusMixVolumeEnabled(busIndex, false);
                soundHandle.SetBusMixVolumeEnabled(busIndex, true);

                // 境界値がないため、パラメータを適当に選び与えます。
                const float VolumeLeft[6] = {1.0f, 0.8f, 0.6f, 0.4f, 0.2f, 0.0f};
                const nn::atk::ChannelMixVolume ChannelVolumeLeft(VolumeLeft, sizeof(VolumeLeft) / sizeof(VolumeLeft[0]));
                const float VolumeRight[6] = {0.0f, 0.1f, 0.2f, 0.3f, 0.4f, 0.5f};
                const nn::atk::ChannelMixVolume ChannelVolumeRight(VolumeRight, sizeof(VolumeRight) / sizeof(VolumeRight[0]));

                soundHandle.SetBusMixVolume(0, busIndex, ChannelVolumeLeft);
                soundHandle.SetBusMixVolume(1, busIndex, ChannelVolumeRight);
                UpdateAndWaitAudioFrame();
            }

            ForceStopSound(soundHandle, soundArchivePlayer);
        }

        g_CustomSubMixSetup.Finalize(g_Allocator);
    }

    g_FsSetup.Finalize();
    g_Allocator.Finalize();
} // NOLINT(impl/function_size)

#ifndef NN_SDK_BUILD_RELEASE
TEST(SoundHandle, SetInvalidBusMixVolumeTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();

    // BusMixVolume が有効でない場合
    {
        g_AtkSetup.Initialize(g_Allocator);
        LoadData();

        nn::atk::SoundHandle soundHandle;
        nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();

        for(int i = 0; i < static_cast<int>(sizeof(TestData) / sizeof(TestData[0])); i++)
        {
            EXPECT_TRUE(soundArchivePlayer.StartSound(&soundHandle, TestData[i]).IsSuccess());
            if(TestData[i] == STRM_MARIOKART)
            {
                UpdateAndWaitForStrmPrepare(soundHandle);
            }

            EXPECT_DEATH_IF_SUPPORTED(soundHandle.SetBusMixVolumeEnabled(0, true), ".*");
            EXPECT_DEATH_IF_SUPPORTED(soundHandle.SetBusMixVolumeEnabled(nn::atk::OutputReceiver::BusCountMax - 1, false), ".*");

            const float Volume[6] = {1.0f, 0.8f, 0.6f, 0.4f, 0.2f, 0.0f};
            const nn::atk::ChannelMixVolume ChannelVolume(Volume, sizeof(Volume) / sizeof(Volume[0]));

            EXPECT_DEATH_IF_SUPPORTED(soundHandle.SetBusMixVolume(0, 0, ChannelVolume), ".*");
            EXPECT_DEATH_IF_SUPPORTED(soundHandle.SetBusMixVolume(1, 0, ChannelVolume), ".*");
            EXPECT_DEATH_IF_SUPPORTED(soundHandle.SetBusMixVolume(0, nn::atk::OutputReceiver::BusCountMax - 1, ChannelVolume), ".*");
            EXPECT_DEATH_IF_SUPPORTED(soundHandle.SetBusMixVolume(1, nn::atk::OutputReceiver::BusCountMax - 1, ChannelVolume), ".*");
            ForceStopSound(soundHandle, soundArchivePlayer);
            UpdateAndWaitAudioFrame();
        }
        g_AtkSetup.Finalize(g_Allocator);
    }

    // BusMixVolume のバスに無効な値を代入した場合
    const bool CustomSubMixCondition[2] = {false, true};
    for(bool isUsedCustomSubMix : CustomSubMixCondition)
    {
        const int SoundSystemBusCountMax = isUsedCustomSubMix ? nn::atk::OutputReceiver::BusCountMax : nn::atk::AuxBus_Count + 1;
        // SoundSystem に設定した最大バス数よりも小さい値でサブミックスを定義します
        const int CustomSubMixBusCountMax = SoundSystemBusCountMax - 1;
        if(isUsedCustomSubMix)
        {
            CustomSubMixSetup::SoundSystemCondition soundSystemCondition;
            soundSystemCondition.busCountMax = SoundSystemBusCountMax;
            soundSystemCondition.totalChannelCount = nn::atk::OutputReceiver::BusCountMax;
            soundSystemCondition.isBusMixVolumeEnabled = true;
            soundSystemCondition.isVolumeThroughModeEnabled = false;
            CustomSubMixSetup::SubMixCondition subMixCondition;
            subMixCondition.srcBusCount = CustomSubMixBusCountMax;
            subMixCondition.srcChannelCount = 1;
            g_CustomSubMixSetup.Initialize(soundSystemCondition, subMixCondition, g_Allocator);
        }
        else
        {
            nnt::atk::util::AtkCommonSetup::InitializeParam initializeParam;
            nn::atk::SoundSystem::SoundSystemParam soundSystemParam;
            soundSystemParam.enableBusMixVolume = true;
            initializeParam.SetSoundSystemParam(soundSystemParam);
            g_CustomSubMixSetup.Initialize(initializeParam, g_Allocator);
        }

        LoadData(g_CustomSubMixSetup);

        nn::atk::SoundHandle soundHandle;
        nn::atk::SoundArchivePlayer& soundArchivePlayer = g_CustomSubMixSetup.GetSoundArchivePlayer();

        for(int i = 0; i < static_cast<int>(sizeof(TestData) / sizeof(TestData[0])); i++)
        {
            EXPECT_TRUE(soundArchivePlayer.StartSound(&soundHandle, TestData[i]).IsSuccess());
            if(TestData[i] == STRM_MARIOKART)
            {
                UpdateAndWaitForStrmPrepare(soundHandle);
            }

            EXPECT_DEATH_IF_SUPPORTED(soundHandle.SetBusMixVolumeEnabled(-1, true), ".*");
            EXPECT_DEATH_IF_SUPPORTED(soundHandle.SetBusMixVolumeEnabled(SoundSystemBusCountMax, false), ".*");
            if(isUsedCustomSubMix)
            {
                EXPECT_DEATH_IF_SUPPORTED(soundHandle.SetBusMixVolumeEnabled(CustomSubMixBusCountMax, false), ".*");
            }

            // 境界値がないため、パラメータを適当に選び与えます。
            const float Volume[6] = {1.0f, 0.8f, 0.6f, 0.4f, 0.2f, 0.0f};
            const nn::atk::ChannelMixVolume ChannelVolume(Volume, sizeof(Volume) / sizeof(Volume[0]));

            EXPECT_DEATH_IF_SUPPORTED(soundHandle.SetBusMixVolume(-1, 0, ChannelVolume), ".*");
            EXPECT_DEATH_IF_SUPPORTED(soundHandle.SetBusMixVolume(nn::atk::WaveChannelMax, 0, ChannelVolume), ".*");
            EXPECT_DEATH_IF_SUPPORTED(soundHandle.SetBusMixVolume(0, -1, ChannelVolume), ".*");
            EXPECT_DEATH_IF_SUPPORTED(soundHandle.SetBusMixVolume(0, SoundSystemBusCountMax, ChannelVolume), ".*");
            if(isUsedCustomSubMix)
            {
                EXPECT_DEATH_IF_SUPPORTED(soundHandle.SetBusMixVolume(0, CustomSubMixBusCountMax, ChannelVolume), ".*");
            }

            ForceStopSound(soundHandle, soundArchivePlayer);
            UpdateAndWaitAudioFrame();
        }

        g_CustomSubMixSetup.Finalize(g_Allocator);
    }

    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}
#endif

TEST(SoundHandle, SetVolumeThroughModeTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();

    const bool CustomSubMixCondition[2] = {false, true};
    for(bool isUsedCustomSubMix : CustomSubMixCondition)
    {
        const int SoundSystemBusCountMax = isUsedCustomSubMix ? nn::atk::OutputReceiver::BusCountMax : nn::atk::AuxBus_Count + 1;
        // SoundSystem に設定した最大バス数よりも小さい値でサブミックスを定義します
        const int CustomSubMixBusCountMax = SoundSystemBusCountMax - 1;
        if(isUsedCustomSubMix)
        {
            CustomSubMixSetup::SoundSystemCondition soundSystemCondition;
            soundSystemCondition.busCountMax = SoundSystemBusCountMax;
            soundSystemCondition.totalChannelCount = nn::atk::OutputReceiver::BusCountMax;
            soundSystemCondition.isBusMixVolumeEnabled = false;
            soundSystemCondition.isVolumeThroughModeEnabled = true;
            CustomSubMixSetup::SubMixCondition subMixCondition;
            subMixCondition.srcBusCount = CustomSubMixBusCountMax;
            subMixCondition.srcChannelCount = 1;
            g_CustomSubMixSetup.Initialize(soundSystemCondition, subMixCondition, g_Allocator);
        }
        else
        {
            nnt::atk::util::AtkCommonSetup::InitializeParam initializeParam;
            nn::atk::SoundSystem::SoundSystemParam soundSystemParam;
            soundSystemParam.enableVolumeThroughMode = true;
            initializeParam.SetSoundSystemParam(soundSystemParam);
            g_CustomSubMixSetup.Initialize(initializeParam, g_Allocator);
        }

        LoadData(g_CustomSubMixSetup);

        nn::atk::SoundHandle soundHandle;
        nn::atk::SoundArchivePlayer& soundArchivePlayer = g_CustomSubMixSetup.GetSoundArchivePlayer();

        for(int i = 0; i < static_cast<int>(sizeof(TestData) / sizeof(TestData[0])); i++)
        {
            EXPECT_TRUE(soundArchivePlayer.StartSound(&soundHandle, TestData[i]).IsSuccess());
            if(TestData[i] == STRM_MARIOKART)
            {
                UpdateAndWaitForStrmPrepare(soundHandle);
            }

            // カスタムサブミックスを使用しない時は SoundSystem の最大バス数までのインデックス、
            // 使用する場合はカスタムサブミックスの持つバス数までで値が代入できるか確認します。
            const int TestBusCount = isUsedCustomSubMix ? CustomSubMixBusCountMax : SoundSystemBusCountMax;
            for (int busIndex = 0; busIndex < TestBusCount; busIndex++)
            {
                soundHandle.SetVolumeThroughMode(busIndex, 0);
                UpdateAndWaitAudioFrame();

                soundHandle.SetVolumeThroughMode(busIndex, nn::atk::VolumeThroughMode_Binary);
                UpdateAndWaitAudioFrame();
            }

            ForceStopSound(soundHandle, soundArchivePlayer);
        }

        g_CustomSubMixSetup.Finalize(g_Allocator);
    }

    g_FsSetup.Finalize();
    g_Allocator.Finalize();
} // NOLINT(impl/function_size)

#ifndef NN_SDK_BUILD_RELEASE
TEST(SoundHandle, SetInvalidVolumeThroughModeTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();

    // VolumeThroughMode が有効でない場合
    {
        g_AtkSetup.Initialize(g_Allocator);
        LoadData();

        nn::atk::SoundHandle soundHandle;
        nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();

        for(int i = 0; i < static_cast<int>(sizeof(TestData) / sizeof(TestData[0])); i++)
        {
            EXPECT_TRUE(soundArchivePlayer.StartSound(&soundHandle, TestData[i]).IsSuccess());
            if(TestData[i] == STRM_MARIOKART)
            {
                UpdateAndWaitForStrmPrepare(soundHandle);
            }

            EXPECT_DEATH_IF_SUPPORTED(soundHandle.SetVolumeThroughMode(0, nn::atk::VolumeThroughMode_Binary), ".*");
            EXPECT_DEATH_IF_SUPPORTED(soundHandle.SetVolumeThroughMode(nn::atk::OutputReceiver::BusCountMax - 1, nn::atk::VolumeThroughMode_Binary), ".*");

            ForceStopSound(soundHandle, soundArchivePlayer);
            UpdateAndWaitAudioFrame();
        }
        g_AtkSetup.Finalize(g_Allocator);
    }

    // VolumeThroughMode のバスに無効な値を代入した場合(カスタムサブミックスの有効無効によって変化)
    const bool CustomSubMixCondition[2] = {false, true};
    for(bool isUsedCustomSubMix : CustomSubMixCondition)
    {
        const int SoundSystemBusCountMax = isUsedCustomSubMix ? nn::atk::OutputReceiver::BusCountMax : nn::atk::AuxBus_Count + 1;
        // SoundSystem に設定した最大バス数よりも小さい値でサブミックスを定義します
        const int CustomSubMixBusCountMax = SoundSystemBusCountMax - 1;
        if(isUsedCustomSubMix)
        {
            CustomSubMixSetup::SoundSystemCondition soundSystemCondition;
            soundSystemCondition.busCountMax = SoundSystemBusCountMax;
            soundSystemCondition.totalChannelCount = nn::atk::OutputReceiver::BusCountMax;
            soundSystemCondition.isBusMixVolumeEnabled = false;
            soundSystemCondition.isVolumeThroughModeEnabled = true;
            CustomSubMixSetup::SubMixCondition subMixCondition;
            subMixCondition.srcBusCount = CustomSubMixBusCountMax;
            subMixCondition.srcChannelCount = 1;
            g_CustomSubMixSetup.Initialize(soundSystemCondition, subMixCondition, g_Allocator);
        }
        else
        {
            nnt::atk::util::AtkCommonSetup::InitializeParam initializeParam;
            nn::atk::SoundSystem::SoundSystemParam soundSystemParam;
            soundSystemParam.enableVolumeThroughMode = true;
            initializeParam.SetSoundSystemParam(soundSystemParam);
            g_CustomSubMixSetup.Initialize(initializeParam, g_Allocator);
        }

        LoadData(g_CustomSubMixSetup);

        nn::atk::SoundHandle soundHandle;
        nn::atk::SoundArchivePlayer& soundArchivePlayer = g_CustomSubMixSetup.GetSoundArchivePlayer();

        for(int i = 0; i < static_cast<int>(sizeof(TestData) / sizeof(TestData[0])); i++)
        {
            EXPECT_TRUE(soundArchivePlayer.StartSound(&soundHandle, TestData[i]).IsSuccess());
            if(TestData[i] == STRM_MARIOKART)
            {
                UpdateAndWaitForStrmPrepare(soundHandle);
            }

            EXPECT_DEATH_IF_SUPPORTED(soundHandle.SetVolumeThroughMode(-1, nn::atk::VolumeThroughMode_Binary), ".*");
            EXPECT_DEATH_IF_SUPPORTED(soundHandle.SetVolumeThroughMode(SoundSystemBusCountMax, nn::atk::VolumeThroughMode_Binary), ".*");
            if(isUsedCustomSubMix)
            {
                EXPECT_DEATH_IF_SUPPORTED(soundHandle.SetVolumeThroughMode(CustomSubMixBusCountMax, nn::atk::VolumeThroughMode_Binary), ".*");
            }

            ForceStopSound(soundHandle, soundArchivePlayer);
        }

        g_CustomSubMixSetup.Finalize(g_Allocator);
    }

    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}
#endif
