﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "common.fsid"

#include <nnt.h>
#include <nnt/atkUtil/testAtk_Util.h>
#include <nnt/atkUtil/testAtk_CommonSetup.h>
#include <nn/atk.h>
#include <nn/nn_Log.h>
#include <nn/mem.h>

namespace {

const int MemoryHeapSize = 32 * 1024 * 1024;

const auto WaitTime = nn::TimeSpan::FromMilliSeconds(16);

nnt::atk::util::FsCommonSetup   g_FsSetup;
nnt::atk::util::AtkCommonSetup  g_AtkSetup;

static char g_HeapMemory[MemoryHeapSize];
nn::mem::StandardAllocator  g_Allocator;

void UpdateAndWaitForSeqStart(nn::atk::SequenceSoundHandle& sequenceSoundHandle)
{
    // 実際に再生が行われていることをパラメータ上で確認できるまで待つ
    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    soundArchivePlayer.Update();

    const auto TimeOut = nn::TimeSpan::FromMilliSeconds(5000);
#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
    const auto WaitTimeToUpdateSequenceHandle = nn::TimeSpan::FromMilliSeconds(20);

    //  VoiceCommandProcess  を呼ばないとパラメータは更新されないはず
    nn::os::SleepThread(WaitTimeToUpdateSequenceHandle);
    EXPECT_EQ(static_cast<uint32_t>(0), sequenceSoundHandle.GetTick()) << "Processed voice commands.";
#endif

    for(auto elapsed = nn::TimeSpan::FromMilliSeconds(0); elapsed < TimeOut; elapsed += WaitTime)
    {
#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
        nn::atk::SoundSystem::VoiceCommandProcess(6);
#endif
        nn::os::SleepThread(WaitTime);
        if(sequenceSoundHandle.GetTick() != 0)
        {
            break;
        }
    }
    EXPECT_NE(static_cast<uint32_t>(0), sequenceSoundHandle.GetTick()) << "Start sequence sound timed out.";
}

}

TEST(SequenceSoundHandle, TickCounterTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();
    g_AtkSetup.Initialize(g_Allocator);
    g_AtkSetup.LoadData(SEQ_MARIOKART, "SEQ_MARIOKART");

    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundHandle soundHandle;

    EXPECT_TRUE(soundArchivePlayer.PrepareSound(&soundHandle, SEQ_MARIOKART).IsSuccess());
    nn::atk::SequenceSoundHandle sequenceSoundHandle(&soundHandle);

    uint32_t currentTick = sequenceSoundHandle.GetTick();
    EXPECT_EQ(0, currentTick);
    sequenceSoundHandle.StartPrepared();
    UpdateAndWaitForSeqStart(sequenceSoundHandle);

    uint32_t commandTag = 0;
    while(!nnt::atk::util::SendDummyCommand(&commandTag))
    {
        soundArchivePlayer.Update();
#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
        nn::atk::SoundSystem::VoiceCommandProcess(6);
#endif
        nn::os::SleepThread(WaitTime);
    }

    uint32_t previousTick = currentTick;
    bool isCommandSend = true;
    int testCount = 0;
    const int TestCountMax = 10;

    while(testCount < TestCountMax)
    {
        // 音源の再生が終了していた場合はそのまま終了
        if(!soundHandle.IsAttachedSound())
        {
            break;
        }

        if(isCommandSend)
        {
            if(nnt::atk::util::IsCommandFinished(commandTag))
            {
                currentTick = sequenceSoundHandle.GetTick();
                EXPECT_GT(currentTick, previousTick);
                previousTick = currentTick;
                ++testCount;
                isCommandSend = nnt::atk::util::SendDummyCommand(&commandTag);
            }
        }
        else
        {
            isCommandSend = nnt::atk::util::SendDummyCommand(&commandTag);
        }
        soundArchivePlayer.Update();
#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
        nn::atk::SoundSystem::VoiceCommandProcess(6);
#endif
        nn::os::SleepThread(WaitTime);
    }

    g_AtkSetup.Finalize(g_Allocator);
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}

TEST(SequenceSoundHandle, SequenceVariableTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();
    g_AtkSetup.Initialize(g_Allocator);
    g_AtkSetup.LoadData(SEQ_MARIOKART, "SEQ_MARIOKART");

    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundHandle soundHandle;
    nn::atk::SequenceSoundHandle invalidSequenceSoundHandle(&soundHandle);

    // サウンドハンドルが無効な場合の書きこみチェック
    for(int16_t i = 0; i <= nn::atk::SequenceSoundHandle::VariableIndexMax; ++i)
    {
        // testVariable は int16_t のうちどのような値でもよい
        int16_t testVariable = i;
        // グローバル変数のチェック
        EXPECT_TRUE(invalidSequenceSoundHandle.WriteGlobalVariable(i, testVariable));

        // ローカル変数のチェック
        EXPECT_FALSE(invalidSequenceSoundHandle.WriteVariable(i, testVariable));

        // ローカルトラック変数のチェック
        for(int j = 0; j <= nn::atk::SequenceSoundHandle::TrackIndexMax; ++j)
        {
            EXPECT_FALSE(invalidSequenceSoundHandle.WriteTrackVariable(j, i, testVariable));
        }
    }

    // 書き込んだパラメータの反映が行われるまで待つ
    soundArchivePlayer.Update();
#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
    nn::atk::SoundSystem::VoiceCommandProcess(6);
#endif
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(200));

    // サウンドハンドルが無効な場合の読み込みチェック
    for(int16_t i = 0; i <= nn::atk::SequenceSoundHandle::VariableIndexMax; ++i)
    {
        int16_t variableResult;

        EXPECT_TRUE(invalidSequenceSoundHandle.ReadGlobalVariable(&variableResult, i));
        EXPECT_EQ(i, variableResult);

        // ローカル変数のチェック
        EXPECT_FALSE(invalidSequenceSoundHandle.ReadVariable(&variableResult, i));

        // ローカルトラック変数のチェック
        for(int j = 0; j <= nn::atk::SequenceSoundHandle::TrackIndexMax; ++j)
        {
            EXPECT_FALSE(invalidSequenceSoundHandle.ReadTrackVariable(&variableResult, j, i));
        }
    }

    EXPECT_TRUE(soundArchivePlayer.PrepareSound(&soundHandle, SEQ_MARIOKART).IsSuccess());
    nn::atk::SequenceSoundHandle sequenceSoundHandle(&soundHandle);

    // サウンドハンドルが有効な場合の書きこみチェック
    for(int16_t i = 0; i <= nn::atk::SequenceSoundHandle::VariableIndexMax; ++i)
    {
        // testVariable は int16_t のうちどのような値でもよい
        int16_t testVariable = i;

        // ローカル変数のチェック
        EXPECT_TRUE(sequenceSoundHandle.WriteVariable(i, testVariable));

        // ローカルトラック変数のチェック
        // TODO: j = 8, 12, 13, 14, 15 の場合にトラック変数が書き込まれない理由を調べる
        for(int j = 0; j < 8; ++j)
        {
            EXPECT_TRUE(sequenceSoundHandle.WriteTrackVariable(j, i, testVariable));
        }
    }

    // 書き込んだパラメータの反映が行われるまで待つ
    soundArchivePlayer.Update();
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(200));

#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
    //  VoiceCommand ではまだ反映されていないはず
    for(int16_t i = 0; i <= nn::atk::SequenceSoundHandle::VariableIndexMax; ++i)
    {
        int16_t variableResult;

        // ローカル変数のチェック
        EXPECT_TRUE(sequenceSoundHandle.ReadVariable(&variableResult, i));
        EXPECT_EQ(nn::atk::detail::driver::SequenceSoundPlayer::VariableDefaultValue, variableResult);

        // ローカルトラック変数のチェック
        // TODO: j = 8, 12, 13, 14, 15 の場合にトラック変数が書き込まれない理由を調べる
        for(int j = 0; j < 8; ++j)
        {
            EXPECT_TRUE(sequenceSoundHandle.ReadTrackVariable(&variableResult, j, i));
            EXPECT_EQ(nn::atk::detail::driver::SequenceSoundPlayer::VariableDefaultValue, variableResult);
        }
    }

    //  VoiceCommand を処理
    nn::atk::SoundSystem::VoiceCommandProcess(6);
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(200));

#endif

    // サウンドハンドルが有効な場合の読み込みチェック
    for(int16_t i = 0; i <= nn::atk::SequenceSoundHandle::VariableIndexMax; ++i)
    {
        int16_t variableResult;

        // ローカル変数のチェック
        EXPECT_TRUE(sequenceSoundHandle.ReadVariable(&variableResult, i));
        EXPECT_EQ(i, variableResult);

        // ローカルトラック変数のチェック
        // TODO: j = 8, 12, 13, 14, 15 の場合にトラック変数が書き込まれない理由を調べる
        for(int j = 0; j < 8; ++j)
        {
            EXPECT_TRUE(sequenceSoundHandle.ReadTrackVariable(&variableResult, j, i));
            EXPECT_EQ(i, variableResult);
            if(i != variableResult)
            {
                NN_LOG("i %d, j %d, result %d\n", i, j, variableResult);
            }
        }
    }

    g_AtkSetup.Finalize(g_Allocator);
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}

#ifndef NN_SDK_BUILD_RELEASE
TEST(SequenceSoundHandle, InvalidSequenceVariableTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();
    g_AtkSetup.Initialize(g_Allocator);
    g_AtkSetup.LoadData(SEQ_MARIOKART, "SEQ_MARIOKART");

    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundHandle soundHandle;

    EXPECT_TRUE(soundArchivePlayer.PrepareSound(&soundHandle, SEQ_MARIOKART).IsSuccess());
    nn::atk::SequenceSoundHandle sequenceSoundHandle(&soundHandle);


    // testVariable は int16_t のうちどのような値でもよい
    int16_t testVariable = 2;
    int16_t variableResult;
    // グローバル変数のチェック
    EXPECT_DEATH_IF_SUPPORTED(sequenceSoundHandle.WriteGlobalVariable(-1, testVariable), ".*");
    EXPECT_DEATH_IF_SUPPORTED(sequenceSoundHandle.WriteGlobalVariable(nn::atk::SequenceSoundHandle::VariableIndexMax + 1, testVariable), ".*");

    EXPECT_DEATH_IF_SUPPORTED(sequenceSoundHandle.ReadGlobalVariable(&variableResult, -1), ".*");
    EXPECT_DEATH_IF_SUPPORTED(sequenceSoundHandle.ReadGlobalVariable(&variableResult, nn::atk::SequenceSoundHandle::VariableIndexMax + 1), ".*");

    // ローカル変数のチェック
    EXPECT_DEATH_IF_SUPPORTED(sequenceSoundHandle.WriteVariable(-1, testVariable), ".*");
    EXPECT_DEATH_IF_SUPPORTED(sequenceSoundHandle.WriteVariable(nn::atk::SequenceSoundHandle::VariableIndexMax + 1, testVariable), ".*");

    EXPECT_DEATH_IF_SUPPORTED(sequenceSoundHandle.ReadVariable(&variableResult, -1), ".*");
    EXPECT_DEATH_IF_SUPPORTED(sequenceSoundHandle.ReadVariable(&variableResult, nn::atk::SequenceSoundHandle::VariableIndexMax + 1), ".*");

    // ローカルトラック変数のチェック
    EXPECT_DEATH_IF_SUPPORTED(sequenceSoundHandle.WriteTrackVariable(-1, 0, testVariable), ".*");
    EXPECT_DEATH_IF_SUPPORTED(sequenceSoundHandle.WriteTrackVariable(0, -1, testVariable), ".*");
    EXPECT_DEATH_IF_SUPPORTED(sequenceSoundHandle.WriteTrackVariable(nn::atk::SequenceSoundHandle::TrackIndexMax + 1, nn::atk::SequenceSoundHandle::VariableIndexMax, testVariable), ".*");
    EXPECT_DEATH_IF_SUPPORTED(sequenceSoundHandle.WriteTrackVariable(nn::atk::SequenceSoundHandle::TrackIndexMax, nn::atk::SequenceSoundHandle::VariableIndexMax + 1, testVariable), ".*");

    EXPECT_DEATH_IF_SUPPORTED(sequenceSoundHandle.ReadTrackVariable(&variableResult, -1, 0), ".*");
    EXPECT_DEATH_IF_SUPPORTED(sequenceSoundHandle.ReadTrackVariable(&variableResult, 0, -1), ".*");
    EXPECT_DEATH_IF_SUPPORTED(sequenceSoundHandle.ReadTrackVariable(&variableResult, nn::atk::SequenceSoundHandle::TrackIndexMax + 1, nn::atk::SequenceSoundHandle::VariableIndexMax), ".*");
    EXPECT_DEATH_IF_SUPPORTED(sequenceSoundHandle.ReadTrackVariable(&variableResult, nn::atk::SequenceSoundHandle::TrackIndexMax, nn::atk::SequenceSoundHandle::VariableIndexMax + 1), ".*");

    g_AtkSetup.Finalize(g_Allocator);
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}

#endif

TEST( SequenceSoundHandle, StartResultTest )
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize( g_HeapMemory, MemoryHeapSize );
    g_FsSetup.Initialize();
    g_AtkSetup.Initialize( g_Allocator );

    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundHandle soundHandle;

    nn::atk::SequenceSoundHandle sequenceSoundHandle( &soundHandle );

    // 存在しない ID を指定
    EXPECT_EQ( nn::atk::SoundStartable::StartResult::ResultCode_ErrorInvalidSoundId, soundArchivePlayer.StartSound( &soundHandle, 0xdeadbeef ).GetCode() );

    // シーケンスデータをロードせずに StartSound
    EXPECT_EQ( nn::atk::SoundStartable::StartResult::ResultCode_ErrorNotSequenceLoaded, soundArchivePlayer.StartSound( &soundHandle, SEQ_MARIOKART ).GetCode() );

    // シーケンスデータだけをロードして StartSound
    // バンクが未ロードでも StartSound は成功する
    // 経緯は http://www-sdd.zelda.nintendo.co.jp/project/nintendoware3/kagemai/html/user.cgi?project=nw3_snd&action=view_report&id=1079
    g_AtkSetup.GetSoundDataManager().LoadData( SEQ_MARIOKART, &g_AtkSetup.GetSoundHeap(), static_cast<uint32_t>(nn::atk::SoundDataManager::LoadFlag_Seq) );
    EXPECT_TRUE( soundArchivePlayer.StartSound( &soundHandle, SEQ_MARIOKART ).IsSuccess() );
    // 現実装では以下の Result を返すケースは存在しない。
    soundHandle.Stop( 0 );

    // 追加でバンクをロードして StartSound
    g_AtkSetup.GetSoundDataManager().LoadData( BANK_BGM, &g_AtkSetup.GetSoundHeap(), static_cast<uint32_t>(nn::atk::SoundDataManager::LoadFlag_Bank) );
    EXPECT_EQ( nn::atk::SoundStartable::StartResult::ResultCode_ErrorNotWarcLoaded, soundArchivePlayer.StartSound( &soundHandle, SEQ_MARIOKART ).GetCode() );

    // 追加で波形アーカイブごとロードした場合
    g_AtkSetup.GetSoundDataManager().LoadData( BANK_BGM, &g_AtkSetup.GetSoundHeap(), static_cast<uint32_t>(nn::atk::SoundDataManager::LoadFlag_All) );
    EXPECT_TRUE( soundArchivePlayer.StartSound( &soundHandle, SEQ_MARIOKART ).IsSuccess() );
    soundHandle.Stop(0);

    // 外部ファイルを利用した際に、不正なラベルを指定した場合
    nn::atk::SoundStartable::StartInfo startInfo;
    startInfo.enableFlag |= nn::atk::SoundStartable::StartInfo::EnableFlagBit_SequenceSoundInfo;
    startInfo.sequenceSoundInfo.sequenceDataAddress = g_AtkSetup.GetSoundDataManager().detail_GetFileAddress( SEQ_MARIOKART );
    startInfo.sequenceSoundInfo.startLocationLabel = "inexistent_label";
    EXPECT_EQ( nn::atk::SoundStartable::StartResult::ResultCode_ErrorInvalidSequenceStartLocationLabel, soundArchivePlayer.StartSound( &soundHandle, SEQ_MARIOKART, &startInfo ).GetCode() );

    g_AtkSetup.Finalize( g_Allocator );
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}

TEST( SequenceSoundHandle, StartResultWithPlayerHeapTest )
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize( g_HeapMemory, MemoryHeapSize );
    g_FsSetup.Initialize();
    g_AtkSetup.Initialize( g_Allocator );

    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundHandle soundHandle;

    nn::atk::SequenceSoundHandle sequenceSoundHandle( &soundHandle );

    // 存在しない ID を指定
    EXPECT_EQ( nn::atk::SoundStartable::StartResult::ResultCode_ErrorInvalidSoundId, soundArchivePlayer.StartSound( &soundHandle, 0xdeadbeef ).GetCode() );

    // シーケンスデータをロードせずに StartSound
    EXPECT_TRUE( soundArchivePlayer.StartSound( &soundHandle, SEQ_MARIOKART_PLAYERHEAP ).IsSuccess() );
    soundHandle.Stop( 0 );

    // シーケンスデータだけをロードして StartSound
    g_AtkSetup.GetSoundDataManager().LoadData( SEQ_MARIOKART_PLAYERHEAP, &g_AtkSetup.GetSoundHeap(), static_cast<uint32_t>(nn::atk::SoundDataManager::LoadFlag_Seq) );
    EXPECT_TRUE( soundArchivePlayer.StartSound( &soundHandle, SEQ_MARIOKART_PLAYERHEAP ).IsSuccess() );
    soundHandle.Stop( 0 );

    // 追加でバンクをロードして StartSound
    g_AtkSetup.GetSoundDataManager().LoadData( BANK_BGM, &g_AtkSetup.GetSoundHeap(), static_cast<uint32_t>(nn::atk::SoundDataManager::LoadFlag_Bank) );
    EXPECT_TRUE( soundArchivePlayer.StartSound( &soundHandle, SEQ_MARIOKART_PLAYERHEAP ).IsSuccess() );
    soundHandle.Stop( 0 );

    // 追加で波形アーカイブごとロードした場合
    g_AtkSetup.GetSoundDataManager().LoadData( BANK_BGM, &g_AtkSetup.GetSoundHeap(), static_cast<uint32_t>(nn::atk::SoundDataManager::LoadFlag_All) );
    EXPECT_TRUE( soundArchivePlayer.StartSound( &soundHandle, SEQ_MARIOKART_PLAYERHEAP ).IsSuccess() );
    soundHandle.Stop( 0 );

    // 外部ファイルを利用した際に、不正なラベルを指定した場合
    nn::atk::SoundStartable::StartInfo startInfo;
    startInfo.enableFlag |= nn::atk::SoundStartable::StartInfo::EnableFlagBit_SequenceSoundInfo;
    startInfo.sequenceSoundInfo.sequenceDataAddress = g_AtkSetup.GetSoundDataManager().detail_GetFileAddress( SEQ_MARIOKART_PLAYERHEAP );
    startInfo.sequenceSoundInfo.startLocationLabel = "inexistent_label";
    EXPECT_EQ( nn::atk::SoundStartable::StartResult::ResultCode_ErrorInvalidSequenceStartLocationLabel, soundArchivePlayer.StartSound( &soundHandle, SEQ_MARIOKART_PLAYERHEAP, &startInfo ).GetCode() );

    g_AtkSetup.Finalize( g_Allocator );
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}

#ifndef NN_SDK_BUILD_RELEASE
TEST(SequenceSoundHandle, InvalidMixVolumeChannelTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();
    g_AtkSetup.Initialize(g_Allocator);
    g_AtkSetup.LoadData(SEQ_MARIOKART, "SEQ_MARIOKART");

    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundHandle soundHandle;

    EXPECT_TRUE(soundArchivePlayer.PrepareSound(&soundHandle, SEQ_MARIOKART).IsSuccess());
    nn::atk::SequenceSoundHandle sequenceSoundHandle(&soundHandle);

    // 5.1ch 音量設定のチェック
    nn::atk::MixVolume mixVolume;
    nn::atk::SequenceSoundHandle::TrackBitFlagSet bitFlag;
    bitFlag.Reset();
    bitFlag[0] = true;
    EXPECT_DEATH_IF_SUPPORTED(sequenceSoundHandle.SetMixVolume(-1, mixVolume), ".*");
    EXPECT_DEATH_IF_SUPPORTED(sequenceSoundHandle.SetMixVolume(nn::atk::WaveChannelMax, mixVolume), ".*");
    EXPECT_DEATH_IF_SUPPORTED(sequenceSoundHandle.SetTrackMixVolume(bitFlag, -1, mixVolume), ".*");
    EXPECT_DEATH_IF_SUPPORTED(sequenceSoundHandle.SetTrackMixVolume(bitFlag, nn::atk::WaveChannelMax, mixVolume), ".*");

    g_AtkSetup.Finalize(g_Allocator);
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}
#endif
