﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "common.fsid"
#include "testAtk_TestEffectUtil.h"

#include <nnt.h>
#include <nnt/atkUtil/testAtk_Util.h>
#include <nnt/atkUtil/testAtk_CommonSetup.h>
#include <nn/atk.h>
#include <nn/nn_Log.h>

#include <nn/mem.h>

namespace {
const int MemoryHeapSize = 32 * 1024 * 1024;

nnt::atk::util::FsCommonSetup g_FsSetup;
nnt::atk::util::AtkCommonSetup g_AtkSetup;

static char g_HeapMemory[MemoryHeapSize];
nn::mem::StandardAllocator  g_Allocator;

bool IsI3dl2ReverbChannelModeSupported(nn::atk::EffectBase::ChannelMode channelMode)
{
    return channelMode == nn::atk::EffectBase::ChannelMode_1Ch || channelMode == nn::atk::EffectBase::ChannelMode_2Ch
        || channelMode == nn::atk::EffectBase::ChannelMode_4Ch || channelMode == nn::atk::EffectBase::ChannelMode_6Ch;
}
}

#ifndef NN_SDK_BUILD_RELEASE
TEST(Effect, I3dl2ReverbChannelSettingDeathTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);

    for (auto& channelMode : nnt::atk::effectUtil::TestEffectChannelMode)
    {
        nn::atk::EffectI3dl2Reverb reverb;

        if (!IsI3dl2ReverbChannelModeSupported(channelMode))
        {
            continue;
        }

        reverb.SetChannelMode(channelMode);
        const int ChannelCount = nnt::atk::effectUtil::ConvertChannelModeToInt(channelMode);
        nn::atk::ChannelIndex* pInvalidChannelSetting = reinterpret_cast<nn::atk::ChannelIndex*>(nnt::atk::util::AllocateUninitializedMemory(g_Allocator, sizeof(nn::atk::ChannelIndex) * ChannelCount));
        NN_ABORT_UNLESS_NOT_NULL(pInvalidChannelSetting);
        // チャンネルの重複
        if (ChannelCount > 1)
        {
            for (int i = 0; i < ChannelCount; ++i)
            {
                if (i == 0 || i == 1)
                {
                    pInvalidChannelSetting[i] = nn::atk::ChannelIndex_FrontLeft;
                }
                else
                {
                    pInvalidChannelSetting[i] = static_cast<nn::atk::ChannelIndex>(i);
                }
            }
            EXPECT_DEATH_IF_SUPPORTED(reverb.SetChannelIndex(pInvalidChannelSetting, channelMode), ".*");
        }

        // チャンネル設定の値に -1 が含まれる
        for (int i = 0; i < ChannelCount; ++i)
        {
            pInvalidChannelSetting[i] = static_cast<nn::atk::ChannelIndex>(i - 1);
        }
        EXPECT_DEATH_IF_SUPPORTED(reverb.SetChannelIndex(pInvalidChannelSetting, channelMode), ".*");

        // チャンネル設定の値に nn::atk::ChannelIndex_Count が含まれる
        for (int i = 0; i < ChannelCount; ++i)
        {
            if (i == ChannelCount - 1)
            {
                pInvalidChannelSetting[i] = nn::atk::ChannelIndex_Count;
            }
            else
            {
                pInvalidChannelSetting[i] = static_cast<nn::atk::ChannelIndex>(i);
            }
        }
        EXPECT_DEATH_IF_SUPPORTED(reverb.SetChannelIndex(pInvalidChannelSetting, channelMode), ".*");

        g_Allocator.Free(reinterpret_cast<void*>(pInvalidChannelSetting));
    }
    g_Allocator.Finalize();
}
#endif

TEST(Effect, I3dl2ReverbSetChannelCountTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);

    for (auto& channelMode : nnt::atk::effectUtil::TestEffectChannelMode)
    {
        nn::atk::EffectI3dl2Reverb reverb;
        if (IsI3dl2ReverbChannelModeSupported(channelMode))
        {
            EXPECT_TRUE(reverb.SetChannelMode(channelMode));
            ASSERT_EQ(channelMode, reverb.GetChannelMode());
            const int ChannelCount = nnt::atk::effectUtil::ConvertChannelModeToInt(channelMode);
            nn::atk::ChannelIndex* pReverbChannelSetting = reinterpret_cast<nn::atk::ChannelIndex*>(nnt::atk::util::AllocateUninitializedMemory(g_Allocator, sizeof(nn::atk::ChannelIndex) * ChannelCount));
            NN_ABORT_UNLESS_NOT_NULL(pReverbChannelSetting);
            reverb.GetChannelIndex(pReverbChannelSetting, ChannelCount);
            for (int i = 0; i < ChannelCount; ++i)
            {
                EXPECT_EQ(static_cast<nn::atk::ChannelIndex>(i), pReverbChannelSetting[i]);
            }
        }
    }

    g_Allocator.Finalize();
}

TEST(Effect, I3dl2ReverbSetChannelIndexTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);

    for (auto& channelMode : nnt::atk::effectUtil::TestEffectChannelMode)
    {
        nn::atk::EffectI3dl2Reverb reverb;
        const int ChannelCount = nnt::atk::effectUtil::ConvertChannelModeToInt(channelMode);

        if (IsI3dl2ReverbChannelModeSupported(channelMode))
        {
            // サポートしているチャンネル数の場合
            EXPECT_TRUE(reverb.SetChannelMode(channelMode));
            EXPECT_EQ(channelMode, reverb.GetChannelMode());

            nn::atk::ChannelIndex* pChannelSetting = reinterpret_cast<nn::atk::ChannelIndex*>(nnt::atk::util::AllocateUninitializedMemory(g_Allocator, sizeof(nn::atk::ChannelIndex) * ChannelCount));
            NN_ABORT_UNLESS_NOT_NULL(pChannelSetting);
            // 初期化
            for (int i = 0; i < ChannelCount; ++i)
            {
                pChannelSetting[i] = nn::atk::ChannelIndex_FrontLeft;
            }

            // 適切なチャンネル設定となるものについて総当たり検査
            while (pChannelSetting[ChannelCount - 1] != nn::atk::ChannelIndex_Count)
            {
                // テストケースとして適切か(チャンネル設定の被りがないか)確認
                bool isValidChannelSetting = true;
                for (int i = 0; i < ChannelCount; ++i)
                {
                    for (int j = i + 1; j < ChannelCount; ++j)
                    {
                        isValidChannelSetting &= (pChannelSetting[i] != pChannelSetting[j]);
                    }
                }

                // テスト本体
                if (isValidChannelSetting)
                {
                    EXPECT_TRUE(reverb.SetChannelIndex(pChannelSetting, channelMode));

                    ASSERT_EQ(channelMode, reverb.GetChannelMode());
                    nn::atk::ChannelIndex* pReverbChannelSetting = reinterpret_cast<nn::atk::ChannelIndex*>(nnt::atk::util::AllocateUninitializedMemory(g_Allocator, sizeof(nn::atk::ChannelIndex) * ChannelCount));
                    NN_ABORT_UNLESS_NOT_NULL(pReverbChannelSetting);
                    NN_ABORT_UNLESS_NOT_NULL(pReverbChannelSetting);
                    reverb.GetChannelIndex(pReverbChannelSetting, ChannelCount);
                    for (int i = 0; i < ChannelCount; ++i)
                    {
                        EXPECT_EQ(pChannelSetting[i], pReverbChannelSetting[i]);
                    }
                    g_Allocator.Free(reinterpret_cast<void*>(pReverbChannelSetting));
                }

                // テストケースの更新
                for (int i = 0; i < ChannelCount; ++i)
                {
                    pChannelSetting[i] = static_cast<nn::atk::ChannelIndex>(pChannelSetting[i] + 1);
                    // 繰り上がりしない場合
                    if (pChannelSetting[i] != nn::atk::ChannelIndex_Count || i == ChannelCount - 1)
                    {
                        break;
                    }

                    pChannelSetting[i] = nn::atk::ChannelIndex_FrontLeft;
                }
            }

            g_Allocator.Free(reinterpret_cast<void*>(pChannelSetting));
        }
        else
        {
            // サポートしていないチャンネル数の場合
            EXPECT_FALSE(reverb.SetChannelMode(channelMode));
        }
    }

    g_Allocator.Finalize();
}

#ifndef NN_SDK_BUILD_RELEASE
TEST(Effect, I3dl2AppendRemoveDeathTest)
{
    nnt::atk::util::OnPreAtkTest();
    // エフェクトの SampleRate と SoundSystem の SampleRate が異なる場合
    for (auto& effectSampleRate : nnt::atk::effectUtil::TestEffectSampleRate)
    {
        for (auto& soundSystemSampleRate : nnt::atk::effectUtil::TestEffectSampleRate)
        {
            if(effectSampleRate == soundSystemSampleRate)
            {
                continue;
            }

            g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
            g_FsSetup.Initialize();
            nnt::atk::util::AtkCommonSetup::InitializeParam param;
            param.GetSoundSystemParam().rendererSampleRate = nnt::atk::effectUtil::ConvertSampleRateToInt(soundSystemSampleRate);
            g_AtkSetup.Initialize(param, g_Allocator);

            nn::atk::EffectI3dl2Reverb reverb;

            nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();

            // エフェクトの Append
            EXPECT_TRUE(reverb.SetSampleRate(effectSampleRate));
            size_t reverbBufferSize = nn::util::align_up(reverb.GetRequiredMemSize(), nn::audio::MemoryPoolType::SizeGranularity);
            void* reverbBuffer = nnt::atk::util::AllocateUninitializedMemory(g_Allocator, reverbBufferSize, nn::audio::MemoryPoolType::AddressAlignment);
            NN_ABORT_UNLESS_NOT_NULL(reverbBuffer);
            nn::audio::MemoryPoolType memoryPoolForReverb;
            nn::atk::SoundSystem::AttachMemoryPool(&memoryPoolForReverb, reverbBuffer, reverbBufferSize);

            reverb.SetEnabled(true);
            EXPECT_DEATH_IF_SUPPORTED(nnt::atk::effectUtil::AppendEffectAndWait(nn::atk::AuxBus_A, &reverb, reverbBuffer, reverbBufferSize, soundArchivePlayer), ".*");

            nn::atk::SoundSystem::DetachMemoryPool(&memoryPoolForReverb);

            g_Allocator.Free(reverbBuffer);
            g_AtkSetup.Finalize(g_Allocator);
            g_FsSetup.Finalize();
            g_Allocator.Finalize();
        }
    }

    // Append 後 IsRemovable() が false のまま Remove
    {
        g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
        g_FsSetup.Initialize();
        g_AtkSetup.Initialize(g_Allocator);

        nn::atk::EffectI3dl2Reverb reverb;

        nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();

        // エフェクトの Append
        EXPECT_TRUE(reverb.SetSampleRate(nn::atk::EffectBase::SampleRate_48000));
        size_t reverbBufferSize = nn::util::align_up(reverb.GetRequiredMemSize(), nn::audio::MemoryPoolType::SizeGranularity);
        void* reverbBuffer = nnt::atk::util::AllocateUninitializedMemory(g_Allocator, reverbBufferSize, nn::audio::MemoryPoolType::AddressAlignment);
        NN_ABORT_UNLESS_NOT_NULL(reverbBuffer);
        nn::audio::MemoryPoolType memoryPoolForReverb;
        nn::atk::SoundSystem::AttachMemoryPool(&memoryPoolForReverb, reverbBuffer, reverbBufferSize);

        reverb.SetEnabled(true);
        EXPECT_TRUE(nn::atk::SoundSystem::AppendEffect(nn::atk::AuxBus_A, &reverb, reverbBuffer, reverbBufferSize));

        // Append で送ったコマンドが処理されるのを待つ
        nnt::atk::util::WaitForProcessCommand();

        // nn::audio のエフェクト追加 API 呼出後、
        // RequestUpdateAudioRendrer が呼ばれて IsRemovable が false になるまで待つ
        nnt::atk::effectUtil::WaitForEffectAppend(reverb);

        EXPECT_DEATH_IF_SUPPORTED(nnt::atk::effectUtil::RemoveEffectAndWait(nn::atk::AuxBus_A, &reverb, soundArchivePlayer), ".*");

        reverb.SetEnabled(false);
        nnt::atk::effectUtil::WaitForEffectClear(reverb);

        nnt::atk::effectUtil::RemoveEffectAndWait(nn::atk::AuxBus_A, &reverb, soundArchivePlayer);
        nn::atk::SoundSystem::DetachMemoryPool(&memoryPoolForReverb);

        g_Allocator.Free(reverbBuffer);
        g_AtkSetup.Finalize(g_Allocator);
        g_FsSetup.Finalize();
        g_Allocator.Finalize();
    }
}
#endif


TEST(Effect, I3dl2ReverbAppendTest)
{
    nnt::atk::util::OnPreAtkTest();
    for (auto& sampleRate : nnt::atk::effectUtil::TestEffectSampleRate)
    {
        for (auto& channelMode : nnt::atk::effectUtil::TestEffectChannelMode)
        {
            g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
            g_FsSetup.Initialize();
            nnt::atk::util::AtkCommonSetup::InitializeParam param;
            param.GetSoundSystemParam().rendererSampleRate = nnt::atk::effectUtil::ConvertSampleRateToInt(sampleRate);
            g_AtkSetup.Initialize(param, g_Allocator);

            nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();

            nn::atk::EffectI3dl2Reverb reverb;
            const int ChannelCount = nnt::atk::effectUtil::ConvertChannelModeToInt(channelMode);

            // 使用するチャンネル設定の初期化
            nn::atk::ChannelIndex* pValidChannelSetting = reinterpret_cast<nn::atk::ChannelIndex*>(nnt::atk::util::AllocateUninitializedMemory(g_Allocator, sizeof(nn::atk::ChannelIndex) * ChannelCount));
            NN_ABORT_UNLESS_NOT_NULL(pValidChannelSetting);
            // 初期化
            for (int i = 0; i < ChannelCount; ++i)
            {
                pValidChannelSetting[i] = static_cast<nn::atk::ChannelIndex>(i);
            }

            // Append 前に変更できることをテストする
            if (IsI3dl2ReverbChannelModeSupported(channelMode))
            {
                EXPECT_TRUE(reverb.SetChannelMode(channelMode));
                EXPECT_TRUE(reverb.SetChannelIndex(pValidChannelSetting, channelMode));
            }
            else
            {
                // 変更できない場合はそれを確認して終了処理を行い、次のテスト項目へ
                EXPECT_FALSE(reverb.SetChannelMode(channelMode));
                EXPECT_FALSE(reverb.SetChannelIndex(pValidChannelSetting, channelMode));
                g_AtkSetup.Finalize(g_Allocator);
                g_FsSetup.Finalize();
                g_Allocator.Finalize();
                continue;
            }

            // エフェクトの Append
            EXPECT_TRUE(reverb.SetSampleRate(sampleRate));
            EXPECT_EQ(sampleRate, reverb.GetSampleRate());
            size_t reverbBufferSize = nn::util::align_up(reverb.GetRequiredMemSize(), nn::audio::MemoryPoolType::SizeGranularity);
            void* reverbBuffer = nnt::atk::util::AllocateUninitializedMemory(g_Allocator, reverbBufferSize, nn::audio::MemoryPoolType::AddressAlignment);
            NN_ABORT_UNLESS_NOT_NULL(reverbBuffer);
            nn::audio::MemoryPoolType memoryPoolForReverb;
            nn::atk::SoundSystem::AttachMemoryPool(&memoryPoolForReverb, reverbBuffer, reverbBufferSize);
            reverb.SetEnabled(true);
            EXPECT_TRUE(nn::atk::SoundSystem::AppendEffect(nn::atk::AuxBus_A, &reverb, reverbBuffer, reverbBufferSize));

            // Append で送ったコマンドが処理されるのを待つ
            nnt::atk::util::WaitForProcessCommand();

            // nn::audio のエフェクト追加 API 呼出後、
            // RequestUpdateAudioRendrer が呼ばれて IsRemovable が false になるまで待つ
            nnt::atk::effectUtil::WaitForEffectAppend(reverb);

            // Append 完了後に変更ができないことをテストする
            EXPECT_FALSE(reverb.SetSampleRate(sampleRate));
            EXPECT_FALSE(reverb.SetChannelMode(channelMode));
            EXPECT_FALSE(reverb.SetChannelIndex(pValidChannelSetting, channelMode));

            reverb.SetEnabled(false);
            soundArchivePlayer.Update();
            nnt::atk::effectUtil::WaitForEffectClear(reverb);
            nn::atk::SoundSystem::RemoveEffect(nn::atk::AuxBus_A, &reverb);
            nn::atk::SoundSystem::DetachMemoryPool(&memoryPoolForReverb);

            // Remove 後に変更できることをテストする
            EXPECT_TRUE(reverb.SetSampleRate(sampleRate));
            EXPECT_TRUE(reverb.SetChannelMode(channelMode));
            EXPECT_TRUE(reverb.SetChannelIndex(pValidChannelSetting, channelMode));

            g_Allocator.Free(reinterpret_cast<void*>(pValidChannelSetting));
            g_Allocator.Free(reverbBuffer);

            g_AtkSetup.Finalize(g_Allocator);
            g_FsSetup.Finalize();
            g_Allocator.Finalize();
        }
    }
}

TEST(Effect, I3dl2ReverbGetSetParameterTest)
{
    nnt::atk::util::OnPreAtkTest();
    nn::atk::EffectI3dl2Reverb reverb;

    // デフォルト値テスト, 境界値テスト, 有効範囲内の値を設定したときに正しく反映されるかのテストを行う
    EXPECT_EQ(nn::atk::EffectI3dl2Reverb::DefaultRoomGain, reverb.GetRoomGain());
    reverb.SetRoomGain(nn::atk::EffectI3dl2Reverb::RoomGainMin - 1);
    EXPECT_EQ(nn::atk::EffectI3dl2Reverb::RoomGainMin, reverb.GetRoomGain());
    reverb.SetRoomGain((nn::atk::EffectI3dl2Reverb::RoomGainMin + nn::atk::EffectI3dl2Reverb::RoomGainMax) / 2);
    EXPECT_EQ((nn::atk::EffectI3dl2Reverb::RoomGainMin + nn::atk::EffectI3dl2Reverb::RoomGainMax) / 2, reverb.GetRoomGain());
    reverb.SetRoomGain(nn::atk::EffectI3dl2Reverb::RoomGainMax + 1);
    EXPECT_EQ(nn::atk::EffectI3dl2Reverb::RoomGainMax, reverb.GetRoomGain());

    EXPECT_EQ(nn::atk::EffectI3dl2Reverb::DefaultRoomHfGain, reverb.GetRoomHfGain());
    reverb.SetRoomHfGain(nn::atk::EffectI3dl2Reverb::RoomHfGainMin - 1);
    EXPECT_EQ(nn::atk::EffectI3dl2Reverb::RoomHfGainMin, reverb.GetRoomHfGain());
    reverb.SetRoomHfGain((nn::atk::EffectI3dl2Reverb::RoomHfGainMin + nn::atk::EffectI3dl2Reverb::RoomHfGainMax) / 2);
    EXPECT_EQ((nn::atk::EffectI3dl2Reverb::RoomHfGainMin + nn::atk::EffectI3dl2Reverb::RoomHfGainMax) / 2, reverb.GetRoomHfGain());
    reverb.SetRoomHfGain(nn::atk::EffectI3dl2Reverb::RoomHfGainMax + 1);
    EXPECT_EQ(nn::atk::EffectI3dl2Reverb::RoomHfGainMax, reverb.GetRoomHfGain());

    EXPECT_EQ(nn::atk::EffectI3dl2Reverb::GetDefaultLateReverbDecayTime(), reverb.GetLateReverbDecayTime());
    reverb.SetLateReverbDecayTime(nn::atk::EffectI3dl2Reverb::GetLateReverbDecayTimeMin() - nn::TimeSpan::FromSeconds(1));
    EXPECT_EQ(nn::atk::EffectI3dl2Reverb::GetLateReverbDecayTimeMin(), reverb.GetLateReverbDecayTime());
    reverb.SetLateReverbDecayTime(nn::TimeSpan::FromMilliSeconds(150));
    EXPECT_EQ(nn::TimeSpan::FromMilliSeconds(150), reverb.GetLateReverbDecayTime());
    reverb.SetLateReverbDecayTime(nn::atk::EffectI3dl2Reverb::GetLateReverbDecayTimeMax() + nn::TimeSpan::FromSeconds(1));
    EXPECT_EQ(nn::atk::EffectI3dl2Reverb::GetLateReverbDecayTimeMax(), reverb.GetLateReverbDecayTime());

    EXPECT_EQ(nn::atk::EffectI3dl2Reverb::DefaultLateReverbHfDecayRatio, reverb.GetLateReverbHfDecayRatio());
    reverb.SetLateReverbHfDecayRatio(nn::atk::EffectI3dl2Reverb::LateReverbHfDecayRatioMin - 1);
    EXPECT_EQ(nn::atk::EffectI3dl2Reverb::LateReverbHfDecayRatioMin, reverb.GetLateReverbHfDecayRatio());
    reverb.SetLateReverbHfDecayRatio((nn::atk::EffectI3dl2Reverb::LateReverbHfDecayRatioMin + nn::atk::EffectI3dl2Reverb::LateReverbHfDecayRatioMax) / 2);
    EXPECT_EQ((nn::atk::EffectI3dl2Reverb::LateReverbHfDecayRatioMin + nn::atk::EffectI3dl2Reverb::LateReverbHfDecayRatioMax) / 2, reverb.GetLateReverbHfDecayRatio());
    reverb.SetLateReverbHfDecayRatio(nn::atk::EffectI3dl2Reverb::LateReverbHfDecayRatioMax + 1);
    EXPECT_EQ(nn::atk::EffectI3dl2Reverb::LateReverbHfDecayRatioMax, reverb.GetLateReverbHfDecayRatio());

    EXPECT_EQ(nn::atk::EffectI3dl2Reverb::DefaultReflectionsGain, reverb.GetReflectionsGain());
    reverb.SetReflectionsGain(nn::atk::EffectI3dl2Reverb::ReflectionsGainMin - 1);
    EXPECT_EQ(nn::atk::EffectI3dl2Reverb::ReflectionsGainMin, reverb.GetReflectionsGain());
    reverb.SetReflectionsGain((nn::atk::EffectI3dl2Reverb::ReflectionsGainMin + nn::atk::EffectI3dl2Reverb::ReflectionsGainMax) / 2);
    EXPECT_EQ((nn::atk::EffectI3dl2Reverb::ReflectionsGainMin + nn::atk::EffectI3dl2Reverb::ReflectionsGainMax) / 2, reverb.GetReflectionsGain());
    reverb.SetReflectionsGain(nn::atk::EffectI3dl2Reverb::ReflectionsGainMax + 1);
    EXPECT_EQ(nn::atk::EffectI3dl2Reverb::ReflectionsGainMax, reverb.GetReflectionsGain());

    EXPECT_EQ(nn::atk::EffectI3dl2Reverb::GetDefaultReflectionDelayTime(), reverb.GetReflectionDelayTime());
    reverb.SetReflectionDelayTime(nn::atk::EffectI3dl2Reverb::GetReflectionDelayTimeMin() - nn::TimeSpan::FromSeconds(1));
    EXPECT_EQ(nn::atk::EffectI3dl2Reverb::GetReflectionDelayTimeMin(), reverb.GetReflectionDelayTime());
    reverb.SetReflectionDelayTime(nn::TimeSpan::FromMilliSeconds(150));
    EXPECT_EQ(nn::TimeSpan::FromMilliSeconds(150), reverb.GetReflectionDelayTime());
    reverb.SetReflectionDelayTime(nn::atk::EffectI3dl2Reverb::GetReflectionDelayTimeMax() + nn::TimeSpan::FromSeconds(1));
    EXPECT_EQ(nn::atk::EffectI3dl2Reverb::GetReflectionDelayTimeMax(), reverb.GetReflectionDelayTime());

    EXPECT_EQ(nn::atk::EffectI3dl2Reverb::DefaultReverbGain, reverb.GetReverbGain());
    reverb.SetReverbGain(nn::atk::EffectI3dl2Reverb::ReverbGainMin - 1);
    EXPECT_EQ(nn::atk::EffectI3dl2Reverb::ReverbGainMin, reverb.GetReverbGain());
    reverb.SetReverbGain((nn::atk::EffectI3dl2Reverb::ReverbGainMin + nn::atk::EffectI3dl2Reverb::ReverbGainMax) / 2);
    EXPECT_EQ((nn::atk::EffectI3dl2Reverb::ReverbGainMin + nn::atk::EffectI3dl2Reverb::ReverbGainMax) / 2, reverb.GetReverbGain());
    reverb.SetReverbGain(nn::atk::EffectI3dl2Reverb::ReverbGainMax + 1);
    EXPECT_EQ(nn::atk::EffectI3dl2Reverb::ReverbGainMax, reverb.GetReverbGain());

    EXPECT_EQ(nn::atk::EffectI3dl2Reverb::GetDefaultLateReverbDelayTime(), reverb.GetLateReverbDelayTime());
    reverb.SetLateReverbDelayTime(nn::atk::EffectI3dl2Reverb::GetLateReverbDelayTimeMin() - nn::TimeSpan::FromSeconds(1));
    EXPECT_EQ(nn::atk::EffectI3dl2Reverb::GetLateReverbDelayTimeMin(), reverb.GetLateReverbDelayTime());
    reverb.SetLateReverbDelayTime(nn::TimeSpan::FromMilliSeconds(50));
    EXPECT_EQ(nn::TimeSpan::FromMilliSeconds(50), reverb.GetLateReverbDelayTime());
    reverb.SetLateReverbDelayTime(nn::atk::EffectI3dl2Reverb::GetLateReverbDelayTimeMax() + nn::TimeSpan::FromSeconds(1));
    EXPECT_EQ(nn::atk::EffectI3dl2Reverb::GetLateReverbDelayTimeMax(), reverb.GetLateReverbDelayTime());

    EXPECT_EQ(nn::atk::EffectI3dl2Reverb::DefaultLateReverbDiffusion, reverb.GetLateReverbDiffusion());
    reverb.SetLateReverbDiffusion(nn::atk::EffectI3dl2Reverb::LateReverbDiffusionMin - 1);
    EXPECT_EQ(nn::atk::EffectI3dl2Reverb::LateReverbDiffusionMin, reverb.GetLateReverbDiffusion());
    reverb.SetLateReverbDiffusion((nn::atk::EffectI3dl2Reverb::LateReverbDiffusionMin + nn::atk::EffectI3dl2Reverb::LateReverbDiffusionMax) / 2);
    EXPECT_EQ((nn::atk::EffectI3dl2Reverb::LateReverbDiffusionMin + nn::atk::EffectI3dl2Reverb::LateReverbDiffusionMax) / 2, reverb.GetLateReverbDiffusion());
    reverb.SetLateReverbDiffusion(nn::atk::EffectI3dl2Reverb::LateReverbDiffusionMax + 1);
    EXPECT_EQ(nn::atk::EffectI3dl2Reverb::LateReverbDiffusionMax, reverb.GetLateReverbDiffusion());

    EXPECT_EQ(nn::atk::EffectI3dl2Reverb::DefaultLateReverbDensity, reverb.GetLateReverbDensity());
    reverb.SetLateReverbDensity(nn::atk::EffectI3dl2Reverb::LateReverbDensityMin - 1);
    EXPECT_EQ(nn::atk::EffectI3dl2Reverb::LateReverbDensityMin, reverb.GetLateReverbDensity());
    reverb.SetLateReverbDensity((nn::atk::EffectI3dl2Reverb::LateReverbDensityMin + nn::atk::EffectI3dl2Reverb::LateReverbDensityMax) / 2);
    EXPECT_EQ((nn::atk::EffectI3dl2Reverb::LateReverbDensityMin + nn::atk::EffectI3dl2Reverb::LateReverbDensityMax) / 2, reverb.GetLateReverbDensity());
    reverb.SetLateReverbDensity(nn::atk::EffectI3dl2Reverb::LateReverbDensityMax + 1);
    EXPECT_EQ(nn::atk::EffectI3dl2Reverb::LateReverbDensityMax, reverb.GetLateReverbDensity());

    EXPECT_EQ(nn::atk::EffectI3dl2Reverb::DefaultHfReference, reverb.GetHfReference());
    reverb.SetHfReference(nn::atk::EffectI3dl2Reverb::HfReferenceMin - 1);
    EXPECT_EQ(nn::atk::EffectI3dl2Reverb::HfReferenceMin, reverb.GetHfReference());
    reverb.SetHfReference((nn::atk::EffectI3dl2Reverb::HfReferenceMin + nn::atk::EffectI3dl2Reverb::HfReferenceMax) / 2);
    EXPECT_EQ((nn::atk::EffectI3dl2Reverb::HfReferenceMin + nn::atk::EffectI3dl2Reverb::HfReferenceMax) / 2, reverb.GetHfReference());
    reverb.SetHfReference(nn::atk::EffectI3dl2Reverb::HfReferenceMax + 1);
    EXPECT_EQ(nn::atk::EffectI3dl2Reverb::HfReferenceMax, reverb.GetHfReference());

    EXPECT_EQ(nn::atk::EffectI3dl2Reverb::DefaultDryGain, reverb.GetDryGain());
    reverb.SetDryGain(nn::atk::EffectI3dl2Reverb::DryGainMin - 1);
    EXPECT_EQ(nn::atk::EffectI3dl2Reverb::DryGainMin, reverb.GetDryGain());
    reverb.SetDryGain((nn::atk::EffectI3dl2Reverb::DryGainMin + nn::atk::EffectI3dl2Reverb::DryGainMax) / 2);
    EXPECT_EQ((nn::atk::EffectI3dl2Reverb::DryGainMin + nn::atk::EffectI3dl2Reverb::DryGainMax) / 2, reverb.GetDryGain());
    reverb.SetDryGain(nn::atk::EffectI3dl2Reverb::DryGainMax + 1);
    EXPECT_EQ(nn::atk::EffectI3dl2Reverb::DryGainMax, reverb.GetDryGain());
}
