﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// std
#include <cstddef> // nullptr, std::size_t
#include <cstdlib> // std::malloc, std::free

// sdk
#include <nn/nn_Assert.h>
#include <nn/codec/detail/codec_OpusPacketInternal.h>

// Externals/libopus
#include <opus.h>
#include <opus_multistream.h>

// Tests
#include <nnt.h>
#include <nnt/codecUtil/testCodec_UtilOpus.h>
#include <nnt/codecUtil/testCodec_NativeOpusDecoder.h>

namespace nnt {
namespace codec {

//------------------------------------------------------------------------------
// OpusDecoder
//------------------------------------------------------------------------------
#define DecoderType ::OpusDecoder

template <>
std::size_t NativeOpusDecoder<DecoderType>::GetWorkBufferSize(int sampleRate, int streamCount, int coupledStreamCount) NN_NOEXCEPT
{
    NN_UNUSED(sampleRate);
    NN_ASSERT(sampleRate == 8000 || sampleRate == 12000 || sampleRate == 16000 || sampleRate == 24000 || sampleRate == 48000);
    NN_ASSERT_EQUAL(streamCount, 1);
    NN_ASSERT(coupledStreamCount == 0 || coupledStreamCount == 1);
    return static_cast<std::size_t>(::opus_decoder_get_size(streamCount + coupledStreamCount));
}

template <>
nn::codec::OpusResult NativeOpusDecoder<DecoderType>::Initialize(
    int sampleRate,
    int channelCount,
    int streamCount,
    int coupledStreamCount,
    const uint8_t* channelMapping,
    void* workBufferAddress,
    std::size_t workBufferSize) NN_NOEXCEPT
{
    NN_UNUSED(streamCount);
    NN_UNUSED(coupledStreamCount);
    NN_UNUSED(channelMapping);
    NN_ASSERT_NOT_EQUAL(sampleRate, 0);
    NN_ASSERT_NOT_EQUAL(channelCount, 0);
    //NN_ASSERT_NOT_EQUAL(streamCount, 0);
    //NN_ASSERT_NOT_EQUAL(coupledStreamCount, 0);
    //NN_ASSERT_NOT_NULL(channelMapping);
    NN_ASSERT_NOT_NULL(workBufferAddress);
    NN_ASSERT_GREATER_EQUAL(workBufferSize, static_cast<std::size_t>(::opus_decoder_get_size(channelCount)));
    // Initializing
    m_WorkBufferAddress = workBufferAddress;
    m_WorkBufferSize = workBufferSize;
    int status = ::opus_decoder_init(static_cast<DecoderType*>(m_WorkBufferAddress), sampleRate, channelCount);
    if (OPUS_OK != status)
    {
        return util::ConvertResult(status);
    }
    m_SampleRate = sampleRate;
    m_ChannelCount = channelCount;
    m_StreamCount = streamCount;
    m_CoupledStreamCount = coupledStreamCount;
    m_IsInitialized = true;
    return nn::codec::OpusResult_Success;
}

template <>
nn::codec::OpusResult NativeOpusDecoder<DecoderType>::DecodeInterleaved(size_t* pOutConsumed, int* pOutSampleCount, int16_t* outputBuffer, size_t outputSize, const void* inputBuffer, size_t inputSize) NN_NOEXCEPT
{
    NN_ASSERT(IsInitialized());
    NN_ASSERT_NOT_NULL(pOutConsumed);
    NN_ASSERT_NOT_NULL(pOutSampleCount);
    NN_ASSERT_NOT_NULL(outputBuffer);
    NN_ASSERT_NOT_NULL(inputBuffer);
    *pOutConsumed = 0;
    using OpusHeader = nn::codec::detail::OpusPacketInternal::Header;
    const auto headerSize = sizeof(OpusHeader);
    if (inputSize <= headerSize)
    {
        return nn::codec::OpusResult_InsufficientOpusBuffer;
    }
    OpusHeader header;
    nn::codec::detail::OpusPacketInternal::GetHeaderFromPacket(&header, static_cast<const uint8_t*>(inputBuffer));
    int status = opus_decode(
        static_cast<DecoderType*>(m_WorkBufferAddress),
        static_cast<const uint8_t*>(inputBuffer) + headerSize,
        static_cast<opus_int32>(header.packetSize),
        outputBuffer,
        static_cast<int>(outputSize / sizeof(int16_t)),
        0
    );
    if (status < 0)
    {
        *pOutSampleCount = 0;
        return util::ConvertResult(status);
    }
    *pOutSampleCount = status;
    return nn::codec::OpusResult_Success;
}

// Instantiation
template class NativeOpusDecoder<DecoderType>;

#undef DecoderType

//------------------------------------------------------------------------------
// OpusMSDecoder
//------------------------------------------------------------------------------
#define DecoderType ::OpusMSDecoder
template <>
std::size_t NativeOpusDecoder<DecoderType>::GetWorkBufferSize(int sampleRate, int streamCount, int coupledStreamCount) NN_NOEXCEPT
{
    NN_UNUSED(sampleRate);
    NN_ASSERT_NOT_EQUAL(sampleRate, 0);
    NN_ASSERT_NOT_EQUAL(streamCount, 0);
    NN_ASSERT_LESS_EQUAL(coupledStreamCount, streamCount);
    return static_cast<std::size_t>(::opus_multistream_decoder_get_size(streamCount, coupledStreamCount));
}

template <>
nn::codec::OpusResult NativeOpusDecoder<DecoderType>::Initialize(
    int sampleRate,
    int channelCount,
    int streamCount,
    int coupledStreamCount,
    const uint8_t* channelMapping,
    void* workBufferAddress,
    std::size_t workBufferSize) NN_NOEXCEPT
{
    NN_UNUSED(!IsInitialized());
    NN_ASSERT_NOT_EQUAL(sampleRate, 0);
    NN_ASSERT_NOT_EQUAL(channelCount, 0);
    NN_ASSERT_NOT_EQUAL(streamCount, 0);
    NN_ASSERT_LESS_EQUAL(coupledStreamCount, streamCount);
    NN_ASSERT_NOT_NULL(channelMapping);
    NN_ASSERT_NOT_NULL(workBufferAddress);
    NN_ASSERT_GREATER_EQUAL(workBufferSize, static_cast<std::size_t>(::opus_multistream_decoder_get_size(streamCount, coupledStreamCount)));
    // Initializing
    m_WorkBufferAddress = workBufferAddress;
    m_WorkBufferSize = workBufferSize;
    const int status = ::opus_multistream_decoder_init(
        static_cast<DecoderType*>(m_WorkBufferAddress),
        sampleRate,
        channelCount,
        streamCount,
        coupledStreamCount,
        channelMapping);
    if (OPUS_OK != status)
    {
        return util::ConvertResult(status);
    }
    m_SampleRate = sampleRate;
    m_ChannelCount = channelCount;
    m_StreamCount = streamCount;
    m_CoupledStreamCount = coupledStreamCount;
    m_IsInitialized = true;
    return nn::codec::OpusResult_Success;
}

template <>
nn::codec::OpusResult NativeOpusDecoder<DecoderType>::DecodeInterleaved(size_t* pOutConsumed, int* pOutSampleCount, int16_t* outputBuffer, size_t outputSize, const void* inputBuffer, size_t inputSize) NN_NOEXCEPT
{
    NN_ASSERT(IsInitialized());
    NN_ASSERT_NOT_NULL(pOutConsumed);
    NN_ASSERT_NOT_NULL(pOutSampleCount);
    NN_ASSERT_NOT_NULL(outputBuffer);
    NN_ASSERT_NOT_NULL(inputBuffer);
    *pOutConsumed = 0;
    using OpusHeader = nn::codec::detail::OpusPacketInternal::Header;
    const auto headerSize = sizeof(OpusHeader);
    if (inputSize <= headerSize)
    {
        return nn::codec::OpusResult_InsufficientOpusBuffer;
    }
    OpusHeader header;
    nn::codec::detail::OpusPacketInternal::GetHeaderFromPacket(&header, static_cast<const uint8_t*>(inputBuffer));
    const int status = ::opus_multistream_decode(
        static_cast<DecoderType*>(m_WorkBufferAddress),
        static_cast<const uint8_t*>(inputBuffer) + headerSize,
        static_cast<opus_int32>(header.packetSize),
        outputBuffer,
        static_cast<int>(outputSize / sizeof(int16_t)),
        0
    );
    if (status < 0)
    {
        *pOutSampleCount = 0;
        util::DumpOpusPacket(static_cast<const uint8_t*>(inputBuffer), inputSize, GetStreamCount());
        return util::ConvertResult(status);
    }
    *pOutSampleCount = status;
    return nn::codec::OpusResult_Success;
}

// Instantiation
template class NativeOpusDecoder<DecoderType>;

#undef DecoderType

}} // nnt::codec
