﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/audioUtil/testAudio_Util.h>
#include <nnt/audioUtil/testAudio_Constants.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_MessageQueue.h>
#include <nn/fs/fs_Result.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>

#include <algorithm> // std::max, std::min
#include <memory> // std::unique_ptr
#include <cstdio> //snprintf
#if defined(NN_BUILD_CONFIG_TOOLCHAIN_VC)
#define snprintf _snprintf
#endif
#include <string>

namespace nnt {
namespace audio {
namespace util {

namespace
{

// Recorder Worker Thread
const size_t RecorderThreadStackSize = 64 * 1024;
NN_ALIGNAS(4096) char g_RecorderThreadStack[ RecorderThreadStackSize ];

bool g_FileSystemInitialized = false;

}

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
bool IsFirmwareSettingEnabled(const char* key) NN_NOEXCEPT
{
    bool isEnabled = false;

    if(nn::settings::fwdbg::GetSettingsItemValue(&isEnabled, sizeof(isEnabled), "audio", key) != sizeof(isEnabled))
    {
        NN_LOG("[audio] Fail to read %s's value in firmware debug settings.\n", key);
    }

    return isEnabled;
}
#endif

////////////////////////////////////////////////////////////////////////////////
// Audio File Utility class
////////////////////////////////////////////////////////////////////////////////

#define SIGWORD(a,b,c,d)        \
    static_cast<uint32_t>(          \
          (static_cast<uint8_t>(a) <<  0)        \
        | (static_cast<uint8_t>(b) <<  8)        \
        | (static_cast<uint8_t>(c) << 16)        \
        | (static_cast<uint8_t>(d) << 24)        \
    )

static inline uint32_t CreateID(const char* c)
{
    return SIGWORD(c[0],c[1],c[2],c[3]);
}

static inline bool checkID(int8_t* id, const char* desiredID)
{
    uint32_t desired = CreateID(desiredID);
    uint32_t _id = SIGWORD(id[0],id[1],id[2],id[3]);
    return (_id == desired);
}

WaveFile::WaveFile()
    : m_SampleRate(0)
    , m_ChannelCount(0)
    , m_BitRate(0)
    , m_DataStartOffset(0)
    , m_CurrentReadWriteOffset(0)
    , m_SampleDataSize(0)
    , m_FileSize(0)
    , m_Endian(WaveEndian_Invalid)
{
}

WaveFile::~WaveFile()
{
}

// parameter accessors
int WaveFile::GetSampleRate() const
{
    return static_cast<int>(m_SampleRate);
}

int WaveFile::GetChannelCount() const
{
    return static_cast<int>(m_ChannelCount);
}

int WaveFile::GetBitRate() const
{
    return static_cast<int>(m_BitRate);
}

size_t WaveFile::GetSampleDataSize() const
{
    return static_cast<size_t>(m_SampleDataSize);
}


bool WaveFile::Open(const char* filename, uint32_t sampleRate, uint16_t channels, uint16_t bitRate)
{
    m_OpenMode = nn::fs::OpenMode_Write;

    NN_ASSERT(bitRate == 16);
    NN_ASSERT(channels >= 1 && channels <= 6);
    NN_ASSERT(sampleRate == 48000 || sampleRate == 32000);
    NN_ASSERT(filename != nullptr);

    nn::Result rt;
    rt = nn::fs::OpenFile(&m_Handle, filename, m_OpenMode);
    NN_ASSERT(rt.IsSuccess());

    m_SampleRate = sampleRate;
    m_ChannelCount = channels;
    m_BitRate = bitRate;
    m_Endian = WaveEndian_Little; // TODO: little endian only for now.

    SetWavInfo(m_WavFormat, m_SampleRate, m_BitRate, m_ChannelCount, 0, m_Endian);
    nn::fs::WriteOption option = {};
    rt = nn::fs::WriteFile(m_Handle, 0, m_WavFormat, sizeof(WaveFileFormat), option);
    m_CurrentReadWriteOffset += sizeof(WaveFileFormat);
    NN_ASSERT(rt.IsSuccess());

    return true;
}

bool WaveFile::Open(const char* filename)
{
    m_OpenMode = nn::fs::OpenMode_Read;
    NN_ASSERT(filename != nullptr);
    nn::Result rt = nn::fs::OpenFile(&m_Handle, filename, m_OpenMode);
    NN_ASSERT(rt.IsSuccess());
    rt = nn::fs::GetFileSize(&m_FileSize, m_Handle);
    NN_ASSERT(rt.IsSuccess());

    // assume that no extension region.
    int8_t readBuf[sizeof(WaveFileFormat)];
    memset(readBuf, 0, sizeof(readBuf));
    size_t read = Read(readBuf,sizeof(WaveFileFormat));
    NN_ASSERT(read == sizeof(WaveFileFormat));
    if (WaveFileResult_Success != GetWaveInfo(readBuf, sizeof(WaveFileFormat),
                                              &m_ChannelCount, &m_SampleRate, &m_BitRate,
                                              &m_DataStartOffset, &m_SampleDataSize, &m_Endian))
    {
        //TODO: chunk 走査失敗時の処理
    }

    return true;
}

size_t WaveFile::Write(const int8_t* buf, const size_t dataSize)
{
    NN_ASSERT(m_OpenMode & nn::fs::OpenMode_Write);
    nn::Result rt;
    nn::fs::WriteOption option = {};
    rt = nn::fs::WriteFile(m_Handle, m_CurrentReadWriteOffset, buf, dataSize, option);
    NN_ASSERT(rt.IsSuccess());

    m_CurrentReadWriteOffset += dataSize;
    m_SampleDataSize += dataSize;
    return dataSize;
}

size_t WaveFile::Read(int8_t* buf, const size_t readSize)
{
    NN_ASSERT(m_OpenMode == nn::fs::OpenMode_Read);

    nn::Result rt;
    rt = nn::fs::ReadFile(m_Handle, m_CurrentReadWriteOffset, buf, readSize);
    NN_ASSERT(rt.IsSuccess());
    // TODO: failure handle case.

    m_CurrentReadWriteOffset += readSize;

    return readSize;
}


void WaveFile::Seek( const uint32_t offset )
{
    m_CurrentReadWriteOffset = offset;
}

void WaveFile::Close()
{
    if (m_OpenMode == nn::fs::OpenMode_Write)
    {
        nn::Result rt;
        rt = nn::fs::FlushFile(m_Handle);
        NN_ASSERT(rt.IsSuccess());

        // update RiffHeader & DataChunk
        SetWavInfo(m_WavFormat, m_SampleRate, m_BitRate, m_ChannelCount, m_SampleDataSize, m_Endian);
        nn::fs::WriteOption option = {};
        rt = nn::fs::WriteFile(m_Handle, 0, m_WavFormat, sizeof(WaveFileFormat), option);
        m_CurrentReadWriteOffset += sizeof(WaveFileFormat);
        NN_ASSERT(rt.IsSuccess());

        // close stream
        rt = nn::fs::FlushFile(m_Handle);
        NN_ASSERT(rt.IsSuccess());
    }

    nn::fs::CloseFile(m_Handle);

    return;
}

uint32_t WaveFile::CalcRiffChunkSize(uint32_t dataSize)
{
    return
        sizeof(WaveFileFormat) // wave file header header
        - 8                    // RIFF ID, RIFF Size
        + dataSize;            // sample data size
}

void WaveFile::SetWavInfo( int8_t* buf, const int32_t samplingFreq, const int16_t bits, const int16_t numChannels, const int64_t size, const WaveEndian endian )
{
    NN_ASSERT(endian == WaveEndian_Little); // Big Endian is not supported for now.
    NN_ASSERT(size >= 0);

    WaveFileFormat* wavFormat = (WaveFileFormat*)buf;

    // RIFF Header
    wavFormat->riff.RIFF = SIGWORD('R','I','F','F');
    wavFormat->riff.riffKind = SIGWORD('W','A','V','E');
    wavFormat->fmt.fmtId = SIGWORD('f','m','t',' ');
    wavFormat->data.dataId = SIGWORD('d','a','t','a');

    // data size
    wavFormat->riff.riffSize = static_cast<const uint32_t>(size) + CalcRiffChunkSize(0);
    wavFormat->data.Nbyte = static_cast<const int32_t>(size);
    wavFormat->fmt.chunkSize  = sizeof(FormatChunk) - 8; // ChunkID と chunkSize 分

    // wav params
    switch (endian)
    {
    case WaveEndian_Little:
        wavFormat->fmt.id  = 1;
        break;
    default:
        break;
    }
    wavFormat->fmt.channelCount = numChannels;
    wavFormat->fmt.samplingFreq  = samplingFreq;
    wavFormat->fmt.bytePerSec = ( bits / 8 ) * numChannels * samplingFreq;
    wavFormat->fmt.blockSize = ( bits / 8 ) * numChannels;
    wavFormat->fmt.bitsPerSample = bits;
}

WaveFile::WaveFileResult WaveFile::GetWaveInfo( int8_t* buf,
                           uint32_t bufferSize,
                           uint16_t* nChannel,
                           uint32_t* samplingFreq,
                           uint16_t* bitsPerSample,
                           uint32_t* dataStartOffset,
                           int64_t* dataSize,
                           WaveEndian* endian)
{
    uint32_t n = 0;
    uint16_t nCh = 0, bps = 0;
    uint32_t fs = 0, start = 0;
    size_t size = 0;
    WaveEndian end = WaveEndian_Invalid;
    bool chkFmt = false, chkData = false, chkLen = false;

    if ( !checkID(&(buf[0]), "RIFF") || !checkID(&(buf[8]), "WAVE") )
    {
        return WaveFileResult_InvalidDataFormat;
    }

    n += 3; // skip RIFF header

    while( n < bufferSize )
    {
        if ( (n + 8) > bufferSize )
        {
            chkLen = true;
            break;
        }

        if ( checkID(&buf[n], "fmt ") )
        {
            // extract fmt parameter
            FormatChunk* fmt = (FormatChunk*)&buf[n];

            nCh = fmt->channelCount;
            fs = fmt->samplingFreq;
            bps = fmt->bitsPerSample;
            switch(fmt->id)
            {
            case 0x01:
                end = WaveEndian_Little;
                break;
            case 0xfe02:
                end = WaveEndian_Big;
                break;
            default:
                end = WaveEndian_Invalid;
                break;
            }

            chkFmt = true;
            n += sizeof(FormatChunk);
            //n++;
        }
        else if ( checkID(&buf[n], "data") )
        {
            // extract data size
            DataChunk* data = (DataChunk*)&buf[n];
            size = static_cast<uint32_t>(data->Nbyte); // TODO:
            n += sizeof(DataChunk);
            start = n;

            chkData = true;
            break; // end up check.
        }
        else
        {
            n++; // move next.
        }
    }

    if (chkFmt && chkData)
    {
        *nChannel = nCh;
        *samplingFreq = fs;
        *bitsPerSample = bps;
        *dataStartOffset = start;
        *dataSize = size;
        *endian = end;
    }
    else if (chkLen)
    {
        return WaveFileResult_BufferSizeTooSmall;
    }
    else
    {
        return WaveFileResult_UnexpectedError;
    }

    return WaveFileResult_Success;
}

const size_t WaveFile::GetRequiredFileSize( int32_t sampleRate, int32_t channelCount, int32_t bitRate, nn::TimeSpan duration )
{
    return static_cast<size_t>((sampleRate * channelCount * (bitRate / 8) * duration.GetMilliSeconds() ) / 1000) + sizeof(WaveFile::WaveFileFormat);
}

////////////////////////////////////////////////////////////////////////////////
// debug APIs
////////////////////////////////////////////////////////////////////////////////

void WaveFile::DumpInfo()
{
    NN_LOG("sampleRate:%d\n",  this->m_SampleRate);
    NN_LOG("channel:%d\n",     this->m_ChannelCount);
    NN_LOG("bitRate:%d\n",     this->m_BitRate);
    NN_LOG("dataSize:%d\n",    this->m_SampleDataSize);
    NN_LOG("sampleCount:%d\n", this->m_SampleDataSize / sizeof(int16_t));
}

////////////////////////////////////////////////////////////////////////////////
// Audio Output Capture
////////////////////////////////////////////////////////////////////////////////
Recorder::Recorder()
    : m_ChannelCount(0)
    , m_FrameSampleCountPerChannel(0)
    , m_FrameSampleCount(0)
    , m_ReadBuffer(nullptr)
    , m_ReadBufferSize(0)
    , m_RecordingDurationSize(0)
    , m_pAux(nullptr)
    , m_pWaveFile(nullptr)
    , m_WorkerBooting(true)
    , m_WorkerRunning(true)
    , m_TrimLeadSilence(false)
    , m_TotalSendDataSize(0)
    , m_RecorderBuffer(nullptr)
    , m_RecorderBufferSize(0)
    , m_RecorderBufferPosition(0)
{
};

Recorder::~Recorder()
{
    std::free(m_ReadBuffer);
    m_ReadBuffer = nullptr;
};

nn::Result Recorder::Prepare(const nn::audio::AudioRendererParameter* param,
                             nn::audio::AuxType* pAux,
                             WaveFile* pWaveFile,
                             size_t bufferSize,
                             nn::TimeSpan recordingDuration,
                             void* recorderBuffer,
                             size_t recorderBufferSize)
{
    NN_ABORT_UNLESS_NOT_NULL(pAux);

    // set channel Count
    int8_t in[nn::audio::MixBufferCountMax] = { 0 }; // dummy not used.
    int8_t out[nn::audio::MixBufferCountMax] = { 0 }; // dummy not used.
    nn::audio::GetAuxInputOutput(pAux, in, out, &m_ChannelCount, nn::audio::MixBufferCountMax);

    // set up internal fields
    m_FrameSampleCountPerChannel = param->sampleCount;
    m_ReadBufferSize = bufferSize;
    m_ReadBuffer = static_cast<int32_t*>(malloc(m_ReadBufferSize));
    NN_ABORT_UNLESS_NOT_NULL(m_ReadBuffer);
    m_pAux = pAux;
    m_FrameSampleCount = m_FrameSampleCountPerChannel * m_ChannelCount;
    m_pWaveFile = pWaveFile;
    m_RecordingDurationSize = static_cast<size_t>((param->sampleRate * recordingDuration.GetMilliSeconds()) / 1000) * m_ChannelCount * sizeof(int16_t);

    m_RecorderBuffer = static_cast<int8_t*>(recorderBuffer);
    m_RecorderBufferSize = recorderBufferSize;
    m_RecorderBufferPosition = 0;

    nn::os::InitializeMessageQueue( &m_MessageQueue, m_MessageQueueBuffer, MessageQueueSize);

    if (m_RecorderBuffer)
    {
        NN_RESULT_SUCCESS;
    }
    else
    {
        return nn::os::CreateThread( &m_WorkerThread,
                                     &WorkerThreadFunction,
                                     this,
                                     g_RecorderThreadStack,
                                     RecorderThreadStackSize,
                                     nn::os::DefaultThreadPriority );
    }
};

void Recorder::EnableLeadSilenceTrimming(bool enable)
{
    m_TrimLeadSilence = enable;
}

namespace {
int GetNonSilenceSampleOffset(int16_t* samples, int sampleCount)
{
    int count = -1;
    for (auto i = 0; i < sampleCount; ++i)
    {
        if (samples[i] != 0)
        {
            return i;
        }
    }
    return count;
}
}

bool Recorder::Record()
{
    // read sample data. (no need to return buffer.)
    int32_t readCount = nn::audio::ReadAuxSendBuffer(m_pAux, m_ReadBuffer, static_cast<int32_t>(m_ReadBufferSize / sizeof(int32_t)));
    int32_t writeCount = nn::audio::WriteAuxReturnBuffer(m_pAux, m_ReadBuffer, readCount);
    NN_UNUSED(writeCount);

    if (readCount % m_FrameSampleCount != 0)
    {
        // ReadAuxSendBuffer() で読み込めるサンプル数は、frameSampleCount 単位であることを期待している。
        NN_ABORT("Aux Send Buffer returned incorrect sample count.\n");
    }

    // write into wav file per audio frame.
    size_t frameDataSize = m_FrameSampleCount * sizeof(int16_t);
    std::unique_ptr<int16_t> bufferForInterleave(new int16_t[m_FrameSampleCount]);
    int32_t* src = m_ReadBuffer;

    while (readCount > 0)
    {
        int16_t* interleavedData = bufferForInterleave.get();
        memset(bufferForInterleave.get(), 0, frameDataSize);

        // interleave samples
        for (auto idx = 0; idx < m_FrameSampleCountPerChannel; idx++)
        {
            for (auto ch = 0; ch < m_ChannelCount; ++ch)
            {
                int16_t smp = static_cast<int16_t>(src[m_FrameSampleCountPerChannel * ch + idx]);
                smp = std::max(std::numeric_limits<int16_t>::min(), smp);
                smp = std::min(std::numeric_limits<int16_t>::max(), smp);
                *interleavedData++ = smp;
            }
        }

        size_t silenceSampleSize = 0;
        if (m_TrimLeadSilence)
        {
            auto count = GetNonSilenceSampleOffset(bufferForInterleave.get(), static_cast<int32_t>(frameDataSize / sizeof(int16_t)));
            if (count >= 0)
            {
                int frame = count / m_ChannelCount;
                silenceSampleSize = sizeof(int16_t) * m_ChannelCount * (frame - 1);
                m_TrimLeadSilence = false;

                if (m_RecorderBuffer)
                {
                    NN_ASSERT(frameDataSize - silenceSampleSize <= m_RecorderBufferSize - m_RecorderBufferPosition);
                    std::memcpy(m_RecorderBuffer + m_RecorderBufferPosition, bufferForInterleave.get() + silenceSampleSize, frameDataSize - silenceSampleSize);
                    m_RecorderBufferPosition += frameDataSize - silenceSampleSize;
                }
                else
                {
                    auto check = SendBufferToWorker(reinterpret_cast<int8_t*>(bufferForInterleave.get()) + silenceSampleSize, frameDataSize - silenceSampleSize);
                    NN_ASSERT(check, "Recorder buffer is full. Sample buffer is shortage.");
                }
            }
        }
        else
        {
            if (m_RecorderBuffer)
            {
                NN_ASSERT(frameDataSize<= m_RecorderBufferSize - m_RecorderBufferPosition);
                std::memcpy(m_RecorderBuffer + m_RecorderBufferPosition, bufferForInterleave.get(), frameDataSize);
                m_RecorderBufferPosition += frameDataSize;
            }
            else
            {
                auto check = SendBufferToWorker(reinterpret_cast<int8_t*>(bufferForInterleave.get()), frameDataSize);
                NN_ASSERT(check, "Recorder buffer is full. Sample buffer is shortage.");
            }
        }

        readCount -= m_FrameSampleCount; // count-down
        src += m_FrameSampleCount; // move to next frame

    }

    if (m_RecorderBuffer)
    {
        if (m_RecorderBufferPosition >= m_RecordingDurationSize)
        {
            auto size = std::min(m_RecorderBufferPosition, m_RecordingDurationSize);
            m_pWaveFile->Write(m_RecorderBuffer, size);
            m_WorkerRunning = false;
        }
    }
    return m_WorkerRunning;
}

nn::Result Recorder::Start()
{
    if (!m_RecorderBuffer)
    {
        nn::os::StartThread(&m_WorkerThread);
        while (m_WorkerBooting)
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
        }
    }
    NN_RESULT_SUCCESS;
};

void Recorder::Stop()
{
    if (!m_RecorderBuffer)
    {
        nn::os::WaitThread(&m_WorkerThread);
        nn::os::DestroyThread(&m_WorkerThread);
    }
    nn::os::FinalizeMessageQueue(&m_MessageQueue);
}

bool Recorder::SendBufferToWorker(int8_t* buffer, size_t size)
{
    m_TotalSendDataSize += size;
    int8_t* sendBuffer = new int8_t[size];
    RecordItem* item = new RecordItem(size, sendBuffer);
    memcpy(sendBuffer, buffer, size);
    // Using TimedSend API, because sometimes on Windows message queue becomes full.
    return nn::os::TimedSendMessageQueue(&m_MessageQueue, reinterpret_cast<uintptr_t>(item), nn::TimeSpan::FromSeconds(1));
}

void Recorder::WorkerThreadFunction(void *arg)
{
    Recorder* rec = reinterpret_cast<Recorder*>(arg);
    size_t totalWriteCount = 0;
    rec->m_WorkerBooting = false;

    while (rec->m_WorkerRunning)
    {
        uintptr_t msg;
        nn::os::ReceiveMessageQueue(&msg, &rec->m_MessageQueue);
        RecordItem* item = reinterpret_cast<RecordItem*>(msg);
        auto writeSize = std::min(rec->m_RecordingDurationSize - totalWriteCount, item->size);
        rec->m_pWaveFile->Write(item->buffer, writeSize);
        totalWriteCount += writeSize;

        delete[] item->buffer;
        item->buffer = nullptr;
        delete item;
        item = nullptr;

        if (totalWriteCount >= rec->m_RecordingDurationSize)
        {
            rec->m_WorkerRunning = false;
        }
    }
}

////////////////////////////////////////////////////////////////////////////////
// Verification Rules
////////////////////////////////////////////////////////////////////////////////

Checker::WaveFormCheckError Checker::ParamCheck(const WaveFile* A, const WaveFile* B)
{
    // check parameters,
    // size, sampleFreq, bitRate, channel
    if (A->GetSampleRate() != B->GetSampleRate())
    {
        return WaveFormCheckError_SampleRate;
    }

    if ( A->GetChannelCount() != B->GetChannelCount())
    {
        return WaveFormCheckError_ChannelCount;
    }

    if ( A->GetBitRate() != B->GetBitRate())
    {
        return WaveFormCheckError_BitRate;
    }

    if ( A->GetSampleDataSize() != B->GetSampleDataSize())
    {
        return WaveFormCheckError_DataSize;
    }

    return WaveFormCheckError_NoError;
}

Checker::WaveFormCheckError Checker::SimpleCheck( WaveFile* A, WaveFile* B, int32_t* status, int diff )
{
    // check prames
    WaveFormCheckError rt;

    if( (rt = ParamCheck(A, B)) != Checker::WaveFormCheckError_NoError)
    {
        return rt;
    }

    // check data
    std::unique_ptr<int16_t> dataA(new int16_t[static_cast<size_t>(A->GetSampleDataSize() / sizeof(int16_t))]);
    std::unique_ptr<int16_t> dataB(new int16_t[static_cast<size_t>(B->GetSampleDataSize() / sizeof(int16_t))]);

    if (A->GetSampleDataSize() != A->Read(reinterpret_cast<int8_t*>(dataA.get()), static_cast<uint32_t>(A->GetSampleDataSize()))
        || B->GetSampleDataSize() != B->Read(reinterpret_cast<int8_t*>(dataB.get()), static_cast<uint32_t>(B->GetSampleDataSize())))
    {
        return WaveFormCheckError_FileAccess;
    }

    int16_t* pA = dataA.get();
    int16_t* pB = dataB.get();
    for (uint32_t i = 0; i < A->GetSampleDataSize() / sizeof(int16_t); i++)
    {
        if (std::abs(*pA++ - *pB++) > diff)
        {
            *status = i;
            return WaveFormCheckError_WaveForm;
        }
    }
    return WaveFormCheckError_NoError;
}

std::string Checker::ResultString(WaveFormCheckError error, int status)
{
    static const int StringLengthMax = 255;
    static char message[StringLengthMax + 1];

    std::string rt("*** ");
    switch(error)
    {
    case WaveFormCheckError_NoError:
        rt = "No Error";
        break;
    case WaveFormCheckError_SampleRate:
        rt += "invalid sample rate";
        break;
    case WaveFormCheckError_ChannelCount:
        rt += "invalid channel count";
        break;
    case WaveFormCheckError_BitRate:
        rt += "invalid bit rate";
        break;
    case WaveFormCheckError_DataSize:
        rt += "invalid data size";
        break;
    case WaveFormCheckError_FileAccess:
        rt += "File access failed";
        break;
    case WaveFormCheckError_WaveForm:
        rt += "invalid wave form";
        snprintf(message, StringLengthMax, " | different position:%d", status);
        rt += message;
        break;
    default:
        break;
    }

    return rt;
}


////////////////////////////////////////////////////////////////////////////////
// File system
////////////////////////////////////////////////////////////////////////////////

namespace {

void* Allocate(size_t size)
{
    return std::malloc(size);
};

void Deallocate(void* p, size_t size)
{
    NN_UNUSED(size);
    std::free(p);
};

};

SimpleFsUtility::SimpleFsUtility()
{
    // Pass
};

SimpleFsUtility::~SimpleFsUtility()
{
    nn::fs::Unmount(m_MountPathName);
};

nn::Result SimpleFsUtility::InitializeFileSystem(const char* name, const char* mountPath)
{
    nn::Result result;
    std::strncpy(m_MountPathName, name, nn::fs::MountNameLengthMax);
    if (!g_FileSystemInitialized)
    {
        nn::fs::SetAllocator(Allocate, Deallocate);
    }

    result = nn::fs::MountHost(name, mountPath);
    if(result.IsFailure())
    {
        NN_LOG("*** MountHost Failed w/ %d\n", result.GetInnerValueForDebug());
    }

    g_FileSystemInitialized = true;

    return result;
};

nn::Result SimpleFsUtility::CreateDirectory(const char* path, PathType type)
{
    std::string entryPath = std::string(path);

    // Find MountPath
    auto folderIndex = entryPath.find_first_of('/', 0);
    if(folderIndex == std::string::npos)
    {
        folderIndex = entryPath.find_first_of("\\", folderIndex);

        if(folderIndex == std::string::npos)
        {
            NN_ABORT("%s is not path.\n", path);
        }
    }

    // Create Directory Recursively
    while(true)
    {
        folderIndex = entryPath.find_first_of('/', folderIndex + 1);
        if(folderIndex == std::string::npos)
        {
            folderIndex = entryPath.find_first_of("\\", folderIndex + 1);

            if(folderIndex == std::string::npos)
            {
                if(type == PathType_Directory)
                {
                    auto result = nn::fs::CreateDirectory(entryPath.c_str());
                    if (result.IsFailure() && !nn::fs::ResultPathAlreadyExists::Includes(result))
                    {
                        return result;
                    }
                }

                NN_RESULT_SUCCESS;
            }
        }

        const std::string directoryPath = entryPath.substr(0, folderIndex);

        auto result = nn::fs::CreateDirectory(directoryPath.c_str());
        if (result.IsFailure() && !nn::fs::ResultPathAlreadyExists::Includes(result))
        {
            return result;
        }
    }
}

nn::Result SimpleFsUtility::PrepareNewFile(const char* filename, const int64_t filesize)
{
    nn::Result result;
    result = CreateDirectory(filename, PathType_File);
    if (result.IsFailure())
    {
        NN_LOG("*** CreateFolder Failed w/ %d\n", result.GetInnerValueForDebug());
        return result;
    }

    result = nn::fs::DeleteFile(filename);
    if (result.IsFailure() && !nn::fs::ResultPathNotFound::Includes(result))
    {
        NN_LOG("*** DeleteFile Failed w/ %d\n", result.GetInnerValueForDebug());
        return result;
    }

    result = nn::fs::CreateFile(filename, filesize);
    if (result.IsFailure() && !nn::fs::ResultPathAlreadyExists::Includes(result))
    {
        NN_LOG("*** CreateFile Failed w/ %d\n", result.GetInnerValueForDebug());
        return result;
    }

    NN_RESULT_SUCCESS;
}

////////////////////////////////////////////////////////////////////////////////
// Renderer
////////////////////////////////////////////////////////////////////////////////

MinimalRenderer::MinimalRenderer(void* address, size_t size)
    : m_channelCount(6)
    , m_Allocator(address, size)
{
    m_MainOutIndex[0] = 0;
    m_MainOutIndex[1] = 1;
    m_MainOutIndex[2] = 2;
    m_MainOutIndex[3] = 3;
    m_MainOutIndex[4] = 4;
    m_MainOutIndex[5] = 5;
};

MinimalRenderer::~MinimalRenderer()
{
    if (m_workBuffer)
    {
        m_Allocator.Free(m_workBuffer);
    }
    if (m_configBuffer)
    {
        m_Allocator.Free(m_configBuffer);
    }
};

nn::audio::SubMixType* MinimalRenderer::GetSubMix()
{
    return &m_SubMix;
}

nn::audio::FinalMixType* MinimalRenderer::GetFinalMix()
{
    return &m_finalMix;
};

nn::audio::AudioRendererConfig* MinimalRenderer::GetConfig()
{
    return &m_config;
}

nn::audio::AudioRendererHandle* MinimalRenderer::GetHandle()
{
    return &m_handle;
}

void MinimalRenderer::Initialize(nn::audio::AudioRendererParameter& params)
{
    m_workBufferSize = nn::audio::GetAudioRendererWorkBufferSize(params);
    m_workBuffer = m_Allocator.Allocate(m_workBufferSize, nn::os::MemoryPageSize);
    ASSERT_TRUE(m_workBuffer != nullptr);
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenAudioRenderer(&m_handle, &m_Event, params, m_workBuffer, m_workBufferSize));
    m_configBufferSize = nn::audio::GetAudioRendererConfigWorkBufferSize(params);
    m_configBuffer = m_Allocator.Allocate(m_configBufferSize);
    ASSERT_TRUE(m_configBuffer != nullptr);
    nn::audio::InitializeAudioRendererConfig(&m_config, params, m_configBuffer, m_configBufferSize);
    ASSERT_TRUE(nn::audio::AcquireSubMix(&m_config, &m_SubMix, params.sampleRate, m_channelCount));
    ASSERT_TRUE(nn::audio::AcquireFinalMix(&m_config, &m_finalMix, m_channelCount));
    nn::audio::SetSubMixDestination(&m_config, &m_SubMix, &m_finalMix);
    for (auto i = 0; i < m_channelCount; ++i)
    {
        nn::audio::SetSubMixMixVolume(&m_SubMix, &m_finalMix,  1.0f, i, i);
    }

    if(params.executionMode != nn::audio::AudioRendererExecutionMode_ManualExecution)
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::audio::AddDeviceSink(&m_config, &m_deviceSink, &m_finalMix, m_MainOutIndex, m_channelCount, "MainAudioOut"));
    }
}

void MinimalRenderer::Start()
{
    ASSERT_TRUE(nn::audio::RequestUpdateAudioRenderer(m_handle, &m_config).IsSuccess());
    ASSERT_TRUE(nn::audio::StartAudioRenderer(m_handle).IsSuccess());
}
void MinimalRenderer::Stop()
{
    nn::audio::StopAudioRenderer(m_handle);
    nn::audio::CloseAudioRenderer(m_handle);
}
void MinimalRenderer::Update()
{
    ASSERT_TRUE(nn::audio::RequestUpdateAudioRenderer(m_handle, &m_config).IsSuccess());
}

void MinimalRenderer::Wait()
{
    m_Event.Wait();
}

void MinimalRenderer::ExecuteRendering()
{
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::ExecuteAudioRendererRendering(m_handle));
}

#if 1
void GetMountPath(std::string& mountPath) // TODO: fs の実装が安定するまでの仮関数
{
    std::string sigloRootPath;

    // get application binary path
    char path[nn::fs::EntryNameLengthMax];
    char* applicationPath = nnt::GetHostArgv()[ 0 ];

    strncpy(path, applicationPath, strlen(applicationPath) + 1);

    // cut off path str at root folder.
    const char* SearchTargets[] =
            {
                "Tests\\Outputs",
                "Tests/Outputs",
                "Externals\\TestBinaries\\Audio",
                "Externals/TestBinaries/Audio",
            };
    char* tmp = nullptr;
    for (auto& target : SearchTargets)
    {
        tmp = std::strstr(path, target);
        if (tmp != nullptr)
        {
            break;
        }
    }
    ASSERT_TRUE(tmp!= nullptr);
    *tmp = 0; // null terminate at end of Root directory

    // concatenate & create mount path.
    sigloRootPath = path;
    mountPath = sigloRootPath + "Externals\\TestBinaries\\Audio\\AudioOutputComp";
}
#else
// TODO: 現状の nn::htc::GetEnvironmentVariable() はシェル上で指定される NINTENDO_SDK_ROOT の値ではなく常に環境変数の値を取得するため使用できない。
// これを改善する SIGLO-22859 が完了したら下記を有効にする
void GetMountPath(std::string& mountPath)
{
    nn::htc::Initialize();
    NN_UTIL_SCOPE_EXIT
    {
        nn::htc::Finalize();
    };

    // Target Manager と接続されたことを通知するイベントを取得します
    nn::os::SystemEvent connectionEvent;
    nn::htc::BindHostConnectionEvent(&connectionEvent);

    // Target Manager との接続を待機します
    NN_LOG("Waiting for connection from Host.\n");
    connectionEvent.Wait();

    // ひとまず MAX_PATH (260) よりも十分に大きい値にしておく
    const size_t pathLength = 1024 * 100;
    char sdkRootPath[pathLength];
    size_t size = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::htc::GetEnvironmentVariable(&size, sdkRootPath, sizeof(sdkRootPath), "NINTENDO_SDK_ROOT"));
    if(nn::htc::GetEnvironmentVariable(&size, sdkRootPath, sizeof(sdkRootPath), "NINTENDO_SDK_ROOT").IsFailure())
    {
        NN_LOG("Fail to get NINTENDO_SDK_ROOT from environment variable.\n");
        return;
    }

    // concatenate & create mount path.
    mountPath = std::string(sdkRootPath) + "\\Externals\\TestBinaries\\Audio\\AudioOutputComp";
}
#endif

size_t GetWaveSampleSize(const std::string &SourceFile)
{
    std::unique_ptr<nnt::audio::util::WaveFile> waveFile(new nnt::audio::util::WaveFile());
    if (!waveFile->Open(SourceFile.c_str()))
    {
        return 0;
    }
    waveFile->Close();
    return waveFile->GetSampleDataSize();
}

size_t LoadSourceSamples(const std::string &SourceFile, void* data, size_t size, int* sampleRate)
{
    std::unique_ptr<nnt::audio::util::WaveFile> waveFile(new nnt::audio::util::WaveFile());
    auto isOpenSucceeded = waveFile->Open(SourceFile.c_str());
    NN_ASSERT(isOpenSucceeded);
    if (!isOpenSucceeded)
    {
        return 0;
    }
    size_t readSize = waveFile->Read(static_cast<int8_t*>(data), size);
    NN_ASSERT(readSize > 0);
    *sampleRate = waveFile->GetSampleRate();
    waveFile->Close();
    return readSize;
}

void SimpleVerify(const std::string &TargetFile, const std::string &GoldenFile, int diff)
{
    // TODO: In future, to reduce test execution time, we should verify them much simpler way. For example, hash comparison.
    std::unique_ptr<nnt::audio::util::WaveFile> A(new nnt::audio::util::WaveFile());
    std::unique_ptr<nnt::audio::util::WaveFile> B(new nnt::audio::util::WaveFile());
    std::unique_ptr<nnt::audio::util::Checker> checker(new nnt::audio::util::Checker());
    nnt::audio::util::Checker::WaveFormCheckError checkResult = nnt::audio::util::Checker::WaveFormCheckError_NoError;
    int32_t sts = 0;

    A->Open(TargetFile.c_str());
    A->DumpInfo();
    B->Open(GoldenFile.c_str());
    checkResult = checker->SimpleCheck(A.get(), B.get(), &sts, diff);

    EXPECT_EQ(checkResult, nnt::audio::util::Checker::WaveFormCheckError_NoError) << nnt::audio::util::Checker::ResultString(checkResult, sts).c_str();

    A->Close();
    B->Close();
}

const char* GetBuildType()
{
#if defined(NN_SDK_BUILD_DEBUG)
    static const char BuildType[] = "Debug";
#elif defined(NN_SDK_BUILD_DEVELOP)
    static const char BuildType[] = "Develop";
#elif defined(NN_SDK_BUILD_RELEASE)
    static const char BuildType[] = "Release";
#else
#error "Build target is not defined."
#endif
    return BuildType;
}

const char* GetTargetName()
{
    static const char TargetName[] = NNT_AUDIO_TARGETNAME;
    return TargetName;
}

std::string GetResultDataDirectory()
{
    std::string result = g_ResultDataFolder;
    result += "/";
    result += nnt::audio::util::GetTargetName();
    result += "/";
    result += nnt::audio::util::GetBuildType();
    return result;
}

NN_ALIGNAS(4096) char g_ScopedAudioRendererWorkBuffer[1024 * 1024 * 32];  // 4096 == nn::os::MemoryPageSize

NN_IMPLICIT ScopedAudioRenderer::ScopedAudioRenderer(const nn::audio::AudioRendererParameter& parameter /*= g_DefaultParameter*/, nn::os::SystemEvent* pSystemEvent /*= nullptr*/) NN_NOEXCEPT
{
    m_Allocator.Initialize(g_ScopedAudioRendererWorkBuffer, sizeof(g_ScopedAudioRendererWorkBuffer));

    size_t workBufferSize = nn::audio::GetAudioRendererWorkBufferSize(parameter);
    m_WorkBuffer = m_Allocator.Allocate(workBufferSize, nn::os::MemoryPageSize);

    if (pSystemEvent)
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenAudioRenderer(&m_Handle, pSystemEvent, parameter, m_WorkBuffer, workBufferSize));
    }
    else
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenAudioRenderer(&m_Handle, parameter, m_WorkBuffer, workBufferSize));
    }

    size_t configBufferSize = nn::audio::GetAudioRendererConfigWorkBufferSize(parameter);
    m_ConfigBuffer = m_Allocator.Allocate(configBufferSize);
    nn::audio::InitializeAudioRendererConfig(&m_Config, parameter, m_ConfigBuffer, configBufferSize);
}

ScopedAudioRenderer::~ScopedAudioRenderer() NN_NOEXCEPT
{
    if (nn::audio::GetAudioRendererState(m_Handle) == nn::audio::AudioRendererState_Started)
    {
        nn::audio::StopAudioRenderer(m_Handle);
    }
    nn::audio::CloseAudioRenderer(m_Handle);

    m_Allocator.Free(m_ConfigBuffer);
    m_Allocator.Free(m_WorkBuffer);
    m_Allocator.Finalize();
}

void ScopedAudioRenderer::Start() NN_NOEXCEPT
{
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::StartAudioRenderer(m_Handle));
}

nn::audio::AudioRendererHandle ScopedAudioRenderer::GetHandle() const NN_NOEXCEPT
{
    return m_Handle;
}
nn::audio::AudioRendererConfig& ScopedAudioRenderer::GetConfig() NN_NOEXCEPT
{
    return m_Config;
}

void ScopedAudioRenderer::Update() NN_NOEXCEPT
{
    NNT_ASSERT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(m_Handle, &m_Config));
}

nn::audio::AudioRendererParameter GetAudioRendererParameterForWaveComparison() NN_NOEXCEPT
{
    nn::audio::AudioRendererParameter parameter;
    nn::audio::InitializeAudioRendererParameter(&parameter);
    parameter.sampleRate = 48000;
    parameter.sampleCount = 240;
    parameter.mixBufferCount = 12;
    parameter.subMixCount = 1;
    parameter.voiceCount = 24;
    parameter.sinkCount = 1;
    parameter.effectCount= 4;
    return parameter;
}

nn::audio::AudioRendererParameter GetAudioRendererParameterForDefault() NN_NOEXCEPT
{
    nn::audio::AudioRendererParameter parameter;
    nn::audio::InitializeAudioRendererParameter(&parameter);
    parameter.sampleRate = 48000;
    parameter.sampleCount = 240;
    parameter.mixBufferCount = 6;
    parameter.subMixCount = 1;
    parameter.voiceCount = 24;
    parameter.sinkCount = 1;
    parameter.effectCount= 4;
    return parameter;
}

} // namespace nnt
} // namespace audio
} // namespace util
