﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/atkUtil/testAtk_Util.h>
#include <nnt/nnt_Argument.h>

#include <nn/atk.h>
#include <nn/mem.h>
#include <nn/nn_Abort.h>
#include <nn/util/util_FormatString.h>
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
#include <cstdlib>
#endif
#include <vector>

#include <nn/atk/atk_DriverCommand.h>

namespace nnt{ namespace atk { namespace util {

namespace {
    static const size_t MountRomCacheBufferSize = 4 * 1024;
    char g_MountRomCacheBuffer[MountRomCacheBufferSize];

    bool GetRootPath(char* path, size_t pathSize)
    {
        char* applicationPath = nnt::GetHostArgv()[ 0 ];
        strncpy(path, applicationPath, pathSize);

        // cut off path str at root folder.
        const char* SearchTargets[] =
                {
                    "Tests\\Outputs",
                    "Tests/Outputs",
                    "Externals\\TestBinaries\\",
                    "Externals/TestBinaries/",
                };
        char* tmp = nullptr;
        for (auto& target : SearchTargets)
        {
            tmp = std::strstr(path, target);
            if (tmp != nullptr)
            {
                break;
            }
        }
        if (tmp == nullptr)
        {
            return false;
        }

        *tmp = 0; // null terminate at end of Root directory
        return true;
    }

    size_t FindDeliminator(std::string string, size_t pos)
    {
        auto result = string.find(":/", pos);
        if(result == std::string::npos)
        {
            result = string.find(":\\", pos);
        }
        return result;
    }

    size_t FindSeparator(std::string string, size_t pos)
    {
        auto result = string.find("/", pos);
        if(result == std::string::npos)
        {
            result = string.find("\\", pos);
        }
        return result;
    }
}

////////////////////////////////////////////////////////////////////////////////////
// Path Utility
////////////////////////////////////////////////////////////////////////////////////
bool GetTargetName(char *path, size_t pathSize) NN_NOEXCEPT
{
    if(pathSize - 1 < strlen(NNT_ATK_TARGETNAME))
    {
        return false;
    }

    nn::util::SNPrintf(path, pathSize, "%s", NNT_ATK_TARGETNAME);
    return true;
}

const char* const GetTargetName() NN_NOEXCEPT
{
    static const char* const targetName = NNT_ATK_TARGETNAME;
    return targetName;
}

bool GetContentsDirectoryPath(char *path, size_t pathSize) NN_NOEXCEPT
{
    // Contents ディレクトリを取得
    // アプリケーションバイナリの配置されているディレクトリの隣に、Content ディレクトリがある
    // + 1 は終端文字の分
    char contentsDirectory[nn::fs::EntryNameLengthMax + 1];
    char* pPath = nnt::GetHostArgv()[ 0 ];
    char* pEnd = strrchr( pPath, '\\' );
    NN_ABORT_UNLESS_NOT_NULL( pEnd );
    ptrdiff_t length = pEnd - pPath + 1;
    strncpy( contentsDirectory, pPath, length );
    strcpy( contentsDirectory + length, "..\\Contents\\" );

    if(pathSize - 1 < strlen(contentsDirectory))
    {
        return false;
    }

    nn::util::SNPrintf(path, pathSize, "%s", contentsDirectory);
    return true;
}

bool GetCommonSoundArchivePath(char *path, size_t pathSize, const char *mountName) NN_NOEXCEPT
{
    const char archivePath[] = "common.bfsar";

    if(pathSize < strlen(mountName) + strlen(":/") + strlen(archivePath))
    {
        return false;
    }

    nn::util::SNPrintf(path, pathSize, "%s:/%s", mountName, archivePath);
    return true;
}

bool GetTestResultsDirectoryPath(char *path, size_t pathSize) NN_NOEXCEPT
{
    // Externals/TestBinaries/Atk ディレクトリを取得
    // アプリケーションバイナリの配置されているディレクトリから参照する
    // + 1 は終端文字の分
    char contentsDirectory[nn::fs::EntryNameLengthMax + 1];
    if (GetRootPath(contentsDirectory, nn::fs::EntryNameLengthMax) == false)
    {
        return false;
    }
    nn::util::SNPrintf(path, pathSize, "%s%s", contentsDirectory, "Externals\\TestBinaries\\Atk\\TestResults\\");
    return true;
}

bool GetAtkTestBinariesDirectoryPath(char *path, size_t pathSize) NN_NOEXCEPT
{
    // Externals/TestBinaries/Atk ディレクトリを取得
    // アプリケーションバイナリの配置されているディレクトリから参照する
    // + 1 は終端文字の分
    char contentsDirectory[nn::fs::EntryNameLengthMax + 1];
    if (GetRootPath(contentsDirectory, nn::fs::EntryNameLengthMax) == false)
    {
        return false;
    }
    nn::util::SNPrintf(path, pathSize, "%s%s", contentsDirectory, "Externals\\TestBinaries\\Atk\\");
    return true;
}

////////////////////////////////////////////////////////////////////////////////////
// Fs Setup Utility
////////////////////////////////////////////////////////////////////////////////////
void UnmountFs(const char* mountName) NN_NOEXCEPT
{
    nn::fs::Unmount(mountName);
}


////////////////////////////////////////////////////////////////////////////////////
// HostPcFs Setup Utility
////////////////////////////////////////////////////////////////////////////////////
void MountHostPcFs(const char* mountName, const char* mountPath) NN_NOEXCEPT
{
    nn::Result result = nn::fs::MountHost(mountName, mountPath);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
}

////////////////////////////////////////////////////////////////////////////////////
// RomFs Setup Utility
////////////////////////////////////////////////////////////////////////////////////
void MountRomFs(const char* mountName) NN_NOEXCEPT
{
    size_t mountRomCacheUseSize = 0;
    nn::Result result = nn::fs::QueryMountRomCacheSize(&mountRomCacheUseSize);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    NN_ABORT_UNLESS_LESS_EQUAL(mountRomCacheUseSize, MountRomCacheBufferSize);

    result = nn::fs::MountRom(mountName, g_MountRomCacheBuffer, MountRomCacheBufferSize);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
}

////////////////////////////////////////////////////////////////////////////////////
// File Operation Utility
////////////////////////////////////////////////////////////////////////////////////

bool GetFileDirectory(char* directoryName, const size_t directoryNameSize, const char* filePath) NN_NOEXCEPT
{
    int pathCount = 0;
    int nameCount = 0;

    while(filePath[pathCount] != '\0')
    {
        if(filePath[pathCount] == '\\' || filePath[pathCount] == '/')
        {
            nameCount = pathCount;
        }
        ++pathCount;
    }

    if(nameCount == 0 || static_cast<size_t>(nameCount + 1) >= directoryNameSize)
    {
        return false;
    }
    nameCount++;
    memcpy(directoryName, filePath, nameCount);
    directoryName[nameCount] = '\0';
    return true;
}

bool GetFileName(char* fileName, const size_t fileNameSize, const char* filePath) NN_NOEXCEPT
{
    const char* pFileName = filePath;
    int nameCount = 0;
    for(const char* pFile = filePath; *pFile != '\0'; ++pFile)
    {
        if(*pFile == '\\' || *pFile == '/')
        {
            nameCount = 0;
            while(*pFile == '\\' || *pFile == '/')
            {
                ++pFile;
            }

            if(*pFile == '\0')
            {
                break;
            }
            pFileName = pFile;
            ++nameCount;
            continue;
        }
        else
        {
            if(static_cast<size_t>(nameCount) >= fileNameSize)
            {
                return false;
            }
        }

        ++nameCount;
    }

    if(nameCount <= 0)
    {
        return false;
    }

    memcpy(fileName, pFileName, nameCount);
    fileName[nameCount] = '\0';
    return true;
}

void CreateDirectory(const char* path) NN_NOEXCEPT
{
    nn::fs::DirectoryEntryType entryType;
    std::string pathString(path);
    std::vector<std::string> directoryList = {};
    size_t findPos = 0;

    findPos = FindDeliminator(pathString, findPos);
    NN_ABORT_UNLESS(findPos != std::string::npos);
    // separator を探すために、検索開始位置を 2 文字分 (":\\" or ":/") 進める
    findPos += 2;

    for(;;)
    {
        findPos = FindSeparator(pathString, findPos);
        if(findPos == std::string::npos)
        {
            break;
        }
        std::string directoryPath = pathString.substr(0, findPos);
        directoryList.push_back(directoryPath);
        // 次の separator を探すために、検索開始位置を 1 文字分 ("\\" or "/") 進める
        ++findPos;
    }
    directoryList.push_back(pathString);

    for( auto directoryPath : directoryList )
    {
        nn::Result result = nn::fs::GetEntryType(&entryType, directoryPath.c_str());
        if(nn::fs::ResultDataCorrupted::Includes(result))
        {
            NN_ABORT("Entry is corrrupted.");
        }

        if(nn::fs::ResultPathNotFound::Includes(result))
        {
            result = nn::fs::CreateDirectory(directoryPath.c_str());
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        }
    }
}

void CreateNewFile(const char* path, size_t size) NN_NOEXCEPT
{
    nn::fs::DirectoryEntryType entryType;
    nn::Result result = nn::fs::GetEntryType(&entryType, path);
    if(result.IsSuccess())
    {
        result = nn::fs::DeleteFile(path);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }
    else if(!nn::fs::ResultPathNotFound::Includes(result))
    {
        NN_ABORT("Unexpected error.");
    }

    nn::fs::CreateFile(path, size);
}

void CopyFile(const char* sourcePath, const char* destinationPath) NN_NOEXCEPT
{
    // ファイルの読み込み
    nn::fs::FileHandle readHandle;
    nn::Result result = nn::fs::OpenFile(&readHandle, sourcePath, nn::fs::OpenMode_Read);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    int64_t fileSize = 0;
    nn::fs::GetFileSize(&fileSize, readHandle);
    void* buffer = malloc(static_cast<size_t>(fileSize));
    result = nn::fs::ReadFile(readHandle, 0, buffer, static_cast<size_t>(fileSize));
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    nn::fs::CloseFile(readHandle);

    // + 1 は終端文字の分
    char destinationDirectory[nn::fs::EntryNameLengthMax + 1];
    GetFileDirectory(destinationDirectory, sizeof(destinationDirectory), destinationPath);

    CreateDirectory(destinationDirectory);
    CreateNewFile(destinationPath, static_cast<size_t>(fileSize));

    nn::fs::FileHandle writeHandle;
    result = nn::fs::OpenFile(&writeHandle, destinationPath, nn::fs::OpenMode_Write);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    result = nn::fs::WriteFile(writeHandle, 0, buffer, static_cast<size_t>(fileSize), nn::fs::WriteOption());
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    result = nn::fs::FlushFile(writeHandle);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    nn::fs::CloseFile(writeHandle);

    free(buffer);
}

////////////////////////////////////////////////////////////////////////////////////
// Command Utility
////////////////////////////////////////////////////////////////////////////////////

bool SendDummyCommand(uint32_t* tag) NN_NOEXCEPT
{
    nn::atk::detail::DriverCommand& cmdmgr = nn::atk::detail::DriverCommand::GetInstance();

    nn::atk::detail::Command* command =
        cmdmgr.AllocCommand<nn::atk::detail::Command>();
#ifdef NNT_ATK_ENABLE_VOICE_COMMAND
    if (command == nullptr)
    {
        return false; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
    }
#else
    NN_SDK_ASSERT_NOT_NULL(command);
#endif
    command->id = nn::atk::detail::DriverCommandId_Dummy;
    *tag = cmdmgr.PushCommand(command);

    return true;
}

bool IsCommandFinished(uint32_t tag) NN_NOEXCEPT
{
    nn::atk::detail::DriverCommand& cmdmgr = nn::atk::detail::DriverCommand::GetInstance();
    return cmdmgr.IsFinishCommand(tag);
}

bool SendReplyCommand(bool* pIsCommandProcessed) NN_NOEXCEPT
{
    nn::atk::detail::DriverCommand& cmdmgr = nn::atk::detail::DriverCommand::GetInstance();

    nn::atk::detail::DriverCommandReply* command =
        cmdmgr.AllocCommand<nn::atk::detail::DriverCommandReply>();
#ifdef NNT_ATK_ENABLE_VOICE_COMMAND
    if (command == nullptr)
    {
        return false; // VoiceCommand 時は AllocCommand に失敗すると nullptr が返る
    }
#else
    NN_SDK_ASSERT_NOT_NULL(command);
#endif
    command->id = nn::atk::detail::DriverCommandId_Reply;
    command->ptr = pIsCommandProcessed;
    uint32_t tag = 0;
    tag = cmdmgr.PushCommand(command);
    // 無効値が来た場合には false を返す
    if (tag == 0)
    {
        return false;
    }

    return true;
}

void FlushCommand(bool isForceFlush) NN_NOEXCEPT
{
    nn::atk::detail::DriverCommand& cmdmgr = nn::atk::detail::DriverCommand::GetInstance();
    cmdmgr.RecvCommandReply();
    cmdmgr.FlushCommand(isForceFlush);
}

uint32_t GetVoiceCommandListCount() NN_NOEXCEPT
{
    return nn::atk::detail::LowLevelVoiceCommand::GetInstance().GetCommandListCount();
}

void* AllocateUninitializedMemory(nn::mem::StandardAllocator& allocator, size_t size)
{
    void* buffer = allocator.Allocate(size);
    NN_ABORT_UNLESS_NOT_NULL(buffer);
    return memset(buffer, 0xCD, size);
}

void* AllocateUninitializedMemory(nn::mem::StandardAllocator& allocator, size_t size, size_t alignment)
{
    void* buffer = allocator.Allocate(size, alignment);
    NN_ABORT_UNLESS_NOT_NULL(buffer);
    return memset(buffer, 0xCD, size);
}

void WaitForProcessCommand() NN_NOEXCEPT
{
    // FlushCommand でダミーコマンドを強制フラッシュし、コマンドが処理されるまで待つ
    nn::atk::detail::DriverCommand& cmdmgr = nn::atk::detail::DriverCommand::GetInstance();
    uint32_t tag = cmdmgr.FlushCommand(true);
    cmdmgr.WaitCommandReply(tag);
}

void UpdateAndWait(nn::atk::SoundArchivePlayer& archivePlayer, const nn::TimeSpan& waitTime) NN_NOEXCEPT
{
    archivePlayer.Update();
#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
    nn::atk::SoundSystem::VoiceCommandProcess(4);
#endif
    nn::os::SleepThread(waitTime);
}

void UpdateAndWait(nn::atk::SoundArchivePlayer& archivePlayer) NN_NOEXCEPT
{
    const nn::TimeSpan WaitTime = nn::TimeSpan::FromMilliSeconds(16);
    UpdateAndWait(archivePlayer, WaitTime);
}

void OnPreAtkTest() NN_NOEXCEPT
{
    PreAtkTestArg arg;
    OnPreAtkTest(arg);
}

void OnPreAtkTest(PreAtkTestArg& arg) NN_NOEXCEPT
{
    nn::audio::SetAudioRendererExclusiveControlLeakageCheckEnabled(arg.isResourceExclusionCheckEnabled);
}

}}}
