﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/atkUtil/testAtk_FileLogger.h>
#include <nnt/atkUtil/testAtk_Util.h>

#include <nn/mem.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_FormatString.h>

#include <mutex>
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
#include <cstdlib>
#endif

namespace nnt{ namespace atk { namespace util {

FileLogger::FileLogger() NN_NOEXCEPT
    : m_Mutex(false, 0)
    , m_IsOpened(false)
{
    strncpy(m_LogBuffer, "", sizeof(m_LogBuffer) - 1);
}

FileLogger::FileLogger(const char* filePath) NN_NOEXCEPT
    : m_Mutex(false, 0)
    , m_IsOpened(false)
{
    strncpy(m_LogBuffer, "", sizeof(m_LogBuffer) - 1);
    m_IsOpened = Open(filePath, OpenOption_OverWrite).IsSuccess();
}

FileLogger::FileLogger(const char* filePath, OpenOption option) NN_NOEXCEPT
    : m_Mutex(false, 0)
    , m_IsOpened(false)
{
    strncpy(m_LogBuffer, "", sizeof(m_LogBuffer) - 1);
    m_IsOpened = Open(filePath, option).IsSuccess();
}

FileLogger::~FileLogger() NN_NOEXCEPT
{
    FileLogger::Close();
}

nn::Result FileLogger::Open(const char* filePath, OpenOption option) NN_NOEXCEPT
{
    NN_ASSERT(!m_IsOpened, "File is already opened.\n");

    nn::Result result;
    switch(option)
    {
    case OpenOption_OverWrite:
        // 該当のファイルが存在した場合、削除を試みたうえでファイルを作成しオープンする
        // 該当のファイルが存在しない場合、そのままファイルを作成しオープンする
        result = nn::fs::DeleteFile(filePath);
        if(result.IsFailure() && !nn::fs::ResultPathNotFound::Includes(result))
        {
            return result;
        }
        break;
    case OpenOption_Append:
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    // + 1 は終端文字の分
    char fileDirectory[nn::fs::EntryNameLengthMax + 1];
    GetFileDirectory(fileDirectory, sizeof(fileDirectory), filePath);
    CreateDirectory(fileDirectory);

    // 該当のファイルが存在しないばあい、ファイル作成を試みたうえでオープンする (OpenOption_OverWrite の場合こちらの条件となる)
    // 該当のファイルが存在したばあい、そのままオープンする
    result = nn::fs::CreateFile(filePath, 0);
    if(result.IsFailure() && !nn::fs::ResultPathAlreadyExists::Includes(result))
    {
        return result;
    }

    NN_RESULT_DO(nn::fs::OpenFile( &m_Handle, filePath, nn::fs::OpenMode::OpenMode_AllowAppend | nn::fs::OpenMode::OpenMode_Write));
    m_IsOpened = true;
    strncpy(m_LogBuffer, "", sizeof(m_LogBuffer) - 1);

    NN_RESULT_SUCCESS;
}

nn::Result FileLogger::Write(const char* pFormat, ...) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    NN_ASSERT(m_IsOpened, "File is not opened.\n");

    NN_ABORT_UNLESS_NOT_NULL(pFormat);

    std::va_list vaList;
    va_start(vaList, pFormat);
    nn::util::VSNPrintf(m_LogBuffer + strlen(m_LogBuffer), sizeof(m_LogBuffer) - strlen(m_LogBuffer), pFormat, vaList);
    va_end(vaList);

    nn::Result result;
    if(strlen(m_LogBuffer) >= WriteThresholdSize)
    {
        int64_t fileSize;
        NN_RESULT_DO(nn::fs::GetFileSize(&fileSize, m_Handle));

        result = nn::fs::WriteFile(m_Handle, fileSize, reinterpret_cast<const void*>(m_LogBuffer), strlen(m_LogBuffer), nn::fs::WriteOption());
        strncpy(m_LogBuffer, "", sizeof(m_LogBuffer) - 1);
    }
    else
    {
        return nn::ResultSuccess();
    }

    return result;
}

nn::Result FileLogger::Flush() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    NN_ASSERT(m_IsOpened, "File is not opened.\n");

    int64_t fileSize;
    NN_RESULT_DO(nn::fs::GetFileSize(&fileSize, m_Handle));
    NN_RESULT_DO(nn::fs::WriteFile(m_Handle, fileSize, reinterpret_cast<const void*>(m_LogBuffer), strlen(m_LogBuffer), nn::fs::WriteOption()));
    strncpy(m_LogBuffer, "", sizeof(m_LogBuffer) - 1);

    return nn::fs::FlushFile(m_Handle);
}

void FileLogger::Close() NN_NOEXCEPT
{
    if(m_IsOpened)
    {
        Flush();
        nn::fs::CloseFile(m_Handle);
        m_IsOpened = false;
    }
}

bool FileLogger::IsOpened() const NN_NOEXCEPT
{
    return m_IsOpened;
}

}}}
