﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>

// テスト対象
#include <nn/sf/sf_ObjectFactory.h>

// SF インターフェイス
#include "testSf_IInheritance.h"

// 実装
#include <nn/nn_Common.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/impl/sf_ExpHeapAllocator.h>
#include <memory>

namespace {

class MyStatelessAllocatorTag
{
};

typedef nn::sf::impl::ExpHeapStaticAllocator<16 * 1024 * 1024, MyStatelessAllocatorTag> MyStatelessAllocator;
typedef MyStatelessAllocator::Policy MyStatelessAllocationPolicy;

struct MyStatelessAllocatorInitializer
{
    MyStatelessAllocatorInitializer() NN_NOEXCEPT
    {
        MyStatelessAllocator::Initialize(nn::lmem::CreationOption_NoOption);
    }
} g_MyStatelessAllocatorInitializer;

class InheritanceBase1Impl
{
public:
    int Get1() NN_NOEXCEPT
    {
        return 1;
    }
};
class InheritanceBase2Impl
{
private:
    static const int N = 2;
public:
    int Get1() NN_NOEXCEPT
    {
        return 1 * N;
    }
    int Get2() NN_NOEXCEPT
    {
        return 2 * N;
    }
};
class InheritanceImpl
{
private:
    static const int N = 3;
public:
    int Get1() NN_NOEXCEPT
    {
        return 1 * N;
    }
    int Get2() NN_NOEXCEPT
    {
        return 2 * N;
    }
    int Get3() NN_NOEXCEPT
    {
        return 3 * N;
    }
};

}

TEST(sf_AllInOne, Inheritance_UnmanagedServiceObject)
{
    {
        nn::sf::UnmanagedServiceObject<nnt::testsf::IInheritanceBase1, InheritanceBase1Impl> as1;
        ASSERT_EQ(1, as1.Get1());

        // Downcast
        nn::sf::SharedPointer<nnt::testsf::IInheritanceBase2> p2(
            static_cast<nnt::testsf::IInheritanceBase2*>(as1.GetShared().Get()), false);
        ASSERT_EQ(1, p2->Get1());
        nn::sf::SharedPointer<nnt::testsf::IInheritance> p3(
            static_cast<nnt::testsf::IInheritance*>(as1.GetShared().Get()), false);
        ASSERT_EQ(1, p3->Get1());
    }
    {
        nn::sf::UnmanagedServiceObject<nnt::testsf::IInheritanceBase1, InheritanceBase2Impl> as1;
        nn::sf::UnmanagedServiceObject<nnt::testsf::IInheritanceBase2, InheritanceBase2Impl> as2;
        ASSERT_EQ(2, as1.Get1());
        ASSERT_EQ(2, as2.Get1());
        ASSERT_EQ(4, as2.Get2());

        // Upcast
        nn::sf::SharedPointer<nnt::testsf::IInheritanceBase1> p1 = as2.GetShared();
        ASSERT_EQ(2, p1->Get1());

        // Downcast
        nn::sf::SharedPointer<nnt::testsf::IInheritance> p3(
            static_cast<nnt::testsf::IInheritance*>(as2.GetShared().Get()), false);
        ASSERT_EQ(2, p3->Get1());
        ASSERT_EQ(4, p3->Get2());
    }
    {
        nn::sf::UnmanagedServiceObject<nnt::testsf::IInheritanceBase1, InheritanceImpl> as1;
        nn::sf::UnmanagedServiceObject<nnt::testsf::IInheritanceBase2, InheritanceImpl> as2;
        nn::sf::UnmanagedServiceObject<nnt::testsf::IInheritance, InheritanceImpl> as3;
        ASSERT_EQ(3, as1.Get1());
        ASSERT_EQ(3, as2.Get1());
        ASSERT_EQ(6, as2.Get2());
        ASSERT_EQ(3, as3.Get1());
        ASSERT_EQ(6, as3.Get2());
        ASSERT_EQ(9, as3.Get3());

        // Upcast
        nn::sf::SharedPointer<nnt::testsf::IInheritanceBase1> p1 = as3.GetShared();
        nn::sf::SharedPointer<nnt::testsf::IInheritanceBase2> p2 = as3.GetShared();
        ASSERT_EQ(3, p1->Get1());
        ASSERT_EQ(3, p2->Get1());
        ASSERT_EQ(6, p2->Get2());
    }
}

TEST(sf_AllInOne, Inheritance_SharedPointer)
{
    typedef nn::sf::ObjectFactory<MyStatelessAllocationPolicy> Factory;
    {
        auto p1 = Factory::CreateSharedEmplaced<nnt::testsf::IInheritanceBase1, InheritanceBase1Impl>();
        ASSERT_EQ(1, p1->Get1());

        // Downcast
        nn::sf::SharedPointer<nnt::testsf::IInheritanceBase2> p2(
            static_cast<nnt::testsf::IInheritanceBase2*>(p1.Get()), true);
        ASSERT_EQ(1, p2->Get1());
        nn::sf::SharedPointer<nnt::testsf::IInheritance> p3(
            static_cast<nnt::testsf::IInheritance*>(p1.Get()), true);
        ASSERT_EQ(1, p3->Get1());
    }
    {
        auto p2 = Factory::CreateSharedEmplaced<nnt::testsf::IInheritanceBase2, InheritanceBase2Impl>();
        ASSERT_EQ(2, p2->Get1());
        ASSERT_EQ(4, p2->Get2());

        // Upcast
        nn::sf::SharedPointer<nnt::testsf::IInheritanceBase1> p1 = p2;
        ASSERT_EQ(2, p1->Get1());

        // Downcast
        nn::sf::SharedPointer<nnt::testsf::IInheritance> p3(
            static_cast<nnt::testsf::IInheritance*>(p2.Get()), true);
        ASSERT_EQ(2, p3->Get1());
        ASSERT_EQ(4, p3->Get2());
    }
    {
        auto p3 = Factory::CreateSharedEmplaced<nnt::testsf::IInheritance, InheritanceImpl>();
        ASSERT_EQ(3, p3->Get1());
        ASSERT_EQ(6, p3->Get2());
        ASSERT_EQ(9, p3->Get3());

        // Upcast
        nn::sf::SharedPointer<nnt::testsf::IInheritanceBase2> p2 = p3;
        ASSERT_EQ(3, p2->Get1());
        ASSERT_EQ(6, p2->Get2());
        nn::sf::SharedPointer<nnt::testsf::IInheritanceBase1> p1 = p3;
        ASSERT_EQ(3, p1->Get1());
    }
}
