﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nn/settings/system/settings_Hid.h>

namespace
{

//!< テスト用コントローラ情報 A
const ::nn::settings::system::NxControllerSettings
    NxControllerSettingsA =
{
    {
        {
            static_cast<uint8_t>(1),
            static_cast<uint8_t>(2),
            static_cast<uint8_t>(3),
            static_cast<uint8_t>(4),
            static_cast<uint8_t>(5),
            static_cast<uint8_t>(6)
        }
    },
    static_cast<uint8_t>(10),
    { 'S', 'e', 't', 't', 'i', 'n', 'g', 's', 'A' },
    {
        {
            static_cast<uint8_t>(1),
            static_cast<uint8_t>(2),
            static_cast<uint8_t>(3),
            static_cast<uint8_t>(4)
        }
    }
};

//!< テスト用コントローラ情報 B
const ::nn::settings::system::NxControllerSettings
    NxControllerSettingsB =
{
    {
        {
            static_cast<uint8_t>(7),
            static_cast<uint8_t>(8),
            static_cast<uint8_t>(9),
            static_cast<uint8_t>(10),
            static_cast<uint8_t>(11),
            static_cast<uint8_t>(12)
        }
    },
    static_cast<uint8_t>(11),
    { 'S', 'e', 't', 't', 'i', 'n', 'g', 's', 'B' },
    {
        {
            static_cast<uint8_t>(5),
            static_cast<uint8_t>(6),
            static_cast<uint8_t>(7),
            static_cast<uint8_t>(8)
        }
    }
};

::nn::settings::system::NxControllerSettings g_NxControllerSettings[
    ::nn::settings::system::NxControllerSettingsCountMax];

//!< コントローラ情報が等しいことを期待します。
void ExpectEqualNxControllerSettings(
    const ::nn::settings::system::NxControllerSettings& lhs,
    const ::nn::settings::system::NxControllerSettings& rhs) NN_NOEXCEPT;

} // namespace

//!< Pro Controller 有線 USB 通信機能が有効か否かを表す値を変更可能か
TEST(HidSuite, UsbFullKeyEnableFlagTest1)
{
    bool value = ::nn::settings::system::IsUsbFullKeyEnabled();

    ::nn::settings::system::SetUsbFullKeyEnabled(true);

    EXPECT_TRUE(::nn::settings::system::IsUsbFullKeyEnabled());

    ::nn::settings::system::SetUsbFullKeyEnabled(false);

    EXPECT_FALSE(::nn::settings::system::IsUsbFullKeyEnabled());

    ::nn::settings::system::SetUsbFullKeyEnabled(true);

    EXPECT_TRUE(::nn::settings::system::IsUsbFullKeyEnabled());

    ::nn::settings::system::SetUsbFullKeyEnabled(value);
}

//!< 振動のマスターボリュームを取得可能か
TEST(HidSuite, VibrationMasterVolumeTest1)
{
    ::nn::settings::system::GetVibrationMasterVolume();
}

//!< 振動のマスターボリュームを変更可能か
TEST(HidSuite, VibrationMasterVolumeTest2)
{
    float value = ::nn::settings::system::GetVibrationMasterVolume();

    ::nn::settings::system::SetVibrationMasterVolume(0.1f);

    EXPECT_FLOAT_EQ(0.1f, ::nn::settings::system::GetVibrationMasterVolume());

    ::nn::settings::system::SetVibrationMasterVolume(0.9f);

    EXPECT_FLOAT_EQ(0.9f, ::nn::settings::system::GetVibrationMasterVolume());

    ::nn::settings::system::SetVibrationMasterVolume(value);
}

//!< コントローラ情報を変更可能か
TEST(HidSuite, NxControllerSettingsTest1)
{
    ::nn::settings::system::SetNxControllerSettings(
        &NxControllerSettingsA, 1);

    ::nn::settings::system::NxControllerSettings* settings =
        &g_NxControllerSettings[0];

    const auto countMax =
        ::nn::settings::system::NxControllerSettingsCountMax;

    EXPECT_EQ(
        1,
        ::nn::settings::system::GetNxControllerSettings(settings, countMax));

    {
        SCOPED_TRACE("");
        ExpectEqualNxControllerSettings(NxControllerSettingsA,
                                            settings[0]);
    }

    for (size_t i = 0; i < countMax; ++i)
    {
        settings[i] = (i % 2 == 1) ? NxControllerSettingsA
                                   : NxControllerSettingsB;
    }

    ::nn::settings::system::SetNxControllerSettings(settings, countMax);

    for (size_t i = 0; i < countMax; ++i)
    {
        settings[i] = ::nn::settings::system::NxControllerSettings();
    }

    EXPECT_EQ(
        countMax,
        ::nn::settings::system::GetNxControllerSettings(settings, countMax));

    for (size_t i = 0; i < countMax; ++i)
    {
        SCOPED_TRACE("");
        ExpectEqualNxControllerSettings(
            (i % 2 == 1) ? NxControllerSettingsA
                         : NxControllerSettingsB, settings[i]);
    }

    ::nn::settings::system::SetNxControllerSettings(nullptr, 0);

    EXPECT_EQ(
        0,
        ::nn::settings::system::GetNxControllerSettings(settings, countMax));
}

namespace {

void ExpectEqualNxControllerSettings(
    const ::nn::settings::system::NxControllerSettings& lhs,
    const ::nn::settings::system::NxControllerSettings& rhs) NN_NOEXCEPT
{
    for (size_t i = 0; i < NN_ARRAY_SIZE(lhs.bd_addr.address); ++i)
    {
        EXPECT_EQ(lhs.bd_addr.address[i], rhs.bd_addr.address[i]);
    }

    EXPECT_EQ(lhs.device_type, rhs.device_type);

    for (size_t i = 0; i < NN_ARRAY_SIZE(lhs.identification_code); ++i)
    {
        EXPECT_EQ(lhs.identification_code[i], rhs.identification_code[i]);
    }

    for (size_t i = 0; i < NN_ARRAY_SIZE(lhs.mainColor.v); ++i)
    {
        EXPECT_EQ(lhs.mainColor.v[i], rhs.mainColor.v[i]);
    }

    for (size_t i = 0; i < NN_ARRAY_SIZE(lhs.subColor.v); ++i)
    {
        EXPECT_EQ(lhs.subColor.v[i], rhs.subColor.v[i]);
    }
}

} // namespace
