﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/settings/system/settings_BluetoothDevices.h>
#include <nnt/nntest.h>

namespace {

//!< テスト用 Bluetooth デバイス設定 A
const ::nn::settings::system::BluetoothDevicesSettings
    BluetoothDevicesSettingsA =
{
    {
        static_cast<uint8_t>(1),
        static_cast<uint8_t>(2),
        static_cast<uint8_t>(3),
        static_cast<uint8_t>(4),
        static_cast<uint8_t>(5),
        static_cast<uint8_t>(6)
    },
    { 'S', 'e', 't', 't', 'i', 'n', 'g', 's', 'A' },
    {
        static_cast<uint8_t>(1),
        static_cast<uint8_t>(2),
        static_cast<uint8_t>(3)
    },
    {
        static_cast<uint8_t>(1),
        static_cast<uint8_t>(2),
        static_cast<uint8_t>(3),
        static_cast<uint8_t>(4),
        static_cast<uint8_t>(5),
        static_cast<uint8_t>(6),
        static_cast<uint8_t>(7),
        static_cast<uint8_t>(8),
        static_cast<uint8_t>(9),
        static_cast<uint8_t>(10),
        static_cast<uint8_t>(11),
        static_cast<uint8_t>(12),
        static_cast<uint8_t>(13),
        static_cast<uint8_t>(14),
        static_cast<uint8_t>(15),
        static_cast<uint8_t>(16)
    },
    static_cast<uint8_t>(1),
    static_cast<uint16_t>(2),
    static_cast<uint32_t>(3),
    static_cast<uint16_t>(4),
    static_cast<uint16_t>(5),
    static_cast<uint8_t>(6),
    static_cast<uint8_t>(7),
    static_cast<uint16_t>(11),
    {
        static_cast<uint8_t>('D'),
        static_cast<uint8_t>('e'),
        static_cast<uint8_t>('s'),
        static_cast<uint8_t>('c'),
        static_cast<uint8_t>('r'),
        static_cast<uint8_t>('i'),
        static_cast<uint8_t>('p'),
        static_cast<uint8_t>('t'),
        static_cast<uint8_t>('o'),
        static_cast<uint8_t>('r'),
        static_cast<uint8_t>('A'),
    },
    static_cast<uint8_t>(9),
    static_cast<uint8_t>(10),
    static_cast<uint16_t>(9),
    {
        static_cast<uint8_t>(1),
        static_cast<uint8_t>(2),
        static_cast<uint8_t>(3),
        static_cast<uint8_t>(4),
        static_cast<uint8_t>(5),
        static_cast<uint8_t>(6),
        static_cast<uint8_t>(7),
        static_cast<uint8_t>(8),
        static_cast<uint8_t>(9)
    },
};

//!< テスト用 Bluetooth デバイス設定 B
const ::nn::settings::system::BluetoothDevicesSettings
    BluetoothDevicesSettingsB =
{
    {
        static_cast<uint8_t>(7),
        static_cast<uint8_t>(8),
        static_cast<uint8_t>(9),
        static_cast<uint8_t>(10),
        static_cast<uint8_t>(11),
        static_cast<uint8_t>(12)
    },
    { 'S', 'e', 't', 't', 'i', 'n', 'g', 's', 'B' },
    {
        static_cast<uint8_t>(4),
        static_cast<uint8_t>(5),
        static_cast<uint8_t>(6)
    },
    {
        static_cast<uint8_t>(17),
        static_cast<uint8_t>(18),
        static_cast<uint8_t>(19),
        static_cast<uint8_t>(20),
        static_cast<uint8_t>(21),
        static_cast<uint8_t>(22),
        static_cast<uint8_t>(23),
        static_cast<uint8_t>(24),
        static_cast<uint8_t>(25),
        static_cast<uint8_t>(26),
        static_cast<uint8_t>(27),
        static_cast<uint8_t>(28),
        static_cast<uint8_t>(29),
        static_cast<uint8_t>(30),
        static_cast<uint8_t>(31),
        static_cast<uint8_t>(32)
    },
    static_cast<uint8_t>(2),
    static_cast<uint16_t>(3),
    static_cast<uint32_t>(4),
    static_cast<uint16_t>(5),
    static_cast<uint16_t>(6),
    static_cast<uint8_t>(7),
    static_cast<uint8_t>(8),
    static_cast<uint16_t>(11),
    {
        static_cast<uint8_t>('D'),
        static_cast<uint8_t>('e'),
        static_cast<uint8_t>('s'),
        static_cast<uint8_t>('c'),
        static_cast<uint8_t>('r'),
        static_cast<uint8_t>('i'),
        static_cast<uint8_t>('p'),
        static_cast<uint8_t>('t'),
        static_cast<uint8_t>('o'),
        static_cast<uint8_t>('r'),
        static_cast<uint8_t>('B'),
    },
    static_cast<uint8_t>(10),
    static_cast<uint8_t>(11),
    static_cast<uint16_t>(9),
    {
        static_cast<uint8_t>(1),
        static_cast<uint8_t>(2),
        static_cast<uint8_t>(3),
        static_cast<uint8_t>(4),
        static_cast<uint8_t>(5),
        static_cast<uint8_t>(6),
        static_cast<uint8_t>(7),
        static_cast<uint8_t>(8),
        static_cast<uint8_t>(9)
    },
};

//!< テスト用 BLE ペアリング設定 A
const ::nn::settings::system::BlePairingSettings
    BlePairingSettingsA =
{
    {
        static_cast<uint8_t>(1),
        static_cast<uint8_t>(2),
        static_cast<uint8_t>(3),
        static_cast<uint8_t>(4),
        static_cast<uint8_t>(5),
        static_cast<uint8_t>(6)
    },
    static_cast<uint16_t>(1),
    {
        static_cast<uint8_t>(1),
        static_cast<uint8_t>(2),
        static_cast<uint8_t>(3)
    },
    {
        static_cast<uint8_t>(4),
        static_cast<uint8_t>(5),
        static_cast<uint8_t>(6)
    },
    static_cast<uint8_t>(2),
};

//!< テスト用 BLE ペアリング設定 B
const ::nn::settings::system::BlePairingSettings
    BlePairingSettingsB =
{
    {
        static_cast<uint8_t>(7),
        static_cast<uint8_t>(8),
        static_cast<uint8_t>(9),
        static_cast<uint8_t>(10),
        static_cast<uint8_t>(11),
        static_cast<uint8_t>(12)
    },
    static_cast<uint16_t>(2),
    {
        static_cast<uint8_t>(4),
        static_cast<uint8_t>(5),
        static_cast<uint8_t>(6)
    },
    {
        static_cast<uint8_t>(7),
        static_cast<uint8_t>(8),
        static_cast<uint8_t>(9)
    },
    static_cast<uint8_t>(3),
};

::nn::settings::system::BluetoothDevicesSettings g_BluetoothDevicesSettings[
    ::nn::settings::system::BluetoothDevicesSettingsCountMax];

::nn::settings::system::BlePairingSettings g_BlePairingSettings[
    ::nn::settings::system::BlePairingSettingsCountMax];

//!< Bluetooth デバイス設定が等しいことを期待します。
void ExpectEqualBluetoothDevicesSettings(
    const ::nn::settings::system::BluetoothDevicesSettings& lhs,
    const ::nn::settings::system::BluetoothDevicesSettings& rhs) NN_NOEXCEPT;

//!< BLE ペアリング設定が等しいことを期待します。
void ExpectEqualBlePairingSettings(
    const ::nn::settings::system::BlePairingSettings& lhs,
    const ::nn::settings::system::BlePairingSettings& rhs) NN_NOEXCEPT;

} // namespace

//!< Bluetooth デバイス設定を取得可能か
TEST(BluetoothDevicesSuite, BluetoothDevicesSettingsTest1)
{
    ::nn::settings::system::GetBluetoothDevicesSettings(
        &g_BluetoothDevicesSettings[0], 1);
}

//!< Bluetooth デバイス設定を変更可能か
TEST(BluetoothDevicesSuite, BluetoothDevicesSettingsTest2)
{
    ::nn::settings::system::SetBluetoothDevicesSettings(
        &BluetoothDevicesSettingsA, 1);

    ::nn::settings::system::BluetoothDevicesSettings* settings =
        &g_BluetoothDevicesSettings[0];

    const auto countMax =
        ::nn::settings::system::BluetoothDevicesSettingsCountMax;

    EXPECT_EQ(1,
              ::nn::settings::system::GetBluetoothDevicesSettings(settings,
                                                                  countMax));

    {
        SCOPED_TRACE("");
        ExpectEqualBluetoothDevicesSettings(BluetoothDevicesSettingsA,
                                            settings[0]);
    }

    for (size_t i = 0; i < countMax; ++i)
    {
        settings[i] = (i % 2 == 1) ? BluetoothDevicesSettingsA
                                   : BluetoothDevicesSettingsB;
    }

    ::nn::settings::system::SetBluetoothDevicesSettings(settings, countMax);

    for (size_t i = 0; i < countMax; ++i)
    {
        settings[i] = ::nn::settings::system::BluetoothDevicesSettings();
    }

    EXPECT_EQ(countMax,
              ::nn::settings::system::GetBluetoothDevicesSettings(settings,
                                                                  countMax));

    for (size_t i = 0; i < countMax; ++i)
    {
        SCOPED_TRACE("");
        ExpectEqualBluetoothDevicesSettings(
            (i % 2 == 1) ? BluetoothDevicesSettingsA
                         : BluetoothDevicesSettingsB, settings[i]);
    }

    ::nn::settings::system::SetBluetoothDevicesSettings(nullptr, 0);

    EXPECT_EQ(0,
              ::nn::settings::system::GetBluetoothDevicesSettings(settings,
                                                                  countMax));
}

//!< Bluetooth が有効か否かを表す値の設定を取得可能か
TEST(BluetoothDevicesSuite, BluetoothDevicesSettingsTest3)
{
    ::nn::settings::system::IsBluetoothEnabled();
}

//!< Bluetooth が有効か否かを表す値の設定を変更可能か
TEST(BluetoothDevicesSuite, BluetoothDevicesSettingsTest4)
{
    ::nn::settings::system::SetBluetoothEnabled(true);

    EXPECT_TRUE(::nn::settings::system::IsBluetoothEnabled());

    ::nn::settings::system::SetBluetoothEnabled(false);

    EXPECT_FALSE(::nn::settings::system::IsBluetoothEnabled());

    ::nn::settings::system::SetBluetoothEnabled(true);

    EXPECT_TRUE(::nn::settings::system::IsBluetoothEnabled());
}

//!< Bluetooth の AFH が有効か否かを表す値の設定を変更可能か
TEST(BluetoothDevicesSuite, BluetoothDevicesSettingsTest5)
{
    bool value = ::nn::settings::system::IsBluetoothAfhEnabled();

    ::nn::settings::system::SetBluetoothAfhEnabled(true);

    EXPECT_TRUE(::nn::settings::system::IsBluetoothAfhEnabled());

    ::nn::settings::system::SetBluetoothAfhEnabled(false);

    EXPECT_FALSE(::nn::settings::system::IsBluetoothAfhEnabled());

    ::nn::settings::system::SetBluetoothAfhEnabled(true);

    EXPECT_TRUE(::nn::settings::system::IsBluetoothAfhEnabled());

    ::nn::settings::system::SetBluetoothAfhEnabled(value);
}

//!< Bluetooth の Boost が有効か否かを表す値の設定を変更可能か
TEST(BluetoothDevicesSuite, BluetoothDevicesSettingsTest6)
{
    bool value = ::nn::settings::system::IsBluetoothBoostEnabled();

    ::nn::settings::system::SetBluetoothBoostEnabled(true);

    EXPECT_TRUE(::nn::settings::system::IsBluetoothBoostEnabled());

    ::nn::settings::system::SetBluetoothBoostEnabled(false);

    EXPECT_FALSE(::nn::settings::system::IsBluetoothBoostEnabled());

    ::nn::settings::system::SetBluetoothBoostEnabled(true);

    EXPECT_TRUE(::nn::settings::system::IsBluetoothBoostEnabled());

    ::nn::settings::system::SetBluetoothBoostEnabled(value);
}

//!< BLE ペアリング設定を変更可能か
TEST(BluetoothDevicesSuite, BlePairingSettingsTest1)
{
    ::nn::settings::system::SetBlePairingSettings(
        &BlePairingSettingsA, 1);

    ::nn::settings::system::BlePairingSettings* settings =
        &g_BlePairingSettings[0];

    const auto countMax =
        ::nn::settings::system::BlePairingSettingsCountMax;

    EXPECT_EQ(
        1,
        ::nn::settings::system::GetBlePairingSettings(settings,countMax));

    {
        SCOPED_TRACE("");
        ExpectEqualBlePairingSettings(BlePairingSettingsA, settings[0]);
    }

    for (size_t i = 0; i < countMax; ++i)
    {
        settings[i] = (i % 2 == 1) ? BlePairingSettingsA : BlePairingSettingsB;
    }

    ::nn::settings::system::SetBlePairingSettings(settings, countMax);

    for (size_t i = 0; i < countMax; ++i)
    {
        settings[i] = ::nn::settings::system::BlePairingSettings();
    }

    EXPECT_EQ(
        countMax,
        ::nn::settings::system::GetBlePairingSettings(settings, countMax));

    for (size_t i = 0; i < countMax; ++i)
    {
        SCOPED_TRACE("");
        ExpectEqualBlePairingSettings(
            (i % 2 == 1) ? BlePairingSettingsA : BlePairingSettingsB,
            settings[i]);
    }

    ::nn::settings::system::SetBlePairingSettings(nullptr, 0);

    EXPECT_EQ(
        0,
        ::nn::settings::system::GetBlePairingSettings(settings, countMax));
}

namespace {

void ExpectEqualBluetoothDevicesSettings(
    const ::nn::settings::system::BluetoothDevicesSettings& lhs,
    const ::nn::settings::system::BluetoothDevicesSettings& rhs) NN_NOEXCEPT
{
    for (size_t i = 0; i < sizeof(lhs.bd_addr) / sizeof(lhs.bd_addr[0]); ++i)
    {
        EXPECT_EQ(lhs.bd_addr[i], rhs.bd_addr[i]);
    }

    for (size_t i = 0; i < sizeof(lhs.device_name) / sizeof(lhs.device_name[0]);
         ++i)
    {
        EXPECT_EQ(lhs.device_name[i], rhs.device_name[i]);
    }

    for (size_t i = 0;
         i < sizeof(lhs.class_of_device) / sizeof(lhs.class_of_device[0]); ++i)
    {
        EXPECT_EQ(lhs.class_of_device[i], rhs.class_of_device[i]);
    }

    for (size_t i = 0; i < sizeof(lhs.link_key) / sizeof(lhs.link_key[0]); ++i)
    {
        EXPECT_EQ(lhs.link_key[i], rhs.link_key[i]);
    }

    EXPECT_EQ(lhs.link_key_present, rhs.link_key_present);
    EXPECT_EQ(lhs.version, rhs.version);
    EXPECT_EQ(lhs.trusted_services, rhs.trusted_services);
    EXPECT_EQ(lhs.vid, rhs.vid);
    EXPECT_EQ(lhs.pid, rhs.pid);
    EXPECT_EQ(lhs.sub_class, rhs.sub_class);
    EXPECT_EQ(lhs.attribute_mask, rhs.attribute_mask);
    EXPECT_EQ(lhs.descriptor_length, rhs.descriptor_length);

    for (size_t i = 0; i < sizeof(lhs.descriptor) / sizeof(lhs.descriptor[0]);
         ++i)
    {
        EXPECT_EQ(lhs.descriptor[i], rhs.descriptor[i]);
    }

    EXPECT_EQ(lhs.key_type, rhs.key_type);
    EXPECT_EQ(lhs.device_type, rhs.device_type);
    EXPECT_EQ(lhs.brr_size, rhs.brr_size);

    for (size_t i = 0; i < sizeof(lhs.brr) / sizeof(lhs.brr[0]); ++i)
    {
        EXPECT_EQ(lhs.brr[i], rhs.brr[i]);
    }

    for (size_t i = 0; i < sizeof(lhs.reserved) / sizeof(lhs.reserved[0]); ++i)
    {
        EXPECT_EQ(lhs.reserved[i], rhs.reserved[i]);
    }
}

void ExpectEqualBlePairingSettings(
    const ::nn::settings::system::BlePairingSettings& lhs,
    const ::nn::settings::system::BlePairingSettings& rhs) NN_NOEXCEPT
{
    for (size_t i = 0; i < NN_ARRAY_SIZE(lhs.bd_addr); ++i)
    {
        EXPECT_EQ(lhs.bd_addr[i], rhs.bd_addr[i]);
    }

    EXPECT_EQ(lhs.manufacturer_id, rhs.manufacturer_id);

    for (size_t i = 0; i < NN_ARRAY_SIZE(lhs.client_id); ++i)
    {
        EXPECT_EQ(lhs.client_id[i], rhs.client_id[i]);
    }

    for (size_t i = 0; i < NN_ARRAY_SIZE(lhs.server_id); ++i)
    {
        EXPECT_EQ(lhs.server_id[i], rhs.server_id[i]);
    }

    EXPECT_EQ(lhs.pairing_type, rhs.pairing_type);
}

} // namespace
