﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <cstdlib>

#include <nn/nn_Log.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/gc/gc.h>  // TODO: TEMP:
#include <nn/fs/fs_GameCard.h>  // TODO: TEMP:
#include <nn/fs/fs_GameCardForInspection.h>
#include <nn/fs/fs_MmcPrivate.h>
#include <nn/gc/detail/gc_AsicHandler.h>  // TODO: TEMP:
#include <nn/gc/detail/gc_AsicHandlerCore.h>  // TODO: TEMP:
#include <nn/gc/detail/gc_DeviceDetector.h>  // TODO: TEMP:
#include <nn/spl/spl_Api.h>

#include "Util.h"
#include "DataStructure.h"
#include "StorageInterface.h"
#include "FileAccessor.h"
#include "OperationExecuter.h"

#define SDMMC_DETAIL_CEILING(value, unit)   ((((value) + (unit) - 1) / (unit)) * (unit))
#define SDMMC_DETAIL_CEILING_FOR_DEVICE_ADDRESS_SPACE(value)    SDMMC_DETAIL_CEILING((value), nn::dd::DeviceAddressSpaceMemoryRegionAlignment)
const size_t g_MmcWorkBufferSize = SDMMC_DETAIL_CEILING_FOR_DEVICE_ADDRESS_SPACE(nn::sdmmc::MmcWorkBufferSize);
const size_t g_SdWorkBufferSize = SDMMC_DETAIL_CEILING_FOR_DEVICE_ADDRESS_SPACE(nn::sdmmc::SdCardWorkBufferSize);
const size_t g_GcWorkBufferSize = SDMMC_DETAIL_CEILING_FOR_DEVICE_ADDRESS_SPACE(nn::gc::GcWorkBufferSize);
NN_DD_ALIGNAS_DEVICE_ADDRESS_SPACE_MEMORY uint8_t g_MmcWorkBuffer[g_MmcWorkBufferSize];
NN_DD_ALIGNAS_DEVICE_ADDRESS_SPACE_MEMORY uint8_t g_SdWorkBuffer[g_SdWorkBufferSize];
NN_DD_ALIGNAS_DEVICE_ADDRESS_SPACE_MEMORY uint8_t g_GcWorkBuffer[g_GcWorkBufferSize];

const uint32_t TestNumSectors = 16 * 1024;
u8* StorageInterface::m_DataBufferAddress = NULL;

nn::dd::DeviceVirtualAddress g_DeviceVirtualAddressOffset = 0;


//--------------------------------------------------------------------------
/*
 * Public API 定義
 */

size_t GetDataBufferSize()
{
    return nn::util::align_up(nn::sdmmc::SectorSize * TestNumSectors, nn::os::MemoryBlockUnitSize);
}

StorageInterface::StorageInterface()
{
    storageName = "Virtual Storage";
    m_StorageType = StorageInterfaceType_NAND;
    m_MmcPort = nn::sdmmc::Port_Mmc0;

    // m_Das
    m_DeviceName = nn::dd::DeviceName::DeviceName_Sdmmc4a;
    m_WorkBufferDeviceVirtualAddress = 0;
    m_DataBufferDeviceVirtualAddress = 0;

    m_StorageSizeNumSectors = 0;

    dataBufferSize = GetDataBufferSize();
    NN_LOG("memory %d byte\n", dataBufferSize);
}

void StorageInterface::MapDataBuffer()
{
    // データバッファをマップする
    m_DataBufferDeviceVirtualAddress = MapDeviceAddressSpaceAligned(&m_Das, reinterpret_cast<uintptr_t>(dataBuffer), dataBufferSize, 0);
    nn::sdmmc::RegisterDeviceVirtualAddress(m_MmcPort, reinterpret_cast<uintptr_t>(dataBuffer), dataBufferSize, m_DataBufferDeviceVirtualAddress);
}

void StorageInterface::UnmapDataBuffer()
{
    // データバッファをアンマップする
    nn::sdmmc::UnregisterDeviceVirtualAddress(m_MmcPort, reinterpret_cast<uintptr_t>(dataBuffer), dataBufferSize, m_DataBufferDeviceVirtualAddress);
    UnmapDeviceAddressSpaceAligned(&m_Das, reinterpret_cast<uintptr_t>(dataBuffer), dataBufferSize, m_DataBufferDeviceVirtualAddress);
}

nn::Result StorageInterface::Speed(nn::sdmmc::SpeedMode targetSpeedMode)
{
    // 接続状態を確認する
    nn::sdmmc::SpeedMode hostSpeedMode;
    nn::sdmmc::BusWidth hostBusWidth;
    NN_RESULT_DO(nn::sdmmc::CheckConnection(&hostSpeedMode, &hostBusWidth, m_MmcPort));
    if(targetSpeedMode != hostSpeedMode)
    {
        NN_SC_DETAIL_ERR_LOG("host speed mode does not match (expected: %d, actual: %d)\n", targetSpeedMode, hostSpeedMode);
        return nn::sdmmc::ResultAbortCommandIssued();   // TEMP:
    }

    // デバイスのスピードモードを調べる
    nn::sdmmc::SpeedMode deviceSpeedMode;
    NN_RESULT_DO(nn::sdmmc::GetDeviceSpeedMode(&deviceSpeedMode, m_MmcPort));

    if(targetSpeedMode != deviceSpeedMode)
    {
        NN_SC_DETAIL_ERR_LOG("device speed mode does not match (expected: %d, actual: %d)\n", targetSpeedMode, deviceSpeedMode);
        return nn::sdmmc::ResultAbortCommandIssued();   // TEMP:
    }

    NN_SC_DETAIL_LOG2("speed mode matched (expected: %d, actual: %d)\n", targetSpeedMode, deviceSpeedMode);
    return nn::ResultSuccess();
}

nn::Result StorageInterface::GetStorageSize()
{
    NN_RESULT_DO(nn::sdmmc::GetDeviceMemoryCapacity(&m_StorageSizeNumSectors, m_MmcPort));
    NN_SC_DETAIL_LOG2("Storage Size: 0x%0x sectors\n", m_StorageSizeNumSectors);
    NN_RESULT_SUCCESS;
}

nn::Result StorageInterface::Reg()
{
    {
        nn::sdmmc::HostBusStatus busStatus;
        nn::sdmmc::GetHostBusStatus(&busStatus, m_MmcPort);
        NN_SC_DETAIL_LOG2("BusPower: %d, BusWidth:%d, DeviceClockFrequencyKHz:%d\n", busStatus.busPower, busStatus.busWidth, busStatus.deviceClockFrequencyKHz);

        nn::sdmmc::SpeedMode devSpeed;
        NN_RESULT_DO(nn::sdmmc::GetDeviceSpeedMode(&devSpeed, m_MmcPort));
        NN_SC_DETAIL_LOG2("Device Speed: %d\n", devSpeed);

        uint32_t numSectors = 0;
        NN_RESULT_DO(nn::sdmmc::GetDeviceMemoryCapacity(&numSectors, m_MmcPort));
        NN_SC_DETAIL_LOG2("Device Memory Capacity: 0x%0x sector(s)\n", numSectors);

        uint32_t devStatus = 0;
        NN_RESULT_DO(nn::sdmmc::GetDeviceStatus(&devStatus, m_MmcPort));
        NN_SC_DETAIL_LOG2("Device Status: %d\n", devStatus);
    }
    {
        NN_SC_DETAIL_LOG2("*** OCR ***\n");
        uint32_t ocr = 0;
        NN_RESULT_DO(nn::sdmmc::GetDeviceOcr(&ocr, m_MmcPort));
        PrintDebugArray((char*)&ocr, sizeof(ocr));

        NN_SC_DETAIL_LOG2("*** RCA ***\n");
        uint16_t rca = 0;
        NN_RESULT_DO(nn::sdmmc::GetDeviceRca(&rca, m_MmcPort));
        PrintDebugArray((char*)&rca, sizeof(rca));
    }
    {
        char buf[nn::sdmmc::DeviceCidSize + nn::sdmmc::DeviceCsdSize + 1];  // 適当に十分な大きさ

        NN_SC_DETAIL_LOG2("*** CID ***\n");
        NN_RESULT_DO(nn::sdmmc::GetDeviceCid(dataBuffer, dataBufferSize, m_MmcPort));
        // 8bit目から格納されており、bit0-7は0埋めする（前インタプリタと挙動を揃える）
        buf[0] = 0;
        memcpy(buf + 1, dataBuffer, nn::sdmmc::DeviceCidSize);
        PrintDebugArray(buf, nn::sdmmc::DeviceCidSize);

        NN_SC_DETAIL_LOG2("*** CSD ***\n");
        NN_RESULT_DO(nn::sdmmc::GetDeviceCsd(dataBuffer, dataBufferSize, m_MmcPort));
        // 8bit目から格納されており、bit0-7は0埋めする（前インタプリタと挙動を揃える）
        buf[0] = 0;
        memcpy(buf + 1, dataBuffer, nn::sdmmc::DeviceCsdSize);
        PrintDebugArray(buf, nn::sdmmc::DeviceCsdSize);
    }

    NN_RESULT_SUCCESS;
}

// *** NAND ***
StorageInterfaceNand::StorageInterfaceNand()
{
    storageName = "NAND";
    m_StorageType = StorageInterfaceType_NAND;
    m_MmcPort = nn::sdmmc::Port_Mmc0;
    m_DeviceName = nn::dd::DeviceName::DeviceName_Sdmmc4a;
    m_CurrentPartition = nn::sdmmc::MmcPartition_UserData;
    dataBuffer = m_DataBufferAddress;
    NN_LOG("Constructor: storage '%s'\n", storageName.c_str());

    nn::sdmmc::Initialize(m_MmcPort);
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

    SetupDeviceAddressSpace(&m_Das, m_DeviceName);
    // ワークバッファの登録
    nn::dd::DeviceAddressSpaceMapInfo info;
    m_WorkBufferDeviceVirtualAddress = MapDeviceAddressSpaceAligned(&m_Das, &info,
        reinterpret_cast<uintptr_t>(g_MmcWorkBuffer), g_MmcWorkBufferSize, g_DeviceVirtualAddressOffset);
    nn::sdmmc::RegisterDeviceVirtualAddress(m_MmcPort, reinterpret_cast<uintptr_t>(g_MmcWorkBuffer), g_MmcWorkBufferSize, m_WorkBufferDeviceVirtualAddress);
    nn::sdmmc::SetMmcWorkBuffer(m_MmcPort, g_MmcWorkBuffer, g_MmcWorkBufferSize);

    // オフセットを保存
    g_DeviceVirtualAddressOffset = m_WorkBufferDeviceVirtualAddress + g_MmcWorkBufferSize;
}

StorageInterfaceNand::~StorageInterfaceNand()
{
    nn::sdmmc::Finalize(m_MmcPort);

    UnmapDeviceAddressSpaceAligned(&m_Das, reinterpret_cast<uintptr_t>(g_MmcWorkBuffer), g_MmcWorkBufferSize, m_WorkBufferDeviceVirtualAddress);

    CleanDeviceAddressSpace(&m_Das, m_DeviceName);
}

nn::Result StorageInterfaceNand::Activate()
{
    NN_SC_LOG("StorageInterfaceNand::Activate\n");
    NN_RESULT_DO(nn::sdmmc::Activate(nn::sdmmc::Port_Mmc0));
    NN_RESULT_DO(CheckSdmmcHiddenError(nn::sdmmc::Port_Mmc0));

    // ストレージ情報の取得
    GetStorageSize();

    NN_RESULT_SUCCESS;
}

void StorageInterfaceNand::Deactivate()
{
    NN_SC_LOG("StorageInterfaceNand::Deactivate\n");
    nn::sdmmc::Deactivate(nn::sdmmc::Port_Mmc0);
}

nn::Result StorageInterfaceNand::Erase()
{
    return nn::sdmmc::EraseMmc(m_MmcPort);
}

nn::Result StorageInterfaceNand::SelectPartition(nn::sdmmc::MmcPartition targetPartition)
{
    NN_RESULT_DO(nn::sdmmc::SelectMmcPartition(m_MmcPort, targetPartition));
    m_CurrentPartition = targetPartition;
    return GetStorageSize();
}

nn::Result StorageInterfaceNand::GetStorageSize()
{
    // Boot Partition の時は API が異なる
    if(m_CurrentPartition != nn::sdmmc::MmcPartition_UserData)
    {
        return nn::sdmmc::GetMmcBootPartitionCapacity(&m_StorageSizeNumSectors, m_MmcPort);
    }

    // UserData のときは普通に処理する
    return StorageInterface::GetStorageSize();
}

nn::Result StorageInterfaceNand::Reg()
{
    NN_RESULT_DO(StorageInterface::Reg());

    NN_SC_DETAIL_LOG2("*** Extended CSD ***\n");
    NN_RESULT_DO(nn::sdmmc::GetMmcExtendedCsd(dataBuffer, dataBufferSize, m_MmcPort));
    PrintDebugArray((const char*)dataBuffer, nn::sdmmc::MmcExtendedCsdSize);

    uint32_t numSectors = 0;
    GetMmcBootPartitionCapacity(&numSectors, m_MmcPort);
    NN_SC_DETAIL_LOG2("Boot Partition Capacity: 0x%0x\n", numSectors);

    NN_SC_DETAIL_LOG2("\n");
    NN_RESULT_SUCCESS;
}

// *** SD ***
StorageInterfaceSd::StorageInterfaceSd()
{
    storageName = "SD";
    m_StorageType = StorageInterfaceType_SD;
    m_MmcPort = nn::sdmmc::Port_SdCard0;
    #ifdef NN_BUILD_CONFIG_HARDWARE_JETSONTK1
        m_DeviceName = nn::dd::DeviceName::DeviceName_Sdmmc3a;
    #else   // NN_BUILD_CONFIG_HARDWARE_JETSONTK2
        // NN_BUILD_CONFIG_HARDWARE_NX
        m_DeviceName = nn::dd::DeviceName::DeviceName_Sdmmc1a;
    #endif
    dataBuffer = m_DataBufferAddress;
    NN_LOG("Constructor: storage '%s'\n", storageName.c_str());

    nn::sdmmc::Initialize(m_MmcPort);
    SetupDeviceAddressSpace(&m_Das, m_DeviceName);

    // ワークバッファの登録
    nn::dd::DeviceAddressSpaceMapInfo info;
    m_WorkBufferDeviceVirtualAddress = MapDeviceAddressSpaceAligned(&m_Das, &info,
        reinterpret_cast<uintptr_t>(g_SdWorkBuffer), g_SdWorkBufferSize, g_DeviceVirtualAddressOffset);
    nn::sdmmc::RegisterDeviceVirtualAddress(m_MmcPort, reinterpret_cast<uintptr_t>(g_SdWorkBuffer), g_SdWorkBufferSize, m_WorkBufferDeviceVirtualAddress);
    nn::sdmmc::SetSdCardWorkBuffer(m_MmcPort, g_SdWorkBuffer, g_SdWorkBufferSize);

    // オフセットを保存
    g_DeviceVirtualAddressOffset = m_WorkBufferDeviceVirtualAddress + g_SdWorkBufferSize;
}

StorageInterfaceSd::~StorageInterfaceSd()
{
    nn::sdmmc::Finalize(m_MmcPort);

    UnmapDeviceAddressSpaceAligned(&m_Das, reinterpret_cast<uintptr_t>(g_SdWorkBuffer), g_SdWorkBufferSize, m_WorkBufferDeviceVirtualAddress);

    CleanDeviceAddressSpace(&m_Das, m_DeviceName);
}


nn::Result StorageInterfaceSd::Activate()
{
    NN_SC_LOG("StorageInterfaceSd::Activate\n");
    NN_RESULT_DO(nn::sdmmc::Activate(nn::sdmmc::Port_SdCard0));
    NN_RESULT_DO(CheckSdmmcHiddenError(nn::sdmmc::Port_SdCard0));

    // ストレージ情報の取得
    GetStorageSize();

    NN_RESULT_SUCCESS;
}

void StorageInterfaceSd::Deactivate()
{
    NN_SC_LOG("StorageInterfaceSd::Deactivate\n");
    nn::sdmmc::Deactivate(nn::sdmmc::Port_SdCard0);
}

nn::Result StorageInterfaceSd::Insert(InsertDirection direction)
{
    NN_LOG("Warning: not yet implemented\n");
    return nn::ResultSuccess();
}

nn::Result StorageInterfaceSd::Reg()
{
    NN_RESULT_DO(StorageInterface::Reg());

    uint32_t numSectors = 0;
    nn::sdmmc::GetSdCardProtectedAreaCapacity(&numSectors, m_MmcPort);

    {
        NN_SC_DETAIL_LOG2("*** SCR ***\n");
        NN_RESULT_DO(nn::sdmmc::GetSdCardScr(dataBuffer, dataBufferSize, m_MmcPort));
        char buf[nn::sdmmc::SdCardScrSize];
        for(int i=0; i<sizeof(buf); i++)
        {
            buf[i] = dataBuffer[nn::sdmmc::SdCardScrSize - i - 1];
        }
        PrintDebugArray(buf, sizeof(buf));
    }

    NN_SC_DETAIL_LOG2("*** SwitchFunctionStatus ***\n");
    nn::Result result = nn::sdmmc::GetSdCardSwitchFunctionStatus(dataBuffer, dataBufferSize,
        m_MmcPort, nn::sdmmc::SdCardSwitchFunction_CheckSupportedFunction);
    if (nn::sdmmc::ResultSdCardNotSupportSwitchFunctionStatus::Includes(result))
    {
        NN_SC_DETAIL_LOG2("SD Card doesn't support Switch Function Status\n");
    }
    else
    {
        NN_RESULT_DO(result);
        NN_SC_DETAIL_LOG2("SD Card Switch Function Status (Check Supported Function):\n");
        PrintDebugArray((const char*)dataBuffer, nn::sdmmc::SdCardSwitchFunctionStatusSize);

        NN_RESULT_DO(nn::sdmmc::GetSdCardSwitchFunctionStatus(dataBuffer, dataBufferSize,
            m_MmcPort, nn::sdmmc::SdCardSwitchFunction_CheckHighSpeed));
        NN_SC_DETAIL_LOG2("SD Card Switch Function Status (Check High Speed):\n");
        PrintDebugArray((const char*)dataBuffer, nn::sdmmc::SdCardSwitchFunctionStatusSize);
    }

    NN_SC_DETAIL_LOG2("*** SdStatus ***\n");
    NN_RESULT_DO(nn::sdmmc::GetSdCardSdStatus(dataBuffer, dataBufferSize, m_MmcPort));
    PrintDebugArray((const char*)dataBuffer, nn::sdmmc::SdCardSdStatusSize);

    NN_RESULT_SUCCESS;
}



// *** GC ***
namespace
{
    nn::os::EventType g_CardDetectionEvent;
}

void ReadGcCalibrationPartitionToSetInternalKeys()
{
    nn::spl::InitializeForFs();

    NN_LOG("Read From Partition For Gc Setting\n");
    const uint32_t BoundaryVersion   = 0x7;
    const size_t VersionAddress      = 0x4;
    const size_t KeyAddress          = 0x3C20;
    const size_t KeySize             = 0x134;
    const size_t OldKeyAddress       = 0x2320;
    const size_t OldKeySize          = 0x110;
    const size_t CertAddress         = 0x2440;
    const size_t CertSize            = 0x400;

    std::unique_ptr<nn::fs::IStorage> storage;
    nn::Result result = nn::fs::OpenBisPartition(&storage, nn::fs::BisPartitionId::CalibrationBinary);
    if(result.IsFailure())
    {
        NN_LOG("Test Failure\n");
        NN_LOG("  OpenBisPartition Failure (Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
    }
    if(storage == nullptr)
    {
        NN_LOG("PTR NULL\n");
    }

    static char keyBuffer[KeySize] = {0};
    static char certBuffer[CertSize] = {0};

    uint32_t version = 0;
    if(storage->Read(VersionAddress, &version, sizeof(version)).IsFailure())
    {
        NN_LOG("[error] calibration data for gc read failure!\n");
    }
    size_t targetKeyAddress = (version >= BoundaryVersion) ? KeyAddress : OldKeyAddress;
    size_t targetKeySize = (version >= BoundaryVersion) ? KeySize : OldKeySize;

    if(storage->Read(targetKeyAddress, keyBuffer, targetKeySize).IsFailure())
    {
        NN_LOG("[error] calibration data for gc read failure!\n");
    }
    if(storage->Read(CertAddress, certBuffer, CertSize).IsFailure())
    {
        NN_LOG("[error] calibration data for gc read failure!\n");
    }

    nn::gc::PresetInternalKeys(keyBuffer, targetKeySize, certBuffer, CertSize);
    memset(keyBuffer, 0, sizeof(keyBuffer));
}

StorageInterfaceGc::StorageInterfaceGc()
{
    m_IsWriter = false;
    m_IsSecure = false;
    m_IsActivated = false;
    m_CardMemorySize = nn::gc::writer::MemorySize_32GB;

    storageName = "GC";
    m_StorageType = StorageInterfaceType_GC;
    m_MmcPort = nn::sdmmc::Port_GcAsic0;
    m_DeviceName = nn::dd::DeviceName::DeviceName_Sdmmc2a;
    dataBuffer = m_DataBufferAddress;
    NN_LOG("Constructor: storage '%s'\n", storageName.c_str());

    // ワークバッファの登録
    SetupDeviceAddressSpace(&m_Das, m_DeviceName);
    nn::dd::DeviceAddressSpaceMapInfo info;
    m_WorkBufferDeviceVirtualAddress = MapDeviceAddressSpaceAligned(&m_Das, &info,
        reinterpret_cast<uintptr_t>(g_GcWorkBuffer), g_GcWorkBufferSize, g_DeviceVirtualAddressOffset);

    // gc 経由で登録する
    nn::gc::Initialize(g_GcWorkBuffer, g_GcWorkBufferSize, m_WorkBufferDeviceVirtualAddress);

    // オフセットを保存
    g_DeviceVirtualAddressOffset = m_WorkBufferDeviceVirtualAddress + g_GcWorkBufferSize;

    // 自信で監視するイベントを用意する
    nn::os::InitializeEvent( &g_CardDetectionEvent, false, nn::os::EventClearMode_ManualClear );
}

StorageInterfaceGc::~StorageInterfaceGc()
{
    nn::sdmmc::Finalize(m_MmcPort);

    UnmapDeviceAddressSpaceAligned(&m_Das, reinterpret_cast<uintptr_t>(g_GcWorkBuffer), g_GcWorkBufferSize, m_WorkBufferDeviceVirtualAddress);

    CleanDeviceAddressSpace(&m_Das, nn::dd::DeviceName::DeviceName_Sdmmc2a);
}

void SignalDetectionEvents(void *pParameter)
{
    nn::os::SignalEvent( &g_CardDetectionEvent );
}

nn::Result StorageInterfaceGc::Activate()
{
    // コールバック関数を登録する
#ifdef NN_SC_ENABLE_GC_THREAD
    if(m_IsActivated)
    {
        nn::gc::Deactivate();
    }
#endif

    NN_SC_LOG("gc Activate\n");
    nn::Result result = nn::gc::Activate();
    if(result.IsFailure())
    {
        if(!nn::gc::IsCardInserted())
        {
            NN_SC_DETAIL_ERR_LOG("Activate GC Error\n");
        }
        return result;
    }
    m_IsActivated = true;

    // ストレージ情報の取得
    GetStorageSize();

    NN_RESULT_SUCCESS;
}

void StorageInterfaceGc::Deactivate()
{
    NN_SC_LOG("StorageInterfaceSd::Deactivate\n");
    nn::gc::Deactivate();
    m_IsActivated = false;
}

nn::Result StorageInterfaceGc::Enable()
{
    m_IsSecure = true;
    if(m_IsWriter)
    {
        // writer ならフラグをいじるだけにする
        NN_RESULT_SUCCESS;
    }
#ifdef NN_SC_ENABLE_GC_THREAD
    nn::Result result = nn::gc::SetCardToSecureMode();
#else
    nn::Result result = nn::gc::ChangeToCardSecureMode();
#endif
    if(result.IsFailure())
    {
        NN_SC_DETAIL_ERR_LOG("Enable GC Error\n");
    }
    return result;
}

nn::Result StorageInterfaceGc::Write(uint32_t sectorIndex, uint32_t numSectors, u8* pDataBuffer, size_t dataBufferSize)
{
    if(m_IsWriter == false)
    {
        // ライトモードに切り替え
        nn::gc::writer::ChangeMode(nn::gc::writer::AsicMode_WriteMode);
        NN_RESULT_DO(nn::gc::writer::ActivateForWriter());
        m_IsWriter = true;
    }

    // ライトモードでは余分な領域が見えるためアドレスがずれる
    static const uint32_t writerModeExtraArea = 0x08;
    return nn::gc::writer::Write(pDataBuffer, dataBufferSize, (int)(sectorIndex) + writerModeExtraArea, numSectors);
}

nn::Result StorageInterfaceGc::Read(u8* pOutDataBuffer, size_t dataBufferSize, uint32_t sectorIndex, uint32_t numSectors)
{
    if(m_IsWriter)
    {
        // リードモードに切り替え
        nn::gc::writer::ChangeMode(nn::gc::writer::AsicMode_ReadMode);
        NN_RESULT_DO(nn::gc::Activate());
        m_IsWriter = false;
        if(m_IsSecure)
        {
#ifdef NN_SC_ENABLE_GC_THREAD
            NN_RESULT_DO(nn::gc::SetCardToSecureMode());
#else
            NN_RESULT_DO(nn::gc::ChangeToCardSecureMode());
#endif
        }
    }
    return nn::gc::Read(pOutDataBuffer, dataBufferSize, sectorIndex, numSectors);
}

nn::Result StorageInterfaceGc::Insert(InsertDirection direction)
{
    nn::gc::detail::DeviceDetector& detector = nn::gc::detail::DeviceDetector::GetInstance();

    // とりあえずチェックして、状態が同じならはや抜けする
    if(direction == InsertDirection_CheckInsert)
    {
        NN_SC_DETAIL_LOG("check card insertion\n");
        if(detector.GetGpioDetectPin() == true)
        {
            NN_RESULT_SUCCESS;
        }
        direction = InsertDirection_Insert;
    }
    else if(direction == InsertDirection_CheckRemove)
    {
        NN_SC_DETAIL_LOG("check card removal\n");
        if(detector.GetGpioDetectPin() == false)
        {
            NN_RESULT_SUCCESS;
        }
        direction = InsertDirection_Remove;
    }

    // 0.1s 間隔のポーリングで状態待ち
    int pollingWaitMs = 100;
    if(direction == InsertDirection_Insert)
    {
        while(detector.GetGpioDetectPin() == false)
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(pollingWaitMs));
        }
    }
    else if(direction == InsertDirection_Remove)
    {
        while(detector.GetGpioDetectPin() == true)
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(pollingWaitMs));
        }
    }
    else
    {
        NN_ABORT("Error: unknown state for insert operation\n");
    }

    NN_RESULT_SUCCESS;
}

nn::Result StorageInterfaceGc::Erase()
{
    if(m_IsWriter == false)
    {
        // ライトモードに切り替え
        nn::gc::writer::ChangeMode(nn::gc::writer::AsicMode_WriteMode);
        NN_RESULT_DO(nn::gc::writer::ActivateForWriter());
        m_IsWriter = true;
    }
    return nn::gc::writer::EraseAndWriteParameter(m_CardMemorySize, 0x80000);// とりあえず決め打ち
}

nn::Result StorageInterfaceGc::WriteParameter(WriteParam param, uint32_t value)
{
    if(param == WriteParam_GcMemorySize)
    {
        m_CardMemorySize = (nn::gc::writer::MemorySize)(value);
    }
    else
    {
        NN_ABORT("Error: unknown writeparam operation");
    }

    NN_RESULT_SUCCESS;
}

nn::Result StorageInterfaceGc::Reg()
{
    // リフレッシュ
    /*
    nn::gc::detail::AsicHandlerCore& gcCore = nn::gc::detail::AsicHandlerCore::GetInstance();
    int refreshTryCount = gcCore.GetRefreshTryCount();
    int refreshSucceededCount = gcCore.GetRefreshSucceededCount();
    NN_SC_DETAIL_LOG2("Refresh Count: (Success: %d / Try: %d)\n", refreshSucceededCount, refreshTryCount);
*/
    NN_RESULT_SUCCESS;
}
