﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <string>
#include <cstring>
#include <cstdlib>

#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/os/os_Thread.h>
#include <nn/nn_TimeSpan.h>
#include <nn/sdmmc/sdmmc_GcAsic.h>
#include <nn/sdmmc/sdmmc_Mmc.h>
#include <nn/sdmmc/sdmmc_SdCard.h>
#include <nn/gc/detail/gc_AsicOperation.h>
#include <nn/gc/detail/gc_Types.h>
#include <nn/fs/fs_SdmmcControl.h>
#include <nn/os/os_Tick.h>

#include "Util.h"
#include "DataStructure.h"
#include "StorageInterface.h"
#include "OperationExecuter.h"
#include "OperationInterpreter.h"
#include "AccessPatternManager.h"
#include "FileAccessor.h"
#include "DisplayData.h"
#include <nn/result/result_HandlingUtility.h>

#define NN_SC_DETAIL_SELECT_MAX_VALUE(a, b) ((a) > (b) ? (a) : (b))
#define NN_SC_DETAIL_SELECT_MIN_VALUE(a, b) ((a) < (b) ? (a) : (b))
#define NN_SC_DETAIL_OPERATION_DATA_STARTS_WITH(op, b) ((op->data.size >= strnlen(b, sizeof(b))) && memcmp((op->data.address), (b), strnlen(b, sizeof(b))) == 0)

using namespace ::nn::fs;

extern char g_VersionInfo[];

//--------------------------------------------------------------------------
/*
 * Private API 定義
 */

OperationExecuter::OperationExecuter()
{
    m_LogIntervalUsec = 10 * 1000 * 1000;
#ifdef SC_ENABLE_DEBUG_TEST_PRINT
    m_LogIntervalUsec = 0;
#endif
    m_WorkBufferSize = 0;
    m_OperationCount = 0;
    m_SkipPageCount = 0;
    m_IsMeasure = false;

#ifdef NN_SC_ENABLE_STORAGE_GC
    // fs プロセスの SDMMC 制御を停止する前に gc に必要な処理を行う
    ReadGcCalibrationPartitionToSetInternalKeys();
#endif

    // fs プロセスの SDMMC 制御を停止する
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::SuspendSdmmcControl());

    // クロック制御は pcv 経由で行う
    nn::sdmmc::SwitchToPcvClockResetControl();

#ifdef NN_SC_ENABLE_STORAGE_GC
    m_pStorageInterfaceArray[StorageInterfaceType_GC]   = new StorageInterfaceGc();
#else
    m_pStorageInterfaceArray[StorageInterfaceType_GC]   = nullptr;
#endif
    m_pStorageInterfaceArray[StorageInterfaceType_NAND] = new StorageInterfaceNand();
    m_pStorageInterfaceArray[StorageInterfaceType_SD]   = new StorageInterfaceSd();
    m_StorageInterfaceType = sizeof_StorageInterfaceType;// 無効な値
}

u32 OperationExecuter::GetAccessPage(ArgumentTypeWriteRead& arg, size_t countIndex)
{
    u32 index;
    switch(arg.accessType)
    {
    case WriteReadAccessType_SEQ:
        {
            index = arg.pageStart + (countIndex * arg.pageUnit);
        }
        break;
    case WriteReadAccessType_ALT:
        {
            if(countIndex % 2 == 0)
            {
                index = arg.pageStart + ((countIndex / 2) * arg.pageUnit);
            }
            else
            {
                index = (arg.pageStart + arg.pageCount) - (((countIndex + 1) / 2) * arg.pageUnit);
            }
        }
        break;
    case WriteReadAccessType_DEC:
        {
            index = arg.pageStart + arg.pageCount - ((countIndex + 1) * arg.pageUnit);
        }
        break;
    default:
        NN_ABORT("Unknown WriteReadAccessType\n");
    }
    return index;
}

void OperationExecuter::Free()
{
    // 終了処理：Deactivate をして data buffer を Unmap
    if(m_StorageInterfaceType != sizeof_StorageInterfaceType)
    {
        GetStorage().Deactivate();
        GetStorage().UnmapDataBuffer();
    }

    for(int i = 0; i < sizeof_StorageInterfaceType; i++)
    {
        if(m_pStorageInterfaceArray[i] != nullptr)
        {
           delete m_pStorageInterfaceArray[i];
        }
    }

    // fs プロセスの SDMMC 制御を再開する
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ResumeSdmmcControl());
}

void OperationExecuter::GenerateSectorData(u8* sectorBuffer, u32 pageAddress, WriteReadPattern pattern)
{
    static const u32 UINT8_MAX_PLUS1 = UINT8_MAX + 1UL;
    static const u64 UINT32_MAX_PLUS1 = UINT32_MAX + 1ULL;
    AccessPatternManager accessPatternManager = AccessPatternManager::GetInstance();
    u8 padPattern = 0;
    switch(pattern)
    {
        // * padPattern を設定して最後に memset する
    case WriteReadPattern_FIX_00:
        {
            padPattern = 0x00;
        }
        break;
    case WriteReadPattern_FIX_FF:
        {
            padPattern = 0xFF;
        }
        break;
    case WriteReadPattern_FIX_55:
        {
            padPattern = 0x55;
        }
        break;
    case WriteReadPattern_FIX_AA:
        {
            padPattern = 0xAA;
        }
        break;
    case WriteReadPattern_FIX_0F:
        {
            padPattern = 0x0F;
        }
        break;
    case WriteReadPattern_FIX_F0:
        {
            padPattern = 0xF0;
        }
        break;
    case WriteReadPattern_FIX_5A:
        {
            padPattern = 0x5A;
        }
        break;
    case WriteReadPattern_FIX_A5:
        {
            padPattern = 0xA5;
        }
        break;

        // * 処理後 return する関数
    case WriteReadPattern_INC8:
        {
            for(size_t i=0; i<nn::sdmmc::SectorSize; i++)
            {
                sectorBuffer[i] = (i % UINT8_MAX_PLUS1);
            }
        }
        return;
    case WriteReadPattern_DEC8:
        {
            for(size_t i=0; i<nn::sdmmc::SectorSize; i++)
            {
                sectorBuffer[i] = UINT8_MAX - (i % UINT8_MAX_PLUS1);
            }
        }
        return;
    case WriteReadPattern_INC32:
        {
            for(size_t i=0; i<nn::sdmmc::SectorSize; i+=U8SIZEOF(u32))
            {
                u32 unitSize = (nn::sdmmc::SectorSize / U8SIZEOF(u32));
                u32 counterValue = (pageAddress % (UINT32_MAX_PLUS1 / unitSize)) * unitSize + i / U8SIZEOF(u32);
                memcpy(sectorBuffer + i, &counterValue, U8SIZEOF(u32));
            }
        }
        return;
    case WriteReadPattern_DEC32:
        {
            for(size_t i=0; i<nn::sdmmc::SectorSize; i+=U8SIZEOF(u32))
            {
                u32 unitSize = (nn::sdmmc::SectorSize / U8SIZEOF(u32));
                u32 counterValue = UINT32_MAX - ((pageAddress % (UINT32_MAX_PLUS1 / unitSize)) * unitSize + i / U8SIZEOF(u32));
                memcpy(sectorBuffer + i, &counterValue, U8SIZEOF(u32));
            }
        }
        return;
    case WriteReadPattern_RANDOM:
        {
            for(size_t i=0; i<nn::sdmmc::SectorSize; i++)
            {
                sectorBuffer[i] = rand() % UINT8_MAX_PLUS1;
            }
        }
        return;
    case WriteReadPattern_AGING:
        {
            accessPatternManager.GetAccessPattern(reinterpret_cast<char*>(sectorBuffer), pageAddress, 1);
        }
        return;
    default:
        {
            padPattern = 0;
        }
        break;
    }
    memset(sectorBuffer, padPattern, nn::sdmmc::SectorSize);
} // NOLINT (readability/fn_size)


nn::Result OperationExecuter::PrepareWriteData(u8* targetBuffer, WriteReadPattern pattern, size_t pageAddress, size_t pageSize)
{
    u8 sectorBuffer[nn::sdmmc::SectorSize];
    for(u32 i = 0; i < pageSize; i++)
    {
        GenerateSectorData(sectorBuffer, pageAddress + i, pattern);
        memcpy(targetBuffer + i * U8SIZEOF(sectorBuffer), sectorBuffer, U8SIZEOF(sectorBuffer));
    }
    return nn::ResultSuccess();
}

bool OperationExecuter::IsReadDataMatch(u8* targetBuffer, WriteReadPattern pattern, size_t pageAddress, size_t pageSize)
{
    if(pattern == WriteReadPattern_NULL)
    {
        return true;
    }
    static const u32 unitLength = 16;
    u8 sectorBuffer[nn::sdmmc::SectorSize];
    for(u32 i = 0; i < pageSize; i++)
    {
        GenerateSectorData(sectorBuffer, pageAddress + i, pattern);
        if( memcmp(targetBuffer + i * U8SIZEOF(sectorBuffer), sectorBuffer, U8SIZEOF(sectorBuffer)) != 0)
        {
            NN_SC_DETAIL_ERR_LOG("Data mismatch! - address: 0x%lX\n", pageAddress + i * nn::sdmmc::SectorSize);
            NN_SC_DETAIL_ERR_LOG("expected:\n");
            // 以下で NN_SC_DETAIL_ERR_LOG や NN_SC_DETAIL_LOG_S を使用すると、エラー画面が見にくくなるため、NN_LOG のまま
            for(u32 jj= 0; jj < U8SIZEOF(sectorBuffer) / unitLength; jj++)
            {
                NN_LOG("  ");
                for(u32 kkk = 0; kkk < unitLength; kkk++)
                {
                    NN_LOG("0x%02x ", sectorBuffer[jj * unitLength + kkk]);
                }
                NN_LOG("\n");
            }
            NN_SC_DETAIL_ERR_LOG("actual:\n");
            for(u32 jj = 0; jj < U8SIZEOF(sectorBuffer) / unitLength; jj++)
            {
                NN_LOG("  ");
                for(u32 kkk = 0; kkk < unitLength; kkk++)
                {
                    NN_LOG("0x%02x ", targetBuffer[i * U8SIZEOF(sectorBuffer) + jj * unitLength + kkk]);
                }
                NN_LOG("\n");
            }
            return false;
        }
    }
    NN_SC_DETAIL_DEBUG_LOG("Data matched!\n");
    return true;
}

// TODO: StorageChecker.cpp の ReadSequenceFileFromHostIO と合併？
// 少なくとも targetPath とかはまとめて定義したい
void WriteToHost(const u8* buffer, const size_t bufferLength, const char* fileName)
{
    if(bufferLength == 0)
    {
        NN_SC_DETAIL_ERR_LOG("NO BUFFER\n");
    }
    FileAccessor& fileAccessor = FileAccessor::GetInstance();
    NN_ABORT_UNLESS(fileAccessor.IsMounted());
    fileAccessor.Write(buffer, bufferLength, fileAccessor.defaultMountPath.GetMountPath(std::string(fileName)).c_str());
}

void ReadFromHost(u8* outBuffer, size_t* outReadLength, const size_t bufferLength, const char* fileName)
{
    FileAccessor& fileAccessor = FileAccessor::GetInstance();
    NN_ABORT_UNLESS(fileAccessor.IsMounted());
    fileAccessor.Read(outBuffer, outReadLength, bufferLength, fileAccessor.defaultMountPath.GetMountPath(std::string(fileName)).c_str());
}

void PrintWriteReadInfo(ArgumentTypeWriteRead arg, bool isForce)
{
    if(isForce || false)
    {
        NN_LOG("Info: Write Read Data Detail\n");
        NN_LOG("pageStart   : 0x%0x\n", arg.pageStart);
        NN_LOG("pageCount   : %d\n", arg.pageCount);
        NN_LOG("pageUnit    : %d\n", arg.pageUnit);
        NN_LOG("mark        : %d\n", arg.mark); // TODO:
        NN_LOG("AccessType  : %d\n", arg.accessType);
        NN_LOG("dataPattern : %d\n", arg.dataPattern);
    }
}

//--------------------------------------------------------------------------
/*
 * Public API 定義
 */

nn::Result OperationExecuter::Power(OperationData* const pOperation)
{
    return nn::ResultSuccess();
}

nn::Result OperationExecuter::Select(OperationData* const pOperation)
{
    auto arg = pOperation->ParseArgument<ArgumentTypeU32>();
    u32 storageType = arg.arg;

    // 引数のチェック
    if(storageType  >= sizeof_StorageInterfaceType)
    {
        NN_SC_DETAIL_ERR_LOG("Invalid argument for Init\n");
        return nn::sdmmc::ResultNotSupported(); // TEMP:
    }
    // 終了処理：Deactivate をして data buffer を Unmap
    if(m_StorageInterfaceType != sizeof_StorageInterfaceType)
    {
        GetStorage().Deactivate();
        GetStorage().UnmapDataBuffer();
    }
    // ストレージを選択する
    m_StorageInterfaceType = (StorageInterfaceType)storageType;
    NN_SC_DETAIL_LOG("Storage [%s] selected\n", GetStorage().storageName.c_str());

    // data buffer を登録
    GetStorage().MapDataBuffer();
    return nn::ResultSuccess();
}

nn::Result OperationExecuter::Init(OperationData* const pOperation)
{
    nn::Result result = GetStorage().Activate();
    if(result.IsFailure())
    {
        pOperation->SetResult(OperationResult_Failure);
        NN_LOG("nn::sdmmc::Activate failed. Module:%d, Description:%d\n", result.GetModule(), result.GetDescription());
    }
    return result;
}

nn::Result OperationExecuter::Enable(OperationData* const pOperation)
{
    nn::Result result = GetStorage().Enable();
    if(result.IsFailure())
    {
        pOperation->SetResult(OperationResult_Failure);
        NN_ABORT("Enable operation failed. Module:%d, Description:%d\n", result.GetModule(), result.GetDescription());
    }
    return result;
}

nn::Result OperationExecuter::Seed(OperationData* const pOperation)
{
    const u32 seedValue = pOperation->ParseArgument<ArgumentTypeU32>().arg;
    srand(seedValue);
    return nn::ResultSuccess();
}

nn::Result OperationExecuter::Wait(OperationData* const pOperation)
{
    const u32 waitMicroSec = pOperation->ParseArgument<ArgumentTypeU32>().arg;
    nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(waitMicroSec));
    return nn::ResultSuccess();
}

nn::Result OperationExecuter::Skip(OperationData* const pOperation)
{
    m_SkipPageCount = pOperation->ParseArgument<ArgumentTypeU32>().arg;
    NN_SC_DETAIL_LOG2("skip count set to 0x%0x\n", m_SkipPageCount);
    return nn::ResultSuccess();
}

nn::Result OperationExecuter::WriteRead(OperationData* const pOperation)
{
    nn::Result result = nn::ResultSuccess();
    DisplayData* pDisplayData = DisplayData::GetInstance();
    auto arg = pOperation->ParseArgument<ArgumentTypeWriteRead>();
    Operation operationId = pOperation->operationId;
    OperationInterpreter& interpreter = OperationInterpreter::GetInstance();

    Time timer;
#ifdef SC_ENABLE_DEBUG_TEST_PRINT
    PrintWriteReadInfo(arg, true);
#endif

    // WriteRead で ALT の時は2回ずつアクセスする
    const u32 numTry = 1 + (u32)((operationId == Operation_WriteRead) && (arg.accessType == WriteReadAccessType_ALT));
    const u32 numLoop = arg.pageCount / arg.pageUnit;
    int64_t totalElapsedReadTime = 0;
    int64_t totalElapsedWriteTime = 0;
    int64_t minimumElapsedReadTime = INT64_MAX;
    int64_t maximumElapsedReadTime = 0;
    int64_t minimumElapsedWriteTime = INT64_MAX;
    int64_t maximumElapsedWriteTime = 0;

    // SEQ or DEC, not write_read のとき skip をチェック：skip がチャンクサイズの倍数なら有効
    u32 numIncrement = numTry;
    if((arg.accessType == WriteReadAccessType_SEQ) || (arg.accessType == WriteReadAccessType_DEC))
    {
        if((m_SkipPageCount > arg.pageUnit) && (m_SkipPageCount % arg.pageUnit == 0))
        {
            numIncrement = (m_SkipPageCount / arg.pageUnit);
            NN_SC_DETAIL_LOG("skip %d is applied - num increment: %d\n", m_SkipPageCount, numIncrement);
        }
    }

    pDisplayData->startSectorIndex = arg.pageStart;
    pDisplayData->totalSectorNum = arg.pageCount;

    // メインループ
    for(u32 i=0; i<numLoop; i+=numIncrement)
    {
        // * Write
        if(operationId == Operation_WriteRead || operationId == Operation_Write)
        {
            for(u32 jj=0; jj<numTry; jj++)
            {
                u32 startPage = GetAccessPage(arg, i + jj);

// 時間でログを出力、後退テスト時はオフ
#ifndef SC_LOG_ALWAYS_FULL
                if(interpreter.logTimer.GetElapsedTime() > m_LogIntervalUsec)
                {
                    interpreter.logTimer.Start();
                    if(interpreter.IsInLoop())
                    {
                        NN_LOG("Write ");
                        interpreter.PrintLoopIndex();
                        PrintWriteReadInfo(arg, true);
                    }
                    NN_SC_DETAIL_LOG2("write start page: 0x%0x\n", startPage);
                }
#endif

                {
                    // Write データの準備（！都度生成）
                    NN_RESULT_DO(PrepareWriteData(GetStorage().dataBuffer, (WriteReadPattern)arg.dataPattern, startPage, arg.pageUnit));
                    pDisplayData->currentSecoterIndex = startPage;
                    pDisplayData->chunkSize = arg.pageUnit;
                    timer.Start();
                    result = GetStorage().Write(startPage, arg.pageUnit, GetStorage().dataBuffer, GetStorage().dataBufferSize);
                    int64_t stopElapsedTime = timer.GetElapsedTime();
                    totalElapsedWriteTime += stopElapsedTime;
                    if(minimumElapsedWriteTime > stopElapsedTime)
                    {
                        minimumElapsedWriteTime = stopElapsedTime;
                    }
                    if(maximumElapsedWriteTime < stopElapsedTime)
                    {
                        maximumElapsedWriteTime = stopElapsedTime;
                    }
                }
                if(result.IsFailure())
                {
                    NN_SC_DETAIL_ERR_LOG("Write Error - Operation: start page: 0x%0x, size: 0x%0x\n", arg.pageStart, arg.pageCount);
                    NN_SC_DETAIL_ERR_LOG("        Storage Access: start page: 0x%0x, size: 0x%0x, (access count: %d, %d)\n", startPage, arg.pageUnit, i, jj);
                    NN_SC_DETAIL_ERR_LOG("<Storage Access: start page> 0x%0x = <Operation: start page> 0x%0x + <access count> 0x%0x * <Operation: size> 0x%0x\n", startPage, arg.pageStart, i + jj, arg.pageUnit);
                    NN_SC_DETAIL_ERR_LOG("");
                    interpreter.PrintLoopIndex();
                    PrintWriteReadInfo(arg, true);
                    return result;
                }
            }
        }

         // * Read
        if(operationId == Operation_WriteRead || operationId == Operation_Read)
        {
            for(u32 jj=0; jj<numTry; jj++)
            {
                u32 startPage = GetAccessPage(arg, i + jj);

// 時間でログを出力、後退テスト時はオフ
#ifndef SC_LOG_ALWAYS_FULL
                if(interpreter.logTimer.GetElapsedTime() > m_LogIntervalUsec)
                {
                    interpreter.logTimer.Start();
                    if(interpreter.IsInLoop())
                    {
                        NN_LOG("Read ");
                        interpreter.PrintLoopIndex();
                        PrintWriteReadInfo(arg, true);
                    }
                    NN_SC_DETAIL_LOG2("read start page: 0x%0x\n", startPage);
                }
#endif
                {
                    pDisplayData->currentSecoterIndex = startPage;
                    pDisplayData->chunkSize = arg.pageUnit;
                    timer.Start();
                    result = GetStorage().Read(GetStorage().dataBuffer, GetStorage().dataBufferSize, startPage, arg.pageUnit);
                    int64_t stopElapsedTime = timer.GetElapsedTime();
                    totalElapsedReadTime += stopElapsedTime;
                    if(minimumElapsedReadTime > stopElapsedTime)
                    {
                        minimumElapsedReadTime = stopElapsedTime;
                    }
                    if(maximumElapsedReadTime < stopElapsedTime)
                    {
                        maximumElapsedReadTime = stopElapsedTime;
                    }
                }
                if(result.IsFailure())
                {
                    NN_SC_DETAIL_ERR_LOG("Read Error - Operation: start page: 0x%0x, size: 0x%0x\n", arg.pageStart, arg.pageCount);
                    NN_SC_DETAIL_ERR_LOG("        Storage Access: start page: 0x%0x, size: 0x%0x, (aceess count: %d, %d - skip: %d)\n", startPage, arg.pageUnit, i, jj, m_SkipPageCount);
                    NN_SC_DETAIL_ERR_LOG("<Storage Access: start page> 0x%0x = <Operation: start page> 0x%0x + <access count> 0x%0x * <Operation: size> 0x%0x\n", arg.pageStart, startPage, i + jj, arg.pageUnit);
                    NN_SC_DETAIL_ERR_LOG("");
                    interpreter.PrintLoopIndex();
                    PrintWriteReadInfo(arg, true);
                    return result;
                }

                // データの比較結果が NG の場合も処理を中断する必要がある
                bool isMatched = IsReadDataMatch(GetStorage().dataBuffer, (WriteReadPattern)arg.dataPattern, startPage, arg.pageUnit);
                if(!isMatched)
                {
                    return nn::sdmmc::ResultNotSupported(); // TEMP:
                }
            }
        }
    }

    // 時間計測
    if(m_IsMeasure)
    {
        const float speedCoefficient = 488.28125;   // = (512 x 10^6) / (1024 * 1024)
        if(operationId == Operation_WriteRead || operationId == Operation_Write)
        {
            float speed = ((float)arg.pageCount * speedCoefficient) / (totalElapsedWriteTime * numIncrement);
            NN_SC_DETAIL_LOG("speed : %f, total time : %" PRId64 "\n",speed, totalElapsedWriteTime);
            m_WriteMeasuredData.AddData(m_OperationCount, arg.pageStart, arg.pageCount * nn::sdmmc::SectorSize, arg.pageUnit * nn::sdmmc::SectorSize, totalElapsedWriteTime, speed, minimumElapsedWriteTime, maximumElapsedWriteTime, operationId);
        }
        else if(operationId == Operation_WriteRead || operationId == Operation_Read)
        {
            float speed = ((float)arg.pageCount * speedCoefficient) / totalElapsedReadTime;
            NN_SC_DETAIL_LOG("speed : %f, total time : %" PRId64 "\n",speed, totalElapsedReadTime);
            m_ReadMeasuredData.AddData(m_OperationCount, arg.pageStart, arg.pageCount * nn::sdmmc::SectorSize, arg.pageUnit * nn::sdmmc::SectorSize, totalElapsedReadTime, speed , minimumElapsedReadTime, maximumElapsedReadTime, operationId);
        }
    }

    NN_RESULT_SUCCESS;
} // NOLINT (readability/fn_size)

nn::Result OperationExecuter::SecureErase(OperationData* const pOperation)
{
    return nn::ResultSuccess();
}


// outBufferLength は終端文字分の + 1 が必要
nn::Result CopyFileName(char* outBuffer, const u8* inStringBuffer, const size_t inStringLength, const size_t outBufferLength)
{
    if(inStringLength > 256 || outBufferLength < (inStringLength + 1))
    {
        NN_SC_DETAIL_ERR_LOG("File Path size is too long\n");
        // NOTE: ファイル書き出しで result failure にはしていない
        return nn::sdmmc::ResultNotSupported();
    }
    memcpy(outBuffer, inStringBuffer, inStringLength);
    outBuffer[inStringLength] = '\0';

    return nn::ResultSuccess();
}

nn::Result OperationExecuter::Load(OperationData* const pOperation)
{
    nn::Result result = nn::ResultSuccess();
    FileAccessor& fileAccessor = FileAccessor::GetInstance();
    auto arg = pOperation->ParseArgument<ArgumentDumpLoad>();

    // ファイル名を取得する
    char fileName[MAX_FILE_PATH_SIZE];
    NN_RESULT_DO(CopyFileName(fileName, pOperation->data.address, pOperation->data.size, sizeof(fileName)));
    std::string filePath = fileAccessor.defaultMountPath.GetMountPath(std::string(fileName));

    // ファイルサイズを計算する
    const int64_t fileSize = (int64_t)(arg.pageCount) * nn::sdmmc::SectorSize;

    // ファイルを開く
    FileHandle file;
    fileAccessor.Open(&file, filePath.c_str(), fileSize, OpenMode::OpenMode_Read);

    // 大きなファイルの処理
    const size_t memSize = 4 * 1024 * 1024;
    u8* buffer = reinterpret_cast<u8*>(malloc(memSize));

    // 繰り返し処理
    const size_t pageUnit = memSize / nn::sdmmc::SectorSize;
    size_t pageRemain = arg.pageCount;
    for(int i=0; pageRemain > 0; i++)
    {
        // 読み出し
        const size_t numReadPage = (pageRemain < pageUnit) ? pageRemain : pageUnit;
        const size_t readLength = numReadPage * nn::sdmmc::SectorSize;
        result = ReadFile(file, i * pageUnit * nn::sdmmc::SectorSize, buffer, readLength);
        if(result.IsFailure())
        {
            NN_SC_DETAIL_ERR_LOG("Read File '%s' Error - address: 0x%0lx, size: 0x%0zx\n", filePath.c_str(), i * pageUnit * nn::sdmmc::SectorSize, readLength);
            return result;
        }
        pageRemain -= numReadPage;

        // 書き出し
        memcpy(GetStorage().dataBuffer, buffer, readLength);
        const uint32_t pageAddress = (uint32_t)((int)(arg.pageStart) + i * pageUnit);   // load のみマイナスの値を許容する（GC のみ関係する）
        result = GetStorage().Write(pageAddress, numReadPage, GetStorage().dataBuffer, GetStorage().dataBufferSize);
        if(result.IsFailure())
        {
            NN_SC_DETAIL_ERR_LOG("Write Error - address: 0x%0x, size: 0x%0zx\n", pageAddress, numReadPage);
        }
        NN_SC_DETAIL_LOG2("wrote at 0x0%x and dumped (%lu/%d) page to %s\n", pageAddress, numReadPage + i * pageUnit, arg.pageCount, filePath.c_str());
    }

    free(buffer);
    CloseFile(file);
    return result;
}

nn::Result OperationExecuter::Dump(OperationData* const pOperation)
{
    nn::Result result = nn::ResultSuccess();
    FileAccessor& fileAccessor = FileAccessor::GetInstance();
    auto arg = pOperation->ParseArgument<ArgumentDumpLoad>();

    // ファイル名を取得する
    char fileName[MAX_FILE_PATH_SIZE];
    NN_RESULT_DO(CopyFileName(fileName, pOperation->data.address, pOperation->data.size, sizeof(fileName)));
    std::string filePath = fileAccessor.defaultMountPath.GetMountPath(std::string(fileName));

    // ファイルを開く
    FileHandle file;
    fileAccessor.Open(&file, filePath.c_str(), 0, OpenMode(OpenMode::OpenMode_Write + OpenMode::OpenMode_AllowAppend));
    auto option = WriteOption::MakeValue(WriteOptionFlag::WriteOptionFlag_Flush);

    // 大きなファイルの処理
    const size_t memSize = 4 * 1024 * 1024;
    u8* buffer = reinterpret_cast<u8*>(malloc(memSize));

    // 繰り返し処理
    const size_t pageUnit = memSize / nn::sdmmc::SectorSize;
    size_t pageRemain = arg.pageCount;
    for(int i=0; pageRemain > 0; i++)
    {
        // 読み出し
        const size_t numReadPage = (pageRemain < pageUnit) ? pageRemain : pageUnit;
        const size_t pageAddress = arg.pageStart + i * pageUnit;

        DisplayData* pDisplayData = DisplayData::GetInstance();
        pDisplayData->startSectorIndex = pageAddress;
        pDisplayData->totalSectorNum   = numReadPage;

        result = GetStorage().Read(GetStorage().dataBuffer, GetStorage().dataBufferSize, pageAddress, numReadPage);
        if(result.IsFailure())
        {
            NN_SC_DETAIL_ERR_LOG("Read Error - address: 0x%0zx, size: 0x%0zx\n", pageAddress, numReadPage);
            return result;
        }
        pageRemain -= numReadPage;

        // 書き出し
        const size_t readLength = numReadPage * nn::sdmmc::SectorSize;
        memcpy(buffer, GetStorage().dataBuffer, readLength);
        result = WriteFile(file, i * pageUnit * nn::sdmmc::SectorSize, buffer, readLength, option);
        if(result.IsFailure())
        {
            NN_SC_DETAIL_ERR_LOG("Write File '%s' Failure\n", filePath.c_str());
        }
        NN_SC_DETAIL_LOG2("read at 0x0%zx and dumped (%zd/%d) page to %s\n", pageAddress, numReadPage + i * pageUnit, arg.pageCount, filePath.c_str());
    }

    free(buffer);
    CloseFile(file);
    return result;
}

nn::Result OperationExecuter::Insert(OperationData* const pOperation)
{
    auto arg = pOperation->ParseArgument<ArgumentTypeU8>();
    InsertDirection direction = (InsertDirection)(arg.arg);
    return GetStorage().Insert(direction);
}

nn::Result OperationExecuter::Pause(OperationData* const pOperation)
{
    // TODO:
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));
    return nn::ResultSuccess();
}

nn::Result OperationExecuter::Register(OperationData* const pOperation)
{
    return nn::ResultSuccess();
}

nn::Result OperationExecuter::MeasureBegin(OperationData* const pOperation)
{
    m_IsMeasure = true;
    return nn::ResultSuccess();
}

nn::Result OperationExecuter::MeasureEnd(OperationData* const pOperation)
{
    m_IsMeasure = false;
    if(pOperation->data.size > 256)
    {
        NN_SC_DETAIL_ERR_LOG("File Path size is too long\n");
    }
    u8 size = pOperation->data.size;
    char fileName[MAX_FILE_PATH_SIZE];
    memcpy(fileName, pOperation->data.address, size);
    fileName[size] = '\0';

    size_t bufSize = MeasuredDataManager::GetOutputBufferHeader(m_WorkBuffer);
    bufSize += m_ReadMeasuredData.GetOutputBuffer(m_WorkBuffer + bufSize);
    bufSize += m_WriteMeasuredData.GetOutputBuffer(m_WorkBuffer + bufSize);
    if(bufSize != 0)
    {
        WriteToHost(m_WorkBuffer, bufSize, fileName);
    }
    NN_SC_DETAIL_LOG2("measured: output to %s\n", fileName);

    // NOTE: ファイル書き出しで result failure にはしていない
    return nn::ResultSuccess();
}

nn::Result OperationExecuter::Log(OperationData* const pOperation)
{
    auto arg = pOperation->ParseArgument<ArgumentTypeU32>();
    int logOption = arg.arg;
    if( logOption != 0 )
    {
        NN_SC_DETAIL_LOG2("%s\n", pOperation->data.address);
    }
    return nn::ResultSuccess();
}
nn::Result OperationExecuter::Erase(OperationData* const pOperation)
{
    return GetStorage().Erase();
}

nn::Result OperationExecuter::Speed(OperationData* const pOperation)
{
    // 速度を選択（念のため index を StorageChecker 内で FIX しておく）
    auto arg = pOperation->ParseArgument<ArgumentTypeU32>();
    nn::sdmmc::SpeedMode speedModeList[] = {
        nn::sdmmc::SpeedMode_MmcIdentification,
        nn::sdmmc::SpeedMode_MmcLegacySpeed,
        nn::sdmmc::SpeedMode_MmcHighSpeed,
        nn::sdmmc::SpeedMode_MmcHs200,
        nn::sdmmc::SpeedMode_MmcHs400,
        nn::sdmmc::SpeedMode_SdCardDefaultSpeed,
        nn::sdmmc::SpeedMode_SdCardHighSpeed,
        nn::sdmmc::SpeedMode_SdCardSdr12,
        nn::sdmmc::SpeedMode_SdCardSdr25,
        nn::sdmmc::SpeedMode_SdCardSdr50,
        nn::sdmmc::SpeedMode_SdCardSdr104,
        nn::sdmmc::SpeedMode_SdCardDdr50,
        nn::sdmmc::SpeedMode_GcAsicFpgaSpeed,
        nn::sdmmc::SpeedMode_GcAsicSpeed,
        nn::sdmmc::SpeedMode_SdCardIdentification,
    };
    nn::sdmmc::SpeedMode targetSpeedMode = nn::sdmmc::SpeedMode_MmcIdentification;
    if(arg.arg < (sizeof(speedModeList) / sizeof(nn::sdmmc::SpeedMode)))
    {
        targetSpeedMode = speedModeList[arg.arg];
    }
    else
    {
        NN_SC_DETAIL_ERR_LOG("Unknown speed mode %d ... force interpret.\n", arg.arg);
        targetSpeedMode = (nn::sdmmc::SpeedMode)(arg.arg);
    }

    return GetStorage().Speed(targetSpeedMode);
}

nn::Result OperationExecuter::Partition(OperationData* const pOperation)
{
    // 対象ストレージチェック
    auto type = GetStorage().GetStorageType();
    if(type != StorageInterfaceType_NAND)
    {
        NN_SC_DETAIL_ERR_LOG("Operation 'partition' does not support storage type %d.\n", type);
        return nn::sdmmc::ResultNotSupported();
    }

    // パーティションを選択（念のため index を StorageChecker 内で FIX しておく）
    auto arg = pOperation->ParseArgument<ArgumentTypeU32>();
    nn::sdmmc::MmcPartition partitionList[] = {
        nn::sdmmc::MmcPartition_UserData,
        nn::sdmmc::MmcPartition_BootPartition1,
        nn::sdmmc::MmcPartition_BootPartition2,
        nn::sdmmc::MmcPartition_Unknown
    };
    nn::sdmmc::MmcPartition targetPartition = nn::sdmmc::MmcPartition_UserData;
    if(arg.arg < (sizeof(partitionList) / sizeof(nn::sdmmc::MmcPartition)))
    {
        targetPartition = partitionList[arg.arg];
    }
    else
    {
        NN_SC_DETAIL_ERR_LOG("Unknown mmc partition %d ... force interpret.\n", arg.arg);
        targetPartition = (nn::sdmmc::MmcPartition)(arg.arg);
    }

    NN_SC_DETAIL_LOG2("partition %d will be selected\n", targetPartition);
    return GetStorage().SelectPartition(targetPartition);
}

nn::Result OperationExecuter::Reg(OperationData* const pOperation)
{
    return GetStorage().Reg();
}

nn::Result OperationExecuter::WriteParam(OperationData* const pOperation)
{
    auto arg = pOperation->ParseArgument<ArgumentTypeU32_2>();
    return GetStorage().WriteParameter((::WriteParam)arg.arg1, arg.arg2);
}

nn::Result OperationExecuter::Command(OperationData* const pOperation)
{
    NN_SC_DETAIL_LOG2("command: \"%s\"\n", pOperation->data.address);

    static char keyLogInterval[] = "loginterval";
    static char keyVersion[] = "version";

    if( NN_SC_DETAIL_OPERATION_DATA_STARTS_WITH(pOperation, keyLogInterval) )
    {
        int arg = std::atoi(reinterpret_cast<char*>(pOperation->data.address) + sizeof(keyLogInterval)); // null 終端を含むが、ちょうどデリミタ ':' にかぶさっていい感じ
        m_LogIntervalUsec = arg * 1000;
        NN_SC_DETAIL_LOG2("Log Interval: %d ms\n", arg);
    }
    else if( NN_SC_DETAIL_OPERATION_DATA_STARTS_WITH(pOperation, keyVersion) )
    {
        NN_LOG("%s\n", g_VersionInfo);
    }
    else
    {
        NN_SC_DETAIL_ERR_LOG("Unknown command\n");
    }
    return nn::ResultSuccess();
}

nn::Result OperationExecuter::GameCardCommand(OperationData* const pOperation)
{
    auto arg = pOperation->ParseArgument<ArgumentTypeU32_3>();
    switch(arg.arg1)
    {
    case 0: return SendReadCardId1();
    case 1: return SendReadCardId2();
    case 2: return SendReadCardId3();
    case 3: return SendReadCardUid();
    case 4: return SendReadRefresh();
    case 5: return SendReadSetKey();
    case 6: return SendReadSelfRefresh();
    case 7: return SendReadRefreshStatus();
    case 8: return SendReadPage(arg.arg2, arg.arg3);
    default:
        break;
    }
    return nn::ResultSuccess();
}


nn::Result OperationExecuter::SendReadCardId1()
{
    nn::gc::detail::CardId1 cardId1;
    nn::gc::detail::AsicOperation& asicOperation = nn::gc::detail::AsicOperation::GetInstance();
    DisplayData* pDisplayData = DisplayData::GetInstance();
    nn::Result result = asicOperation.SendCardReadId1(&cardId1);
    memcpy(GetStorage().dataBuffer, &cardId1, sizeof(nn::gc::detail::CardId1));
    pDisplayData->bufferSizeForGameCardCommand =  sizeof(nn::gc::detail::CardId1);
    return result;
}

nn::Result OperationExecuter::SendReadCardId2()
{
    nn::gc::detail::CardId2 cardId2;
    nn::gc::detail::AsicOperation& asicOperation = nn::gc::detail::AsicOperation::GetInstance();
    DisplayData* pDisplayData = DisplayData::GetInstance();
    nn::Result result = asicOperation.SendCardReadId2(&cardId2);
    memcpy(GetStorage().dataBuffer, &cardId2, sizeof(nn::gc::detail::CardId2));
    pDisplayData->bufferSizeForGameCardCommand =  sizeof(nn::gc::detail::CardId2);
    return result;
}

nn::Result OperationExecuter::SendReadCardId3()
{
    nn::gc::detail::CardId3 cardId3;
    nn::gc::detail::AsicOperation& asicOperation = nn::gc::detail::AsicOperation::GetInstance();
    DisplayData* pDisplayData = DisplayData::GetInstance();
    nn::Result result = asicOperation.SendCardReadId3(&cardId3);
    memcpy(GetStorage().dataBuffer, &cardId3, sizeof(nn::gc::detail::CardId3));
    pDisplayData->bufferSizeForGameCardCommand =  sizeof(nn::gc::detail::CardId3);
    return result;
}

nn::Result OperationExecuter::SendReadCardUid()
{
    nn::gc::detail::CardUid cardUid;
    nn::gc::detail::AsicOperation& asicOperation = nn::gc::detail::AsicOperation::GetInstance();
    DisplayData* pDisplayData = DisplayData::GetInstance();
    nn::Result result = asicOperation.SendCardReadUid(&cardUid);
    memcpy(GetStorage().dataBuffer, &cardUid, sizeof(nn::gc::detail::CardUid));
    pDisplayData->bufferSizeForGameCardCommand =  sizeof(nn::gc::detail::CardUid);
    return result;
}

nn::Result OperationExecuter::SendReadRefresh()
{
    nn::gc::detail::RefreshResponse refreshResp;
    nn::gc::detail::AsicOperation& asicOperation = nn::gc::detail::AsicOperation::GetInstance();
    DisplayData* pDisplayData = DisplayData::GetInstance();
    nn::Result result = asicOperation.SendCardRefresh(&refreshResp);
    memcpy(GetStorage().dataBuffer, &refreshResp, sizeof(nn::gc::detail::RefreshResponse));
    pDisplayData->bufferSizeForGameCardCommand =  sizeof(nn::gc::detail::RefreshResponse);
    return result;
}

nn::Result OperationExecuter::SendReadSetKey()
{
    nn::gc::detail::AsicOperation& asicOperation = nn::gc::detail::AsicOperation::GetInstance();
    DisplayData* pDisplayData = DisplayData::GetInstance();
    nn::Result result = asicOperation.SendCardSetKey();
    pDisplayData->bufferSizeForGameCardCommand =  0;
    return result;
}

nn::Result OperationExecuter::SendReadSelfRefresh()
{
    nn::gc::detail::SelfRefreshResponse selfRefreshResponse;
    nn::gc::detail::AsicOperation& asicOperation = nn::gc::detail::AsicOperation::GetInstance();
    DisplayData* pDisplayData = DisplayData::GetInstance();
    nn::Result result = asicOperation.SendCardSelfRefresh(&selfRefreshResponse);
    memcpy(GetStorage().dataBuffer, &selfRefreshResponse, sizeof(nn::gc::detail::SelfRefreshResponse));
    pDisplayData->bufferSizeForGameCardCommand =  sizeof(nn::gc::detail::SelfRefreshResponse);
    return result;
}

nn::Result OperationExecuter::SendReadRefreshStatus()
{
    nn::gc::detail::SelfRefreshStatus selfRefreshStatus;
    nn::gc::detail::AsicOperation& asicOperation = nn::gc::detail::AsicOperation::GetInstance();
    DisplayData* pDisplayData = DisplayData::GetInstance();
    nn::Result result = asicOperation.SendCardReadRefreshStatus(&selfRefreshStatus);
    memcpy(GetStorage().dataBuffer, &selfRefreshStatus, sizeof(nn::gc::detail::SelfRefreshStatus));
    pDisplayData->bufferSizeForGameCardCommand =  sizeof(nn::gc::detail::SelfRefreshStatus);
    return result;
}

nn::Result OperationExecuter::SendReadPage(u32 pageAddress, u32 pageCount)
{
    DisplayData* pDisplayData = DisplayData::GetInstance();
    nn::Result result = GetStorage().Read(GetStorage().dataBuffer, GetStorage().dataBufferSize, pageAddress, pageCount);
    pDisplayData->bufferSizeForGameCardCommand =  pageCount * 512;
    return result;
}
