﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <random>
#include <cinttypes>
#include <nn/os.h>
#include <nn/os/os_MemoryHeap.h>
#include <nn/init.h>
#include <nn/fs/fs_SdmmcControl.h>
#include <nn/nn_Log.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sdmmc/sdmmc_Mmc.h>
#include <nn/sdmmc/sdmmc_SdCard.h>
#ifdef NN_SDMMC_TEST_SMMU_ENABLE
    #include <nn/sdmmc/sdmmc_DeviceVirtualAddress.h>
#endif
#include "SdmmcCommon.h"

#define MMC_AGING_MODE
//#define SDCARD_AGING_MODE

//#define ENABLE_ACCESS_LOG

namespace
{
    #ifdef MMC_AGING_MODE
        const size_t WorkBufferSize = nn::sdmmc::MmcWorkBufferSize;
    #else   // SDCARD_AGING_MODE
        const size_t WorkBufferSize = nn::sdmmc::SdCardWorkBufferSize;
    #endif

    #define SDMMC_DETAIL_CEILING(value, unit)   ((((value) + (unit) - 1) / (unit)) * (unit))
    #define SDMMC_DETAIL_CEILING_FOR_DEVICE_ADDRESS_SPACE(value)    SDMMC_DETAIL_CEILING((value), nn::dd::DeviceAddressSpaceMemoryRegionAlignment)
    NN_DD_ALIGNAS_DEVICE_ADDRESS_SPACE_MEMORY uint8_t g_WorkBuffer[SDMMC_DETAIL_CEILING_FOR_DEVICE_ADDRESS_SPACE(WorkBufferSize)];

    const size_t   ChunkSizeMinByte  = 512;                   // このチャンクサイズからエージングを始める
    const size_t   ChunkSizeMaxByte  = 8 * 1024 * 1024;       // このチャンクサイズまでエージングを続ける
    const uint32_t CheckMaxCount     = 128;                   // チャンクあたりのアクセス数
    const size_t   DataBufferSize    = 2 * ChunkSizeMaxByte;  // 作業バッファのサイズは最大チャンクサイズの 2 倍とする
    const uint32_t AccessAlignment   = 16 * 1024;             // アクセス時に守るアライメント
    #ifdef MMC_AGING_MODE
        const uint32_t SystemSectorCount = 0x400000;              // MMC の場合にシステムがあると思われるセクタ数
    #endif

    #ifdef NN_SDMMC_TEST_SMMU_ENABLE
        nn::dd::DeviceAddressSpaceType  g_Das;
        nn::dd::DeviceAddressSpaceMapInfo g_Info;
        nn::dd::DeviceVirtualAddress g_WorkBufferDeviceVirtualAddress;
        nn::dd::DeviceVirtualAddress g_DataBufferDeviceVirtualAddress;
    #endif

    struct AgingConfig
    {
        nn::sdmmc::Port            port;
        uint32_t                    offset;
        size_t                      dataBufferSize;
        uintptr_t                   dataBufferAddress;
        #ifdef NN_SDMMC_TEST_SMMU_ENABLE
            nn::dd::DeviceName          name;
        #endif
    };

    struct MeasuredTicks
    {
        nn::os::Tick min;
        nn::os::Tick max;
        nn::os::Tick sum;
    };

    struct Measurements
    {
        int64_t speed;
        int64_t iops;
        int64_t min;
        int64_t max;
    };

    void SetConfig(AgingConfig* pConfig)
    {
        #ifdef MMC_AGING_MODE
            pConfig->port   = nn::sdmmc::Port_Mmc0;
            pConfig->offset = SystemSectorCount;
        #elif defined SDCARD_AGING_MODE
            pConfig->port   = nn::sdmmc::Port_SdCard0;
            pConfig->offset = 0;
        #else
            #error "Please select the target device"
        #endif

        #ifdef NN_SDMMC_TEST_SMMU_ENABLE
            pConfig->name = (pConfig->port == nn::sdmmc::Port_SdCard0) ?
                nn::dd::DeviceName::DeviceName_Sdmmc1a : nn::dd::DeviceName::DeviceName_Sdmmc4a;
                //nn::dd::DeviceName::DeviceName_Sdmmc1a : nn::dd::DeviceName::DeviceName_Sdmmc2a;
        #endif
    }

    nn::Result Initialize(AgingConfig* pConfig)
    {
        // クロック制御は pcv 経由で行う
        nn::sdmmc::SwitchToPcvClockResetControl();

        nn::sdmmc::Initialize(pConfig->port);

        #ifdef NN_SDMMC_TEST_SMMU_ENABLE
            SetupDeviceAddressSpace(&g_Das, pConfig->name);
            nn::dd::DeviceVirtualAddress deviceVirtualAddressOffset = 0;

            g_WorkBufferDeviceVirtualAddress = MapDeviceAddressSpaceAligned(&g_Das, &g_Info,
                reinterpret_cast<uintptr_t>(g_WorkBuffer), sizeof(g_WorkBuffer), deviceVirtualAddressOffset);
            deviceVirtualAddressOffset = g_WorkBufferDeviceVirtualAddress + sizeof(g_WorkBuffer);
            nn::sdmmc::RegisterDeviceVirtualAddress(pConfig->port, reinterpret_cast<uintptr_t>(g_WorkBuffer), sizeof(g_WorkBuffer), g_WorkBufferDeviceVirtualAddress);
        #endif
        if (pConfig->port == nn::sdmmc::Port_Mmc0)
        {
            nn::sdmmc::SetMmcWorkBuffer(pConfig->port, g_WorkBuffer, sizeof(g_WorkBuffer));
        }
        else
        {
            nn::sdmmc::SetSdCardWorkBuffer(pConfig->port, g_WorkBuffer, sizeof(g_WorkBuffer));
        }

        pConfig->dataBufferSize = nn::util::align_up(DataBufferSize, nn::os::MemoryBlockUnitSize);
        NN_RESULT_DO(nn::os::AllocateMemoryBlock(&pConfig->dataBufferAddress, pConfig->dataBufferSize));
        #ifdef NN_SDMMC_TEST_SMMU_ENABLE
            g_DataBufferDeviceVirtualAddress = MapDeviceAddressSpaceAligned(&g_Das, pConfig->dataBufferAddress, pConfig->dataBufferSize, deviceVirtualAddressOffset);
            nn::sdmmc::RegisterDeviceVirtualAddress(pConfig->port, pConfig->dataBufferAddress, pConfig->dataBufferSize, g_DataBufferDeviceVirtualAddress);
        #endif

        NN_RESULT_SUCCESS;
    }

    void Finalize(AgingConfig* pConfig)
    {
        #ifdef NN_SDMMC_TEST_SMMU_ENABLE
            nn::sdmmc::UnregisterDeviceVirtualAddress(pConfig->port, pConfig->dataBufferAddress, pConfig->dataBufferSize, g_DataBufferDeviceVirtualAddress);
            UnmapDeviceAddressSpaceAligned(&g_Das, pConfig->dataBufferAddress, pConfig->dataBufferSize, g_DataBufferDeviceVirtualAddress);
        #endif
        nn::os::FreeMemoryBlock(pConfig->dataBufferAddress, pConfig->dataBufferSize);

        #ifdef NN_SDMMC_TEST_SMMU_ENABLE
            nn::sdmmc::UnregisterDeviceVirtualAddress(pConfig->port, reinterpret_cast<uintptr_t>(g_WorkBuffer), sizeof(g_WorkBuffer), g_WorkBufferDeviceVirtualAddress);
            UnmapDeviceAddressSpaceAligned(&g_Info);

            CleanDeviceAddressSpace(&g_Das, pConfig->name);
        #endif

        nn::sdmmc::Finalize(pConfig->port);
    }

    nn::Result ReadSectorMaxCount(uint32_t* pOutSectorMaxCount, AgingConfig* pConfig)
    {
        NN_RESULT_DO(nn::sdmmc::GetDeviceMemoryCapacity(pOutSectorMaxCount, pConfig->port));
        NN_LOG("Memory Capacity %u (sectors)\n", *pOutSectorMaxCount);
        NN_LOG("Memory Capacity %u (MB)\n", *pOutSectorMaxCount / 2 / 1024);
        NN_RESULT_SUCCESS;
    }

    nn::Result Activate(AgingConfig* pConfig)
    {
        NN_RESULT_DO(nn::sdmmc::Activate(pConfig->port));

        nn::sdmmc::ErrorInfo errorInfo;
        size_t logSize;
        char logBuffer[256];
        nn::sdmmc::GetAndClearErrorInfo(&errorInfo, &logSize, logBuffer, sizeof(logBuffer), pConfig->port);
        if ((errorInfo.numActivationFailures != 0) || (errorInfo.numActivationErrorCorrections != 0))
        {
            NN_ABORT("Activate had hidden error. %s\n", logBuffer);
        }

        NN_RESULT_SUCCESS;
    }

    nn::Result GetReadTick(uint8_t* pOutBuffer, nn::os::Tick* diffTick, uint32_t sectorIndex, size_t chunkSize, nn::sdmmc::Port port)
    {
        uint32_t sectorCount = chunkSize / nn::sdmmc::SectorSize;
        nn::os::Tick tick = nn::os::GetSystemTick();
        NN_RESULT_DO(nn::sdmmc::Read(pOutBuffer, sectorCount * nn::sdmmc::SectorSize, port, sectorIndex, sectorCount));
        *diffTick = nn::os::GetSystemTick() - tick;
        #ifdef ENABLE_ACCESS_LOG
            NN_LOG("addr = 0x%08X, data = 0x%02X%02X%02X%02X\n",
                   sectorIndex, pOutBuffer[0], pOutBuffer[1], pOutBuffer[2], pOutBuffer[3]);
        #endif

        nn::sdmmc::ErrorInfo errorInfo;
        size_t logSize;
        char logBuffer[256];
        nn::sdmmc::GetAndClearErrorInfo(&errorInfo, &logSize, logBuffer, sizeof(logBuffer), port);
        if ((errorInfo.numReadWriteFailures != 0) || (errorInfo.numReadWriteErrorCorrections != 0))
        {
            NN_ABORT("Read had hidden error. %s\n", logBuffer);
        }

        NN_RESULT_SUCCESS;
    }

    nn::Result GetWriteTick(const uint8_t* pBuffer, nn::os::Tick* diffTick, uint32_t sectorIndex, size_t chunkSize, nn::sdmmc::Port port)
    {
        #ifdef ENABLE_ACCESS_LOG
            NN_LOG("addr = 0x%08X, data = 0x%02X%02X%02X%02X\n",
                   sectorIndex, pBuffer[0], pBuffer[1], pBuffer[2], pBuffer[3]);
        #endif

        uint32_t sectorCount = chunkSize / nn::sdmmc::SectorSize;
        nn::os::Tick tick = nn::os::GetSystemTick();
        NN_RESULT_DO(nn::sdmmc::Write(port, sectorIndex, sectorCount, pBuffer, sectorCount * nn::sdmmc::SectorSize));
        *diffTick = nn::os::GetSystemTick() - tick;

        nn::sdmmc::ErrorInfo errorInfo;
        size_t logSize;
        char logBuffer[256];
        nn::sdmmc::GetAndClearErrorInfo(&errorInfo, &logSize, logBuffer, sizeof(logBuffer), port);
        if ((errorInfo.numReadWriteFailures != 0) || (errorInfo.numReadWriteErrorCorrections != 0))
        {
            NN_ABORT("Write had hidden error. %s\n", logBuffer);
        }

        NN_RESULT_SUCCESS;
    }

    nn::Result Write(uint8_t* pOutBuffer, nn::os::Tick* pTick, uint32_t sectorIndex, size_t chunkSizeByte, AgingConfig* pConfig)
    {
        NN_RESULT_DO(GetWriteTick(pOutBuffer, pTick, sectorIndex, chunkSizeByte, pConfig->port));
        NN_RESULT_SUCCESS;
    }

    nn::Result Read(uint8_t* pOutBuffer, nn::os::Tick* pTick, uint32_t sectorIndex, size_t chunkSizeByte, AgingConfig* pConfig)
    {
        NN_RESULT_DO(GetReadTick(pOutBuffer, pTick, sectorIndex, chunkSizeByte, pConfig->port));
        NN_RESULT_SUCCESS;
    }

    void Verify(uint8_t* pWriteBuffer, uint8_t* pReadBuffer, uint32_t sectorIndex, size_t size)
    {
        for (uint32_t byte = 0; byte < static_cast<uint32_t>(size); byte++)
        {
            NN_ABORT_UNLESS(pWriteBuffer[byte] == pReadBuffer[byte],
                            "0x%08X sector, 0x%08X byte, expected 0x%02X, actual 0x%02X",
                            sectorIndex, byte, pWriteBuffer[byte], pReadBuffer[byte]);
        }
    }

    void Deactivate(AgingConfig* pConfig)
    {
        nn::sdmmc::Deactivate(pConfig->port);
    }

    void DiaplayMeasurements(MeasuredTicks writeTicks, MeasuredTicks readTicks, size_t chunkSizeByte)
    {
        Measurements writeMeasurements;
        writeMeasurements.speed =
            ((static_cast<int64_t>(chunkSizeByte) * CheckMaxCount * 1000 * 1000 * 1000)
             / ConvertToTimeSpan(writeTicks.sum).GetNanoSeconds()) / 1024; // X KByte/s
        writeMeasurements.iops =
            (static_cast<int64_t>(CheckMaxCount) * 1000 * 1000 * 1000)
            / ConvertToTimeSpan(writeTicks.sum).GetNanoSeconds();
        writeMeasurements.min = ConvertToTimeSpan(writeTicks.min).GetMicroSeconds();
        writeMeasurements.max = ConvertToTimeSpan(writeTicks.max).GetMicroSeconds();

        Measurements readMeasurements;
        readMeasurements.speed =
            ((static_cast<int64_t>(chunkSizeByte) * CheckMaxCount * 1000 * 1000 * 1000)
             / ConvertToTimeSpan(readTicks.sum).GetNanoSeconds()) / 1024; // X KByte/s
        readMeasurements.iops =
            (static_cast<int64_t>(CheckMaxCount) * 1000 * 1000 * 1000)
            / ConvertToTimeSpan(readTicks.sum).GetNanoSeconds();
        readMeasurements.min = ConvertToTimeSpan(readTicks.min).GetMicroSeconds();
        readMeasurements.max = ConvertToTimeSpan(readTicks.max).GetMicroSeconds();

        // パフォーマンス表示
        NN_LOG("W %.1f %" PRIu64 " %" PRId64 " %" PRId64 " %" PRId64 "\n",
               chunkSizeByte / 1024.0, writeMeasurements.speed, writeMeasurements.iops,
               writeMeasurements.min, writeMeasurements.max);

        NN_LOG("R %.1f %" PRId64 " %" PRId64 " %" PRId64 " %" PRId64 "\n",
               chunkSizeByte / 1024.0, readMeasurements.speed, readMeasurements.iops,
               readMeasurements.min, readMeasurements.max);
    }

    nn::Result SequentialAging(AgingConfig* pConfig, bool isPositive)
    {
        // 事前準備
        Initialize(pConfig);

        // 初期化処理
        NN_RESULT_DO(Activate(pConfig));

        // バス情報の表示
        LogBusStatus(pConfig->port);

        // デバイスサイズの取得
        uint32_t sectorMaxCount;
        NN_RESULT_DO(ReadSectorMaxCount(&sectorMaxCount, pConfig));

        // 測定結果表示のラベル
        NN_LOG("Chunk(KB) KB/sec IOPS min(usec) max(usec)\n");

        // アクセスするチャンクサイズを 2 倍ずつ増加させる
        for (size_t chunkSizeByte = ChunkSizeMinByte; chunkSizeByte <= ChunkSizeMaxByte; chunkSizeByte *= 2)
        {
            MeasuredTicks writeTicks = { nn::os::Tick(0), nn::os::Tick(0), nn::os::Tick(0) };
            MeasuredTicks readTicks  = { nn::os::Tick(0), nn::os::Tick(0), nn::os::Tick(0) };

            // CheckMaxCount 回のアクセスを行う
            for (uint32_t count = 0; count < CheckMaxCount; count++)
            {
                uint8_t* pWriteBuffer = reinterpret_cast<uint8_t*>(pConfig->dataBufferAddress);
                uint8_t* pReadBuffer  = reinterpret_cast<uint8_t*>(pConfig->dataBufferAddress + pConfig->dataBufferSize / 2);

                // アクセスアライメントより小さいチャンクサイズのときはアライメントに合わせる
                uint32_t stepSectorCount = ((chunkSizeByte < AccessAlignment) ? AccessAlignment : chunkSizeByte) / nn::sdmmc::SectorSize;
                uint32_t sectorIndex =  count * stepSectorCount + pConfig->offset;

                // アクセス先がデバイスサイズを超えたらループを抜ける
                if (sectorIndex + (chunkSizeByte / nn::sdmmc::SectorSize) >  sectorMaxCount)
                {
                    break;
                }

                // テストパターンを生成
                srand(sectorIndex);
                for (uint32_t byte = 0; byte < static_cast<uint32_t>(chunkSizeByte); byte++)
                {
                    pWriteBuffer[byte] = (rand() & 0xFF) ^ (isPositive ? 0x00 : 0xFF);
                }

                nn::os::Tick tick(0);

                // Write
                NN_RESULT_DO(Write(pWriteBuffer, &tick, sectorIndex, chunkSizeByte, pConfig));
                writeTicks.sum += tick;
                writeTicks.min = (writeTicks.min > tick || writeTicks.min == nn::os::Tick(0)) ? tick : writeTicks.min;
                writeTicks.max = (writeTicks.max < tick) ? tick : writeTicks.max;

                // Read
                NN_RESULT_DO(Read(pReadBuffer, &tick, sectorIndex, chunkSizeByte, pConfig));
                readTicks.sum += tick;
                readTicks.min = (readTicks.min > tick || readTicks.min == nn::os::Tick(0)) ? tick : readTicks.min;
                readTicks.max = (readTicks.max < tick) ? tick : readTicks.max;

                // Verify
                Verify(pWriteBuffer, pReadBuffer, sectorIndex, chunkSizeByte);
            }

            // 計測結果の表示
            DiaplayMeasurements(writeTicks, readTicks, chunkSizeByte);
        }

        // 終了処理
        Deactivate(pConfig);

        // 後始末
        Finalize(pConfig);
        NN_RESULT_SUCCESS;
    }

    nn::Result RandomAging(AgingConfig* pConfig, bool isPositive)
    {
        // 事前準備
        Initialize(pConfig);

        // 初期化処理
        NN_RESULT_DO(Activate(pConfig));

        // バス情報の表示
        LogBusStatus(pConfig->port);

        // デバイスサイズの取得
        uint32_t sectorMaxCount;
        NN_RESULT_DO(ReadSectorMaxCount(&sectorMaxCount, pConfig));

        // 測定結果表示のラベル
        NN_LOG("Chunk(KB) KB/sec IOPS min(usec) max(usec)\n");

        // アクセスするチャンクサイズを 2 倍ずつ増加させる
        for (size_t chunkSizeByte = ChunkSizeMinByte; chunkSizeByte <= ChunkSizeMaxByte; chunkSizeByte *= 2)
        {
            MeasuredTicks writeTicks = { nn::os::Tick(0), nn::os::Tick(0), nn::os::Tick(0) };
            MeasuredTicks readTicks  = { nn::os::Tick(0), nn::os::Tick(0), nn::os::Tick(0) };

            // CheckMaxCount 回のアクセスを行う
            for (uint32_t count = 0; count < CheckMaxCount; count++)
            {
                uint8_t* pWriteBuffer = reinterpret_cast<uint8_t*>(pConfig->dataBufferAddress);
                uint8_t* pReadBuffer  = reinterpret_cast<uint8_t*>(pConfig->dataBufferAddress + pConfig->dataBufferSize / 2);

                srand(count);
                uint32_t sectorIndex = ((rand() % (sectorMaxCount - pConfig->offset)) & ~(AccessAlignment / nn::sdmmc::SectorSize - 1)) + pConfig->offset;

                // テストパターンを生成
                srand(sectorIndex);
                for (uint32_t byte = 0; byte < static_cast<uint32_t>(chunkSizeByte); byte++)
                {
                    pWriteBuffer[byte] = (rand() & 0xFF) ^ (isPositive ? 0x00 : 0xFF);
                }

                nn::os::Tick tick(0);

                // Write
                NN_RESULT_DO(Write(pWriteBuffer, &tick, sectorIndex, chunkSizeByte, pConfig));
                writeTicks.sum += tick;
                writeTicks.min = (writeTicks.min > tick || writeTicks.min == nn::os::Tick(0)) ? tick : writeTicks.min;
                writeTicks.max = (writeTicks.max < tick) ? tick : writeTicks.max;

                // Read
                NN_RESULT_DO(Read(pReadBuffer, &tick, sectorIndex, chunkSizeByte, pConfig));
                readTicks.sum += tick;
                readTicks.min = (readTicks.min > tick || readTicks.min == nn::os::Tick(0)) ? tick : readTicks.min;
                readTicks.max = (readTicks.max < tick) ? tick : readTicks.max;

                // Verify
                Verify(pWriteBuffer, pReadBuffer, sectorIndex, chunkSizeByte);
            }

            // 計測結果の表示
            DiaplayMeasurements(writeTicks, readTicks, chunkSizeByte);
        }

        // 終了処理
        Deactivate(pConfig);

        // 後始末
        Finalize(pConfig);
        NN_RESULT_SUCCESS;
    }

    nn::Result SequentialPositivePatternAging(AgingConfig* pConfig)
    {
        NN_LOG("--------- Sequential Positive Pattern Aging ---------\n");

        NN_RESULT_DO(SequentialAging(pConfig, true));
        NN_RESULT_SUCCESS;
    }

    nn::Result SequentialNegativePatternAging(AgingConfig* pConfig)
    {
        NN_LOG("--------- Sequential Negative Pattern Aging ---------\n");

        NN_RESULT_DO(SequentialAging(pConfig, false));
        NN_RESULT_SUCCESS;
    }

    nn::Result RandomPositivePatternAging(AgingConfig* pConfig)
    {
        NN_LOG("--------- Random Positive Pattern Aging ---------\n");

        NN_RESULT_DO(RandomAging(pConfig, true));
        NN_RESULT_SUCCESS;
    }

    nn::Result RandomNegativePatternAging(AgingConfig* pConfig)
    {
        NN_LOG("--------- Random Negative Pattern Aging ---------\n");

        NN_RESULT_DO(RandomAging(pConfig, false));
        NN_RESULT_SUCCESS;
    }

}

extern "C" void nninitStartup()
{
    nn::Result result = nn::os::SetMemoryHeapSize(512 * 1024 * 1024);
    if (result.IsFailure())
    {
        NN_LOG("Failed SetMemoryHeapSize\n");
    }
}

extern "C" void nnMain()
{
    NN_LOG("/// SDMMC AGING ///\n");

    // メモリヒープから malloc で使用するメモリ領域を確保し、設定する
    uintptr_t heapPtrForMalloc;
    const size_t HeapSizeForMalloc = 256 * 1024 * 1024;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::AllocateMemoryBlock(&heapPtrForMalloc, HeapSizeForMalloc));
    nn::init::InitializeAllocator(reinterpret_cast<void*>(heapPtrForMalloc), HeapSizeForMalloc);

    // fs プロセスの SDMMC 制御を停止する
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::SuspendSdmmcControl());

    // エージングコンフィグのセット
    AgingConfig config;
    SetConfig(&config);

    // エージングを無限に行う
    nn::Result result;
    for (uint32_t loop = 0; ; loop++)
    {
        NN_LOG("============== %d loop ==============\n", loop);

        result = SequentialPositivePatternAging(&config);
        if (result.IsFailure())
        {
            break;
        }

        result = SequentialNegativePatternAging(&config);
        if (result.IsFailure())
        {
            break;
        }

        result = RandomPositivePatternAging(&config);
        if (result.IsFailure())
        {
            break;
        }

        result = RandomNegativePatternAging(&config);
        if (result.IsFailure())
        {
            break;
        }
    }

    // エージング結果の表示
    // PASS が表示されるケースは現実装では存在しない（強制停止した場合は本体再起動が必要）
    if (result.IsSuccess())
    {
        NN_LOG("[[ PASSED !! ]]\n");
    }
    else
    {
        NN_LOG("[[ FAILED !! ]]\n");
        NN_LOG("Module:%d Description:%d Inner:%d\n",
               result.GetModule(), result.GetDescription(), result.GetInnerValueForDebug());
    }

    // fs プロセスの SDMMC 制御を再開する
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ResumeSdmmcControl());

    nn::os::FreeMemoryBlock(heapPtrForMalloc, HeapSizeForMalloc);
}

