﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/sdmmc/sdmmc_Mmc.h>
#include <nn/sdmmc/sdmmc_SdCard.h>
#ifdef NN_SDMMC_TEST_SMMU_ENABLE
    #include <nn/sdmmc/sdmmc_DeviceVirtualAddress.h>
#endif
#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/os.h>
#include <nn/os/os_MemoryHeap.h>
#include <nn/init.h>
#include <nn/dd.h>
#include <nn/fs/fs_SdmmcControl.h>
#include <cstring>
#include <cinttypes>
#include "SdmmcCommon.h"

// 注意：初期プロセスの FS の sMMU 設定を無効にした環境でしか動作しません。

namespace
{
    #if 1   // MMC
        const nn::sdmmc::Port TestPort = nn::sdmmc::Port_Mmc0;
        #ifdef NN_SDMMC_TEST_SMMU_ENABLE
            const nn::dd::DeviceName DeviceName = nn::dd::DeviceName::DeviceName_Sdmmc4a;
        #endif
        const size_t WorkBufferSize = nn::sdmmc::MmcWorkBufferSize;
    #else   // SD カード
        const nn::sdmmc::Port TestPort = nn::sdmmc::Port_SdCard0;
        #ifdef NN_SDMMC_TEST_SMMU_ENABLE
            #ifdef NN_BUILD_CONFIG_HARDWARE_JETSONTK1
                const nn::dd::DeviceName DeviceName = nn::dd::DeviceName::DeviceName_Sdmmc3a;
            #else   // NN_BUILD_CONFIG_HARDWARE_JETSONTK2
                    // NN_BUILD_CONFIG_HARDWARE_NX
                const nn::dd::DeviceName DeviceName = nn::dd::DeviceName::DeviceName_Sdmmc1a;
            #endif
        #endif
        const size_t WorkBufferSize = nn::sdmmc::SdCardWorkBufferSize;
    #endif
    const bool isAttachMmcToPortSdCard0ForDebug = false;

    #define SDMMC_DETAIL_CEILING(value, unit)   ((((value) + (unit) - 1) / (unit)) * (unit))
    #define SDMMC_DETAIL_CEILING_FOR_DEVICE_ADDRESS_SPACE(value)    SDMMC_DETAIL_CEILING((value), nn::dd::DeviceAddressSpaceMemoryRegionAlignment)

    NN_DD_ALIGNAS_DEVICE_ADDRESS_SPACE_MEMORY uint8_t g_WorkBuffer[SDMMC_DETAIL_CEILING_FOR_DEVICE_ADDRESS_SPACE(WorkBufferSize)];
    #if (defined(NN_DETAIL_SDMMC_ADMA2_ENABLE))
        NN_DD_ALIGNAS_DEVICE_ADDRESS_SPACE_MEMORY uint8_t g_WorkBufferForHostController[SDMMC_DETAIL_CEILING_FOR_DEVICE_ADDRESS_SPACE(nn::sdmmc::WorkBufferSizeForHostController)];
    #endif

    const uint32_t TestSectorIndex = 0;
    const size_t TotalSize = 64 * 1024 * 1024;
    const size_t BufferSizeMin = 4 * 1024;
    const size_t BufferSizeMax = 8 * 1024 * 1024;
    const size_t DataBufferSize = nn::util::align_up(BufferSizeMax, nn::os::MemoryBlockUnitSize);
    void* g_pDataBuffer;

    nn::Result MeasureRead(size_t bufferSize)
    {
        nn::os::Tick totalTick(0);
        uint32_t sectorIndex = TestSectorIndex;
        uint32_t numSectors = bufferSize / nn::sdmmc::SectorSize;
        NN_ABORT_UNLESS((bufferSize % nn::sdmmc::SectorSize) == 0);    // バッファサイズはセクタの整数倍
        int numSplit = TotalSize / bufferSize;
        NN_ABORT_UNLESS((TotalSize % bufferSize) == 0); // 全体のアクセスサイズはバッファサイズの整数倍
        for (int i = 0; i < numSplit; i++)
        {
            nn::os::Tick tick = nn::os::GetSystemTick();
            nn::Result result = nn::sdmmc::Read(g_pDataBuffer, bufferSize, TestPort, sectorIndex, numSectors);
            totalTick += (nn::os::GetSystemTick() - tick);
            if (result.IsFailure())
            {
                NN_LOG("Read is failure. Module:%d, Description:%d\n", result.GetModule(), result.GetDescription());
                return result;
            }
            nn::sdmmc::ErrorInfo errorInfo;
            size_t logSize;
            char logBuffer[256];
            nn::sdmmc::GetAndClearErrorInfo(&errorInfo, &logSize, logBuffer, sizeof(logBuffer), TestPort);
            if ((errorInfo.numReadWriteFailures != 0) || (errorInfo.numReadWriteErrorCorrections != 0))
            {
                NN_LOG("Read had hidden error. %s\n", logBuffer);
                // 以降のアクセスを続ける
            }
            sectorIndex += numSectors;
        }

        nn::TimeSpan timeSpan = ConvertToTimeSpan(totalTick);
        NN_LOG("%zu (KB) %" PRId64 " (KB/sec)\n", bufferSize / 1024,
            ((static_cast<int64_t>(TotalSize) * 1000) / timeSpan.GetMilliSeconds()) / 1024);
        return nn::ResultSuccess();
    }
}

extern "C" void nninitStartup()
{
    const size_t MemoryHeapSize = 512 * 1024 * 1024;
    nn::Result result = nn::os::SetMemoryHeapSize(MemoryHeapSize);
    NN_ABORT_UNLESS(result.IsSuccess(), "Cannot set MemoryHeapSize.");
}

extern "C" void nnMain()
{
    // メモリヒープから malloc で使用するメモリ領域を確保し、設定する
    uintptr_t heapPtrForMalloc;
    const size_t HeapSizeForMalloc = 256 * 1024 * 1024;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::AllocateMemoryBlock(&heapPtrForMalloc, HeapSizeForMalloc));
    nn::init::InitializeAllocator(reinterpret_cast<void*>(heapPtrForMalloc), HeapSizeForMalloc);

    // fs プロセスの SDMMC 制御を停止する
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::SuspendSdmmcControl());

    // クロック制御は pcv 経由で行う
    nn::sdmmc::SwitchToPcvClockResetControl();

    nn::sdmmc::Initialize(TestPort);

    #ifdef NN_SDMMC_TEST_SMMU_ENABLE
        nn::dd::DeviceAddressSpaceType das;
        SetupDeviceAddressSpace(&das, DeviceName);
        nn::dd::DeviceVirtualAddress deviceVirtualAddressOffset = 0;

        nn::dd::DeviceAddressSpaceMapInfo info;
        nn::dd::DeviceVirtualAddress workBufferDeviceVirtualAddress = MapDeviceAddressSpaceAligned(&das, &info,
            reinterpret_cast<uintptr_t>(g_WorkBuffer), sizeof(g_WorkBuffer), deviceVirtualAddressOffset);
        deviceVirtualAddressOffset = workBufferDeviceVirtualAddress + sizeof(g_WorkBuffer);
        nn::sdmmc::RegisterDeviceVirtualAddress(TestPort, reinterpret_cast<uintptr_t>(g_WorkBuffer), sizeof(g_WorkBuffer), workBufferDeviceVirtualAddress);
    #endif
    if ((TestPort == nn::sdmmc::Port_SdCard0) && (!isAttachMmcToPortSdCard0ForDebug))
    {
        nn::sdmmc::SetSdCardWorkBuffer(TestPort, g_WorkBuffer, sizeof(g_WorkBuffer));
    }
    else
    {
        nn::sdmmc::SetMmcWorkBuffer(TestPort, g_WorkBuffer, sizeof(g_WorkBuffer));
    }

    #if (defined(NN_DETAIL_SDMMC_ADMA2_ENABLE))
        #ifdef NN_SDMMC_TEST_SMMU_ENABLE
            nn::dd::DeviceAddressSpaceMapInfo infoForHostController;
            uintptr_t hostControllerBufferAddress = reinterpret_cast<uintptr_t>(g_WorkBufferForHostController);
            size_t hostControllerBufferSize = sizeof(g_WorkBufferForHostController);
            nn::dd::DeviceVirtualAddress hostControllerBufferDeviceVirtualAddress = MapDeviceAddressSpaceAligned(&das, &infoForHostController,
                hostControllerBufferAddress, hostControllerBufferSize, deviceVirtualAddressOffset);
            deviceVirtualAddressOffset = hostControllerBufferDeviceVirtualAddress + hostControllerBufferSize;
            nn::sdmmc::RegisterDeviceVirtualAddress(TestPort, hostControllerBufferAddress, hostControllerBufferSize, hostControllerBufferDeviceVirtualAddress);
        #endif

        nn::sdmmc::SetWorkBufferForHostController(TestPort, g_WorkBufferForHostController, sizeof(g_WorkBufferForHostController));
    #endif

    uintptr_t dataBufferAddress;
    nn::Result result = nn::os::AllocateMemoryBlock(&dataBufferAddress, DataBufferSize);
    NN_ABORT_UNLESS(result.IsSuccess(), "Cannot allocate memory.");
    #ifdef NN_SDMMC_TEST_SMMU_ENABLE
        nn::dd::DeviceVirtualAddress dataBufferDeviceVirtualAddress = MapDeviceAddressSpaceAligned(&das, dataBufferAddress, DataBufferSize, deviceVirtualAddressOffset);
        nn::sdmmc::RegisterDeviceVirtualAddress(TestPort, dataBufferAddress, DataBufferSize, dataBufferDeviceVirtualAddress);
    #endif
    g_pDataBuffer = reinterpret_cast<uint8_t*>(dataBufferAddress);

    result = nn::sdmmc::Activate(TestPort);
    if (result.IsSuccess())
    {
        nn::sdmmc::ErrorInfo errorInfo;
        size_t logSize;
        char logBuffer[256];
        nn::sdmmc::GetAndClearErrorInfo(&errorInfo, &logSize, logBuffer, sizeof(logBuffer), TestPort);
        if ((errorInfo.numActivationFailures != 0) || (errorInfo.numActivationErrorCorrections != 0))
        {
            NN_LOG("Activate had hidden error. %s\n", logBuffer);
            // 以降のアクセスを続ける
        }

        // バス状態のダンプ
        LogBusStatus(TestPort);

        uint32_t numSectorsOfMemoryCapacity = 0;
        result = nn::sdmmc::GetDeviceMemoryCapacity(&numSectorsOfMemoryCapacity, TestPort);
        NN_ABORT_UNLESS(result.IsSuccess());
        NN_LOG("Memory Capacity %u (sectors)\n", numSectorsOfMemoryCapacity);
        NN_LOG("Memory Capacity %u (MB)\n", numSectorsOfMemoryCapacity / 2 / 1024);

        if ((TestPort == nn::sdmmc::Port_SdCard0) && (!isAttachMmcToPortSdCard0ForDebug))
        {
            uint32_t numSectorsOfProtectedAreaCapacity = 0;
            result = nn::sdmmc::GetSdCardProtectedAreaCapacity(&numSectorsOfProtectedAreaCapacity, TestPort);
            NN_ABORT_UNLESS(result.IsSuccess());
            NN_LOG("SD Card Protected Area %u (MB)\n", numSectorsOfProtectedAreaCapacity / 2 / 1024);
        }

        size_t bufferSize = BufferSizeMin;
        while (bufferSize <= BufferSizeMax)
        {
            result = MeasureRead(bufferSize);
            if (result.IsFailure())
            {
                break;
            }
            bufferSize *= 2;
        }
    }
    else
    {
        NN_LOG("Activate is failure. Module:%d, Description:%d\n", result.GetModule(), result.GetDescription());
    }
    nn::sdmmc::Deactivate(TestPort);

    #ifdef NN_SDMMC_TEST_SMMU_ENABLE
        nn::sdmmc::UnregisterDeviceVirtualAddress(TestPort, dataBufferAddress, DataBufferSize, dataBufferDeviceVirtualAddress);
        UnmapDeviceAddressSpaceAligned(&das, dataBufferAddress, DataBufferSize, dataBufferDeviceVirtualAddress);
    #endif
    nn::os::FreeMemoryBlock(dataBufferAddress, DataBufferSize);

    #if (defined(NN_DETAIL_SDMMC_ADMA2_ENABLE))
        #ifdef NN_SDMMC_TEST_SMMU_ENABLE
            nn::sdmmc::UnregisterDeviceVirtualAddress(TestPort, hostControllerBufferAddress, hostControllerBufferSize, hostControllerBufferDeviceVirtualAddress);
            UnmapDeviceAddressSpaceAligned(&infoForHostController);
        #endif
    #endif

    #ifdef NN_SDMMC_TEST_SMMU_ENABLE
        nn::sdmmc::UnregisterDeviceVirtualAddress(TestPort, reinterpret_cast<uintptr_t>(g_WorkBuffer), sizeof(g_WorkBuffer), workBufferDeviceVirtualAddress);
        UnmapDeviceAddressSpaceAligned(&info);

        CleanDeviceAddressSpace(&das, DeviceName);
    #endif

    nn::sdmmc::Finalize(TestPort);

    // fs プロセスの SDMMC 制御を再開する
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ResumeSdmmcControl());

    nn::os::FreeMemoryBlock(heapPtrForMalloc, HeapSizeForMalloc);
}
