﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/sdmmc/sdmmc_GcAsic.h>
#ifdef NN_SDMMC_TEST_SMMU_ENABLE
    #include <nn/sdmmc/sdmmc_DeviceVirtualAddress.h>
#endif
#include <nn/os/os_MemoryHeap.h>
#include <nn/init.h>
#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/dd.h>
#include <nn/fs/fs_SdmmcControl.h>
#include <cstring>
#include "SdmmcCommon.h"

// 注意：初期プロセスの FS の sMMU 設定を無効にした環境でしか動作しません。

namespace
{
    const nn::sdmmc::Port TestPort = nn::sdmmc::Port_GcAsic0;
    #ifdef NN_SDMMC_TEST_SMMU_ENABLE
        const nn::dd::DeviceName DeviceName = nn::dd::DeviceName::DeviceName_Sdmmc2a;
    #endif

    const uint32_t TestNumSectors = 9;
    const size_t DataBufferSize = nn::util::align_up(nn::sdmmc::SectorSize * TestNumSectors, nn::os::MemoryBlockUnitSize);
    void* g_pDataBuffer;

    const char* GetPortName(nn::sdmmc::Port port)
    {
        switch (port)
        {
        case nn::sdmmc::Port_Mmc0:
            return "Port_Mmc0";
        case nn::sdmmc::Port_SdCard0:
            return "Port_SdCard0";
        case nn::sdmmc::Port_GcAsic0:
            return "Port_GcAsic0";
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    void IssueCommandToGcAsic() NN_NOEXCEPT
    {
        // 容量取得
        uint32_t numSectorsOfMemoryCapacity = 0;
        nn::Result result = nn::sdmmc::GetDeviceMemoryCapacity(&numSectorsOfMemoryCapacity, TestPort);
        NN_ABORT_UNLESS(result.IsSuccess());
        NN_LOG("Memory Capacity %u (sector)\n", numSectorsOfMemoryCapacity);

        // スリープ
        NN_LOG("SleepGcAsic %s ...\n", GetPortName(TestPort));
        result = nn::sdmmc::SleepGcAsic(TestPort);
        if (result.IsSuccess())
        {
            NN_LOG("Done.\n");
        }
        else
        {
            NN_LOG("nn::sdmmc::SleepGcAsic is failure. Module:%d, Description:%d\n", result.GetModule(), result.GetDescription());
        }

        // スリープ復帰
        NN_LOG("FinishGcAsicOperation %s ...\n", GetPortName(TestPort));
        result = nn::sdmmc::FinishGcAsicOperation(TestPort);
        if (result.IsSuccess())
        {
            NN_LOG("Done.\n");
        }
        else
        {
            NN_LOG("nn::sdmmc::FinishGcAsicOperation is failure. Module:%d, Description:%d\n", result.GetModule(), result.GetDescription());
        }

        // 鍵更新
        NN_LOG("UpdateGcAsicKey %s ...\n", GetPortName(TestPort));
        result = nn::sdmmc::UpdateGcAsicKey(TestPort);
        if (result.IsSuccess())
        {
            NN_LOG("Done.\n");
        }
        else
        {
            NN_LOG("nn::sdmmc::UpdateGcAsicKey is failure. Module:%d, Description:%d\n", result.GetModule(), result.GetDescription());
        }

        // オペレーション書き込み
        NN_LOG("WriteGcAsicOperation %s ...\n", GetPortName(TestPort));
        result = nn::sdmmc::WriteGcAsicOperation(TestPort, g_pDataBuffer, DataBufferSize);
        if (result.IsSuccess())
        {
            NN_LOG("Done.\n");
        }
        else
        {
            NN_LOG("nn::sdmmc::WriteGcAsicOperation is failure. Module:%d, Description:%d\n", result.GetModule(), result.GetDescription());
        }

        // 1 セクタのリード
        uint32_t testSectorIndex = 0;
        uint32_t testNumSectors = 1;
        NN_LOG("Read  %s, sectorIndex=%u, numSectors=%u ...\n", GetPortName(TestPort), testSectorIndex, testNumSectors);
        result = nn::sdmmc::Read(g_pDataBuffer, DataBufferSize, TestPort, testSectorIndex, testNumSectors);
        if (result.IsSuccess())
        {
            NN_LOG("Done.\n");
        }
        else
        {
            NN_LOG("nn::sdmmc::Read is failure. Module:%d, Description:%d\n", result.GetModule(), result.GetDescription());
        }

        // オペレーション終了
        NN_LOG("FinishGcAsicOperation %s ...\n", GetPortName(TestPort));
        result = nn::sdmmc::FinishGcAsicOperation(TestPort);
        if (result.IsSuccess())
        {
            NN_LOG("Done.\n");
        }
        else
        {
            NN_LOG("nn::sdmmc::FinishGcAsicOperation is failure. Module:%d, Description:%d\n", result.GetModule(), result.GetDescription());
        }
    }
}

extern "C" void nninitStartup()
{
    const size_t MemoryHeapSize = 512 * 1024 * 1024;
    nn::Result result = nn::os::SetMemoryHeapSize(MemoryHeapSize);
    NN_ABORT_UNLESS(result.IsSuccess(), "Cannot set MemoryHeapSize.");
}

extern "C" void nnMain()
{
    // メモリヒープから malloc で使用するメモリ領域を確保し、設定する
    uintptr_t heapPtrForMalloc;
    const size_t HeapSizeForMalloc = 256 * 1024 * 1024;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::AllocateMemoryBlock(&heapPtrForMalloc, HeapSizeForMalloc));
    nn::init::InitializeAllocator(reinterpret_cast<void*>(heapPtrForMalloc), HeapSizeForMalloc);

    // fs プロセスの SDMMC 制御を停止する
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::SuspendSdmmcControl());

    // クロック制御は pcv 経由で行う
    nn::sdmmc::SwitchToPcvClockResetControl();

    nn::sdmmc::Initialize(TestPort);

    uintptr_t dataBufferAddress;
    nn::Result result = nn::os::AllocateMemoryBlock(&dataBufferAddress, DataBufferSize);
    NN_ABORT_UNLESS(result.IsSuccess(), "Cannot allocate memory.");

    #ifdef NN_SDMMC_TEST_SMMU_ENABLE
        nn::dd::DeviceAddressSpaceType das;
        SetupDeviceAddressSpace(&das, DeviceName);
        nn::dd::DeviceVirtualAddress deviceVirtualAddress = MapDeviceAddressSpaceAligned(&das, dataBufferAddress, DataBufferSize, 0);
        nn::sdmmc::RegisterDeviceVirtualAddress(TestPort, dataBufferAddress, DataBufferSize, deviceVirtualAddress);
    #endif

    g_pDataBuffer = reinterpret_cast<uint8_t*>(dataBufferAddress);

    result = nn::sdmmc::Activate(TestPort);
    if (result.IsSuccess())
    {
        // バス状態のダンプ
        LogBusStatus(TestPort);

        // GC ASIC へのコマンド発行
        IssueCommandToGcAsic();
    }
    else
    {
        NN_LOG("nn::sdmmc::Activate is failure. Module:%d, Description:%d\n", result.GetModule(), result.GetDescription());
    }
    nn::sdmmc::Deactivate(TestPort);

    #ifdef NN_SDMMC_TEST_SMMU_ENABLE
        nn::sdmmc::UnregisterDeviceVirtualAddress(TestPort, dataBufferAddress, DataBufferSize, deviceVirtualAddress);
        UnmapDeviceAddressSpaceAligned(&das, dataBufferAddress, DataBufferSize, deviceVirtualAddress);
        CleanDeviceAddressSpace(&das, DeviceName);
    #endif

    nn::os::FreeMemoryBlock(dataBufferAddress, DataBufferSize);

    nn::sdmmc::Finalize(TestPort);

    // fs プロセスの SDMMC 制御を再開する
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ResumeSdmmcControl());

    nn::os::FreeMemoryBlock(heapPtrForMalloc, HeapSizeForMalloc);
}
