﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <cstring>

#include <nn/fs.h>
#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_IStorage.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>

namespace
{
    // システムを壊す可能性があることに注意
    // http://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=89108751
    #if 0
        nn::fs::BisPartitionId TestPartitionId = nn::fs::BisPartitionId::BootPartition1Root;
        std::int64_t TestOffset = 0x00180000;   // 未使用とされている場所
        const size_t TestBufferSize = 2 * 1024 * 1024;
    #endif
    #if 0
        nn::fs::BisPartitionId TestPartitionId = nn::fs::BisPartitionId::BootPartition2Root;
        std::int64_t TestOffset = 0x00100000;   // 未使用とされている場所
        const size_t TestBufferSize = 2 * 1024 * 1024;
    #endif
    #if 1
        nn::fs::BisPartitionId TestPartitionId = nn::fs::BisPartitionId::UserDataRoot;
        std::int64_t TestOffset = 0x400000ULL * 512;    // 適当な場所
        const size_t TestBufferSize = (8 * 1024 * 1024) + 1024;
    #endif

    #define ABORT_IF_FAILURE(result, ...) \
        if (result.IsFailure()) { NN_LOG(__VA_ARGS__); NN_LOG(" Result: Module %d, Description %d\n", result.GetModule(), result.GetDescription()); NN_ABORT_UNLESS(result.IsSuccess()); }

    char g_Buffer[TestBufferSize];

    const char* GetPartitionString(nn::fs::BisPartitionId partitionId)
    {
        switch (partitionId)
        {
        case nn::fs::BisPartitionId::BootPartition1Root:
            return "Boot Partition 1";
        case nn::fs::BisPartitionId::BootPartition2Root:
            return "Boot Partition 2";
        case nn::fs::BisPartitionId::UserDataRoot:
            return "User Data";
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    void SetData(void* pData, size_t dataSize) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(pData);
        if (dataSize == 0)
        {
            return;
        }
        uint32_t* pCurrentData = reinterpret_cast<uint32_t*>(pData);
        for (size_t i = 0; i < (dataSize / sizeof(uint32_t)); i++)
        {
            pCurrentData[i] = i;
        }
    }

    void CheckData(void* pData, size_t dataSize) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(pData);
        if (dataSize == 0)
        {
            return;
        }
        uint32_t* pCurrentData = reinterpret_cast<uint32_t*>(pData);
        for (size_t i = 0; i < (dataSize / sizeof(uint32_t)); i++)
        {
            NN_ABORT_UNLESS(pCurrentData[i] == i);
        }
    }

    void* Allocate(size_t size)
    {
        return std::malloc(size);
    }

    void Deallocate(void* p, size_t size)
    {
        NN_UNUSED(size);
        std::free(p);
    }
}

extern "C" void nnMain()
{
    nn::fs::SetAllocator(Allocate, Deallocate);

    NN_LOG("Test MMC Access (%s)\n", GetPartitionString(TestPartitionId));

    std::unique_ptr<nn::fs::IStorage> storage;
    nn::Result result = nn::fs::OpenBisPartition(&storage, TestPartitionId);
    ABORT_IF_FAILURE(result, "nn::fs::OpenBisPartition");

    int64_t size;
    result = storage->GetSize(&size);
    ABORT_IF_FAILURE(result, "nn::fs::GetSize");
    NN_LOG("Size of %s: %.1f(MB)\n", GetPartitionString(TestPartitionId), size / 1024.0 / 1024.0);

    NN_LOG("Write %.1f(KB) ...\n", sizeof(g_Buffer) / 1024.0);
    SetData(g_Buffer, sizeof(g_Buffer));
    result = storage->Write(TestOffset, g_Buffer, sizeof(g_Buffer));
    ABORT_IF_FAILURE(result, "nn::fs::Write");

    NN_LOG("Read %.1f(KB) ...\n", sizeof(g_Buffer) / 1024.0);
    std::memset(g_Buffer, 0xA5, sizeof(g_Buffer));
    result = storage->Read(TestOffset, g_Buffer, sizeof(g_Buffer));
    ABORT_IF_FAILURE(result, "nn::fs::Read");

    NN_LOG("Verify ...\n");
    CheckData(g_Buffer, sizeof(g_Buffer));

    NN_LOG("Done.\n");
}
