﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       SixAxisSensor の処理に特化した API の宣言
 */

#pragma once

#include <nn/hid/hid_Xpad.h>
#include <nn/util/util_MathTypes.h>
#include <nn/xcd/xcd_Input.h>
#include <nn/hid/hid_SixAxisSensor.h>

#include "hid_RingLifo.h"
#include "hid_Accelerometer.h"
#include "hid_Gyroscope.h"

namespace nn { namespace hid { namespace detail {

//!< 基本姿勢
const nn::hid::DirectionState  UnitDirection = { {{{1.0f, 0.0f, 0.0f}}},
                                                 {{{0.0f, 1.0f, 0.0f}}},
                                                 {{{0.0f, 0.0f, 1.0f}}} };

//!< 6軸センサーの入力値を格納するLIFO
class SixAxisSensorStateLifo final : public detail::RingLifo<nn::hid::SixAxisSensorState, nn::hid::SixAxisSensorStateCountMax>
{
public:
    SixAxisSensorStateLifo() NN_NOEXCEPT { /* 何もしない */ }

    virtual ~SixAxisSensorStateLifo() NN_NOEXCEPT NN_OVERRIDE { /* 何もしない */ }
};

//!< 6軸センサーの処理を担うクラス
class SixAxisSensorProcessor final
{
    NN_DISALLOW_COPY(SixAxisSensorProcessor);
    NN_DISALLOW_MOVE(SixAxisSensorProcessor);

private:

    SixAxisSensorStateLifo      m_Lifo;                          //!< 加工後のセンサー値を格納するLIFO

    Accelerometer               m_Accelerometer;                 //!< 加速度算出のためのクラス
    Gyroscope                   m_Gyroscope;                     //!< 角速度算出のためのクラス

    nn::util::Float3            m_Angle;                         //!< 現在の回転角度[deg]
    nn::hid::DirectionState     m_Direction;                     //!< 現在の姿勢行列
    nn::hid::DirectionState     m_LastDirection;                 //!< 1サンプル前の姿勢行列
    int64_t                     m_SamplingNumber;                //!< 現在のサンプリング番号

    float                       m_SamplingIntervalSeconds;       //!< センサーのサンプリング周期[sec] (※ 通信周期ではない)
    nn::xcd::PadState           m_LastPadState;                  //!< 前回サンプリングした PadState

    bool                        m_IsSamplingEnabled;             //!< 6軸センサーの利用状態
    bool                        m_IsCalibrated;                  //!< キャリブレーションの状態

    bool                        m_IsBaseDirReviseEnabled;        //!< 基本姿勢による姿勢補正
    bool                        m_IsSixAxisSensorFusionEnabled;  //!< 加速度による姿勢補正

    nn::hid::DirectionState     m_BaseDirection;                 //!< 基本姿勢です
    float                       m_BaseDirectionRevisePower;      //!< 基本姿勢に近づける強さです

public:
    SixAxisSensorProcessor()  NN_NOEXCEPT;
    ~SixAxisSensorProcessor() NN_NOEXCEPT {};

    //!< 6軸センサーの利用状態を制御します。
    void ControlSixAxisSensor( const bool& enableSampling ) NN_NOEXCEPT;

    //!< 基本姿勢による姿勢補正状態を制御します。
    void ControlBaseDirRevise(const bool& enable) NN_NOEXCEPT;
    bool IsBaseDirReviseEnabled() const NN_NOEXCEPT;

    //!< 加速度による姿勢補正状態を制御します。
    void EnableSixAxisSensorFusion(const bool& enable) NN_NOEXCEPT;
    bool IsSixAxisSensorFusionEnabled() const NN_NOEXCEPT;

    //!< 6軸センサーの利用状態を取得します。
    bool IsSamplingEnabled() const NN_NOEXCEPT;

    //!< キャリブレーション状態を返します。
    bool IsCalibrated() const NN_NOEXCEPT;

    //!< 6軸センサーのフルスケールレンジを設定します。
    void SetAccelerometerFsr(const nn::xcd::AccelerometerFsr& accFsr) NN_NOEXCEPT;
    void SetGyroscopeFsr(const nn::xcd::GyroscopeFsr& gyroFsr) NN_NOEXCEPT;

    //!< 6軸センサーのキャリブレーション値を設定します。
    void SetSensorCalibrationValue( const nn::xcd::SensorCalibrationValue& value ) NN_NOEXCEPT;

    //!< 現在の回転角度[deg]を設定します。
    void SetAngle( const  nn::util::Float3& angle ) NN_NOEXCEPT;

    //!< 現在の姿勢を設定します。
    void SetDirection( const nn::hid::DirectionState& direction ) NN_NOEXCEPT;

    //!< PadState を注入して内部ステートを更新します。
    void Sampling( nn::xcd::PadState& state, bool isConnected ) NN_NOEXCEPT;

    //!< センサーのカウント値をサンプリングします。
    void Sampling(const nn::xcd::SensorState& accelerationCount,
                  const nn::xcd::SensorState& angularVelocityCount,
                  const nn::TimeSpan& deltaTime,
                  bool isConnected ) NN_NOEXCEPT;

    //!< SixAxisSensor の入力状態を過去に遡って読み出します。
    int  GetSixAxisSensorStates(SixAxisSensorState* outStates, int count ) const NN_NOEXCEPT;

    //!< 開発用途: 内部状態をダンプします。
    void DumpStateForDev(const SixAxisSensorState* states, int count) const NN_NOEXCEPT;

    //!< 加速度による姿勢補正のパラメータを設定/取得します。
    void SetSixAxisSensorFusionParameters(const float& revisePower,
                                          const float& reviseRange) NN_NOEXCEPT;
    void GetSixAxisSensorFusionParameters(float* pOutRevisePower,
                                          float* pOutReviseRange) const NN_NOEXCEPT;
    void ResetSixAxisSensorFusionParameters() NN_NOEXCEPT;

    //!< 基本姿勢を用いた姿勢補正のパラメータを設定/取得します。
    void SetBaseDirectionReviseParameters(const nn::hid::DirectionState& direction,
                                          const float& revisePower) NN_NOEXCEPT;
    void GetBaseDirectionReviseParameters(nn::hid::DirectionState* pOutDirection,
                                          float* pOutRevisePower) const NN_NOEXCEPT;
    void ResetBaseDirectionReviseParameters() NN_NOEXCEPT;

    //!< 現在の姿勢を基本姿勢を用いて更新します。
    void UpdateDirectionByBaseDirection(nn::hid::DirectionState* pOutDirection,
                                        const nn::hid::DirectionState& lastDirection) const NN_NOEXCEPT;
private:

    //!< Sampling 時に引数指定された PadState と直近のサンプルのサンプリング番号を元に内挿補間します。
    void Interpolate( nn::xcd::PadState& state, bool isConnected ) NN_NOEXCEPT;

    //!< 現在の回転角度[deg] を更新します。
    void UpdateAngle( const nn::util::Float3& angularVelocity ) NN_NOEXCEPT;

};

}}} // namespace nn::hid::detail
