﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>

#include "hid_CalcUtil.h"
#include "hid_SixAxisSensorProcessor.h"

namespace
{

const int SamplesPerState = 3; //!< PadState に含まれるセンサーのサンプル数

const nn::hid::DirectionState    DefaultBaseDirection             = nn::hid::detail::UnitDirection;
const float                      DefaultBaseDirectionRevisePower  = 0.0f;

bool IsSensorValid( const nn::xcd::SensorState& sensor )
{
    return !( sensor.x == 0 &&
              sensor.y == 0 &&
              sensor.z == 0 );
}

int GetLostPackets( int previousNumber, int currentNumber )
{
    const int SamplingNumberMax = 255; //!< PadState.samplingNumber の最大値

    if( currentNumber < previousNumber ) // カウンタが一周回った時
    {
        return currentNumber + SamplingNumberMax - previousNumber;
    }
    return ( currentNumber - previousNumber - 1 );
}

void GetDeltaCountForLinearInterpolate( nn::xcd::SensorState* pOutDelta, const nn::xcd::SensorState& prev, const nn::xcd::SensorState& current, const int16_t lostSampleCount )
{
    NN_SDK_REQUIRES(pOutDelta);

    int16_t n = lostSampleCount + 1;
    pOutDelta->x = ( current.x - prev.x ) / n;
    pOutDelta->y = ( current.y - prev.y ) / n;
    pOutDelta->z = ( current.z - prev.z ) / n;
}

}

namespace nn { namespace hid { namespace detail {

SixAxisSensorProcessor::SixAxisSensorProcessor() NN_NOEXCEPT :
    m_Accelerometer(),
    m_Gyroscope(),
    m_SamplingNumber(0),
    m_SamplingIntervalSeconds(0.005f), // デフォルト 5ms
    m_IsSamplingEnabled(false),
    m_IsCalibrated(false),
    m_IsBaseDirReviseEnabled(false),
    m_IsSixAxisSensorFusionEnabled(true),
    m_BaseDirection(DefaultBaseDirection),
    m_BaseDirectionRevisePower(DefaultBaseDirectionRevisePower)
{
    const nn::util::Float3 InitialAngle = { {{0.0f, 0.0f, 0.0f}} };
    SetAngle(InitialAngle);
    SetDirection(UnitDirection);
}

void SixAxisSensorProcessor::ControlSixAxisSensor( const bool& enableSampling ) NN_NOEXCEPT
{
    m_IsSamplingEnabled = enableSampling;

    if(!enableSampling)
    {
        m_IsCalibrated = false;
    }
}

void SixAxisSensorProcessor::ControlBaseDirRevise(const bool& enable) NN_NOEXCEPT
{
    m_IsBaseDirReviseEnabled = enable;
}

bool SixAxisSensorProcessor::IsBaseDirReviseEnabled() const NN_NOEXCEPT
{
    return m_IsBaseDirReviseEnabled;
}

void SixAxisSensorProcessor::EnableSixAxisSensorFusion(const bool& enable) NN_NOEXCEPT
{
    m_IsSixAxisSensorFusionEnabled = enable;
}

bool SixAxisSensorProcessor::IsSixAxisSensorFusionEnabled () const NN_NOEXCEPT
{
    return m_IsSixAxisSensorFusionEnabled;
}

bool SixAxisSensorProcessor::IsSamplingEnabled() const NN_NOEXCEPT
{
    return m_IsSamplingEnabled;
}

bool SixAxisSensorProcessor::IsCalibrated() const NN_NOEXCEPT
{
    return m_IsCalibrated;
}

void SixAxisSensorProcessor::SetAccelerometerFsr(const nn::xcd::AccelerometerFsr& accFsr) NN_NOEXCEPT
{
    m_Accelerometer.SetSensorFsr(accFsr);
}

void SixAxisSensorProcessor::SetGyroscopeFsr(const nn::xcd::GyroscopeFsr& gyroFsr) NN_NOEXCEPT
{
    m_Gyroscope.SetSensorFsr(gyroFsr);
}

//!< 6軸センサーのキャリブレーションを実行します。
void SixAxisSensorProcessor::SetSensorCalibrationValue( const nn::xcd::SensorCalibrationValue& value ) NN_NOEXCEPT
{
    m_Accelerometer.SetCalibrationValue(value.accelerometerOrigin, value.accelerometerSensitivity );
    m_Gyroscope.SetCalibrationValue(value.gyroscopeOrigin, value.gyroscopeSensitivity);
    m_IsCalibrated = true;
}

void SixAxisSensorProcessor::SetAngle( const ::nn::util::Float3& angle ) NN_NOEXCEPT
{
    m_Angle = angle;
}

void SixAxisSensorProcessor::SetDirection( const nn::hid::DirectionState& direction ) NN_NOEXCEPT
{
    m_Direction = direction;
}

void SixAxisSensorProcessor::SetSixAxisSensorFusionParameters(const float& revisePower,
                                                              const float& reviseRange) NN_NOEXCEPT
{
    m_Accelerometer.SetAccelerationReviseParameters(revisePower, reviseRange);
}

void SixAxisSensorProcessor::GetSixAxisSensorFusionParameters(float* pOutRevisePower,
                                                              float* pOutReviseRange) const NN_NOEXCEPT
{
    m_Accelerometer.GetAccelerationReviseParameters(pOutRevisePower, pOutReviseRange);
}

void SixAxisSensorProcessor::ResetSixAxisSensorFusionParameters() NN_NOEXCEPT
{
    m_Accelerometer.ResetAccelerationReviseParameters();
}

void SixAxisSensorProcessor::SetBaseDirectionReviseParameters(const nn::hid::DirectionState& direction,
                                                 const float& revisePower) NN_NOEXCEPT
{
    m_BaseDirection = direction;
    m_BaseDirectionRevisePower = revisePower;
}

void SixAxisSensorProcessor::GetBaseDirectionReviseParameters(nn::hid::DirectionState* pOutDirection,
                                        float* pOutRevisePower) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutDirection);
    NN_SDK_REQUIRES_NOT_NULL(pOutRevisePower);

    *pOutDirection = m_BaseDirection;
    *pOutRevisePower = m_BaseDirectionRevisePower;
}

void SixAxisSensorProcessor::ResetBaseDirectionReviseParameters() NN_NOEXCEPT
{
    SetBaseDirectionReviseParameters(DefaultBaseDirection, DefaultBaseDirectionRevisePower);
}

void SixAxisSensorProcessor::UpdateDirectionByBaseDirection(nn::hid::DirectionState* pOutDirection,
                                               const nn::hid::DirectionState& lastDirection) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutDirection);
    float f1;
    float f2;

    //----- 各方向を動かす
    f2 = ApproximateDirectionAxis(&pOutDirection->x,
                                  lastDirection.x,
                                  m_BaseDirection.x,
                                  m_BaseDirectionRevisePower);

    f1 = ApproximateDirectionAxis(&pOutDirection->y,
                                  lastDirection.y,
                                  m_BaseDirection.y,
                                  m_BaseDirectionRevisePower);

    if ( f1 > f2 )
    {
        f2 = f1 ;
    }

    f1 = ApproximateDirectionAxis(&pOutDirection->z,
                                  lastDirection.z,
                                  m_BaseDirection.z,
                                  m_BaseDirectionRevisePower);
    if ( f1 > f2 )
    {
        f2 = f1 ;
    }

    //----- 正規化する
    nn::hid::detail::NormalizeDirectionState( pOutDirection, 2.999f ) ;

    // f2 が補正の強さ
}

void SixAxisSensorProcessor::Interpolate( nn::xcd::PadState& state, bool isConnected ) NN_NOEXCEPT
{
    const int LostPackets = GetLostPackets(m_LastPadState.sampleNumber,  state.sampleNumber);
    if( LostPackets == 0 )
    {
        return;
    }
    //NN_SDK_LOG("%d->%d | %d packets lost\n", m_LastPadState.sampleNumber, state.sampleNumber, LostPackets);

    nn::xcd::SensorState deltaAccCount;
    nn::xcd::SensorState deltaGyroCount;
    const int16_t LostSampleCount = static_cast<int16_t>( LostPackets *  SamplesPerState );
    GetDeltaCountForLinearInterpolate( &deltaAccCount, m_LastPadState.accelerometer[2], state.accelerometer[0], LostSampleCount );
    GetDeltaCountForLinearInterpolate( &deltaGyroCount, m_LastPadState.gyroscope[2], state.gyroscope[0], LostSampleCount );

    nn::xcd::PadState work = m_LastPadState;
    //NN_SDK_LOG("DeltaAcc=(%d,%d,%d)\n", deltaAccCount.x, deltaAccCount.y, deltaAccCount.z );
    //NN_SDK_LOG("\t(%d,%d,%d)\n", m_LastPadState.accelerometer[2].x,
    //                             m_LastPadState.accelerometer[2].y,
    //                             m_LastPadState.accelerometer[2].z );
    for(int i=0; i < LostPackets; i++ )
    {
        for(int j=0; j < SamplesPerState; j++ )
        {
            int prevIndex = ( j==0 ) ? 2 : j - 1;
            work.accelerometer[j].x = work.accelerometer[prevIndex].x + deltaAccCount.x;
            work.accelerometer[j].y = work.accelerometer[prevIndex].y + deltaAccCount.y;
            work.accelerometer[j].z = work.accelerometer[prevIndex].z + deltaAccCount.z;
            work.gyroscope[j].x = work.gyroscope[prevIndex].x + deltaGyroCount.x;
            work.gyroscope[j].y = work.gyroscope[prevIndex].y + deltaGyroCount.y;
            work.gyroscope[j].z = work.gyroscope[prevIndex].z + deltaGyroCount.z;

            //NN_SDK_LOG("\t(%d,%d,%d)\n", work.accelerometer[j].x, work.accelerometer[j].y, work.accelerometer[j].z );
        }
        Sampling(work, isConnected);
    }
    //NN_SDK_LOG("\t(%d,%d,%d)\n", state.accelerometer[0].x,
    //                             state.accelerometer[0].y,
    //                             state.accelerometer[0].z );

}

void SixAxisSensorProcessor::Sampling( nn::xcd::PadState& state, bool isConnected ) NN_NOEXCEPT
{
    // 前回サンプリングしたデータと同一の場合はスキップ
    if(m_LastPadState.sampleNumber == state.sampleNumber)
    {
        return;
    }

    for(int j=0; j< SamplesPerState; j++)
    {
        if( !m_IsSamplingEnabled )
        {
            // サンプリング開始前は無入力状態にする
            m_Accelerometer.GetNoInputCount( &state.accelerometer[j] );
            m_Gyroscope.GetNoInputCount( &state.gyroscope[j] );
        }
        else if( !IsSensorValid(state.accelerometer[j]) )
        {
            // 加速度センサーの生値がオールゼロのときは不正と見做し、前回サンプルを採用する。
            nn::xcd::SensorState *pAcc = (j==0) ? &m_LastPadState.accelerometer[2] : &state.accelerometer[j - 1];
            nn::xcd::SensorState *pGyro = (j==0) ? &m_LastPadState.gyroscope[2] : &state.gyroscope[j - 1];
            state.accelerometer[j] = *pAcc;
            state.gyroscope[j] = *pGyro;
        }

        // 新しくサンプリングされた生値を元に内部ステートを更新
        Sampling(state.accelerometer[j],
                 state.gyroscope[j],
                 ::nn::TimeSpan::FromMilliSeconds(static_cast<int64_t>(m_SamplingIntervalSeconds * 1000)),
                 isConnected);
    }
    m_LastPadState = state;
}

void SixAxisSensorProcessor::Sampling(const nn::xcd::SensorState& accelerationCount,
                                      const nn::xcd::SensorState& angularVelocityCount,
                                      const nn::TimeSpan& deltaTime,
                                      bool isConnected ) NN_NOEXCEPT
{
    nn::hid::SixAxisSensorState sensorState;

    m_LastDirection = m_Direction;

    sensorState.deltaTime      = deltaTime;
    sensorState.samplingNumber = m_SamplingNumber++;
    sensorState.attributes.Set<SixAxisSensorAttribute::IsConnected>(isConnected);

    // 加速度値[G] の生成
    m_Accelerometer.GetState(&sensorState.acceleration, accelerationCount);

    // 角速度値[dps] の生成
    m_Gyroscope.GetState(&sensorState.angularVelocity, angularVelocityCount);

    // 回転角の更新
    UpdateAngle(sensorState.angularVelocity );

    // 姿勢値の更新
    // 異常値を弾くワークアラウンド
    if( sensorState.acceleration.x != 0.0f ||
        sensorState.acceleration.y != 0.0f ||
        sensorState.acceleration.z != 0.0f )
    {
        // 角速度による姿勢値の補正
        m_Gyroscope.UpdateDirection( &m_Direction, sensorState.angularVelocity, m_SamplingIntervalSeconds );

        // 基本姿勢による姿勢値の補正
        if(m_IsBaseDirReviseEnabled)
        {
            UpdateDirectionByBaseDirection(&m_Direction,
                                           m_LastDirection);
        }

        // 加速度による姿勢値の補正
        if(m_IsSixAxisSensorFusionEnabled)
        {
            m_Accelerometer.UpdateDirection( &m_Direction, sensorState.acceleration );
        }
    }

    sensorState.direction = m_Direction;
    sensorState.angle     = m_Angle;

    m_Lifo.Append(sensorState); // LIFO に格納
}

void SixAxisSensorProcessor::UpdateAngle( const nn::util::Float3& angularVelocity ) NN_NOEXCEPT
{
    m_Angle.x += angularVelocity.x * m_SamplingIntervalSeconds;
    m_Angle.y += angularVelocity.y * m_SamplingIntervalSeconds;
    m_Angle.z += angularVelocity.z * m_SamplingIntervalSeconds;
}

int SixAxisSensorProcessor::GetSixAxisSensorStates(SixAxisSensorState* pOutStates, int count ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutStates);

    auto ret = m_Lifo.Read(pOutStates, count);
    // DumpStateForDev(pOutStates, count);
    return ret;
}

void SixAxisSensorProcessor::DumpStateForDev(const SixAxisSensorState* states, int count) const NN_NOEXCEPT
{
    NN_UNUSED(states);
    NN_SDK_REQUIRES_NOT_NULL(states);

    for(int j=count - 1; 0 <= j; j--)
    {
#if 0
        NN_SDK_LOG("================================\n");
        NN_SDK_LOG("SamplingNumber | 0x%08llx\n", states[j].samplingNumber );
        NN_SDK_LOG("DeltaTime      | %08lld\n", states[j].deltaTime.GetMicroSeconds() );

        // Processed
        NN_SDK_LOG("Acc[G]         | %05.5f,%05.5f,%05.5f\n", states[j].acceleration.x, states[j].acceleration.y, states[j].acceleration.z );
        NN_SDK_LOG("Gyro[/360dps]  | %05.5f,%05.5f,%05.5f\n", states[j].angularVelocity.x, states[j].angularVelocity.y, states[j].angularVelocity.z );
        NN_SDK_LOG("Angle[deg]     | %05.5f,%05.5f,%05.5f\n", states[j].angle.x, states[j].angle.y, states[j].angle.z );
        NN_SDK_LOG("Direction.x    | %05.5f,%05.5f,%05.5f\n", states[j].direction.x.x, states[j].direction.x.y, states[j].direction.x.z );
        NN_SDK_LOG("Direction.y    | %05.5f,%05.5f,%05.5f\n", states[j].direction.y.x, states[j].direction.y.y, states[j].direction.y.z );
        NN_SDK_LOG("Direction.z    | %05.5f,%05.5f,%05.5f\n", states[j].direction.z.x, states[j].direction.z.y, states[j].direction.z.z );
        NN_SDK_LOG("ZeroAcc[count] | %05.5f,%05.5f,%05.5f\n", m_AccelerationZeroCount.x, m_AccelerationZeroCount.y, m_AccelerationZeroCount.z );
        NN_SDK_LOG("ZeroGyro[count]| %05.5f,%05.5f,%05.5f\n", m_AngularVelocityZeroCount.x, m_AngularVelocityZeroCount.y, m_AngularVelocityZeroCount.z );
        NN_SDK_LOG("Attitude.x     | %05.5f,%05.5f,%05.5f\n", states[j].direction.x.x, states[j].direction.x.y, states[j].direction.x.z );
        NN_SDK_LOG("Attitude.y     | %05.5f,%05.5f,%05.5f\n", states[j].direction.y.x, states[j].direction.y.y, states[j].direction.y.z );
        NN_SDK_LOG("Attitude.z     | %05.5f,%05.5f,%05.5f\n", states[j].direction.z.x, states[j].direction.z.y, states[j].direction.z.z );
#else
        #if 1
        // Attitude
        NN_SDK_LOG("%05.5f,%05.5f,%05.5f,", states[j].direction.x.x, states[j].direction.x.y, states[j].direction.x.z );
        NN_SDK_LOG("%05.5f,%05.5f,%05.5f,", states[j].direction.y.x, states[j].direction.y.y, states[j].direction.y.z );
        NN_SDK_LOG("%05.5f,%05.5f,%05.5f,", states[j].direction.z.x, states[j].direction.z.y, states[j].direction.z.z );
        // 3 axis data
        NN_SDK_LOG("%05.5f,%05.5f,%05.5f,", states[j].acceleration.x, states[j].acceleration.y, states[j].acceleration.z );
        NN_SDK_LOG("%05.5f,%05.5f,%05.5f,", states[j].angularVelocity.x, states[j].angularVelocity.y, states[j].angularVelocity.z );
        NN_SDK_LOG("%05.5f,%05.5f,%05.5f,", states[j].angle.x, states[j].angle.y, states[j].angle.z );
        // Gyro zero count
        nn::util::Float3 gyroZeroCount;
        m_Gyroscope.GetZeroCountForDev(&gyroZeroCount);
        NN_SDK_LOG("%05.5f,%05.5f,%05.5f\n", gyroZeroCount.x, gyroZeroCount.y, gyroZeroCount.z );

        #endif
#endif
    }
}

}}} // namespace nn::hid::detail
