﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>

#include "hid_CalcUtil.h"
#include "hid_Accelerometer.h"

namespace {

// TORIAEZU: LSM6DS3 の Typ値
const nn::xcd::SensorState TypicalOrigin       = { 0, 0, 0 };
const nn::xcd::SensorState TypicalSensitivity  = { 16384, 16384, 16384 };

const nn::util::Float3 DefaultLastAcceleration = {{{ 0.0f, 0.0f, -1.0f }}};
const nn::hid::detail::AccelerationPlayMode DefaultMode = nn::hid::detail::AccelerationPlayMode::AccelerationPlayMode_Tight;

const float  DefaultPlayRadius  = 0.0f;
const float  DefaultSensitivity = 1.0f;
const float  DefaultRevisePower = 0.030f;
const float  DefaultReviseRange = 0.400f;
const bool   DefaultReviseFlag  = true;

const nn::xcd::AccelerometerFsr DefaultFsr = nn::xcd::AccelerometerFsr_8G;

float CalculateAccelerationLoose(float currentAcc,
                                 float lastAcc,
                                 float playRadius,
                                 float sensitivity)
{
    const float Delta = currentAcc - lastAcc; //----- 目標値への足りない差分

    float  f1 = ( Delta < 0.0f ) ? -Delta : Delta;

    //----- 遊び内外での引き寄せ率計算
    if ( f1 >= playRadius )
    {
        //----- 遊び外であれば追従性を100%適用
        f1 = 1.0f ;
    }
    else
    {
        //----- 遊び内であれば目的に近いほど追従性を弱くする
        f1 /= playRadius ;
        f1 *= f1 ;      // ２乗
        f1 *= f1 ;      // ４乗
    }

    f1 *= sensitivity ;

    //----- 引き寄せ
    return static_cast<float>(lastAcc + f1 * Delta);
}


float CalculateAccelerationTight(float currentAcc,
                                 float lastAcc,
                                 float playRadius,
                                 float sensitivity)
{
    const float Delta = currentAcc - lastAcc; //----- 目標値への足りない差分
    float  calculatedAcceleration = lastAcc;

    // 遊び範囲外の場合は感度に応じた補正を掛ける
    if ( Delta < -playRadius )
    {
        calculatedAcceleration += ( Delta + playRadius ) * sensitivity;
    }
    else if ( Delta > playRadius )
    {
        calculatedAcceleration += ( Delta - playRadius ) * sensitivity;
    }

    return calculatedAcceleration;
}

float GetFullScaleRangeValue(const nn::xcd::AccelerometerFsr& accFsr)
{
    float accFsrValue = 1.0f;

    switch (accFsr)
    {
    case nn::xcd::AccelerometerFsr_2G:
        accFsrValue = 2.0f;
        break;
    case nn::xcd::AccelerometerFsr_4G:
        accFsrValue = 4.0f;
        break;
    case nn::xcd::AccelerometerFsr_8G:
        accFsrValue = 8.0f;
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
    return accFsrValue;
}

}

namespace nn { namespace hid { namespace detail {

Accelerometer::Accelerometer() NN_NOEXCEPT :
    m_LastAcceleration(DefaultLastAcceleration) ,
    m_PlayMode(DefaultMode)                     ,
    m_PlayRadius(DefaultPlayRadius)             ,
    m_Sensitivity(DefaultSensitivity)           ,
    m_RevisePower(DefaultRevisePower)           ,
    m_ReviseRange(DefaultReviseRange)           ,
    m_ReviseFlag(DefaultReviseFlag)             ,
    m_Fsr(DefaultFsr)
{
    SetCalibrationValue( TypicalOrigin, TypicalSensitivity ); // TORIAEZU: LSM6DS3 の Typ値を設定
}

void Accelerometer::GetNoInputCount( nn::xcd::SensorState* pOutCount ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    const int16_t Count1G = static_cast<int16_t>( ( 1.0f / m_Scale.z ) + m_ZeroCount.z );
    pOutCount->x = 0;
    pOutCount->y = 0;
    pOutCount->z = Count1G;
}

void Accelerometer::GetState(nn::util::Float3* pOutState, const nn::xcd::SensorState& accelerationCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutState);

    nn::util::Float3 calculatedAcceleration;

    ConvertToAcceleration(&calculatedAcceleration, accelerationCount);
    ReviseAccelerationByPlayMode(pOutState, calculatedAcceleration);

    // 値域を制限するためクランプ
    const float AccelerationMax = GetFullScaleRangeValue(m_Fsr);
    const nn::util::Float3 Clamp = {{{ AccelerationMax, AccelerationMax, AccelerationMax }}};
    nn::hid::detail::ClampVector(pOutState, Clamp );

    UpdateLastAcceleration(*pOutState);
}

void Accelerometer::SetScaleValue() NN_NOEXCEPT
{
    const float Delta = 1.0f - 0.0f;   // 1Gスケール - ゼロ点オフセット
    float fsrScale;

    // TORIAEZU: Fsr による係数, 2G のFSRを基準とした CAL 値が取得される
    switch (m_Fsr)
    {
    case nn::xcd::AccelerometerFsr_8G:
        fsrScale = 4.00f;
        break;
    case nn::xcd::AccelerometerFsr_4G:
        fsrScale = 2.00f;
        break;
    case nn::xcd::AccelerometerFsr_2G:
        fsrScale = 1.00f;
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    m_Scale.x = Delta / ( m_SensitivityCount.x - m_ZeroCount.x ) * fsrScale;
    m_Scale.y = Delta / ( m_SensitivityCount.y - m_ZeroCount.y ) * fsrScale;
    m_Scale.z = Delta / ( m_SensitivityCount.z - m_ZeroCount.z ) * fsrScale;
}

void Accelerometer::ConvertToAcceleration( nn::util::Float3* pOut, const nn::xcd::SensorState& count ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOut);

    pOut->x = ( count.x - m_ZeroCount.x ) * m_Scale.x;
    pOut->y = ( count.y - m_ZeroCount.y ) * m_Scale.y;
    pOut->z = ( count.z - m_ZeroCount.z ) * m_Scale.z;
}

void Accelerometer::ReviseAccelerationByPlayMode( nn::util::Float3* pOut, const nn::util::Float3& acceleration ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOut);

    pOut->x = CalcAcceleration( acceleration.x, m_LastAcceleration.x );
    pOut->y = CalcAcceleration( acceleration.y, m_LastAcceleration.y );
    pOut->z = CalcAcceleration( acceleration.z, m_LastAcceleration.z );
}

void Accelerometer::UpdateLastAcceleration( const nn::util::Float3& acceleration ) NN_NOEXCEPT
{
    m_LastAcceleration = acceleration;
}

float Accelerometer::CalcAcceleration(const float& currentAcc, const float& lastAcc ) const NN_NOEXCEPT
{
    switch( m_PlayMode )
    {
    case AccelerationPlayMode::AccelerationPlayMode_Loose:
        return CalculateAccelerationLoose(currentAcc, lastAcc, m_PlayRadius, m_Sensitivity);
    case AccelerationPlayMode::AccelerationPlayMode_Tight:
        return CalculateAccelerationTight(currentAcc, lastAcc, m_PlayRadius, m_Sensitivity);
    default: NN_UNEXPECTED_DEFAULT;
    }
}

float Accelerometer::UpdateDirection( nn::hid::DirectionState* pOutDirection, const nn::util::Float3& acceleration ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutDirection);

    // __VPADReviseDirAcc を参照
    nn::util::Float3          v1;
    nn::util::Float3          v2;
    nn::util::Float3          vec;
    nn::util::Float3          g = {{{ 0.0f, 0.0f, -1.0f }}};
    nn::hid::DirectionState   d1;
    nn::hid::DirectionState   d2;
    float                     f1;
    float                     level;

    if( !m_ReviseFlag )
    {
        return 0.0f;
    }

    f1 = acceleration.x * acceleration.x + acceleration.y * acceleration.y + acceleration.z * acceleration.z ;
    if ( f1 == 0.0f )
    {
        return ( 0.0f ) ;
    }

    f1 = Sqrt( f1 ) ;
    if ( f1 < 1.0f )
    {
        if ( f1 <= (1.0f - m_ReviseRange) )
        {
            return ( 0.0f ) ;
        }
        level = ( f1 - (1.0f - m_ReviseRange) )
            * (1.0f / m_ReviseRange) ;
    }
    else
    {
        if ( f1 >= (1.0f + m_ReviseRange) )
        {
            return ( 0.0f ) ;
        }
        level = ( f1 - (1.0f + m_ReviseRange) )
            * (-1.0f / m_ReviseRange) ;
    }
    level *= level ;
    level *= m_RevisePower ;
    //level = 1.0f; // 重力成分により完全に補正する

    f1 = 1.0f / f1 ;
    vec.x = f1 * acceleration.x ;
    vec.y = f1 * acceleration.y ;
    vec.z = f1 * acceleration.z ;

    v1.x = vec.x * pOutDirection->x.x + vec.y * pOutDirection->y.x + vec.z * pOutDirection->z.x ;
    v1.y = vec.x * pOutDirection->x.y + vec.y * pOutDirection->y.y + vec.z * pOutDirection->z.y ;
    v1.z = vec.x * pOutDirection->x.z + vec.y * pOutDirection->y.z + vec.z * pOutDirection->z.z ;

    v2.x = ( g.x - v1.x ) * level + v1.x ;
    v2.y = ( g.y - v1.y ) * level + v1.y ;
    v2.z = ( g.z - v1.z ) * level + v1.z ;
    if ( nn::hid::detail::Normalize( &v2 ) == 0.0f ) {
        return ( 0.0f ) ;
    }

    nn::hid::detail::MakeVecDir( &d1, v1, v2 );                        // v1 が v2 に向く行列 d1 を作る
    nn::hid::detail::MultiplyDirection( &d2, *pOutDirection, d1 ) ;    // m_Direction を d1 で変換
    *pOutDirection = d2 ;

    nn::hid::detail::NormalizeDirectionState( pOutDirection, 2.999f ) ;

    //----- 補正の強さ計算
    v1.x -= v2.x ;
    v1.y -= v2.y ;
    v1.z -= v2.z ;
    f1 = Sqrt( v1.x * v1.x + v1.y * v1.y + v1.z * v1.z ) ;

    return ( f1 ) ;
}

void Accelerometer::SetSensorFsr(const nn::xcd::AccelerometerFsr& accFsr) NN_NOEXCEPT
{
    m_Fsr = accFsr;
    SetScaleValue();
}

void Accelerometer::GetSensorFsr(nn::xcd::AccelerometerFsr* pOutAccFsr) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutAccFsr);

    *pOutAccFsr = m_Fsr;
}

void Accelerometer::SetCalibrationValue( const nn::xcd::SensorState& accelerometerOrigin,
                                         const nn::xcd::SensorState& accelerometerSensitivity ) NN_NOEXCEPT
{
    m_ZeroCount.x = accelerometerOrigin.x;
    m_ZeroCount.y = accelerometerOrigin.y;
    m_ZeroCount.z = accelerometerOrigin.z;

    m_SensitivityCount.x = accelerometerSensitivity.x;
    m_SensitivityCount.y = accelerometerSensitivity.y;
    m_SensitivityCount.z = accelerometerSensitivity.z;

    SetScaleValue();
}


//!< 加速度センサーの遊びの挙動方法を設定します。
void Accelerometer::SetPlayMode( const AccelerationPlayMode& mode ) NN_NOEXCEPT
{
    m_PlayMode = mode;

}

//!< 加速度センサーの遊びの挙動方法を取得します。
void Accelerometer::GetPlayMode( AccelerationPlayMode* pOutMode ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutMode);

    *pOutMode = m_PlayMode;
}

//!< 加速度センサーの遊びの挙動方法を設定します。
void Accelerometer::SetParameters( const float& playRadius, const float& sensitivity ) NN_NOEXCEPT
{
    m_PlayRadius  = playRadius;
    m_Sensitivity = sensitivity;

}

//!< 加速度センサーの遊びの挙動方法を取得します。
void Accelerometer::GetParameters( float* pOutPlayRadius, float* pOutSensitivity ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutPlayRadius);
    NN_SDK_REQUIRES_NOT_NULL(pOutSensitivity);

    *pOutPlayRadius  = m_PlayRadius;
    *pOutSensitivity = m_Sensitivity;
}

void Accelerometer::SetAccelerationReviseParameters( const float& revisePower, const float& reviseRange ) NN_NOEXCEPT
{
    m_RevisePower = revisePower;
    m_ReviseRange = reviseRange;
}
void Accelerometer::GetAccelerationReviseParameters( float* pOutRevisePower, float* pOutReviseRange ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutRevisePower);
    NN_SDK_REQUIRES_NOT_NULL(pOutReviseRange);

    *pOutRevisePower = m_RevisePower;
    *pOutReviseRange = m_ReviseRange;
}

void Accelerometer::ResetAccelerationReviseParameters() NN_NOEXCEPT
{
    m_RevisePower = DefaultRevisePower;
    m_ReviseRange = DefaultReviseRange;
}

void Accelerometer::ControlAccelerationRevise( bool isEnable ) NN_NOEXCEPT
{
    m_ReviseFlag = isEnable;
}

bool Accelerometer::IsEnableAccelerationRevise() const NN_NOEXCEPT
{
    return m_ReviseFlag;
}

}}} // namespace nn::hid::detail
