﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cstdio>
#include <cstring>

#include <nnt/nntest.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/init.h>
#include <nn/os.h>

#include <nn/TargetConfigs/build_Platform.h>

#include <nn/ssl.h>
#include <nn/socket.h>
#include <nn/nifm.h>

#include <Common/testCommonUtil.h>
#include <Common/testInfraInfo.h>
#include <Common/testServerPki.h>

SslTestCommonUtil        g_CommonUtil;
NN_ALIGNAS(4096) uint8_t g_SocketMemoryPoolBuffer[nn::socket::DefaultSocketMemoryPoolSize];

extern "C" void nninitStartup()
{
    NN_LOG("nninitStartup loaded %p\n", nninitStartup);
    // メモリヒープの全体サイズを設定する
    const size_t MemoryHeapSize = 128 * 1024 * 1024;
    auto result = nn::os::SetMemoryHeapSize( MemoryHeapSize );

    ASSERT_TRUE( result.IsSuccess() );

    // メモリヒープから malloc で使用するメモリ領域を確保
    uintptr_t address = 0;

    result = nn::os::AllocateMemoryBlock( &address, MemoryHeapSize );
    ASSERT_TRUE( result.IsSuccess() );

    // malloc 用のメモリ領域を設定する
    nn::init::InitializeAllocator( reinterpret_cast<void*>(address), MemoryHeapSize );
}

TEST(InitTest, Success)
{
    ASSERT_TRUE(g_CommonUtil.SetupNetwork().IsSuccess());
    ASSERT_TRUE(nn::socket::Initialize(
        g_SocketMemoryPoolBuffer,
        nn::socket::DefaultSocketMemoryPoolSize,
        nn::socket::MinSocketAllocatorSize,
        nn::socket::DefaultConcurrencyLimit).IsSuccess());
    ASSERT_TRUE(nn::ssl::Initialize().IsSuccess());
}

TEST(TimeValidation, Success)
{
    SimpleHttpsClient httpsClient(false, ServerName, ServerPort_ExpiredCert);
#ifdef NO_RESOLVER
    httpsClient.SetIpAddress(ServerIpAddress);
#endif
    do {
        nn::ssl::Connection::VerifyOption verifyOption =
            nn::ssl::Connection::VerifyOption::VerifyOption_None;
        verifyOption |= nn::ssl::Connection::VerifyOption::VerifyOption_PeerCa;
        verifyOption |= nn::ssl::Connection::VerifyOption::VerifyOption_HostName;
        verifyOption |= nn::ssl::Connection::VerifyOption::VerifyOption_DateCheck;

        ASSERT_TRUE(httpsClient.Initialize(
            nn::ssl::Context::SslVersion::SslVersion_Auto,
            verifyOption));

        ASSERT_TRUE(httpsClient.ImportServerPki(
            g_pTestCaCertExpired,
            sizeof(g_pTestCaCertExpired),
            nn::ssl::CertificateFormat_Pem));

        bool ret = httpsClient.PerformSslHandshake(false);
        EXPECT_TRUE(!ret);
        EXPECT_TRUE(nn::ssl::ResultSslErrorExpiredCertificate::Includes(httpsClient.GetLastResult()));

    } while (NN_STATIC_CONDITION(false));

    httpsClient.Finalize();
}

TEST(FinalizeTest, Success)
{
    EXPECT_TRUE(nn::ssl::Finalize().IsSuccess());
    nn::socket::Finalize();
    g_CommonUtil.FinalizeNetwork();
}
