﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nn/os.h>
#include <nn/init.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/socket.h>
#include <nn/ssl.h>
#include <curl/curl.h>
#include <nnt.h>

#include <Common/testCommonUtil.h>
#include <Common/testInfraInfo.h>

extern "C" void nninitStartup()
{
    NN_LOG("nninitStartup loaded %p\n", nninitStartup);
    // メモリヒープの全体サイズを設定する
    const size_t MemoryHeapSize = 128 * 1024 * 1024;
    auto result = nn::os::SetMemoryHeapSize( MemoryHeapSize );

    ASSERT_TRUE( result.IsSuccess() );

    // メモリヒープから malloc で使用するメモリ領域を確保
    uintptr_t address = 0;

    result = nn::os::AllocateMemoryBlock( &address, MemoryHeapSize );
    ASSERT_TRUE( result.IsSuccess() );

    // malloc 用のメモリ領域を設定する
    nn::init::InitializeAllocator( reinterpret_cast<void*>(address), MemoryHeapSize );
}

namespace
{
const char UrlString[] = "https://ntd-net-server1.ntd.nintendo.com";
NN_ALIGNAS(4096) uint8_t g_SocketMemoryPoolBuffer[nn::socket::DefaultSocketMemoryPoolSize];

class TestManager : public testing::Test
{
protected:
    virtual void SetUp()
    {
        nn::Result result = nn::socket::Initialize(
            g_SocketMemoryPoolBuffer,
            nn::socket::DefaultSocketMemoryPoolSize,
            nn::socket::MinSocketAllocatorSize,
            nn::socket::DefaultConcurrencyLimit);
        EXPECT_TRUE(result.IsSuccess());

        ASSERT_TRUE(curl_global_init(CURL_GLOBAL_DEFAULT) == CURLE_OK);
    }

    virtual void TearDown()
    {
        curl_global_cleanup();
        nn::socket::Finalize();
    }
};

size_t CurlWriteFunction(char *pData, size_t blobsize, size_t blobcount, void *userdata) NN_NOEXCEPT
{
    int32_t count = static_cast<int32_t>(blobsize * blobcount);
    return count;
}

size_t CurlSslContextCallback(CURL* pCurl, void* pSslContext, void* pUserData)
{
    // Obtain pointer to the SSL context passed by CURLOPT_SSL_CTX_FUNCTION
    nn::ssl::Context* pContext = reinterpret_cast<nn::ssl::Context*>(pSslContext);

    // Create SSL context
    nn::Result result = pContext->Create(nn::ssl::Context::SslVersion_Auto);
    if( result.IsFailure() )
    {
        return (size_t) - 1;
    }

    return 0;
}
}

//-------------------------------------------------------------------------------------------------
//  Tests
//-------------------------------------------------------------------------------------------------
// START tests for SIGLONTD-8147 ------------------------------------------------------------------
TEST_F(TestManager, test1)
{
}

TEST_F(TestManager, test2)
{
}

TEST_F(TestManager, test3)
{
}

TEST_F(TestManager, test4)
{
}

TEST_F(TestManager, test5)
{
}

TEST_F(TestManager, test6)
{
}

TEST_F(TestManager, test7)
{
}

TEST_F(TestManager, test8)
{
}
// END tests for SIGLONTD-8147 --------------------------------------------------------------------

TEST(TestWithPerform, Success)
{
    SslTestCommonUtil commonUtil;

    ASSERT_TRUE(commonUtil.SetupNetwork().IsSuccess());
    nn::Result result = nn::socket::Initialize(
        g_SocketMemoryPoolBuffer,
        nn::socket::DefaultSocketMemoryPoolSize,
        nn::socket::MinSocketAllocatorSize,
        nn::socket::DefaultConcurrencyLimit);
    EXPECT_TRUE(result.IsSuccess());

    for (int i = 0; i < 64; i++)
    {
        curl_global_init(CURL_GLOBAL_DEFAULT);

        CURL*    curl = nullptr;
        CURLcode res  = CURLE_OK;

        curl = curl_easy_init();
        ASSERT_TRUE(curl != nullptr);

        curl_easy_setopt(curl, CURLOPT_URL, UrlString);
        curl_easy_setopt(curl, CURLOPT_SSL_VERIFYPEER, 0L);
        curl_easy_setopt(curl, CURLOPT_SSL_VERIFYHOST, 0L);
        curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION,  CurlWriteFunction);
        curl_easy_setopt(curl, CURLOPT_SSL_CTX_FUNCTION, CurlSslContextCallback);

        res = curl_easy_perform(curl);
        EXPECT_TRUE(res == CURLE_OK);

        long httpResponseCode = 0;
        curl_easy_getinfo(curl, CURLINFO_RESPONSE_CODE, &httpResponseCode);
        EXPECT_TRUE(httpResponseCode == 200);

        curl_easy_cleanup(curl);
        curl_global_cleanup();
    }

    nn::socket::Finalize();
    commonUtil.FinalizeNetwork();
}
