﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cstdio>
#include <cstring>

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/init.h>
#include <nn/os.h>

#include <curl/curl.h>

#include <nn/ssl.h>
#include <nn/socket.h>
#include <nn/nifm.h>
#include <nn/nifm/nifm_NetworkConnection.h>
#include <nn/nifm/nifm_ApiRequest.h>
#include <nn/nifm/nifm_ApiClientManagement.h>
#include <nn/nifm/nifm_ApiForTest.h>

#include <Common/testCommonUtil.h>

// ------------------------------------------------------------------------------------------------
// Utils
// ------------------------------------------------------------------------------------------------
#define PRINT_HEX_TABLE(pData,len)                                  \
do                                                                  \
{                                                                   \
    unsigned char* pChar = reinterpret_cast<unsigned char*>(pData); \
    for(uint32_t i = 0; i < len; i++) {                             \
        NN_LOG("0x%02x", pChar[i]);                                 \
        if((i + 1) % 16 == 0) {                                     \
            NN_LOG("\n");                                           \
        } else {                                                    \
            if(((i + 1) % 8 == 0) && (i % 16 != 0)) {               \
                NN_LOG(" | ");                                      \
            } else {                                                \
                NN_LOG(" ");                                        \
            }                                                       \
        }                                                           \
    }                                                               \
} while(NN_STATIC_CONDITION(false))

namespace
{

static nn::nifm::NetworkConnection  *g_pNifmConnection = nullptr;

int CreateTcpSocketInternal(
    bool bEstablishConn,
    uint16_t portNumber,
    const char* pInHostName,
    uint32_t ipAddress)
{
    int                tcpSocket;
    nn::socket::InAddr     inetAddr;
    nn::socket::SockAddrIn serverAddr;

    memset(&inetAddr, 0x00, sizeof(inetAddr));

    if(pInHostName == nullptr && ipAddress == 0)
    {
        NN_LOG("Host name and IP address was not passed.\n");
        return -1;
    }

    tcpSocket = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp);
    if(tcpSocket < 0)
    {
        NN_LOG("Failed to create TCP socket (errno: %d)\n", errno);
        return -1;
    }
    NN_LOG("Created TCP socket (sockfd: %d).\n", tcpSocket);

    if (bEstablishConn != true)
    {
        return tcpSocket;
    }

    if(ipAddress == 0)
    {
        NN_LOG("Resolving %s\n", pInHostName);
        nn::socket::HostEnt *pHostEnt = nn::socket::GetHostEntByName(pInHostName);
        if(pHostEnt == nullptr)
        {
            NN_LOG("Failed to resolve host name (errno:%d)\n", errno);
            return -1;
        }

        // Just pick the first one
        memcpy(&inetAddr, pHostEnt->h_addr_list[0], sizeof(nn::socket::InAddr));
        serverAddr.sin_addr.S_addr = inetAddr.S_addr;
    }
    else
    {
        NN_LOG("Use 0x%x for server IP.\n", ipAddress);
        serverAddr.sin_addr.S_addr = ipAddress;
    }

    serverAddr.sin_family = nn::socket::Family::Af_Inet;
    serverAddr.sin_port   = nn::socket::InetHtons(portNumber);

    int rval = nn::socket::Connect(tcpSocket, (nn::socket::SockAddr*)&serverAddr, sizeof(serverAddr));
    if(rval < 0)
    {
        NN_LOG("Failed to establish TCP connection (errno: %d).\n", errno);
        nn::socket::Close(tcpSocket);
        return -1;
    }

    NN_LOG("Established TCP connection (addr:0x%x on port :%d).\n",
         serverAddr.sin_addr.S_addr, nn::socket::InetNtohs(serverAddr.sin_port));

    return tcpSocket;
}

int CurlTransferInfoInternal(
    void *clientp,
    curl_off_t downloadTotal,
    curl_off_t downloadCurrent,
    curl_off_t uploadTotal,
    curl_off_t uploadCurrent)
{
    static curl_off_t lastDownload = downloadCurrent;
    static curl_off_t lastUpload   = uploadCurrent;
    static nn::os::Tick lastTick   = nn::os::GetSystemTick();
    nn::os::Tick currentTick       = nn::os::GetSystemTick();

    if( downloadTotal > 0 )
    {
        static int lastPercent = 0;
        int nowPercent = static_cast<int>((double)downloadCurrent / (double)downloadTotal * 100.0);
        if( nowPercent >= lastPercent + 10 ) // Only print every 10 percent or more
        {
            lastPercent = (nowPercent / 10) * 10; // Round down to the nearest 10.
            NN_LOG("\n * Download Progress: %d%%\n", nowPercent);
        }
    }

    if( uploadTotal > 0 )
    {
        static int lastPercent = 0;
        int nowPercent = static_cast<int>((double)uploadCurrent / (double)uploadTotal * 100.0);
        if( nowPercent >= lastPercent + 10 ) // Only print every 10 percent or more
        {
            lastPercent = (nowPercent / 10) * 10; // Round down to the nearest 10.
            NN_LOG("\n * Upload Progress: %d%%\n", nowPercent);
        }
    }

    // If we have sent or received anything, reset the timer.
    if( (downloadCurrent > lastDownload) || (uploadCurrent > lastUpload) )
    {
        lastTick = currentTick;
    }
    // If we have not sent or received data in the last 'TransactionTimeoutSec' seconds,
    // cancel the transaction.
    else if( nn::os::ConvertToTimeSpan(currentTick - lastTick).GetSeconds() >=
             SimpleCurlHttpsClient::TransactionTimeoutSec )
    {
        NN_LOG("\n * Error: No data sent or received for %d seconds. Canceling transaction..\n\n",
            SimpleCurlHttpsClient::TransactionTimeoutSec);
        return -1; // Return non-zero to cancel transaction.
    }

    return 0;
}
}


// ------------------------------------------------------------------------------------------------
// SslTestCommonUtil
// ------------------------------------------------------------------------------------------------
nn::Result SslTestCommonUtil::SetupNetwork()
{
    nn::Result result = nn::ResultSuccess();
    uint32_t   timeoutSec = 0;
    result = nn::nifm::Initialize();
    if(result.IsFailure())
    {
        NN_LOG("Failed to init nifm: %d-%d\n", result.GetModule(), result.GetDescription());
        return result;
    }

    result = nn::nifm::SetExclusiveClient(nn::nifm::GetClientId());
    if( result.IsFailure() )
    {
        NN_LOG(" * Error: Failed to set exclusive client for nn::nifm. Desc: %d\n\n", result.GetDescription());
        return result;
    }

    ::g_pNifmConnection = new nn::nifm::NetworkConnection;
    if (::g_pNifmConnection == nullptr)
    {
        NN_LOG("Failed to create nifm request\n");
        result = nn::ssl::ResultInsufficientMemory();
        return result;
    }

    nn::nifm::RequestHandle requestHandle = ::g_pNifmConnection->GetRequestHandle();
    nn::nifm::SetRequestConnectionConfirmationOption(requestHandle,
                                                     nn::nifm::ConnectionConfirmationOption_Prohibited);
    nn::nifm::SetRequestPersistent(requestHandle, true);

    ::g_pNifmConnection->SubmitRequest();
    while(!::g_pNifmConnection->IsAvailable())
    {
        NN_LOG("Waiting for network interface availability...\n");
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        if(timeoutSec++ > TimeoutOnNicConfigurationInSec)
        {
            NN_LOG("Failed to setup network interface.\n");
            result = nn::ssl::ResultErrorLower();
            break;
        }
    }

    return result;
}

void SslTestCommonUtil::FinalizeNetwork()
{
    if (::g_pNifmConnection != nullptr)
    {
        ::g_pNifmConnection->CancelRequest();
        delete ::g_pNifmConnection;
        ::g_pNifmConnection = nullptr;
    }
}

int SslTestCommonUtil::CreateTcpSocket(
    bool bEstablishConn,
    uint16_t portNumber,
    const char* pInHostName,
    uint32_t ipAddress)
{
    return CreateTcpSocketInternal(
        bEstablishConn,
        portNumber,
        pInHostName,
        ipAddress);
}

void SslTestCommonUtil::CloseTcpSocket(int socketFd)
{
    if(socketFd >= 0)
    {
        nn::socket::Close(socketFd);
        NN_LOG("Closed TCP socket(sockfd: %d)\n", socketFd);
    }
}

void SslTestCommonUtil::DumpCurlResults(CURL* pInCurlHandle)
{
    CURLcode res;

    long verifyResult = 0;
    res = curl_easy_getinfo(pInCurlHandle, CURLINFO_SSL_VERIFYRESULT, &verifyResult);
    if (res != CURLE_OK)
    {
        NN_LOG("curl_easy_getinfo - CURLINFO_SSL_VERIFYRESULT failed: %d\n", res);
    }
    else
    {
        nn::Result tmpResult;
        if (nn::ssl::GetSslResultFromValue(
                &tmpResult,
                reinterpret_cast<char*>(&verifyResult),
                sizeof(verifyResult)).IsSuccess())
        {
            NN_LOG(" CURLINFO_SSL_VERIFYRESULT: SSL Result Desc - %d\n",
                tmpResult.GetDescription());
        }
        else
        {
            NN_LOG(" Failed to convert SSL result.\n");
        }
    }

    long doHandshakeResult = 0;
    res = curl_easy_getinfo(pInCurlHandle, CURLINFO_SSL_HANDSHAKE_RESULT, &doHandshakeResult);
    if (res != CURLE_OK)
    {
        NN_LOG("curl_easy_getinfo - CURLINFO_SSL_HANDSHAKE_RESULT failed: %d\n", res);
    }
    else
    {
        nn::Result tmpResult;
        if (nn::ssl::GetSslResultFromValue(
                &tmpResult,
                reinterpret_cast<char*>(&doHandshakeResult),
                sizeof(doHandshakeResult)).IsSuccess())
        {
            NN_LOG(" CURLINFO_SSL_HANDSHAKE_RESULT: SSL Result Desc - %d\n",
                tmpResult.GetDescription());
        }
        else
        {
            NN_LOG(" Failed to convert SSL result.\n");
        }
    }
}

// ------------------------------------------------------------------------------------------------
// SimpleHttpsClient
// ------------------------------------------------------------------------------------------------
SimpleHttpsClient::SimpleHttpsClient(bool IsBlocking, const char* pInHostName, uint16_t portNum):
    m_TcpSocket(-1),
    m_IsSocketImported(false),
    m_pReceivedData(nullptr),
    m_ReceivedTotalBytes(0),
    m_IsInitialized(false),
    m_LastResult(nn::ResultSuccess()),
    m_ServerCertStoreId(0),
    m_ClientCertStoreId(0),
    m_ManualIpAddress(0)
{
    strcpy(m_HostName, pInHostName);
    m_IsNonBlocking = IsBlocking;
    m_PortNumber    = portNum;
}

SimpleHttpsClient::~SimpleHttpsClient()
{
    if(m_IsInitialized == true)
    {
        this->Finalize();
    }
}

nn::Result SimpleHttpsClient::GetLastResult()
{
    return m_LastResult;
}

void SimpleHttpsClient::SetIpAddress(uint32_t address)
{
    m_ManualIpAddress = address;
}

int SimpleHttpsClient::SetupTcpSocket()
{
    if(m_HostName[0] == '\0' && m_ManualIpAddress == 0)
    {
        NN_LOG("Host name and IP address are not set\n");
        return -1;
    }

    int tcpSocket = CreateTcpSocketInternal(
        true, // Establish TCP connection
        m_PortNumber,
        (m_ManualIpAddress)?(nullptr):(m_HostName),
        m_ManualIpAddress);
    if(tcpSocket >= 0)
    {
        m_TcpSocket = tcpSocket;
    }
    return tcpSocket;
}

nn::Result SimpleHttpsClient::SetupSslContext(nn::ssl::Context::SslVersion sslVersion)
{
    nn::Result result = nn::ResultSuccess();
    do
    {
        result = m_SslContext.Create(sslVersion);
        if(result.IsFailure())
        {
            NN_LOG("Failed to create context (Description:%d)\n",
                result.GetDescription());
            break;
        }

        nn::ssl::SslContextId contextId;
        result = m_SslContext.GetContextId(&contextId);
        if(result.IsFailure())
        {
            NN_LOG("Failed to get context ID (Description:%d)\n",
                result.GetDescription());
            break;
        }
        else
        {
            NN_LOG("Created SSL context (ID:%d).\n", contextId);
        }
    }
    while (NN_STATIC_CONDITION(false));

    m_LastResult = result;
    return result;
}

nn::Result SimpleHttpsClient::SetupSslConnection(nn::ssl::Connection::VerifyOption verifyOption)
{
    nn::Result result = nn::ResultSuccess();
    do
    {
        result = m_SslConnection.Create(&m_SslContext);
        if(result.IsFailure())
        {
            NN_LOG("pSslConnection->Create failed (Description:%d)\n",
                result.GetDescription());
            break;
        }

        result = m_SslConnection.SetSocketDescriptor(m_TcpSocket);
        if(result.IsFailure())
        {
            NN_LOG("pSslConnection->SetSocketDescriptor failed (Description:%d)\n",
                result.GetDescription());
            break;
        }

        // Mark it as imported not to close it upon cleanup of SimpleHttpsClient
        m_IsSocketImported = true;

        uint32_t hostNameLen = (uint32_t)strlen(m_HostName);
        result = m_SslConnection.SetHostName(m_HostName, hostNameLen);
        if(result.IsFailure())
        {
            NN_LOG("pSslConnection->SetHostName failed (Description:%d)\n",
                result.GetDescription());
            break;
        }

        result = m_SslConnection.SetOption(nn::ssl::Connection::OptionType_SkipDefaultVerify, true);
        if(result.IsFailure())
        {
            NN_LOG("pSslConnection->SetOption failed (Description:%d)\n",
                result.GetDescription());
            break;
        }

        result = m_SslConnection.SetVerifyOption(verifyOption);
        if(result.IsFailure())
        {
            NN_LOG("pSslConnection->SetVerifyOption failed (Description:%d)\n",
                result.GetDescription());
            break;
        }

        if(m_IsNonBlocking)
        {
            result = m_SslConnection.SetIoMode(nn::ssl::Connection::IoMode_NonBlocking);
            if(result.IsFailure())
            {
                NN_LOG("Failed to configure I/O blocking mode (Description:%d)\n",
                    result.GetDescription());
                break;
            }
        }

        nn::ssl::SslConnectionId connectiontId;
        result = m_SslConnection.GetConnectionId(&connectiontId);
        if(result.IsFailure())
        {
            NN_LOG("Failed to get connection ID (Description:%d)\n",
                result.GetDescription());
            break;
        }
        else
        {
            NN_LOG("Created SSL connection (ID:%d).\n", connectiontId);
        }
    }
    while (NN_STATIC_CONDITION(false));

    m_LastResult = result;
    return result;
}

bool SimpleHttpsClient::Initialize(
    nn::ssl::Context::SslVersion SslVersion,
    nn::ssl::Connection::VerifyOption verifyOption)
{
    bool       IsSuccess = true;

    if(m_IsInitialized == true)
    {
        return false; // already initialized
    }

    do
    {
        m_pReceivedData = new char[ReadBufferSize];
        if(m_pReceivedData == nullptr)
        {
            NN_LOG("Failed to allocate memory for read buffer.\n");
            IsSuccess = false;
            break;
        }
        NN_LOG("Allocated %d bytes for receive buffer.\n", ReadBufferSize);
        memset(m_pReceivedData, 0x00, ReadBufferSize);

        if(this->SetupTcpSocket() < 0)
        {
            NN_LOG("Failed to setup TCP socket.\n");
            IsSuccess = false;
            break;
        }

        if(SetupSslContext(SslVersion).IsFailure())
        {
            NN_LOG("Failed to setup SSL context.\n");
            IsSuccess = false;
            break;
        }

        if(SetupSslConnection(verifyOption).IsFailure())
        {
            NN_LOG("Failed to setup SSL connection context.\n");
            IsSuccess = false;
            break;
        }
    }
    while (NN_STATIC_CONDITION(false));

    if((m_pReceivedData!= nullptr) && (IsSuccess == false))
    {
        delete m_pReceivedData;
        m_pReceivedData = nullptr;
    }

    NN_LOG("Initialized SimpleHttpsClient.\n");
    m_IsInitialized = true;

    return IsSuccess;
}

void SimpleHttpsClient::Finalize()
{
    // Cloes TCP socket
    if(m_TcpSocket > 0 && m_IsSocketImported != true)
    {
        nn::socket::Close(m_TcpSocket);
    }

    if(m_ServerCertStoreId != 0)
    {
        m_SslContext.RemovePki(m_ServerCertStoreId);
        m_ServerCertStoreId = 0;
    }

    // Destroy nn::ssl::Connection first
    nn::ssl::SslConnectionId connectiontId;
    if(m_SslConnection.GetConnectionId(&connectiontId).IsSuccess())
    {
        if(connectiontId != 0)
        {
            m_SslConnection.Destroy();
        }
    }

    // Destroy nn::ssl::Context after nn::ssl::Connection is destroyed
    nn::ssl::SslContextId contextId;
    if(m_SslContext.GetContextId(&contextId).IsSuccess())
    {
        if(contextId != 0)
        {
            m_SslContext.Destroy();
        }
    }

    // Free memory
    if(m_pReceivedData != nullptr)
    {
        delete m_pReceivedData;
        m_pReceivedData = nullptr;
    }

    m_IsInitialized = false;
    NN_LOG("Finalized SimpleHttpsClient.\n");
}

bool SimpleHttpsClient::PerformSslHandshake(bool g_IsPrintServerCert)
{
    nn::Result result              = nn::ResultSuccess();
    char*      serverCertBuff      = nullptr;
    uint32_t   asyncHandshakeCount = 0;
    uint32_t   serverCertLength    = 0;

    if(m_IsInitialized == false)
    {
        return false; // not initialized
    }

    if(g_IsPrintServerCert == true)
    {
        // Allocate buffer for server certificate and pass it to the SSL library
        serverCertBuff = new char[this->ServerCertBufferSize];
        if(serverCertBuff == nullptr)
        {
            return false; // Failed to allocate buffer
        }
        memset(serverCertBuff, 0x00, this->ServerCertBufferSize);

        result = m_SslConnection.SetServerCertBuffer(serverCertBuff, this->ServerCertBufferSize);
        if(result.IsFailure())
        {
            NN_LOG("SetServerCertBuffer failed (Description:%d)\n", result.GetDescription());
            delete serverCertBuff;
            m_LastResult = result;
            return false;
        }
    }

    do
    {
        result = (serverCertBuff)?
            m_SslConnection.DoHandshake(&serverCertLength, nullptr):m_SslConnection.DoHandshake();
        if(result.IsSuccess())
        {
            NN_LOG("SSL Handshake completed.\n");
            break;
        }

        if(nn::ssl::ResultBufferTooShort::Includes(result))
        {
            // Failed to store server certificate, but it is ok to continue
            NN_LOG("Provided buffer was not long enough to store server cert\n");
        }
        else if(nn::ssl::ResultIoWouldBlock::Includes(result))
        {
            if(asyncHandshakeCount++ < this->MaxAsyncDoHandshakeRetry)
            {
                NN_LOG("Handshake would block, try again after 100 msec (retry count:%d)\n",
                    asyncHandshakeCount);
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
                continue;
            }
            else
            {
                NN_LOG("Handshake failed after %d times of execution\n", asyncHandshakeCount);
                break;
            }
        }
        else
        {
            NN_LOG("SSL Handshake failed (Description:%d)\n", result.GetDescription());
            if(nn::ssl::ResultVerifyCertFailed::Includes(result))
            {
                nn::Result verifyCertError;
                result = m_SslConnection.GetVerifyCertError(&verifyCertError);
                if(result.IsFailure())
                {
                    NN_LOG("Failed to get verify cert error (Description:%d)\n",
                        result.GetDescription());
                }
                else
                {
                    NN_LOG("Certificate validation failed (Description:%d)\n",
                        verifyCertError.GetDescription());
                    result = verifyCertError;
                }
            }
            break;
        }
    }
    while (NN_STATIC_CONDITION(true));

    if(result.IsSuccess() && (serverCertLength > 0))
    {
        NN_LOG("Obtained server certiticate (%d bytes) ---------------------\n", serverCertLength);
        PRINT_HEX_TABLE(serverCertBuff, serverCertLength);
        NN_LOG("\n\n");
    }

    if(serverCertBuff != nullptr)
    {
        delete serverCertBuff;
    }

    m_LastResult = result;
    return (result.IsSuccess())?(true):(false);
}

bool SimpleHttpsClient::SendHttpGetOverSsl()
{
    char httpReqBuff[128] = {0};

    if(m_IsInitialized == false)
    {
        return false; // not initialized
    }

    sprintf(httpReqBuff, "GET / HTTP/1.0\r\nHost: %s\r\n\r\n", m_HostName);

    int sentBytes     = 0;
    nn::Result result = m_SslConnection.Write(httpReqBuff, &sentBytes, (uint32_t)strlen(httpReqBuff));
    if(result.IsFailure() || sentBytes < 0)
    {
        NN_LOG("Failed to write data. (Description:%d)\n", result.GetDescription());
    }
    else
    {
        NN_LOG("Sent HTTP request via SSL (%d bytes).\n", sentBytes);
    }

    m_LastResult = result;
    return (result.IsSuccess())?(true):(false);
}

bool SimpleHttpsClient::ReceiveAllHttpData()
{
    nn::Result                     result = nn::ResultSuccess();
    nn::ssl::Connection::PollEvent pollEvent;
    nn::ssl::Connection::PollEvent pollOutEvent;

    if(m_IsInitialized == false)
    {
        return false; // not initialized
    }

    memset(m_pReceivedData, 0x00, ReadBufferSize);
    do
    {
        int receivedBytes  = 0;
        int ReadSize       = ReadBufferSize - m_ReceivedTotalBytes;

        if(ReadSize <= 0)
        {
            NN_LOG("There's no more space to store received data (%d bytes received).\n",
                m_ReceivedTotalBytes);
            break;
        }

        if(m_IsNonBlocking)
        {
            pollEvent = nn::ssl::Connection::PollEvent::PollEvent_None;  // reset
            pollEvent |= nn::ssl::Connection::PollEvent::PollEvent_Read; // set read event

            result = m_SslConnection.Poll(&pollOutEvent, &pollEvent, MsecPollTimeout);
            if(result.IsFailure())
            {
                NN_LOG("Failed due to Poll error (Description:%d)\n", result.GetDescription());
                break;
            }

            if(nn::ssl::ResultIoTimeout::Includes(result))
            {
                NN_LOG("Finished receiving data for the timeout.\n");
                break;
            }

            if((pollOutEvent & nn::ssl::Connection::PollEvent::PollEvent_Read)
               == nn::ssl::Connection::PollEvent::PollEvent_Read)
            {
                result = m_SslConnection.Read(
                    (m_pReceivedData + m_ReceivedTotalBytes),
                    &receivedBytes,
                    ReadSize);
                if(nn::ssl::ResultIoWouldBlock::Includes(result))
                {
                    continue;
                }
            }
        }
        else
        {
            result = m_SslConnection.Read(
                (m_pReceivedData + m_ReceivedTotalBytes),
                &receivedBytes,
                ReadSize);
        }

        if(result.IsFailure() || receivedBytes < 0)
        {
            NN_LOG("Failed to read data (Description:%d)\n", result.GetDescription());
            break;
        }

        if(receivedBytes == 0)
        {
            NN_LOG("Connection closed by the server.\n");
            break;
        }

        m_ReceivedTotalBytes += receivedBytes;
        NN_LOG("Received %d bytes (total:%d bytes)\n", receivedBytes, m_ReceivedTotalBytes);
    } while(NN_STATIC_CONDITION(true));

    m_LastResult = result;
    return (result.IsSuccess())?(true):(false);
}

void SimpleHttpsClient::PrintReceivedData()
{
    if(m_IsInitialized == false)
    {
        return; // not initialized
    }

    if(m_ReceivedTotalBytes == 0)
    {
        NN_LOG("Nothing is received yet.\n");
        return;
    }

    NN_LOG("\n------------ Received body (total: %d bytes received)---------------\n",
            m_ReceivedTotalBytes);
    NN_LOG("%s", m_pReceivedData);
    NN_LOG("\n--------------------------------------------------------------------\n");
}

bool SimpleHttpsClient::ImportServerPki(
    const char* pInCertData,
    uint32_t certDataSize,
    nn::ssl::CertificateFormat certFormat)
{
    if(m_IsInitialized == false)
    {
        return false; // not initialized
    }

    nn::Result result = m_SslContext.ImportServerPki(
        &m_ServerCertStoreId,
        pInCertData,
        certDataSize,
        certFormat);

    if(result.IsSuccess())
    {
        return true;
    }
    else
    {
        m_LastResult = result;
        return false;
    }
}

bool SimpleHttpsClient::ImportClientPki(
    const char* pInP12Data,
    const char* pInPwData,
    uint32_t  p12DataSize,
    uint32_t  pwDataSize)
{
    if(m_IsInitialized == false)
    {
        return false; // not initialized
    }

    nn::Result result = m_SslContext.ImportClientPki(
        &m_ClientCertStoreId,
        pInP12Data,
        pInPwData,
        p12DataSize,
        pwDataSize);
    if(result.IsSuccess())
    {
        return true;
    }
    else
    {
        m_LastResult = result;
        return false;
    }
}

bool SimpleHttpsClient::RegisterDeviceClientCert()
{
    if(m_IsInitialized == false)
    {
        NN_LOG("%s: not initialized\n", __FUNCTION__);
        return false; // not initialized
    }

    nn::Result result =
        m_SslContext.RegisterInternalPki(
            &m_ClientCertStoreId,
            nn::ssl::Context::InternalPki_DeviceClientCertDefault);
    if(result.IsSuccess())
    {
        return true;
    }
    else
    {
        m_LastResult = result;
        return false;
    }
}

nn::ssl::Connection* SimpleHttpsClient::GetSslConnection()
{
    return &m_SslConnection;
}

bool SimpleHttpsClient::EnableSessionCache()
{
    nn::Result result = m_SslConnection.SetSessionCacheMode(nn::ssl::Connection::SessionCacheMode_SessionId);
    if(result.IsSuccess())
    {
        return true;
    }
    else
    {
        m_LastResult = result;
        return false;
    }
}

//-------------------------------------------------------------------------------------------------
//  class SimpleCurlHttpsClient
//-------------------------------------------------------------------------------------------------
SimpleCurlHttpsClient::SimpleCurlHttpsClient(
    const char* pInHostName,
    const char* pInServerCert,
    const char* pInClientPki,
    const char* pInPassword,
    uint32_t serverCertSize,
    uint32_t clientPkiSize,
    uint32_t passwordSize,
    InitMode initMode) :
    m_IsPerformPeerCaValidation(true),
    m_IsPeformNameValidation(true),
    m_IsCurlVerbose(false),
    m_pCurlCtxCallback(nullptr)
{
    strncpy(m_HostName, pInHostName, sizeof(m_HostName));
    m_pServerCert       = pInServerCert;
    m_ServerCertSize    = serverCertSize;
    m_pClientPki        = pInClientPki;
    m_ClientPkiSize     = clientPkiSize;
    m_pPassword         = pInPassword;
    m_PasswordSize      = passwordSize;
    m_InitMode          = initMode;
    m_ServerCertStoreId = 0;
    m_ClientCertStoreId = 0;

    memset(&m_ProxyInfo, 0x00, sizeof(m_ProxyInfo));
}

SimpleCurlHttpsClient::~SimpleCurlHttpsClient()
{
}

void SimpleCurlHttpsClient::ConfigureValidation(bool peerCaValidation, bool hostNameValidation)
{
    m_IsPerformPeerCaValidation = peerCaValidation;
    m_IsPeformNameValidation    = hostNameValidation;
}

void SimpleCurlHttpsClient::EnableCurlVerbose()
{
    m_IsCurlVerbose = true;
}

bool SimpleCurlHttpsClient::IsImportServerPki()
{
    return (m_pServerCert)?(true):(false);
}
bool SimpleCurlHttpsClient::IsImportClientPki()
{
    return (m_pClientPki)?(true):(false);
}

const char* SimpleCurlHttpsClient::GetServerCert()
{
    return m_pServerCert;
}
const char* SimpleCurlHttpsClient::GetClientPki()
{
    return m_pClientPki;
}

const char* SimpleCurlHttpsClient::GetPassword()
{
    return m_pPassword;
}

uint32_t SimpleCurlHttpsClient::GetServerCertSize()
{
    return m_ServerCertSize;
}

uint32_t SimpleCurlHttpsClient::GetClientPkiSize()
{
    return m_ClientPkiSize;
}

uint32_t SimpleCurlHttpsClient::GetPasswordSize()
{
    return m_PasswordSize;
}
void SimpleCurlHttpsClient::SetServerCertStoreId(nn::ssl::CertStoreId id)
{
    m_ServerCertStoreId = id;
}

void SimpleCurlHttpsClient::SetCurlCtxFunction(CurlSslCtxFunction pCallback)
{
    m_pCurlCtxCallback = pCallback;
}

void SimpleCurlHttpsClient::SetClientCertStoreId(nn::ssl::CertStoreId id)
{
    m_ClientCertStoreId = id;
}

bool SimpleCurlHttpsClient::SetupProxy(
    const char* pInProxyAddress,
    const char* pInUserName,
    const char* pInPassword,
    uint16_t portNumber)
{
    if(pInProxyAddress == nullptr)
    {
        return false; // This parameter is mandatory
    }

    strncpy(m_ProxyInfo.address, pInProxyAddress, ProxyStrBuffLength);
    if(pInUserName != nullptr)
    {
        strncpy(m_ProxyInfo.userName, pInUserName, ProxyStrBuffLength);
    }
    if(pInPassword != nullptr)
    {
        strncpy(m_ProxyInfo.password, pInPassword, ProxyStrBuffLength);
    }
    m_ProxyInfo.port = portNumber;
    m_ProxyInfo.isConfigured = true;

    return true;
}


bool SimpleCurlHttpsClient::InitializeManual()
{
    if(m_InitMode == InitMode_Manual)
    {
        return (curl_global_init(CURL_GLOBAL_DEFAULT) == CURLE_OK)?(true):(false);
    }
    return true;
}

void SimpleCurlHttpsClient::FinalizeManual()
{
    if(m_InitMode == InitMode_Manual)
    {
        curl_global_cleanup();
    }
}

#define BREAK_ON_CURL_FAILURE(res) \
    do{ \
        if(res != CURLE_OK) {\
            isFailed = true; \
            break; \
        } \
    } while (NN_STATIC_CONDITION(0))

bool SimpleCurlHttpsClient::Perform()
{
    CURL*    curl;
    CURLcode res;
    bool     isFailed = false;

    if(m_InitMode == InitMode_Auto)
    {
        res = curl_global_init(CURL_GLOBAL_DEFAULT);
        if(res != CURLE_OK)
        {
            return false;
        }
    }

    curl = curl_easy_init();
    if(curl == nullptr)
    {
        return false;
    }

    do
    {
        res = curl_easy_setopt(curl, CURLOPT_URL, m_HostName);
        BREAK_ON_CURL_FAILURE(res);
        res = curl_easy_setopt(curl, CURLOPT_CONNECTTIMEOUT, 15L);
        BREAK_ON_CURL_FAILURE(res);
        res = curl_easy_setopt(curl, CURLOPT_NOPROGRESS, 0L);
        BREAK_ON_CURL_FAILURE(res);
        res = curl_easy_setopt(curl, CURLOPT_XFERINFOFUNCTION, CurlTransferInfoInternal);
        BREAK_ON_CURL_FAILURE(res);
        res = curl_easy_setopt(curl, CURLOPT_SSL_VERIFYPEER, (m_IsPerformPeerCaValidation)?1L:0L);
        BREAK_ON_CURL_FAILURE(res);
        res = curl_easy_setopt(curl, CURLOPT_SSL_VERIFYHOST, (m_IsPeformNameValidation)?2L:0L);
        BREAK_ON_CURL_FAILURE(res);
        res = curl_easy_setopt(curl, CURLOPT_SSL_CTX_FUNCTION, m_pCurlCtxCallback);
        BREAK_ON_CURL_FAILURE(res);
        res = curl_easy_setopt(curl, CURLOPT_SSL_CTX_DATA, reinterpret_cast<void*>(this));
        BREAK_ON_CURL_FAILURE(res);

        if(m_IsCurlVerbose == true)
        {
            res = curl_easy_setopt(curl, CURLOPT_VERBOSE, 1L);
            BREAK_ON_CURL_FAILURE(res);
        }

        if(m_ProxyInfo.isConfigured == true)
        {
            res = curl_easy_setopt(curl, CURLOPT_PROXYAUTOCONFIG, 0);
            BREAK_ON_CURL_FAILURE(res);
            res = curl_easy_setopt(curl, CURLOPT_PROXYAUTH, CURLAUTH_BASIC);
            BREAK_ON_CURL_FAILURE(res);

            res = curl_easy_setopt(curl, CURLOPT_PROXY, m_ProxyInfo.address);
            BREAK_ON_CURL_FAILURE(res);
            res = curl_easy_setopt(curl, CURLOPT_PROXYPORT, m_ProxyInfo.port);
            BREAK_ON_CURL_FAILURE(res);

            if(m_ProxyInfo.userName[0] != '\0')
            {
                res = curl_easy_setopt(curl, CURLOPT_PROXYUSERNAME, m_ProxyInfo.userName);
                BREAK_ON_CURL_FAILURE(res);
            }

            if(m_ProxyInfo.password[0] != '\0')
            {
                res = curl_easy_setopt(curl, CURLOPT_PROXYPASSWORD, m_ProxyInfo.password);
                BREAK_ON_CURL_FAILURE(res);
            }
        }


        NN_LOG("Fetching URL: %s\n", this, m_HostName);
        res = curl_easy_perform(curl);
        if(m_IsCurlVerbose)
        {
            if (res != CURLE_OK)
            {
                NN_LOG("curl_easy_perform FAILED: curl error: %d\n", res);
                SslTestCommonUtil::DumpCurlResults(curl);
            }
            else
            {
                NN_LOG("curl_easy_perform SUCCEEDED.\n");
            }
        }
    } while (NN_STATIC_CONDITION(0));

    curl_easy_cleanup(curl);

    if(m_InitMode == InitMode_Auto)
    {
        curl_global_cleanup();
    }

    return (isFailed)?(false):(true);
}
