﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ro.h>
#include <nn/ro/ro_Types.h>

#include "util_Common.h"

#include <nn/nn_Log.h>
#include <nn/svc/svc_Base.h>

uintptr_t g_ConstructorObj;
uintptr_t g_DestructorObj;

namespace {
    const char* BinTemplate1Name = "nro/BinTemplate1.nro";
    const char* BinTemplate2Name = "nro/BinTemplate2.nro";
    const char* BinTemplate3Name = "nro/BinTemplate3.nro";
    const char* BinTemplate4Name = "nro/BinTemplate4.nro";

    typedef uintptr_t (*ReturnUintptr)();
    typedef int (*ReturnInt)();

    const char* GetInstanceAddress1Name = "_Z19GetInstanceAddress1v";
    const char* GetInstanceAddress2Name = "_Z19GetInstanceAddress2v";
    const char* GetInstanceFuncAddress1Name = "_Z20GetInstanceFuncAddr1v";
    const char* GetInstanceFuncAddress2Name = "_Z20GetInstanceFuncAddr2v";

    const char* GetOverlapInstanceAddrFrom1Name = "_Z27GetOverlapInstanceAddrFrom1v";
    const char* GetOverlapItemFrom1Name = "_Z19GetOverlapItemFrom1v";

    const char* GetOverlapInstanceAddrFrom2Name = "_Z27GetOverlapInstanceAddrFrom2v";
    const char* GetOverlapItemFrom2Name = "_Z19GetOverlapItemFrom2v";

    const char* OverlapValueName = "g_Template";

    class TemplateTest : public ::testing::TestWithParam<nn::ro::BindFlag>
    {
    protected:
        virtual void SetUp()
        {
            m_Allocator = &TestAllocator::GetInstance();
            m_Nro1.SetUp(BinTemplate1Name, m_Allocator->GetAllocator());
            m_Nro2.SetUp(BinTemplate2Name, m_Allocator->GetAllocator());
            m_Nro3.SetUp(BinTemplate3Name, m_Allocator->GetAllocator());
            m_Nro4.SetUp(BinTemplate4Name, m_Allocator->GetAllocator());
            nn::ro::Initialize();
        }

        virtual void TearDown()
        {
            nn::ro::Finalize();
        }

        TestAllocator* m_Allocator;
        TestNro m_Nro1;
        TestNro m_Nro2;
        TestNro m_Nro3;
        TestNro m_Nro4;
    };

    void CheckNone(uintptr_t addr)
    {
        nn::svc::MemoryInfo info;
        nn::svc::PageInfo pageInfo;

        auto result = nn::svc::QueryMemory(&info, &pageInfo, addr);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_EQ(info.state, nn::svc::MemoryState_Free);
        ASSERT_EQ(info.permission, nn::svc::MemoryPermission_None);
        ASSERT_EQ(info.attribute, 0);
    }

    void CheckText(uintptr_t addr)
    {
        nn::svc::MemoryInfo info;
        nn::svc::PageInfo pageInfo;

        auto result = nn::svc::QueryMemory(&info, &pageInfo, addr);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_EQ(info.state, nn::svc::MemoryState_AliasCode);
        ASSERT_EQ(info.permission, nn::svc::MemoryPermission_ReadExecute);
        ASSERT_EQ(info.attribute, 0);
    }
} // namespace

INSTANTIATE_TEST_CASE_P(ManualDll, TemplateTest, ::testing::Values(nn::ro::BindFlag_Lazy, nn::ro::BindFlag_Now));

TEST_P(TemplateTest, InlineLocal)
{
    g_ConstructorObj = 0;
    g_DestructorObj = 0;

    // 関数ローカルの静的オブジェクトのコンストラクタは呼び出し時に呼び出される
    auto result = m_Nro1.Load(GetParam());
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(g_ConstructorObj, 0);

    result = m_Nro2.Load(GetParam());
    ASSERT_RESULT_SUCCESS(result);

    uintptr_t addr;

    result = nn::ro::LookupSymbol(&addr, GetInstanceAddress1Name);
    ASSERT_RESULT_SUCCESS(result);
    ReturnUintptr getAddr1 = reinterpret_cast<ReturnUintptr>(addr);

    result = nn::ro::LookupSymbol(&addr, GetInstanceFuncAddress1Name);
    ASSERT_RESULT_SUCCESS(result);
    ReturnUintptr getFuncAddr1 = reinterpret_cast<ReturnUintptr>(addr);

    result = nn::ro::LookupSymbol(&addr, GetInstanceAddress2Name);
    ASSERT_RESULT_SUCCESS(result);
    ReturnUintptr getAddr2 = reinterpret_cast<ReturnUintptr>(addr);

    result = nn::ro::LookupSymbol(&addr, GetInstanceFuncAddress2Name);
    ASSERT_RESULT_SUCCESS(result);
    ReturnUintptr getFuncAddr2 = reinterpret_cast<ReturnUintptr>(addr);

    ASSERT_EQ(getFuncAddr1(), getFuncAddr2());
    uintptr_t funcAddr = getFuncAddr2();
    CheckText(funcAddr);

    // 初めての呼び出しの時に静的オブジェクトのコンストラクタが呼び出される
    g_ConstructorObj = 0;
    uintptr_t instanceAddr1 = getAddr1();
    ASSERT_NE(g_ConstructorObj, 0);

    g_ConstructorObj = 0;
    // 次の呼び出しの時にはコンストラクタは呼び出されない
    uintptr_t instanceAddr2 = getAddr2();
    ASSERT_EQ(g_ConstructorObj, 0);

    ASSERT_EQ(instanceAddr1, instanceAddr2);

    m_Nro1.Unload();
    ASSERT_NE(g_DestructorObj, 0);
    g_DestructorObj = 0;

    CheckNone(instanceAddr1);
    CheckNone(funcAddr);

    // 参照が元に戻され、新たに参照解決がされることで、アクセスできるようになっている
    ASSERT_NE(funcAddr, getFuncAddr2());
    CheckText(getFuncAddr2());

    // 参照が元に戻されたことによって、関数ローカルの静的オブジェクトのコンストラクタが呼び出される
    g_ConstructorObj = 0;
    ASSERT_NE(instanceAddr2, getAddr2());
    ASSERT_NE(g_ConstructorObj, 0);

    result = m_Nro1.Load(GetParam());
    ASSERT_RESULT_SUCCESS(result);

    result = nn::ro::LookupSymbol(&addr, GetInstanceAddress1Name);
    ASSERT_RESULT_SUCCESS(result);
    getAddr1 = reinterpret_cast<ReturnUintptr>(addr);

    result = nn::ro::LookupSymbol(&addr, GetInstanceFuncAddress1Name);
    ASSERT_RESULT_SUCCESS(result);
    getFuncAddr1 = reinterpret_cast<ReturnUintptr>(addr);

    ASSERT_EQ(getFuncAddr1(), getFuncAddr2());
    ASSERT_EQ(getAddr1(), getAddr2());

    // 参照している側のモジュールをアンロードしてもデストラクタは呼び出されない
    g_DestructorObj = 0;
    m_Nro1.Unload();
    ASSERT_EQ(g_DestructorObj, 0);

    g_DestructorObj = 0;
    m_Nro2.Unload();
    ASSERT_NE(g_DestructorObj, 0);
}

TEST_P(TemplateTest, StaticObject)
{
    g_ConstructorObj = 0;
    g_DestructorObj = 0;

    auto result = m_Nro3.Load(GetParam());
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_NE(g_ConstructorObj, 0);
    uintptr_t consAddr = g_ConstructorObj;
    g_ConstructorObj = 0;

    // 重複シンボルの場合、同じオブジェクトに対してコンストラクタが呼び出されてしまう
    result = m_Nro4.Load(GetParam());
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(g_ConstructorObj, consAddr);

    uintptr_t addr;

    result = m_Nro3.FindSymbol(&addr, OverlapValueName);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(consAddr, addr);

    result = nn::ro::LookupSymbol(&addr, GetOverlapInstanceAddrFrom1Name);
    ASSERT_RESULT_SUCCESS(result);
    ReturnUintptr getAddr1 = reinterpret_cast<ReturnUintptr>(addr);

    result = nn::ro::LookupSymbol(&addr, GetOverlapItemFrom1Name);
    ASSERT_RESULT_SUCCESS(result);
    ReturnInt getItem1 = reinterpret_cast<ReturnInt>(addr);

    result = nn::ro::LookupSymbol(&addr, GetOverlapInstanceAddrFrom2Name);
    ASSERT_RESULT_SUCCESS(result);
    ReturnUintptr getAddr2 = reinterpret_cast<ReturnUintptr>(addr);

    result = nn::ro::LookupSymbol(&addr, GetOverlapItemFrom2Name);
    ASSERT_RESULT_SUCCESS(result);
    ReturnInt getItem2 = reinterpret_cast<ReturnInt>(addr);

    ASSERT_EQ(getAddr1(), getAddr2());
    uintptr_t itemAddr = getAddr2();

    ASSERT_EQ(getItem1(), getItem2());
    ASSERT_EQ(getItem1(), 2);

    g_DestructorObj = 0;
    m_Nro3.Unload();
    ASSERT_NE(g_DestructorObj, 0);
    ASSERT_EQ(g_DestructorObj, consAddr);

    CheckNone(itemAddr);
    ASSERT_NE(itemAddr, getAddr2());
    itemAddr = getAddr2();

    // 参照が戻され、未初期化のオブジェクトを参照するようになる
    // BSS 領域の未初期化オブジェクトにアクセス
    ASSERT_EQ(getItem2(), 0);

    // 再ロードされたときに、モジュールのロード順にシンボルが解決される
    result = m_Nro3.Load(GetParam());
    ASSERT_RESULT_SUCCESS(result);

    result = nn::ro::LookupSymbol(&addr, GetOverlapInstanceAddrFrom1Name);
    ASSERT_RESULT_SUCCESS(result);
    getAddr1 = reinterpret_cast<ReturnUintptr>(addr);

    result = nn::ro::LookupSymbol(&addr, GetOverlapItemFrom1Name);
    ASSERT_RESULT_SUCCESS(result);
    getItem1 = reinterpret_cast<ReturnInt>(addr);

    ASSERT_EQ(getAddr1(), getAddr2());
    ASSERT_EQ(getItem1(), getItem2());
    ASSERT_EQ(getAddr1(), itemAddr);

    // m_Nro4 にあるオブジェクトに対してデストラクタが呼び出されてしまう。
    result = m_Nro4.FindSymbol(&addr, OverlapValueName);
    ASSERT_RESULT_SUCCESS(result);
    uintptr_t destAddr = addr;

    g_DestructorObj = 0;
    m_Nro3.Unload();
    ASSERT_EQ(g_DestructorObj, destAddr);

    g_DestructorObj = 0;
    m_Nro4.Unload();
    ASSERT_EQ(g_DestructorObj, destAddr);
}

