﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ro.h>
#include <nn/ro/ro_Types.h>

#include "util_Common.h"

#include <nn/nn_Log.h>

namespace {
    const char* BinHiddenName = "nro/BinHidden.nro";
    const char* BinAccessHiddenName = "nro/BinAccessHidden.nro";

    typedef int (*ReturnInt)();
    typedef uintptr_t (*ReturnUintptr)();
    typedef void (*SetInt)(int);
    const char* HiddenFuncName = "_Z13HiddenFuncv";
    const char* CallHiddenFuncName = "_Z14CallHiddenFuncv";
    const char* GetHiddenVariableAddressName = "_Z24GetHiddenVariableAddressv";
    const char* HiddenValueName = "g_HiddenVariable";

    class HiddenTest : public ::testing::TestWithParam<nn::ro::BindFlag>
    {
    protected:
        virtual void SetUp()
        {
            m_Allocator = &TestAllocator::GetInstance();
            m_Nro1.SetUp(BinHiddenName, m_Allocator->GetAllocator());
            m_Nro2.SetUp(BinAccessHiddenName, m_Allocator->GetAllocator());
            nn::ro::Initialize();
        }

        virtual void TearDown()
        {
            nn::ro::Finalize();
        }

        TestAllocator* m_Allocator;
        TestNro m_Nro1;
        TestNro m_Nro2;
    };

} // namespace

INSTANTIATE_TEST_CASE_P(ManualDll, HiddenTest, ::testing::Values(nn::ro::BindFlag_Lazy, nn::ro::BindFlag_Now));

TEST_P(HiddenTest, Bind)
{
    auto result = m_Nro1.Load(GetParam());
    ASSERT_RESULT_SUCCESS(result);

    result = m_Nro2.Load(GetParam());
    ASSERT_RESULT_SUCCESS(result);

    uintptr_t addr;

    // Hidden は外部参照が出来ない
    {
        result = nn::ro::LookupSymbol(&addr, HiddenFuncName);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::ro::ResultNotFound());

        result = nn::ro::LookupSymbol(&addr, HiddenValueName);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::ro::ResultNotFound());

        result = nn::ro::LookupSymbol(&addr, CallHiddenFuncName);
        ASSERT_RESULT_SUCCESS(result);
        uintptr_t callAddr1 = addr;

        result = nn::ro::LookupSymbol(&addr, GetHiddenVariableAddressName);
        ASSERT_RESULT_SUCCESS(result);
        uintptr_t getAddrFunc1 = addr;

        result = m_Nro1.FindSymbol(&addr, HiddenFuncName);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::ro::ResultNotFound());

        result = m_Nro1.FindSymbol(&addr, HiddenValueName);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::ro::ResultNotFound());

        result = m_Nro1.FindSymbol(&addr, CallHiddenFuncName);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_EQ(callAddr1, addr);
        ReturnInt call1 = reinterpret_cast<ReturnInt>(addr);
        ASSERT_EQ(1, call1());

        result = m_Nro1.FindSymbol(&addr, GetHiddenVariableAddressName);
        ASSERT_RESULT_SUCCESS(result);
        ASSERT_EQ(getAddrFunc1, addr);
        ReturnUintptr getAddr1 = reinterpret_cast<ReturnUintptr>(addr);

        result = m_Nro2.FindSymbol(&addr, HiddenFuncName);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::ro::ResultNotFound());

        result = m_Nro2.FindSymbol(&addr, HiddenValueName);
        ASSERT_RESULT_FAILURE_VALUE(result, nn::ro::ResultNotFound());

        result = m_Nro2.FindSymbol(&addr, CallHiddenFuncName);
        ASSERT_RESULT_SUCCESS(result);
        uintptr_t callAddr2 = addr;
        ASSERT_NE(callAddr1, callAddr2);
        ReturnInt call2 = reinterpret_cast<ReturnInt>(addr);
        ASSERT_NE(call1(), call2());

        result = m_Nro2.FindSymbol(&addr, GetHiddenVariableAddressName);
        ASSERT_RESULT_SUCCESS(result);
        uintptr_t getAddrFunc2 = addr;
        ASSERT_NE(getAddrFunc1, getAddrFunc2);
        ReturnUintptr getAddr2 = reinterpret_cast<ReturnUintptr>(addr);
        ASSERT_NE(getAddr1(), getAddr2());
    }

    m_Nro1.Unload();
    m_Nro2.Unload();
}

