﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System.Reflection;
using System.IO;
using System.Collections.Generic;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using CsTestAssistants;
using System.Text;
using System.Xml.Linq;
using System.Xml.XPath;

namespace RidCompatibilityTest
{
    using ExpectEvaluator = ThrowEvaluator<UnexpectFailureException>;

    [TestClass]
    public class RidCompatibilityTest : TestClassBase
    {
        public TestContext TestContext { get; set; }

        private const string LogPrefix = "[RidCompatibilityTest]";

        private readonly string TestIntermediateDirPath = Path.Combine(SigloHelper.Path.GetTestsRoot(),
                    @"Rid\Sources\Tests\RidCompatibilityTest\Intermediate");
        private readonly ulong TestMenuID = 0x01001a500005e011;

        private readonly string FailureOptionString = "--pattern-failure-exit \"\\[  FAILED  \\]\"";

        // テスト対象バージョンの Enum 定義
        private enum TestVersion
        {
            develop,
            v3,
            v4,
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// クラスが生成され、呼び出される前に一度だけ実施。
        /// </summary>
        /// <param name="context"></param>
        //!----------------------------------------------------------------------------
        [ClassInitialize]
        public static void TestClassInitialize(TestContext context)
        {
            ClassInitialize(context, LogPrefix);
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// クラスが破棄されるタイミングで一度だけ実施。
        /// </summary>
        //!----------------------------------------------------------------------------
        [ClassCleanup]
        public static void TestClassCleanup()
        {
            ClassCleanup();
        }

        //!----------------------------------------------------------------------------
        /// <summary>
        /// `TestProperty属性の "JIRA" キーの値` と `Sigloコンフィギュレーションコンテキスト` に依存した作業用ワーキングディレクトリパスを生成します.
        /// </summary>
        /// <param name="context">Sigloコンフィギュレーションコンテキスト</param>
        /// <param name="methodName">
        /// 参照対象TestProperty属性を保持するメソッド名.
        /// 指定を省略した場合、呼び出されたカレントメソッド名が自動で選択されます.
        /// 明示的な null 指定の場合、もしくは "JIRA" キーの TestPropertyが見つからない場合には、TestPropertyに依存しないベース作業用ディレクトリが返却されます。
        /// </param>
        /// <returns>
        /// TestProperty( "JIRA", * ) に依存した作業用ディレクトリパス、もしくはベース作業用ディレクトリパス。
        /// </returns>
        //!----------------------------------------------------------------------------
        private string GenerateIntermediateDirectoryAsMethod(SigloHelper.Configuration.Context context,
            [System.Runtime.CompilerServices.CallerMemberName] string methodName = null)
        {
            var runAssembly = Assembly.GetExecutingAssembly();
            var asmTitle = (AssemblyTitleAttribute)System.Attribute.GetCustomAttribute(runAssembly, typeof(AssemblyTitleAttribute));
            return GenerateIntermediateDirectory(context, asmTitle.Title, methodName);
        }

        private void CleanupTarget(SigloHelper.CommodityExecutor.Context context)
        {
            // 念のためクリーンアップ処理とターゲット機を再起動させておく
            ExpectEvaluator.IsTrue(context.RunDevMenuCommandSystem(
                "application clear-task-status-list AND " +
                "application uninstall --all AND " +
                "application reset-required-version --all"));
            ExpectEvaluator.IsTrue(context.ResetTarget());
        }

        // 実行時に指定されたターゲットバージョンの値を Enum の形で返します
        private TestVersion GetTargetVersion()
        {
            if (TestContext.Properties.Contains("TargetVersion"))
            {
                var version = ((string)TestContext.Properties["TargetVersion"]).ToLower();
                if (version == "dev" || version == "develop")
                {
                    return TestVersion.develop;
                }
                else if (version == "3" || version == "v3")
                {
                    return TestVersion.v3;
                }
                else if (version == "4" || version == "v4")
                {
                    return TestVersion.v4;
                }
            }
            // 無指定または無効な指定値の場合はデフォルト値である develop を返す
            return TestVersion.develop;
        }

        // テスト対象の nsp ファイルパスを取得
        private string GetTargetTestNspPath()
        {
            var testVersion = GetTargetVersion();
            var targetPath = Path.Combine(SigloHelper.Path.GetTestsRoot(),
                    @"Outputs\NX-NXFP2-a64\TargetTools\RidCompatibilityVerifier\Release\RidCompatibilityVerifier.nsp");
            if (testVersion != TestVersion.develop)
            {
                // develop 以外のテスト用の nsp ファイルは Externals\TestBinaries\Rid\RidCompatibilityVerifier 以下に配置する
                var basePath = Path.Combine(SigloHelper.Path.GetSigloRoot(), @"Externals\TestBinaries\Rid\RidCompatibilityVerifier");
                // TargetVersion オプションの引数値でターゲットとするパスを確定させる
                targetPath = Path.Combine(basePath,
                    string.Format(@"{0}\RidCompatibilityVerifier.nsp", testVersion.ToString()));
            }
            return targetPath;
        }

        // テスト対象のアプリをターゲット上にインストールする
        private void InstallMenuNsp(SigloHelper.CommodityExecutor.Context context)
        {
            var targetPath = GetTargetTestNspPath();
            // メニューに相当するアプリのインストール先は NAND へ
            ExpectEvaluator.IsTrue(context.RunDevMenuCommandSystem(
                string.Format("application install \"{0}\" -s builtin", targetPath)));
        }

        // テスト対象のアプリをアンインストールする
        private void UninstallMenuApp(SigloHelper.CommodityExecutor.Context context)
        {
            context.RunDevMenuCommandSystem(
                string.Format("application uninstall 0x{0:x16}", TestMenuID));
        }

        // テスト対象アプリのパッチを作成する
        // 内部処理としては変更せずにバージョン情報のみを更新
        private string CreateCurrentMenuPatchFile()
        {
            var patchMetaFile = Path.Combine(TestIntermediateDirPath,
                @"RidCompatibilityVerifierForPatch.nmeta");
            // nmeta ファイルをパッチ用に編集し保持しておく
            {
                var originalMetaFile = Path.Combine(SigloHelper.Path.GetTestsRoot(),
                    @"Rid\Sources\TargetTools\RidCompatibilityVerifier\RidCompatibilityVerifier.nmeta");
                var xdoc = XDocument.Load(originalMetaFile);
                var appElem = xdoc.XPathSelectElement("/NintendoSdkMeta/Application");

                // ReleaseVersion と DisplayVersion を更新
                appElem.XPathSelectElement("ReleaseVersion").Value = "1";
                appElem.XPathSelectElement("DisplayVersion").Value = "1.0.1";

                // アイコンパスは絶対パス指定にしておく
                var iconPath = appElem.XPathSelectElement("Icon/IconPath");
                iconPath.Value = Path.Combine(SigloHelper.Path.GetProgramsRoot(),
                   @"Iris\Resources\SpecFiles\NintendoSDK_Application.bmp");

                xdoc.Save(patchMetaFile);
            }

            var outPatchFilePath = Path.Combine(TestIntermediateDirPath, @"menuPatch.nsp");

            var testDescFile = Path.Combine(SigloHelper.Path.GetTestsRoot(),
                @"Common\Resources\TestApplication.autogen.desc");

            var targetCurrentCodePath = Path.Combine(SigloHelper.Path.GetTestsRoot(),
                @"Outputs\NX-NXFP2-a64\TargetTools\RidCompatibilityVerifier\Release\code");

            var targetPath = GetTargetTestNspPath();

            // MakeTestApplication のオプション引数文字列を生成
            var b = new StringBuilder(1024);
            b.Append(" --type Patch --verbose ");
            b.Append(" --output-file-path \"").Append(outPatchFilePath).Append('\"');
            b.Append(" --meta \"").Append(patchMetaFile).Append('\"');
            b.Append(" --desc \"").Append(testDescFile).Append('\"');
            b.Append(" --code \"").Append(targetCurrentCodePath).Append('\"');
            b.Append(" --original-application \"").Append(targetPath).Append('\"');

            // MakeTestApplication を使って nsp (パッチ)を生成
            if (0 != CommandLineExecutor.ExecuteOnProcess(TestApplication.GetExecutionToolPath(), b.ToString()))
            {
                throw new UnexpectFailureException("MakeTestApplication operation failed.");
            }

            // 生成した nsp ファイルのパスを返す
            return outPatchFilePath;
        }

        // 厳密にはテスト関数ではないが、テスト全体のセットアップ処理のみ単体で実施できるように TestMethod としておく
        [TestMethod]
        [Timeout(90 * 1000)]
        public void SetupForTest()
        {
            using (var scope = new TestMethodLog())
            {
                // 無線接続が必要であるため設定しておく
                var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);
                var systemUpdateTestPath = Path.Combine(SigloHelper.Path.GetTestsRoot(), @"Ns\Sources\Tests\SystemUpdateBasic");
                var networkSetting = Path.Combine(systemUpdateTestPath, "NetworkSettingsForCI.txt");
                // Wireless-LAN 設定インポート
                ExpectEvaluator.IsTrue(executor.RunSettingsManager("Import " + networkSetting));

                // SD カードのフォーマット処理を実施しておく
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("sdcard format"));

                // 再起動しておく
                ExpectEvaluator.IsTrue(executor.ResetTarget());

                // 本テスト用の中間ディレクトリが存在しない場合は作成する
                if (Directory.Exists(TestIntermediateDirPath) == false)
                {
                    Directory.CreateDirectory(TestIntermediateDirPath);
                }
            }
        }

        // 厳密にはテスト関数ではないが、アップロード処理のみ単体で実施できるように TestMethod としておく
        [TestMethod]
        [Timeout(10 * 60 * 1000)]
        public void UploadContentsForTest()
        {
            var UploadTimeout = new System.TimeSpan(0, 5, 0);         // 失敗時のリトライも含めて長めに設定

            using (var scope = new TestMethodLog())
            {
                // TestForMenuUpdate() 用のアップロード処理
                var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);

                // 作業用ディレクトリ
                var intermediate = GenerateIntermediateDirectoryAsMethod(executor);

                // upload contents
                var uploader = new D4cHelper.NspUploader(
                    intermediate,
                    ProxyConfiguration,
                    ServerEnvironment);

                var account = CliTokenAccount;
                var timeout = UploadTimeout.TotalSeconds;
                var menuId = new ID64(TestMenuID);
                uploader.RejectRoms(account, menuId);

                var patchPath = CreateCurrentMenuPatchFile();

                uploader.Publish(account, patchPath, D4cHelper.NspUploader.UploadOptions.Constants.Patch, timeout);
                uploader.RegisterVersion(patchPath);

                // ================================================================
                // TestForApplicationUpdate() 用のアップロード処理
                var demoId = new ID64(0x01001a500005e012);

                var demoApp = new TestApplication.ApplicationParameter(demoId, 0);
                demoApp.UseSmallCode = true;

                var aoc1v1 = new TestApplication.AddonParameter(menuId, 1, 1, 4 * 1024);
                var aoc10v0 = new TestApplication.AddonParameter(menuId, 0, 10, 4 * 1024);

                var request = new List<TestApplication.GenerateParameter<int>>()
                {
                    demoApp,
                    aoc1v1,
                    aoc10v0,
                };
                var contents = TestApplication.MakeContents(intermediate, request);
                var demo1   = contents[0];
                var Aoc1V1  = contents[1];
                var Aoc10V0 = contents[2];

                uploader.Publish(account, demo1.NspPath, D4cHelper.NspUploader.UploadOptions.Constants.Normal, timeout);
                uploader.RegisterVersion(demo1.NspPath);

                uploader.Publish(CliTokenAccount, Aoc1V1.NspPath, D4cHelper.NspUploader.UploadOptions.MakeUniqueAocArchiveNumberBy(Aoc1V1));
                uploader.RegisterVersion(Aoc1V1.NspPath);
                uploader.Publish(CliTokenAccount, Aoc10V0.NspPath, D4cHelper.NspUploader.UploadOptions.MakeUniqueAocArchiveNumberBy(Aoc10V0));
                uploader.RegisterVersion(Aoc10V0.NspPath);
            }
        }

        [TestMethod]
        [Timeout(90 * 1000)]
        public void TestForPrepare()
        {
            using (var scope = new TestMethodLog())
            {
                // Target command executor
                var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);

                // テスト実行前にクリーンアップ処理を実施
                CleanupTarget(executor);

                // テスト用のメニューアプリをターゲット上へインストール
                InstallMenuNsp(executor);
                // 本テスト終了時にアンインストールするようにしておく
                scope.AddDisposer(() => { UninstallMenuApp(executor); });

                // QuestフラグをONにする
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("debug enable-rid-mode"));
                // テスト終了時にOFFに戻しておく
                scope.AddDisposer(() => { executor.RunDevMenuCommandSystem("debug disable-rid-mode"); });

                var options = FailureOptionString;
                var menuIdString = string.Format("0x{0:x16}", TestMenuID);

                // Prepare 処理を実施
                ExpectEvaluator.IsTrue(executor.RunOnTarget(menuIdString, options, "--prepare"));

                // ToDo：Miiデータやアルバムデータ等の Prepare 処理で削除されるデータの確認処理を入れる予定
                // (現状は DevMenuCommand 等で動作確認を簡便に実施することが難しい)
            }
        }

        [TestMethod]
        [Timeout(120 * 1000)]
        public void TestForDisplay()
        {
            using (var scope = new TestMethodLog())
            {
                // Target command executor
                var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);

                // テスト実行前にクリーンアップ処理を実施
                CleanupTarget(executor);

                // テストにはユーザアカウントが必要なため追加処理を入れる
                // テスト完了時にクリアする
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("account add"));
                scope.AddDisposer(() => { executor.RunDevMenuCommandSystem("account clear_all"); });

                // テスト用のメニューアプリを SDEV へインストール
                InstallMenuNsp(executor);
                // 本テスト終了時にアンインストールするようにしておく
                scope.AddDisposer(() => { UninstallMenuApp(executor); });

                // QuestフラグをONにする
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("debug enable-rid-mode"));
                // テスト終了時にOFFに戻しておく
                scope.AddDisposer(() => { executor.RunDevMenuCommandSystem("debug disable-rid-mode"); });

                // ターゲット機の自動スリープ設定を適当な値に設定しておく(--display実行後に変更しているのを確認するため)
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    "settings set-auto-sleep --handheld 3m AND " +
                    "settings set-auto-sleep --console 1h AND " +
                    "settings set-auto-sleep --mediaplay enabled"));

                var targetVersion = GetTargetVersion();
                var isCheckErrorReportSetting = (targetVersion != TestVersion.v3 && targetVersion != TestVersion.v4);
                if (isCheckErrorReportSetting)
                {
                    // v5 以降に必要となるチェック処理
                    // ターゲット機のエラーレポート共有権限設定を適当な値に設定しておく(--display実行後に変更しているのを確認するため)
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("error set-report-share-permission NotConfirmed"));
                }

                var options = FailureOptionString;
                var menuIdString = string.Format("0x{0:x16}", TestMenuID);

                // 試遊モードの処理を実施
                ExpectEvaluator.IsTrue(executor.RunOnTarget(menuIdString, options, "--display"));

                // 自動スリープ設定が無効となっていることを確認
                {
                    executor.OutputStream.Standard.Clear();
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("settings get-auto-sleep --all"));
                    ExpectEvaluator.IsTrue(executor.OutputStream.Standard.ToString().Contains(
                        "handheld: never"));
                    ExpectEvaluator.IsTrue(executor.OutputStream.Standard.ToString().Contains(
                        "console: never"));
                    ExpectEvaluator.IsTrue(executor.OutputStream.Standard.ToString().Contains(
                        "mediaplay: disabled"));
                }

                if (isCheckErrorReportSetting)
                {
                    // エラーレポート共有権限設定が Granted となっていることを確認
                    executor.OutputStream.Standard.Clear();
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("error get-report-share-permission"));
                    ExpectEvaluator.IsTrue(executor.OutputStream.Standard.ToString().Contains(
                        "Granted"));
                }

                // ToDo：セーブデータ削除の確認処理を入れる予定
                // (現状は DevMenuCommand 等で動作確認を簡便に実施することが難しい)
            }
        }

        [TestMethod]
        [Timeout(6 * 60 * 1000)]
        public void TestForSystemUpdate()
        {
            using (var scope = new TestMethodLog())
            {
                var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);

                // 不明なシステムプログラムを削除しておく(クリーンアップ処理の一環)
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("systemprogram uninstall all-unknown"));
                // テスト実行前にクリーンアップ処理を実施
                CleanupTarget(executor);

                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("systemprogram list"));

                var systemUpdateTestPath = Path.Combine(SigloHelper.Path.GetTestsRoot(), @"Ns\Sources\Tests\SystemUpdateBasic");
                var nupTestPsScriptDirPath = Path.Combine(systemUpdateTestPath, "Scripts");

                var deviceIdFilePath = Path.Combine(TestIntermediateDirPath, @"GetDeviceId.Result.txt");
                var targetNameArg = string.Empty;
                var targetAddrArg = string.Empty;

                if (TestContext.Properties.Contains("TargetName"))
                {
                    targetNameArg = string.Format("-TargetName {0}", (string)TestContext.Properties["TargetName"]);
                }
                if (TestContext.Properties.Contains("TargetAddress"))
                {
                    targetAddrArg = string.Format("-TargetAddress {0}", (string)TestContext.Properties["TargetAddress"]);
                }

                // Ns のスクリプト処理を借りるため PowerShell でテスト環境を構築
                var ps = new PowerShellExecutor();
                // 64ビットのレジストリを参照させるための回避措置
                ps.IsSetSysNativePath = true;
                {
                    var getDevaiceIdCommandPath = Path.Combine(nupTestPsScriptDirPath, "GetDeviceId.ps1");
                    var psarg = new List<string>() {
                        string.Format("-OutputDeviceIdFile {0}", deviceIdFilePath),
                        targetNameArg,
                        targetAddrArg,
                    };
                    // デバイスIDの取得
                    ExpectEvaluator.IsEqual<int>(0, ps.Execute(getDevaiceIdCommandPath, psarg));
                }

                // td1 サーバー環境インポート
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("servicediscovery import-all td1pass"));

                // 配信対象デバイス設定 ( Version 0 )
                var systemUpdateId = "01001a500005e080";
                var deliveryId = "4";
                var setupWeatherCommandPath = Path.Combine(nupTestPsScriptDirPath, "SetupWetherDelivery.ps1");
                {
                    var psarg = new List<string>() {
                        "-InputDeviceIdFile " + deviceIdFilePath,
                        "-TitleId " + systemUpdateId,
                        "-TitleVersion 0",
                        "-DeliveryId " + deliveryId,
                    };
                    ExpectEvaluator.IsEqual<int>(0, ps.Execute(setupWeatherCommandPath, psarg));
                }
                scope.AddDisposer(() => {
                    var stopWeatherCommandPath = Path.Combine(nupTestPsScriptDirPath, "StopWetherDelivery.ps1");
                    var psarg = new List<string>() {
                        string.Format("-DeliveryId {0}", deliveryId),
                    };
                    ps.Execute(stopWeatherCommandPath, psarg);
                });

                // デバイス再起動
                ExpectEvaluator.IsTrue(executor.ResetTarget());

                // QuestフラグをONにする
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("debug enable-rid-mode"));
                // テスト終了時にOFFに戻しておく
                scope.AddDisposer(() => { executor.RunDevMenuCommandSystem("debug disable-rid-mode"); });

                // テスト用のメニューアプリをターゲット上へインストール
                InstallMenuNsp(executor);
                // 本テスト終了時にアンインストールするようにしておく
                scope.AddDisposer(() => { UninstallMenuApp(executor); });

                var options = FailureOptionString;
                var menuIdString = string.Format("0x{0:x16}", TestMenuID);

                // SystemUpdate を実行 (正常に終了)
                executor.OutputStream.Standard.Clear();
                ExpectEvaluator.IsTrue(executor.RunOnTarget(menuIdString, options, "--system-update"));
                ExpectEvaluator.IsTrue(executor.OutputStream.Standard.ToString().Contains(
                    "RunCommandSystemUpdate() : State = Completed"));

                // 本テスト終了時点でインストールした(不要なシステムプログラム)を削除しておく
                scope.AddDisposer(() => { executor.RunDevMenuCommandSystem("systemprogram uninstall all-unknown"); });

                // SystemUpdate を再度実行してみる
                // アップデート不要となるはず
                executor.OutputStream.Standard.Clear();
                ExpectEvaluator.IsTrue(executor.RunOnTarget(menuIdString, options, "--system-update"));
                ExpectEvaluator.IsTrue(executor.OutputStream.Standard.ToString().Contains(
                    "RunCommandSystemUpdate() : State = NeedNoUpdate"));

                // 配信対象デバイス設定 ( Version 1 )
                {
                    var psarg = new List<string>() {
                        string.Format("-InputDeviceIdFile {0}", deviceIdFilePath),
                        string.Format("-TitleId {0}", systemUpdateId),
                        "-TitleVersion 1",
                        string.Format("-DeliveryId {0}", deliveryId),
                    };
                    ExpectEvaluator.IsEqual<int>(0, ps.Execute(setupWeatherCommandPath, psarg));
                }

                // SystemUpdate を実行 (正常に終了)
                executor.OutputStream.Standard.Clear();
                ExpectEvaluator.IsTrue(executor.RunOnTarget(menuIdString, options, "--system-update"));
                ExpectEvaluator.IsTrue(executor.OutputStream.Standard.ToString().Contains(
                    "RunCommandSystemUpdate() : State = Completed"));

                // SystemUpdate を再度実行してみる (アップデート不要)
                executor.OutputStream.Standard.Clear();
                ExpectEvaluator.IsTrue(executor.RunOnTarget(menuIdString, options, "--system-update"));
                ExpectEvaluator.IsTrue(executor.OutputStream.Standard.ToString().Contains(
                    "RunCommandSystemUpdate() : State = NeedNoUpdate"));
            }
        }

        [TestMethod]
        [Timeout(3 * 60 * 1000)]
        public void TestForMenuUpdate()
        {
            using (var scope = new TestMethodLog())
            {
                // Target command executor
                var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);

                // テスト実行前にクリーンアップ処理を実施
                CleanupTarget(executor);

                // QuestフラグをONにする
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("debug enable-rid-mode"));
                // テスト終了時にOFFに戻しておく
                scope.AddDisposer(() => { executor.RunDevMenuCommandSystem("debug disable-rid-mode"); });

                // ターゲット上にパッチは存在しないことを確認
                executor.OutputStream.Standard.Clear();
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("patch list"));
                ExpectEvaluator.IsTrue(executor.OutputStream.Standard.ToString().Contains(
                    "0 patch(es) found."));

                // テスト用のメニューアプリをターゲット上へインストール
                InstallMenuNsp(executor);
                // 本テスト終了時にアンインストールするようにしておく
                scope.AddDisposer(() => { UninstallMenuApp(executor); });

                var options = FailureOptionString;
                var menuIdString = string.Format("0x{0:x16}", TestMenuID);

                // アプリを実行してメニューのアップデート処理を実行(パッチをダウンロード)
                // 実行後は自動的にアプリは終了する
                executor.OutputStream.Standard.Clear();
                ExpectEvaluator.IsTrue(executor.RunOnTarget(menuIdString, options, "--menu-update"));
                // State == Completed 状態となることを想定
                ExpectEvaluator.IsTrue(executor.OutputStream.Standard.ToString().Contains(
                    "RunCommandMenuUpdate() : State = Completed"));

                // パッチがダウンロードできていることを確認
                executor.OutputStream.Standard.Clear();
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("patch list"));
                ExpectEvaluator.IsTrue(executor.OutputStream.Standard.ToString().Contains(
                    "1 patch(es) found."));

                // 再度メニューのアップデート処理を実行 (アップデート不要となることを想定)
                executor.OutputStream.Standard.Clear();
                ExpectEvaluator.IsTrue(executor.RunOnTarget(menuIdString, options, "--menu-update"));
                ExpectEvaluator.IsTrue(executor.OutputStream.Standard.ToString().Contains(
                    "RunCommandMenuUpdate() : State = NeedNoUpdate"));
            }
        }

        [TestMethod]
        [Timeout(8 * 60 * 1000)]
        public void TestForApplicationUpdate()
        {
            using (var scope = new TestMethodLog())
            {
                // Target command executor
                var executor = new SigloHelper.CommodityExecutor.Context(ActiveConfiguration);

                // テスト実行前にクリーンアップ処理を実施
                CleanupTarget(executor);

                // 作業用ディレクトリ
                var intermediate = GenerateIntermediateDirectoryAsMethod(executor);

                // ローカルインストール用のコンテンツ生成
                var demo2Id = new ID64(0x01001a500005e013);

                var demoApp2 = new TestApplication.ApplicationParameter(demo2Id, 0);
                demoApp2.UseSmallCode = true;

                var menuId = new ID64(TestMenuID);

                var aoc1v0 = new TestApplication.AddonParameter(menuId, 0, 1, 4 * 1024);
                var aoc2v0 = new TestApplication.AddonParameter(menuId, 0, 2, 4 * 1024);

                var request = new List<TestApplication.GenerateParameter<int>>()
                {
                    demoApp2,
                    aoc1v0,
                    aoc2v0,
                };
                var contents = TestApplication.MakeContents(intermediate, request);
                var demo2   = contents[0];
                var Aoc1V0  = contents[1];
                var Aoc2V0  = contents[2];

                // テスト用のメニューアプリをターゲット上へインストール
                InstallMenuNsp(executor);

                // テスト用のアプリや Aoc をローカルインストールしてテスト環境を整備
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    string.Format("application install \"{0}\"", demo2.NspPath)));

                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                    string.Format("addoncontent install \"{0}\"", Aoc1V0.NspPath) + " AND " +
                    string.Format("addoncontent install \"{0}\"", Aoc2V0.NspPath)));

                // 念のためテスト完了後にクリーンアップ処理を実施しておく
                scope.AddDisposer(() => { CleanupTarget(executor); });

                // QuestフラグをONにする
                ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem("debug enable-rid-mode"));
                // テスト終了時にOFFに戻しておく
                scope.AddDisposer(() => { executor.RunDevMenuCommandSystem("debug disable-rid-mode"); });

                // テスト実行前のターゲット上のアプリを確認
                {
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        "application list"));
                    ExpectEvaluator.IsTrue(executor.OutputStream.Standard.ToString().Contains(
                        "2 application(s) found."));
                    ExpectEvaluator.IsTrue(executor.OutputStream.Standard.ToString().Contains(
                        string.Format("0x{0:x16}", menuId.Value)));
                    ExpectEvaluator.IsTrue(executor.OutputStream.Standard.ToString().Contains(
                        string.Format("0x{0:x16}", demo2Id.Value)));
                }
                // テスト実行前の追加コンテンツを確認
                {
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        "addoncontent list"));
                    ExpectEvaluator.IsTrue(executor.OutputStream.Standard.ToString().Contains(
                        "2 add-on-content(s) found."));
                    ExpectEvaluator.IsTrue(executor.OutputStream.Standard.ToString().Contains(
                        string.Format("0x{0:x16}      1", menuId.Value)));
                    ExpectEvaluator.IsTrue(executor.OutputStream.Standard.ToString().Contains(
                        string.Format("0x{0:x16}      2", menuId.Value)));
                }

                var options = FailureOptionString;
                var menuIdString = string.Format("0x{0:x16}", TestMenuID);
                var targetVersion = GetTargetVersion();

                // アプリを実行してアプリケーションと追加コンテンツのアップデート処理を実行
                // 本テスト用の固定処理を実行する
                // ・アプリ(demo2) 0x01001a500005e013 を削除
                // ・アプリ(demo1) 0x01001a500005e012 を新たにダウンロード
                // ・追加コンテンツ index=1 をアップデートダウンロード
                // ・追加コンテンツ index=2 を削除
                // ・追加コンテンツ index=10 を新たにダウンロード
                executor.OutputStream.Standard.Clear();
                var runResult = executor.RunOnTarget(menuIdString, options, "--application-update");
                if (targetVersion == TestVersion.v3)
                {
                    // v3 では失敗することを想定
                    ExpectEvaluator.IsFalse(runResult);
                    ExpectEvaluator.IsTrue(executor.OutputStream.Standard.ToString().Contains(
                        "RunCommandApplicationUpdate() : State = Failed"));
                }
                else
                {
                    // v3 以降のバージョンでは成功することを確認
                    ExpectEvaluator.IsTrue(runResult);
                    ExpectEvaluator.IsTrue(executor.OutputStream.Standard.ToString().Contains(
                        "RunCommandApplicationUpdate() : State = Completed"));
                }

                // 0x01001a500005e012 のデモアプリが落ちてくるはず
                var demoId = new ID64(0x01001a500005e012);

                // テスト実行後のターゲット上のアプリを確認
                {
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        "application list"));
                    ExpectEvaluator.IsTrue(executor.OutputStream.Standard.ToString().Contains(
                        "2 application(s) found."));
                    ExpectEvaluator.IsTrue(executor.OutputStream.Standard.ToString().Contains(
                        string.Format("0x{0:x16}", menuId.Value)));
                    ExpectEvaluator.IsTrue(executor.OutputStream.Standard.ToString().Contains(
                        string.Format("0x{0:x16}", demoId.Value)));
                }
                // テスト実行後の追加コンテンツを確認 (v3 ではAocのダウンロードが失敗するため確認しない)
                if (targetVersion != TestVersion.v3)
                {
                    ExpectEvaluator.IsTrue(executor.RunDevMenuCommandSystem(
                        "addoncontent list"));
                    ExpectEvaluator.IsTrue(executor.OutputStream.Standard.ToString().Contains(
                        "2 add-on-content(s) found."));
                    ExpectEvaluator.IsTrue(executor.OutputStream.Standard.ToString().Contains(
                        string.Format("0x{0:x16}      1", menuId.Value)));
                    ExpectEvaluator.IsTrue(executor.OutputStream.Standard.ToString().Contains(
                        string.Format("0x{0:x16}     10", menuId.Value)));
                }
            }
        }
    }
}
