﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nn/nn_Log.h>

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>

#include <nn/rid.h>

#include "RidCompatibilityVerifier_Option.h"
#include "RidCompatibilityVerifier_RunCommand.h"
#include "RidCompatibilityVerifier_SystemUpdateTest.h"
#include "RidCompatibilityVerifier_MenuUpdateTest.h"
#include "RidCompatibilityVerifier_ApplicationUpdateTest.h"

void RunCommandPrepare()
{
    NN_LOG("Start RunCommandPrepare()\n");

    // SDカードチェック
    auto result = nn::rid::CheckSdCard();
    NNT_EXPECT_RESULT_SUCCESS(result);

    // コンテンツの破損チェック
    result = nn::rid::CheckContentsIntegrity();
    NNT_EXPECT_RESULT_SUCCESS(result);

    // Mii 削除
    result = nn::rid::DeleteAllMii();
    NNT_EXPECT_RESULT_SUCCESS(result);

    // アルバム削除
    result = nn::rid::DeleteAllAlbums();
    NNT_EXPECT_RESULT_SUCCESS(result);
}

void RunCommandDisplay()
{
    NN_LOG("Start RunCommandDisplay()\n");

    // ひとまず Startup() を呼び出すのみ
    auto result = nn::rid::StartUp();
    NNT_EXPECT_RESULT_SUCCESS(result);
}

void RunCommandSystemUpdate()
{
    NN_LOG("Start RunCommandSystemUpdate()\n");

    RidTestSystemUpdater systemUpdaterTest;
    systemUpdaterTest.Start();

    // テストが終了状態となるまで待つ
    while (NN_STATIC_CONDITION(true))
    {
        if (systemUpdaterTest.IsUpdateProcessDone() == true)
        {
            break;
        }
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(500));
    }

    NNT_EXPECT_RESULT_SUCCESS(systemUpdaterTest.GetLastResult());
    // テストの結果(状態)を表示する
    NN_LOG("Run RunCommandSystemUpdate() : State = %s\n", systemUpdaterTest.GetState().c_str());
}

void RunCommandMenuUpdate()
{
    NN_LOG("Start RunCommandMenuUpdate()\n");

    RidTestMenuUpdater menuUpdaterTest;
    menuUpdaterTest.Start();

    // テストが終了状態となるまで待つ
    while (NN_STATIC_CONDITION(true))
    {
        if (menuUpdaterTest.IsUpdateProcessDone() == true)
        {
            break;
        }
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(500));
    }

    NNT_EXPECT_RESULT_SUCCESS(menuUpdaterTest.GetLastResult());
    // テストの結果(状態)を表示する
    NN_LOG("Run RunCommandMenuUpdate() : State = %s\n", menuUpdaterTest.GetState().c_str());
}

void RunCommandApplicationUpdate()
{
    NN_LOG("Start RunCommandApplicationUpdate()\n");

    RidTestApplicationUpdater appUpdaterTest;
    appUpdaterTest.Start();

    // テストが終了状態となるまで待つ
    while (NN_STATIC_CONDITION(true))
    {
        if (appUpdaterTest.IsUpdateProcessDone() == true)
        {
            break;
        }
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(500));
    }

    NNT_EXPECT_RESULT_SUCCESS(appUpdaterTest.GetLastResult());
    // テストの結果(状態)を表示する
    NN_LOG("Run RunCommandApplicationUpdate() : State = %s\n", appUpdaterTest.GetState().c_str());
}

struct CommandPair
{
    std::string commandStr;
    void (*function)();
} CommanddPairList[] = {
    { "--prepare" , RunCommandPrepare },
    { "--display" , RunCommandDisplay },
    { "--system-update" , RunCommandSystemUpdate },
    { "--menu-update" , RunCommandMenuUpdate },
    { "--application-update" , RunCommandApplicationUpdate },
};

TEST(RidCompatibilityVerifier, RunTest)
{
    auto firstArg = Option::GetFirstCommandString();

    for (auto& command : CommanddPairList)
    {
        if (firstArg == command.commandStr)
        {
            command.function();
        }
    }
}
