﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>

#include <nn/psc.h>

namespace nnt { namespace psm {

//! USB 電源関連の状態を表す列挙型
enum class UsbPowerState
{
    Inactive,
    He15V,
    He13V,
    Lt13V,

    Dcp1500Ma,
    Cdp1500Ma,
    Sdp500Ma,

    Apple500Ma,
    Apple1000Ma,
    Apple2000Ma,

    StartOutput,
    ActiveOutput,
    EndOutput,

    ErrorOverVoltage,
    ErrorCradleUsbHubUndetected,
};

//! VDD50 の状態を表す列挙型
enum Vdd50Condition
{
    Vdd50Condition_None,
    Vdd50Condition_Vdd50A,
    Vdd50Condition_Vdd50B,
};

//! PSM ドライバライブラリ、PSM ライブラリ、テスト用の変数等を初期化します。
void Initialize() NN_NOEXCEPT;

//! PSM ドライバライブラリ、PSM ライブラリを終了します。
void Finalize() NN_NOEXCEPT;

//! VDD50 の状態が指定した状態であるか確認します。内部で EXPECT 文を実行します。
void CheckVdd50Condition(Vdd50Condition vdd50Condition) NN_NOEXCEPT;

//! PSM ドライバライブラリの持つ電池温度を変更します。
void ChangeBatteryTemperature(double temperature) NN_NOEXCEPT;

//! PSM ドライバライブラリの持つ電池電圧を変更します。
void ChangeBatteryVoltageMilliVolt(int batteryVoltage) NN_NOEXCEPT;

//! PSM ドライバライブラリの持つ電池残量値を変更します。
void ChangeBatteryChargePercentage(double batteryChargePercentage) NN_NOEXCEPT;

//! PSM ドライバライブラリの持つ平均充電電流を変更します。
void ChangeAverageCurrentMilliAmpere(int averageCurrentMilliAmpere) NN_NOEXCEPT;

//! PSM ドライバライブラリの持つ電圧ベース電池残量値を変更します。
void ChangeVoltageFuelGaugePercentage(double voltageFuelGaugePercentage) NN_NOEXCEPT;

//! PSM ライブラリを使用して給電エミュレーションの設定を変更します。
void ChangeEnoughPowerChargeEmulation(bool enabled) NN_NOEXCEPT;

//! PSM ドライバライブラリの持つ USB の電源関連情報を変更します。
void ChangeUsbPowerState(UsbPowerState usbPowerState) NN_NOEXCEPT;

//! PSM ドライバライブラリの持つシステム電源状態を変更します。
void ChangePmState(::nn::psc::PmState pmState) NN_NOEXCEPT;

//! PSM ドライバライブラリの持つシステム電源状態をステートマシンの定義に沿って遷移させます。
void TransitPmState(::nn::psc::PmState pmState) NN_NOEXCEPT;

//! PSM ドライバライブラリの持つシステム電源状態をステートマシンの定義に沿って遷移させます。stopOnActiveState == true の場合 PSM ドライバライブラリがアクティブになるシステム電源状態に入った時に停止します。
void TransitPmState(::nn::psc::PmState pmState, bool stopOnActiveState) NN_NOEXCEPT;

}} // namespace nnt::psm
