﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>

#include <nn/psc/psc_Types.h>
#include <nn/psm/psm_SystemApi.h>

#include <nnt/nntest.h>

#include "testPsm_Common.h"

namespace nnt { namespace psm {

namespace {

const uint16_t AcceptableDifferenceOfSocRep = 1;

struct BatteryChargePercentageConversion
{
    int convertedPercentage;
    uint16_t minSocRep;
    uint16_t maxSocRep;
};

const double SocRepMultiplier = 0.00390625;

double ConvertSocRepToBatteryChargePercentage(uint16_t socRep) NN_NOEXCEPT
{
    return static_cast<double>(socRep) * SocRepMultiplier;
}

const BatteryChargePercentageConversion BatteryChargePercentageConversionList[] =
{
    { 1, 0x0000, 0x03ee },
    { 2, 0x03ef, 0x04e4 },
    { 3, 0x04e5, 0x05db },
    { 4, 0x05dc, 0x06d1 },

    // 実行時間短縮のために中間値は省略します。
    /*
    { 5, 0x06d2, 0x07c7 },
    { 6, 0x07c8, 0x08bd },
    { 7, 0x08be, 0x09b3 },
    { 8, 0x09b4, 0x0aa9 },
    { 9, 0x0aaa, 0x0b9f },
    { 10, 0x0ba0, 0x0c96 },
    { 11, 0x0c97, 0x0d8c },
    { 12, 0x0d8d, 0x0e82 },
    { 13, 0x0e83, 0x0f78 },
    { 14, 0x0f79, 0x106e },
    { 15, 0x106f, 0x1164 },
    { 16, 0x1165, 0x125b },
    { 17, 0x125c, 0x1351 },
    { 18, 0x1352, 0x1447 },
    { 19, 0x1448, 0x153d },
    { 20, 0x153e, 0x1633 },
    { 21, 0x1634, 0x1729 },
    { 22, 0x172a, 0x181f },
    { 23, 0x1820, 0x1916 },
    { 24, 0x1917, 0x1a0c },
    { 25, 0x1a0d, 0x1b02 },
    { 26, 0x1b03, 0x1bf8 },
    { 27, 0x1bf9, 0x1cee },
    { 28, 0x1cef, 0x1de4 },
    { 29, 0x1de5, 0x1edb },
    { 30, 0x1edc, 0x1fd1 },
    { 31, 0x1fd2, 0x20c7 },
    { 32, 0x20c8, 0x21bd },
    { 33, 0x21be, 0x22b3 },
    { 34, 0x22b4, 0x23a9 },
    { 35, 0x23aa, 0x249f },
    { 36, 0x24a0, 0x2596 },
    { 37, 0x2597, 0x268c },
    { 38, 0x268d, 0x2782 },
    { 39, 0x2783, 0x2878 },
    { 40, 0x2879, 0x296e },
    { 41, 0x296f, 0x2a64 },
    { 42, 0x2a65, 0x2b5b },
    { 43, 0x2b5c, 0x2c51 },
    { 44, 0x2c52, 0x2d47 },
    { 45, 0x2d48, 0x2e3d },
    { 46, 0x2e3e, 0x2f33 },
    { 47, 0x2f34, 0x3029 },
    { 48, 0x302a, 0x311f },
    { 49, 0x3120, 0x3216 },
    { 50, 0x3217, 0x330c },
    { 51, 0x330d, 0x3402 },
    { 52, 0x3403, 0x34f8 },
    { 53, 0x34f9, 0x35ee },
    { 54, 0x35ef, 0x36e4 },
    { 55, 0x36e5, 0x37db },
    { 56, 0x37dc, 0x38d1 },
    { 57, 0x38d2, 0x39c7 },
    { 58, 0x39c8, 0x3abd },
    { 59, 0x3abe, 0x3bb3 },
    { 60, 0x3bb4, 0x3ca9 },
    { 61, 0x3caa, 0x3d9f },
    { 62, 0x3da0, 0x3e96 },
    { 63, 0x3e97, 0x3f8c },
    { 64, 0x3f8d, 0x4082 },
    { 65, 0x4083, 0x4178 },
    { 66, 0x4179, 0x426e },
    { 67, 0x426f, 0x4364 },
    { 68, 0x4365, 0x445b },
    { 69, 0x445c, 0x4551 },
    { 70, 0x4552, 0x4647 },
    { 71, 0x4648, 0x473d },
    { 72, 0x473e, 0x4833 },
    { 73, 0x4834, 0x4929 },
    { 74, 0x492a, 0x4a1f },
    { 75, 0x4a20, 0x4b16 },
    { 76, 0x4b17, 0x4c0c },
    { 77, 0x4c0d, 0x4d02 },
    { 78, 0x4d03, 0x4df8 },
    { 79, 0x4df9, 0x4eee },
    { 80, 0x4eef, 0x4fe4 },
    { 81, 0x4fe5, 0x50db },
    { 82, 0x50dc, 0x51d1 },
    { 83, 0x51d2, 0x52c7 },
    { 84, 0x52c8, 0x53bd },
    { 85, 0x53be, 0x54b3 },
    { 86, 0x54b4, 0x55a9 },
    { 87, 0x55aa, 0x569f },
    { 88, 0x56a0, 0x5796 },
    { 89, 0x5797, 0x588c },
    { 90, 0x588d, 0x5982 },
    { 91, 0x5983, 0x5a78 },
    { 92, 0x5a79, 0x5b6e },
    { 93, 0x5b6f, 0x5c64 },
    { 94, 0x5c65, 0x5d5b },
    { 95, 0x5d5c, 0x5e51 },
    { 96, 0x5e52, 0x5f47 },
    */

    { 97, 0x5f48, 0x603d },
    { 98, 0x603e, 0x6133 },
    { 99, 0x6134, 0x6229 },
    { 100, 0x622a, 0x6400 },
};

struct PsmBatteryChargePercentagePolicyTestSetting
{
    double baseRawPercentage;
};

const PsmBatteryChargePercentagePolicyTestSetting PsmBatteryChargePercentagePolicyTestSettings[] =
{
    { 11.0 },
    { 16.0 },
    { 0.0 },
    { 3.0 },
};

class PsmBatteryChargePercentagePolicyTest : public ::testing::TestWithParam<PsmBatteryChargePercentagePolicyTestSetting>
{
public:
    static void SetUpTestCase()
    {
        ::nnt::psm::Initialize();
    }

    static void TearDownTestCase()
    {
        ::nnt::psm::Finalize();
    }
};

INSTANTIATE_TEST_CASE_P(PsmBatteryChargePercentagePolicyTestName, PsmBatteryChargePercentagePolicyTest, ::testing::ValuesIn(PsmBatteryChargePercentagePolicyTestSettings));

} // namespace

TEST_F(PsmBatteryChargePercentagePolicyTest, DefaultOffset)
{
    TransitPmState(::nn::psc::PmState_FullAwake);

    // 電池電圧、電池残量値生値を更新してライブラリ内部に持つ電池残量値生値のオフセットを更新します。
    ChangeBatteryChargePercentage(3.0);
    ChangeBatteryVoltageMilliVolt(3199);
    ChangeBatteryVoltageMilliVolt(3200);

    for ( const BatteryChargePercentageConversion pair : BatteryChargePercentageConversionList )
    {
        auto minRawPercentage = ConvertSocRepToBatteryChargePercentage(pair.minSocRep);
        ChangeBatteryChargePercentage(ConvertSocRepToBatteryChargePercentage(pair.minSocRep + AcceptableDifferenceOfSocRep));
        NN_LOG("MinRaw: %d.%04d %% -> %d %%\n", static_cast<int>(minRawPercentage), static_cast<int>(minRawPercentage * 10000) % 10000, ::nn::psm::GetBatteryChargePercentage());
        EXPECT_EQ(pair.convertedPercentage, ::nn::psm::GetBatteryChargePercentage());

        auto maxRawPercentage = ConvertSocRepToBatteryChargePercentage(pair.maxSocRep);
        ChangeBatteryChargePercentage(ConvertSocRepToBatteryChargePercentage(pair.maxSocRep - AcceptableDifferenceOfSocRep));
        NN_LOG("MaxRaw: %d.%04d %% -> %d %%\n", static_cast<int>(maxRawPercentage), static_cast<int>(maxRawPercentage * 10000) % 10000, ::nn::psm::GetBatteryChargePercentage());
        EXPECT_EQ(pair.convertedPercentage, ::nn::psm::GetBatteryChargePercentage());
    }

    ChangeBatteryChargePercentage(0.0);
}

TEST_P(PsmBatteryChargePercentagePolicyTest, CalibratedOffsetLower)
{
    TransitPmState(::nn::psc::PmState_FullAwake);

    // 電池電圧、電池残量値生値を更新してライブラリ内部に持つ電池残量値生値のオフセットを更新します。
    ChangeBatteryChargePercentage(GetParam().baseRawPercentage); // 11 %(表示値約 8 %) まで補正が可能です。
    ChangeBatteryVoltageMilliVolt(3199);
    ChangeBatteryVoltageMilliVolt(3200);

    if ( GetParam().baseRawPercentage <= 11.0 )
    {
        // スレッショルド以下ならば初期値 1 を保証します。
        EXPECT_EQ(1, ::nn::psm::GetBatteryChargePercentage());
    }

    int previousPercentage = ::nn::psm::GetBatteryChargePercentage();

    NN_LOG("Raw: %d %% -> %d %%\n", static_cast<int>(GetParam().baseRawPercentage), ::nn::psm::GetBatteryChargePercentage());

    for ( double offsetRawPercentage = 1.0; offsetRawPercentage <= 2.0; offsetRawPercentage += 1.0 )
    {
        ChangeBatteryChargePercentage(GetParam().baseRawPercentage + offsetRawPercentage);
        NN_LOG("Raw: %d %% -> %d %%\n", static_cast<int>(GetParam().baseRawPercentage + offsetRawPercentage), ::nn::psm::GetBatteryChargePercentage());

        if ( ::nn::psm::GetBatteryChargePercentage() == 1 )
        {
            EXPECT_EQ(0, ::nn::psm::GetBatteryChargePercentage() - previousPercentage);
        }
        else
        {
            EXPECT_LE(1, ::nn::psm::GetBatteryChargePercentage() - previousPercentage);
            EXPECT_GE(2, ::nn::psm::GetBatteryChargePercentage() - previousPercentage);
        }

        previousPercentage = ::nn::psm::GetBatteryChargePercentage();
    }

    ChangeBatteryChargePercentage(0.0);
}

TEST_P(PsmBatteryChargePercentagePolicyTest, CalibratedOffsetHigher)
{
    TransitPmState(::nn::psc::PmState_FullAwake);

    // 電池電圧、電池残量値生値を更新してライブラリ内部に持つ電池残量値生値のオフセットを更新します。
    ChangeBatteryChargePercentage(GetParam().baseRawPercentage); // 11 %(表示値約 8 %) まで補正が可能
    ChangeBatteryVoltageMilliVolt(3199);
    ChangeBatteryVoltageMilliVolt(3200);

    if ( GetParam().baseRawPercentage <= 11.0 )
    {
        // スレッショルド以下ならば初期値 1 を保証します。
        EXPECT_EQ(1, ::nn::psm::GetBatteryChargePercentage());
    }

    const double BaseRawPercentage = 98.0;

    ChangeBatteryChargePercentage(BaseRawPercentage);

    int previousPercentage = ::nn::psm::GetBatteryChargePercentage();

    NN_LOG("Raw: %d %% -> %d %%\n", static_cast<int>(BaseRawPercentage), ::nn::psm::GetBatteryChargePercentage());

    for ( double offsetRawPercentage = 1.0; offsetRawPercentage <= 2.0; offsetRawPercentage += 1.0 )
    {
        ChangeBatteryChargePercentage(BaseRawPercentage + offsetRawPercentage);
        NN_LOG("Raw: %d %% -> %d %%\n", static_cast<int>(BaseRawPercentage + offsetRawPercentage), ::nn::psm::GetBatteryChargePercentage());

        if ( previousPercentage == 100 )
        {
            EXPECT_EQ(0, ::nn::psm::GetBatteryChargePercentage() - previousPercentage);
        }
        else
        {
            EXPECT_LE(1, ::nn::psm::GetBatteryChargePercentage() - previousPercentage);
            EXPECT_GE(2, ::nn::psm::GetBatteryChargePercentage() - previousPercentage);
        }

        previousPercentage = ::nn::psm::GetBatteryChargePercentage();
    }

    ChangeBatteryChargePercentage(0.0);
}

TEST_F(PsmBatteryChargePercentagePolicyTest, CalibratedOffsetDecrease)
{
    TransitPmState(::nn::psc::PmState_FullAwake);

    ChangeBatteryChargePercentage(11.0); // 11 % で一度補正をします。
    ChangeBatteryVoltageMilliVolt(3199);
    ChangeBatteryVoltageMilliVolt(3200);

    const double BaseRawPercentage = 4.0;

    ChangeBatteryChargePercentage(4.0); // 11 % 未満に下げて電圧は変えずにカウントアップするか確認します。

    int previousPercentage = ::nn::psm::GetBatteryChargePercentage();

    NN_LOG("Raw: %d %% -> %d %%\n", static_cast<int>(BaseRawPercentage), ::nn::psm::GetBatteryChargePercentage());

    for ( double offsetRawPercentage = 1.0; offsetRawPercentage <= 2.0; offsetRawPercentage += 1.0 )
    {
        ChangeBatteryChargePercentage(BaseRawPercentage + offsetRawPercentage);
        NN_LOG("Raw: %d %% -> %d %%\n", static_cast<int>(BaseRawPercentage + offsetRawPercentage), ::nn::psm::GetBatteryChargePercentage());

        EXPECT_LE(1, ::nn::psm::GetBatteryChargePercentage() - previousPercentage); // 生値のオフセットが更新されるので最低でも 1% 上昇します。
        EXPECT_GE(2, ::nn::psm::GetBatteryChargePercentage() - previousPercentage);

        previousPercentage = ::nn::psm::GetBatteryChargePercentage();
    }

    ChangeBatteryChargePercentage(0.0);
}

}} // namespace nnt::psm
