﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/fs.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

#include <nn/prepo.h>
#include <nn/prepo/prepo_ApiAdmin.h>
#include <nn/prepo/prepo_ApiConfig.h>
#include <nn/prepo/prepo_ApiDebug.h>
#include <nn/prepo/prepo_SystemPlayReport.h>

#include "../Common/testPrepo_Common.h"

namespace
{
    nn::Result SendPlayReport(int count)
    {
        for (auto i = 0; i < count; i++)
        {
            const char eventId[] = "test_prepo_statistics";
            nn::prepo::SystemPlayReport report(eventId);

            nn::ApplicationId appId = { 0x0100000000003006 };
            NN_RESULT_DO(report.SetApplicationId(appId));

            char buffer[nn::prepo::SystemPlayReport::BufferSizeMin];
            report.SetBuffer(buffer, sizeof (buffer));

            NN_RESULT_DO(report.Save());
        }

        EXPECT_TRUE(nnt::prepo::WaitForTransmissionStatusChanged(nn::prepo::TransmissionStatus_Processing,
            nn::TimeSpan::FromSeconds(20)));

        EXPECT_TRUE(nnt::prepo::WaitForTransmissionStatusChanged(nn::prepo::TransmissionStatus_Idle,
            nn::TimeSpan::FromSeconds(30)));

        NN_RESULT_SUCCESS;
    }
}

TEST(PlayReportStatistics, Initialize)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountHost("prepo-stats", "C:\\siglo\\prepo-stats"));

    NNT_ASSERT_RESULT_SUCCESS(nn::prepo::SetUserAgreementCheckEnabled(false));
}

TEST(PlayReportStatistics, Main)
{
    const char FilePath[] = "prepo-stats:/statistics";

    if (nnt::prepo::FileExists(FilePath))
    {
        // 再起動後

        nn::prepo::Statistics saved;
        NNT_ASSERT_RESULT_SUCCESS(nnt::prepo::ReadFile(&saved, FilePath));

        nn::prepo::Statistics current;
        nn::prepo::GetStatistics(&current);

        for (int i = 0; i < nn::prepo::ReportCategory_Count; i++)
        {
            const auto category = static_cast<nn::prepo::ReportCategory>(i);

            EXPECT_GE(current.groups[category].uploaded.size, saved.groups[category].uploaded.size);
            EXPECT_GE(current.groups[category].uploaded.count, saved.groups[category].uploaded.count);
            EXPECT_GE(current.groups[category].lostByBufferShortage.size, saved.groups[category].lostByBufferShortage.size);
            EXPECT_GE(current.groups[category].lostByBufferShortage.count, saved.groups[category].lostByBufferShortage.count);
            EXPECT_GE(current.groups[category].lostByStorageShortage.size, saved.groups[category].lostByStorageShortage.size);
            EXPECT_GE(current.groups[category].lostByStorageShortage.count, saved.groups[category].lostByStorageShortage.count);
            EXPECT_GE(current.groups[category].lostByDisagree.size, saved.groups[category].lostByDisagree.size);
            EXPECT_GE(current.groups[category].lostByDisagree.count, saved.groups[category].lostByDisagree.count);
        }
    }
    else
    {
        // 再起動前

        NNT_ASSERT_RESULT_SUCCESS(nn::prepo::ClearStatistics());

        nn::prepo::Statistics before;
        nn::prepo::GetStatistics(&before);

        for (int i = 0; i < nn::prepo::ReportCategory_Count; i++)
        {
            const auto category = static_cast<nn::prepo::ReportCategory>(i);

            EXPECT_EQ(0, before.groups[category].uploaded.size);
            EXPECT_EQ(0, before.groups[category].uploaded.count);
            EXPECT_EQ(0, before.groups[category].lostByBufferShortage.size);
            EXPECT_EQ(0, before.groups[category].lostByBufferShortage.count);
            EXPECT_EQ(0, before.groups[category].lostByStorageShortage.size);
            EXPECT_EQ(0, before.groups[category].lostByStorageShortage.count);
            EXPECT_EQ(0, before.groups[category].lostByDisagree.size);
            EXPECT_EQ(0, before.groups[category].lostByDisagree.count);
        }

        const int reportCount = 100;

        NNT_ASSERT_RESULT_SUCCESS(SendPlayReport(reportCount));

        nn::prepo::Statistics after;
        nn::prepo::GetStatistics(&after);

        // 1 件もロスせずに送信されることを期待している。

        for (int i = 0; i < nn::prepo::ReportCategory_Count; i++)
        {
            const auto category = static_cast<nn::prepo::ReportCategory>(i);

            if (category == nn::prepo::ReportCategory_Normal)
            {
                EXPECT_GT(after.groups[category].uploaded.size, before.groups[category].uploaded.size);
                EXPECT_GE(after.groups[category].uploaded.count, before.groups[category].uploaded.count + reportCount);
            }
            else
            {
                EXPECT_EQ(after.groups[category].uploaded.size, before.groups[category].uploaded.size);
                EXPECT_EQ(after.groups[category].uploaded.count, before.groups[category].uploaded.count);
            }

            EXPECT_EQ(after.groups[category].lostByBufferShortage.size, before.groups[category].lostByBufferShortage.size);
            EXPECT_EQ(after.groups[category].lostByBufferShortage.count, before.groups[category].lostByBufferShortage.count);
            EXPECT_EQ(after.groups[category].lostByStorageShortage.size, before.groups[category].lostByStorageShortage.size);
            EXPECT_EQ(after.groups[category].lostByStorageShortage.count, before.groups[category].lostByStorageShortage.count);
            EXPECT_EQ(after.groups[category].lostByDisagree.size, before.groups[category].lostByDisagree.size);
            EXPECT_EQ(after.groups[category].lostByDisagree.count, before.groups[category].lostByDisagree.count);
        }
    }
}

TEST(PlayReportStatistics, Finalize)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::prepo::SetUserAgreementCheckEnabled(true));
}
