﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/fs.h>
#include <nn/prepo.h>
#include <nn/prepo/prepo_Result.h>
#include <nn/prepo/detail/prepo_ApiDetail.h>
#include <nn/prepo/detail/prepo_PlayReportGenerator.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>
#include <cstdlib>
#include <nn/time.h>
#include <nn/account.h>
#include <nn/account/account_ApiForSystemServices.h>
#include "../Common/testPrepo_Common.h"

TEST(PlayReportHostPc, Initialize)
{
    nn::account::InitializeForSystemService();
    nn::time::Initialize();

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountHost("prepo-dump", "C:\\siglo\\prepo-dump"));
}

template <size_t BufferSize> class DebugPlayReport
{
public:
    //
    nn::Bit8 buffer[BufferSize];
    //
    size_t position;

public:
    DebugPlayReport() NN_NOEXCEPT
    {
        Clear();
    }

    void Clear() NN_NOEXCEPT
    {
        nn::prepo::detail::PlayReportGenerator::Initialize(&position, buffer, BufferSize);
    }

    nn::Result Add(const char* key, int64_t value) NN_NOEXCEPT
    {
        return nn::prepo::detail::PlayReportGenerator::AddKeyValue(&position, key, value, buffer, BufferSize, position);
    }

    nn::Result Add(const char* key, const nn::prepo::Any64BitId& value) NN_NOEXCEPT
    {
        return nn::prepo::detail::PlayReportGenerator::AddKeyValue(&position, key, value, buffer, BufferSize, position);
    }

    nn::Result Add(const char* key, double value) NN_NOEXCEPT
    {
        return nn::prepo::detail::PlayReportGenerator::AddKeyValue(&position, key, value, buffer, BufferSize, position);
    }

    nn::Result Add(const char* key, const char* value) NN_NOEXCEPT
    {
        return nn::prepo::detail::PlayReportGenerator::AddKeyValue(&position, key, value, buffer, BufferSize, position);
    }
};

TEST(PlayReportHostPc, Save)
{
    DebugPlayReport<2048> report;

    NNT_EXPECT_RESULT_SUCCESS(report.Add("integer", static_cast<int64_t>(123)));
    EXPECT_TRUE(nn::prepo::detail::VerifyReport(report.buffer, report.position));

    NNT_EXPECT_RESULT_SUCCESS(report.Add("double", 1.23));
    EXPECT_TRUE(nn::prepo::detail::VerifyReport(report.buffer, report.position));

    NNT_EXPECT_RESULT_SUCCESS(report.Add("string", "123"));
    EXPECT_TRUE(nn::prepo::detail::VerifyReport(report.buffer, report.position));

    nn::prepo::Any64BitId id = {123};
    NNT_EXPECT_RESULT_SUCCESS(report.Add("id", id));
    EXPECT_TRUE(nn::prepo::detail::VerifyReport(report.buffer, report.position));

    nn::fs::DeleteFile("prepo-dump:/report.msgpack");

    NN_ABORT_UNLESS_RESULT_SUCCESS(nnt::prepo::WriteFile("prepo-dump:/report.msgpack", report.buffer, report.position));
}
