﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <functional>
#include <unistd.h>
#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/nn_TimeSpan.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/ae.h>
#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/teamcity/testTeamcity_Logger.h>
#include <nnt/nnt_Argument.h>

#include "TestFunctions.h"

#ifdef NNT_POWERSTATE_TEST_SWTICH_LOG_TO_UART
extern "C" void nndiagStartup()
{
}
#endif

namespace {

    struct SleepWakeSequenceTestParam
    {
        // Preparation func to be called before entring sleep.
        // Argument gives incrementing loop index starting from 0.
        std::function<void(int)>    funcBeforeSleep;

        // Check func to be called after wake-up.
        // Argument gives incrementing loop index starting from 0.
        std::function<void(int)>    funcAfterWake;

        // Number of loops to run. Set 1 unless you have strong reason with your test.
        int                         loopCountMax;
    };

    // ------------------------------------------------------------------------
    // Specific test parameters

    const SleepWakeSequenceTestParam SleepWakeSequenceTestParamNxSet[] =
    {
        // Plain loop test
        { nullptr, nullptr, 3 },

        // BPC test
        { nnt::BpcTestFuncBeforeSleep, nnt::BpcTestFuncAfterWake, 1},

        // Add your test here
    };

    //
    // ------------------------------------------------------------------------

    nn::os::SystemEventType g_SystemAppletEvent;

    NN_ALIGNAS(4096) char g_MallocBuffer[0x200000];
    extern "C" void nninitStartup()
    {
        nn::os::SetMemoryHeapSize( 0x200000 );
        nn::init::InitializeAllocator(g_MallocBuffer, sizeof(g_MallocBuffer));
    }

    bool CheckAndProcessMessage(nn::os::SystemEventType* pEvent, nn::ae::Message expectMessage, nn::TimeSpan timeout = 0) NN_NOEXCEPT
    {
        nn::ae::Message message;
        if (timeout == 0)
        {
            message = nn::ae::WaitForNotificationMessage(pEvent);
        }
        else
        {
            if (!nn::os::TimedWaitSystemEvent(pEvent, timeout))
            {
                NN_LOG("Timed out expecting message=0x%08x\n", expectMessage);
                return false;
            }
            message = nn::ae::GetNotificationMessage();
        }

        if (message != expectMessage)
        {
            NN_LOG("Received unexpected expect=0x%08x message=0x%08x", expectMessage, message);
            return false;
        }

        switch (message)
        {
            case nn::ae::Message_ChangeIntoForeground:
                    NN_LOG("Received Message_ChangeIntoForeground\n");
                    nn::ae::AcquireForegroundRights();
                    NN_LOG("Done AcquireForegroundRights()\n");
                    break;

            case nn::ae::Message_ChangeIntoBackground:
                    NN_LOG("Received Message_ChangeIntoBackground\n");
                    nn::ae::ReleaseForegroundRights();
                    NN_LOG("Done ReleaseForegroundRights()\n");
                    break;

            case nn::ae::Message_Exit:
                    NN_LOG("Received Message_Exit\n");
                    break;

            case nn::ae::Message_ApplicationExited:
                    NN_LOG("Received Message_ApplicationExited\n");
                    break;

            case nn::ae::Message_FinishedSleepSequence:
                    NN_LOG("Received Message_FinishedSleepSequence\n");
                    break;

            case nn::ae::Message_DetectShortPressingHomeButton:
                    NN_LOG("Received Message_DetectShortPressingHomeButton\n");
                    break;

            case nn::ae::Message_DetectLongPressingHomeButton:
                    NN_LOG("Received Message_DetectLongPressingHomeButton\n");
                    break;

            case nn::ae::Message_DetectShortPressingPowerButton:
                    NN_LOG("Received Message_DetectShortPressingPowerButton\n");
                    break;

            case nn::ae::Message_DetectMiddlePressingPowerButton:
                    NN_LOG("Received Message_DetectMiddlePressingPowerButton\n");
                    break;

            case nn::ae::Message_DetectLongPressingPowerButton:
                    NN_LOG("Received Message_DetectLongPressingPowerButton\n");
                    break;

            default:
                    NN_ABORT("Received unknown message= 0x%08x", message);
                    break;
        }

        return true;
    }

    bool LoopSleepWakeSequence(nn::os::SystemEventType& event, const SleepWakeSequenceTestParam& param)
    {
        if (param.loopCountMax > 0)
        {
            NN_LOG("Loops %d times\n", param.loopCountMax);
        }
        else
        {
            NN_LOG("Loops forever\n");
        }

        // TODO: Call Auto-wake debug API

        for (int loopCount = 0; param.loopCountMax <= 0 || loopCount < param.loopCountMax; ++loopCount)
        {
            NN_LOG("\n** Loop #%d **\n", loopCount);

            if (param.funcBeforeSleep)
            {
                param.funcBeforeSleep(loopCount);
                NN_LOG("Called funcBeforeSleep.\n");
                if (::testing::Test::HasFatalFailure())
                {
                    NN_LOG("Test failed in funcBeforeSleep(). Aborting.\n");
                    return false;
                }
            }

            // 通常の SA と同じように LockFG する
            nn::ae::LockForeground();
            NN_LOG("Called nn::ae::LockForeground().\n");

            nn::ae::StartSleepSequence();
            NN_LOG("Called nn::ae::StartSleepSequence().\n");

            // Sleep からの復帰メッセージを待機
            EXPECT_TRUE(CheckAndProcessMessage(&event, nn::ae::Message_FinishedSleepSequence, nn::TimeSpan::FromSeconds(5)));
            NN_LOG("Processed nn::ae::Message_FinishedSleepSequence.\n");

            // スリープから起床したので UnlockFG しておく
            nn::ae::UnlockForeground();
            NN_LOG("Called nn::ae::UnlockForeground().\n");

            if (param.funcAfterWake)
            {
                param.funcAfterWake(loopCount);
                NN_LOG("Called funcAfterWake.\n");
                if (::testing::Test::HasFatalFailure())
                {
                    NN_LOG("Test failed in funcAfterWake(). Aborting.\n");
                    return false;
                }
            }
        }

        return true;
    }
} // anonymous namespace

#ifdef NNT_POWERSTATE_TEST_AS_GOOGLE_TEST

class SleepWakeSequenceTest : public ::testing::TestWithParam<SleepWakeSequenceTestParam>
{};

TEST_P(SleepWakeSequenceTest, LoopSleepWakeSequence)
{
    EXPECT_TRUE(LoopSleepWakeSequence(g_SystemAppletEvent, GetParam()));
}

INSTANTIATE_TEST_CASE_P(
    SleepWakeSequenceTestNx,
    SleepWakeSequenceTest,
    ::testing::ValuesIn(SleepWakeSequenceTestParamNxSet)
);

// Google test global Set-Up and Tear-Down feature
class TestEnvironment : public ::testing::Environment {
public:
    virtual void SetUp()
    {
        nn::ae::InitializeNotificationMessageEvent(&g_SystemAppletEvent);
        NN_LOG("Initialization done.\n");

        // SA 起動直後は FG 遷移要求がある
        NN_ABORT_UNLESS(CheckAndProcessMessage(&g_SystemAppletEvent, nn::ae::Message_ChangeIntoForeground));
    }

    virtual void TearDown()
    {
    }
} g_TestEnvironment;

void SystemAppletMenuMain(nn::ae::SystemAppletParameters* param)
{
    int argc = ::nnt::GetHostArgc();
    char** argv = ::nnt::GetHostArgv();

    NN_LOG("Running nnMain() from testMain_Main.cpp\n");
    ::testing::InitGoogleTest(&argc, argv);

    ::testing::TestEventListeners& listeners =
        ::testing::UnitTest::GetInstance()->listeners();
    ::testing::TestEventListener* defaultResultPrinter =
        listeners.Release(listeners.default_result_printer());
#if defined(NN_BUILD_CONFIG_HARDWARE_BDSLIMX6) || defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK1) || defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || defined(NN_BUILD_CONFIG_HARDWARE_NX)
    listeners.Append(new ::nnt::teamcity::ServiceMessageLogger());
#endif
    listeners.Append(defaultResultPrinter);

    // *Customized from default main* : Added Global SetUp and TearDown
    ::testing::AddGlobalTestEnvironment(&g_TestEnvironment);

    int result = RUN_ALL_TESTS();
    ::nnt::Exit(result);
}

#else

bool ParseProgramOption(int* pOutloopCountMax)
{
    int argc = nnt::GetHostArgc();
    char** argv = nnt::GetHostArgv();
    int option;

    *pOutloopCountMax = 1; // Default value

    while ((option = getopt(argc, argv, "c:fh")) != -1)
    {
        switch(option)
        {
        case 'c':
            *pOutloopCountMax = std::atoi(optarg);
            break;
        case 'f':
            *pOutloopCountMax = 0;
            break;
        case 'h':
            return false;
        default:
            NN_LOG("Error: detected unknown option:[%c].\n", option);
            return false;
        }
    }
    return true;
}

void SystemAppletMenuMain(nn::ae::SystemAppletParameters* param)
{
    int totalLoopCountMax = 1;
    if (!ParseProgramOption(&totalLoopCountMax))
    {
        NN_LOG("Options:\n"
            "-c [loop count] : Number of loops. Default is 1. Setting 0 or negative value makes test run forever.\n"
            "-f              : Loop forever.\n");
        NN_ABORT("Aborting test.");
    }

    NN_LOG("SystemAppletMenuMain: Launched Sleep Wake Sequence Test running as a system applet.\n");

    nn::ae::InitializeNotificationMessageEvent(&g_SystemAppletEvent);
    NN_LOG("Initialization done.\n");

    // SA 起動直後は FG 遷移要求がある
    NN_ABORT_UNLESS(CheckAndProcessMessage(&g_SystemAppletEvent, nn::ae::Message_ChangeIntoForeground));

    for (int i = 0; i < totalLoopCountMax; ++i)
    {
        NN_LOG("\n\n=================================\n");
        NN_LOG("= Whole set loop %d\n", i);
        NN_LOG("=================================\n\n");

        for (auto& param : SleepWakeSequenceTestParamNxSet )
        {
            NN_ABORT_UNLESS(LoopSleepWakeSequence(g_SystemAppletEvent, param));
        }
    }

    NN_LOG("\n\n [DONE] Exiting Sleep Wake Sequence Test.\n");
}

#endif

extern "C" void nnMain()
{
    nn::ae::InvokeSystemAppletMain(nn::ae::AppletId_SystemAppletMenu, SystemAppletMenuMain);
}
